/*
 * wbiff.c, a part of the W Window System
 *
 * Copyright (C) 1994-1999 by Torsten Scherer, Benjamin Lorenz,
 * Jan Paul Schmidt and Eero Tamminen
 *
 * This package is free software; you can redistribute it and/or modify it
 * under the terms specified in the docs/COPYRIGHTS file coming with this
 * package.
 *
 * -- shows when there's new mail
 *
 * CHANGES
 * Benjamin Lorenz 9/94:
 * - bitmap support
 * TeSche 12/94:
 * - changed for W1R1
 * jps 12/97:
 * - added geometry and help option
 * - long option support
 * - support for wlaunch swallow option
 * - wbiff now takes a command via -c or --command, which
 *   is invoked when clicking on its window
 * ++eero, 4/99:
 * - fix width bug
 * - add debug option
 * - add monitor option to set what file to watch
 * - use w_putblock() instead of w_plot(). for that I had to pad
 *   every image line with two additional bytes to long-align it...
 * - alloc mailbox path instead of using char[64]...
 * ++eero, 5/99:
 * - add (slow) blinking when when there's mail
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <pwd.h>
#include <signal.h>
#include <sys/stat.h>
#include <getopt.h>
#include <Wlib.h>


#define	MAILBOXPATH	"/var/spool/mail/%s"
#define CHECK_INTERVAL	10	/* seconds */


static struct option long_options [] = {
  { "command", required_argument, NULL, 'c' },
  { "geometry", required_argument, NULL, 'g' },
  { "monitor", required_argument, NULL, 'm' },
  { "parent", required_argument, NULL, 'p'},
  { "debug", no_argument, NULL, 'd' },
  { "help", no_argument, NULL, 'r'},
  { NULL, 0, NULL, 0}
};


/* image size (not that the bitmap data width is 64 because it has
 * to be long (32-bit) aligned).
 */
#define BM_W 48
#define BM_H 48

static uchar mailfull_data[] = {
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x03,0xff,0xff,0xff,0xe0,0x00,
 0x00,0x03,0xff,0xff,0xff,0xf8,0x20,0x00,0x00,0x02,0x00,0x00,0x00,0x0b,0xa0,
 0x00,0x00,0x02,0x00,0x00,0x00,0xea,0xbe,0x00,0x00,0x02,0x78,0x00,0x00,0xaa,
 0xbe,0x00,0x00,0x02,0x00,0x00,0x00,0xab,0xbe,0x00,0x00,0x06,0x40,0x00,0x00,
 0xe8,0x3e,0x00,0x00,0x0e,0x00,0x7e,0x00,0x08,0x3e,0x00,0x00,0x1e,0x00,0x00,
 0x00,0x08,0x7e,0x00,0x00,0x3e,0x00,0x78,0x00,0x08,0xfe,0x00,0x00,0x7e,0x00,
 0x00,0x00,0x09,0xfe,0x00,0x00,0x7e,0x00,0x7c,0xe0,0x0b,0xfe,0x00,0x00,0x7e,
 0x00,0x00,0x00,0x0f,0xfe,0x00,0x00,0x7e,0x00,0x00,0x00,0x0f,0xf6,0x00,0x00,
 0x7e,0x00,0x00,0x00,0x0f,0xe6,0x00,0x00,0x7f,0xff,0xff,0xff,0xff,0xc6,0x00,
 0x00,0x7d,0x55,0x55,0x55,0x57,0x86,0x00,0x00,0x7f,0xff,0xff,0xff,0xff,0x06,
 0x00,0x00,0x7f,0xff,0xff,0xff,0xfe,0x06,0x00,0x00,0x60,0x60,0x00,0x00,0x06,
 0x06,0x00,0x00,0x60,0x60,0x00,0x00,0x06,0x06,0x00,0x00,0x60,0x60,0x00,0x00,
 0x06,0x06,0x00,0x00,0x60,0x60,0x00,0x00,0x06,0x06,0x00,0x00,0x60,0x60,0x00,
 0x00,0x06,0x06,0x00,0x00,0x60,0xff,0xff,0xff,0xff,0xfe,0x00,0x00,0x61,0xff,
 0xff,0xff,0xff,0xfe,0x00,0x00,0x63,0xf5,0x55,0x55,0x57,0x5e,0x00,0x00,0x67,
 0xea,0xaa,0xaa,0xae,0xbe,0x00,0x00,0x6f,0xf5,0x55,0x55,0x57,0x7e,0x00,0x00,
 0x7f,0xea,0xaa,0xaa,0xae,0xfe,0x00,0x00,0x7f,0xf5,0x55,0x55,0x57,0xfe,0x00,
 0x00,0x7f,0xff,0xff,0xff,0xff,0xfe,0x00,0x00,0x7f,0xff,0xff,0xff,0xff,0xfe,
 0x00,0x00,0x7f,0xd5,0x55,0x55,0x57,0xfe,0x00,0x00,0x7f,0xaa,0xaa,0xaa,0xaf,
 0xf6,0x00,0x00,0x7f,0x55,0x55,0x55,0x57,0xe6,0x00,0x00,0x7e,0xaa,0xaa,0xaa,
 0xaf,0xc6,0x00,0x00,0x7d,0x55,0x55,0x55,0x57,0x86,0x00,0x00,0x7f,0xff,0xff,
 0xff,0xff,0x06,0x00,0x00,0x7f,0xff,0xff,0xff,0xfe,0x06,0x00,0x00,0x60,0x60,
 0x00,0x00,0x06,0x06,0x00,0x00,0x60,0x00,0x00,0x00,0x06,0x00,0x00,0x00,0x60,
 0x77,0x39,0xec,0x06,0x00,0x00,0x00,0x60,0x7f,0x6c,0xcc,0x06,0x00,0x00,0x00,
 0x60,0x6b,0x7c,0xcc,0x06,0x00,0x00,0x00,0x60,0x63,0x6d,0xef,0x06,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00};

static uchar mailempty_data[] = {
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0xff,0xff,0xff,0xff,0xfe,0x00,0x00,0x01,0xff,0xff,0xff,0xff,
 0xfe,0x00,0x00,0x03,0xf5,0x55,0x55,0x55,0x5e,0x00,0x00,0x07,0xea,0xaa,0xaa,
 0xaa,0xbe,0x00,0x00,0x0f,0xf5,0x55,0x55,0x55,0x7e,0x00,0x00,0x1f,0xea,0xaa,
 0xaa,0xaa,0xfe,0x00,0x00,0x3f,0xf5,0x55,0x55,0x55,0xfe,0x00,0x00,0x7f,0xff,
 0xff,0xff,0xff,0xfe,0x00,0x00,0x7f,0xff,0xff,0xff,0xff,0xfe,0x00,0x00,0x7f,
 0xd5,0x55,0x55,0x57,0xfe,0x00,0x00,0x7f,0xaa,0xaa,0xaa,0xaf,0xf6,0x00,0x00,
 0x7f,0x55,0x55,0x55,0x57,0xe6,0x00,0x00,0x7e,0xaa,0xaa,0xaa,0xaf,0xc6,0x00,
 0x00,0x7d,0x55,0x55,0x55,0x57,0x86,0x00,0x00,0x7f,0xff,0xff,0xff,0xff,0x06,
 0x00,0x00,0x7f,0xff,0xff,0xff,0xfe,0x06,0x00,0x00,0x60,0x60,0x00,0x00,0x06,
 0x06,0x00,0x00,0x60,0x60,0x00,0x00,0x06,0x06,0x00,0x00,0x60,0x60,0x00,0x00,
 0x06,0x06,0x00,0x00,0x60,0x60,0x00,0x00,0x06,0x06,0x00,0x00,0x60,0x60,0x00,
 0x00,0x06,0x06,0x00,0x00,0x60,0x7f,0xff,0xff,0xff,0xfe,0x00,0x00,0x60,0xff,
 0xff,0xff,0xff,0xfe,0x00,0x00,0x61,0xf5,0x55,0x55,0x57,0x5e,0x00,0x00,0x63,
 0xea,0xaa,0xaa,0xae,0xbe,0x00,0x00,0x67,0xf5,0x55,0x55,0x57,0x7e,0x00,0x00,
 0x6f,0xea,0xaa,0xaa,0xae,0xfe,0x00,0x00,0x7f,0xf5,0x55,0x55,0x57,0xfe,0x00,
 0x00,0x7f,0xff,0xff,0xff,0xff,0xfe,0x00,0x00,0x7f,0xff,0xff,0xff,0xff,0xfe,
 0x00,0x00,0x7f,0xd5,0x55,0x55,0x57,0xfe,0x00,0x00,0x7f,0xaa,0xaa,0xaa,0xaf,
 0xf6,0x00,0x00,0x7f,0x55,0x55,0x55,0x57,0xe6,0x00,0x00,0x7e,0xaa,0xaa,0xaa,
 0xaf,0xc6,0x00,0x00,0x7d,0x55,0x55,0x55,0x57,0x86,0x00,0x00,0x7f,0xff,0xff,
 0xff,0xff,0x06,0x00,0x00,0x7f,0xff,0xff,0xff,0xfe,0x06,0x00,0x00,0x60,0x60,
 0x00,0x00,0x06,0x00,0x00,0x00,0x60,0x00,0x00,0x00,0x06,0x00,0x00,0x00,0x60,
 0x00,0x00,0x00,0x06,0x00,0x00,0x00,0x60,0x00,0x00,0x00,0x06,0x00,0x00,0x00,
 0x60,0x00,0x00,0x00,0x06,0x00,0x00,0x00,0x60,0x00,0x00,0x00,0x06,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00};

/* note that image width is less than data width */
static BITMAP mailfull = {
  BM_W, BM_H, BM_PACKEDMONO, 4, (BM_W+31) >> 5, 1, mailfull_data
};

static BITMAP mailempty = {
  BM_W, BM_H, BM_PACKEDMONO, 4, (BM_W+31) >> 5, 1, mailempty_data
};


static void usage (char *name) {
  fprintf (stderr, "\
Usage: %s [OPTION]...\n\
The `Am I important?' program.\n\
\n\
  -c, --command              command executed when clicking on wbiff\n\
  -d, --debug                set Wlib debugging on\n\
  -g, --geometry             set size and position\n\
  -m, --monitor              set which file you want to monitor\n\
  -h, --help                 display this help and exit\n\
\n\
", name);
}



int main (int argc, char *argv []) {
  struct passwd *pw;
  char *mboxpath;
  struct stat st;
  short is_shown;
  WEVENT *ev;
  WWIN *win;
  ushort win_flags = EV_MOUSE;
  ulong parentid = 0;
  short x = UNDEF;
  short y = UNDEF;
  short w = mailfull.width;
  short h = mailfull.height;
  short w_off, h_off;
  short show_usage = 0;
  char *command = NULL;
  int c;

  /* just in case we've started in the background */
  signal(SIGTTOU, SIG_IGN);

  if (!(pw = getpwuid (getuid ()))) {
    fprintf (stderr, "%s: can't determine your login name\n", argv[0]);
    return 1;
  }
  if (!(mboxpath = malloc(strlen(MAILBOXPATH) + strlen(pw->pw_name)))) {
    fprintf (stderr, "%s: malloc failed\n", argv[0]);
    w_exit ();
    return 1;
  }
  sprintf(mboxpath, MAILBOXPATH, pw->pw_name);


  while ((c = getopt_long (argc, argv, "c:dg:hm:p:", long_options, NULL)) != -1) {
    switch (c) {

      case 'c':
        command = optarg;
        win_flags = EV_MOUSE;
        break;

      case 'd':
	w_trace(1);
        break;

      case 'g':
        scan_geometry (optarg, &w, &h, &x, &y);
        break;

      case 'm':
	free(mboxpath);
        mboxpath = optarg;
        break;

      case 'p':
        sscanf (optarg, "%lx", &parentid);
        break;

      default:
        show_usage = 1;
        break;
    }
  }

  if (show_usage) {
    usage (argv[0]);
    return 0;
  }

  if (!w_init ()) {
    fprintf (stderr, "%s: can't connect to wserver\n", argv[0]);
    return 1;
  }

  if (parentid) {
    WWIN *parent = w_winFromID(parentid);
    w = parent->width;
    h = parent->height;
    x = y = 0;
    if (!(win = w_createChild (parent, w, h, W_NOBORDER | win_flags))) {
      fprintf (stderr, "%s: can't create child window\n", argv[0]);
      w_exit ();
      return 1;
    }
  } else {
    if (w < mailfull.width) {
      w = mailfull.width;
    }
    if (h < mailfull.height) {
      h = mailfull.height;
    }
    if (!(win = w_create(w, h, W_MOVE | win_flags))) {
      fprintf (stderr, "%s: can't create window\n", argv[0]);
      w_exit ();
      return 1;
    }
  }

  if (w_open (win, x, y)) {
    fprintf (stderr, "%s: can't open window\n", argv[0]);
    w_exit ();
    return 1;
  }
  
  is_shown = 2;

  w_off = (w - mailfull.width) / 2;
  h_off = (h - mailfull.height) / 2;
  while (42) {

    if (stat(mboxpath, &st)) {

      fprintf(stderr, "unable to stat '%s'\n", mboxpath);
      w_exit();
      return 1;

    } else {

      if (st.st_mtime > st.st_atime) {

	if (is_shown != 1) {
	  is_shown = 1;
	  w_putblock(&mailfull, win, w_off, h_off);
	} else {
	  /* reverse image */
	  w_pbox(win, 0, 0, win->width, win->height);
	}

      } else {

	if (is_shown != 0) {
	  is_shown = 0;
	  w_putblock(&mailempty, win, w_off, h_off);
	}
      }
    }

    /* watch events with 10 sec. timeout */
    if ((ev = w_queryevent(NULL, NULL, NULL, CHECK_INTERVAL*1000))) {
      switch (ev->type) {

        case EVENT_GADGET:
          if (ev->key == GADGET_EXIT) {
            w_exit ();
            return 0;
          }
          break;

        case EVENT_MPRESS:
          if (command) {
/* This simplifies everything a bit instead of a fork plus exec combination. */
            system (command);
          }
          break;
      }
    }
  }
}
