#ifndef TXLIB_H
#define TXLIB_H
//
//                     TxWin, Textmode Windowing Library
//
//   Original code Copyright (c) 1995-2005 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
// This file contains Original Code and/or Modifications of Original Code as
// defined in and that are subject to the GNU Lesser General Public License.
// You may not use this file except in compliance with the License.
// BY USING THIS FILE YOU AGREE TO ALL TERMS AND CONDITIONS OF THE LICENSE.
// A copy of the License is provided with the Original Code and Modifications,
// and is also available at http://www.dfsee.com/txwin/lgpl.htm
//
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as published
// by the Free Software Foundation; either version 2.1 of the License,
// or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public License
// along with this library; (lgpl.htm) if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
//
// Questions on TxWin licensing can be directed to: txwin@fsys.nl
//
// ==========================================================================
//
//
// TX library interfaces
//
// Author: J. van Wijk
//
// Developed for LPT/DFSee utilities
//

#include <stdarg.h>                             // variable vfprintf
#include <stdio.h>                              // std/file i/o
#include <conio.h>                              // console  i/o
#include <stdlib.h>                             // std C stuff
#include <stddef.h>                             // threadid and errno
#include <string.h>                             // fast strings
#include <ctype.h>                              // type tests & conversions
#include <malloc.h>                             // memory allocation
#include <process.h>                            // threading etc
#include <limits.h>                             // number limits
#include <time.h>                               // time functions
#include <errno.h>                              // UNIX like error handling



//+++++++++++++++ OEM and version specific definitions

#if defined    (__WATCOMC__)                    // Auto TX target macro defs
   #if defined      (__NT__)
      #define WIN32
   #elif defined   (__OS2__)
      #define DEV32
   #elif defined   (__DOS__)
      #define DOS32
   #elif defined (__LINUX__)
      #define LINUX
   #endif
#endif

#if defined (TXOEM) || defined (TXMIN)
   #undef  USEWINDOWING
   #undef  HAVEMOUSE
   #undef  HAVETHREADS
#else
   #if defined   (LINUX)
      #undef  HAVEMOUSE
   #else
      #define HAVEMOUSE
   #endif
   #define USEWINDOWING
   #if defined (DEV32) || defined (WIN32)
      #define HAVETHREADS
   #else
      #undef  HAVETHREADS
   #endif
#endif



#if defined (LINUX)
   #include <sys/types.h>
   #include <sys/ioctl.h>
   #include <sys/stat.h>
   #include <fcntl.h>
   #include <unistd.h>
#else
   #include <io.h>                              // special  i/o
   #include <direct.h>                          // directory functions
#endif

#define TX_ERROR                     200        // Generic TX error
#define TX_INVALID_FILE              202        // File cannot be opened
#define TX_INVALID_PATH              203        // Path not found
#define TX_ACCESS_DENIED             205        // File access denied (locked)
#define TX_INVALID_HANDLE            206        // Invalid handle specified
#define TX_INVALID_DATA              207        // Invalid data (structure)
#define TX_ALLOC_ERROR               208        // Memory allocation error
#define TX_INVALID_DRIVE             215        // Drive (letter) invalid
#define TX_PENDING                   217        // Function pending
#define TX_FAILED                    218        // Function failed
#define TX_WRITE_PROTECT             219        // Write protected (readonly)
#define TX_CMD_UNKNOWN               222        // cmd not known, unhandled
#define TX_NO_COMPRESS               223        // No compression achieved
#define TX_NO_INITIALIZE             224        // No compress init
#define TX_ABORTED                   225        // Function aborted
#define TX_BAD_OPTION_CHAR           226        // option char out of range
#define TX_TOO_MANY_ARGS             227        // too many arguments for cmd
#define TX_DISPLAY_CHANGE            228        // display mode changed

#ifndef NOTRACE                                 // by default, include tracing
  #define DUMP 1
#endif

#if   defined (WIN32)
   #define WIN32_LEAN_AND_MEAN 1
   #include <windows.h>
   #if defined (_DEBUG)
      #define DUMP 1
   #endif
#elif defined (DOS32)                           // 32-bit extended DOS (WATCOM)
   #include <dos.h>
   #include <i86.h>
#elif defined (LINUX)                           // LINUX native        (WATCOM)
#else
   #ifndef DEV32
     #define DEV32
   #endif
   #define INCL_BASE
   #define INCL_NOPMAPI                         // avoid the real PM defs
   #define INCL_LONGLONG                        // Use 64bit compiler support
   #define INCL_WINSHELLDATA
   #define INCL_DOSDEVIOCTL                     // OS2 device definitions
   #include <os2.h>
   #define INCL_REXXSAA
   #include <rexxsaa.h>                         // REXX API's
#endif

#if defined (TXFTRC)                            // TXlib low impact Flow Trace
   #define TXFLOW(cond,nr,hex,str)                                         \
   if (cond)                                                               \
   {                                                                       \
      fprintf(stderr, "\n%12.12s %5u: ", __FILE__, __LINE__);              \
      fprintf(stderr, "%9lu %8.8lX %s", (ULONG) nr, (ULONG) hex, str);     \
   }
#else
   #define TXFLOW(cond,nr,hex,str)
#endif


#if   defined (WIN32)
   #define  TXFMODE                ""
   #define  USHORT                 unsigned short int
   #define  TXINVALID              ((ULONG) -1)
   #define  TXCURTHREAD            ((ULONG)(*(ULONG  *) _threadid))
   #define  TXHTHREAD              HANDLE
   #define  TXHFILE                HANDLE
   #define  TxRead(hf,b,z,w)       (ULONG) !ReadFile(hf,b,(z),w,NULL)
   #define  TxWrite(hf,b,z,w)      (ULONG) !WriteFile(hf,b,(z),w,NULL)
   #define  TxSetSize(hf,s)        0, SetFilePointer(hf,s,NULL,FILE_BEGIN); \
                                   SetEndOfFile(hf)
   #define  TxClose(hf)            (ULONG) !CloseHandle(hf)
   #define  TxDeleteFile(f)        remove(f)
   #define  TxSleep(msec)          Sleep(msec)
   #define  TxAlloc(num,size)      calloc((size_t) (num), (size_t) (size))
   #define  TxFreeMem(ptr)         if (ptr != NULL) {free(ptr); ptr = NULL;}
   #define  TxThreadPrioMin()      SetThreadPriority(GetCurrentThread(), \
                                      THREAD_PRIORITY_LOWEST)
   #define  TxThreadPrioLow()      SetThreadPriority(GetCurrentThread(), \
                                      THREAD_PRIORITY_BELOW_NORMAL)
   #define  TxThreadPrioStd()      SetThreadPriority(GetCurrentThread(), \
                                      THREAD_PRIORITY_NORMAL)
   #define  TxThreadPrioHigh()     SetThreadPriority(GetCurrentThread(), \
                                      THREAD_PRIORITY_ABOVE_NORMAL)
   #define  TxThreadPrioMax()      SetThreadPriority(GetCurrentThread(), \
                                      THREAD_PRIORITY_HIGHEST)
   #define  TxBeginThread(f,s,p)   (TXHTHREAD) _beginthread(f,s,p)
   #define  TxKillThread(ht)       TerminateThread(ht, 0)
   #define  FATTR_ARCHIVED         FILE_ATTRIBUTE_ARCHIVE
   #define  FATTR_SYSTEM           FILE_ATTRIBUTE_SYSTEM
   #define  FATTR_HIDDEN           FILE_ATTRIBUTE_HIDDEN
   #define  FATTR_READONLY         FILE_ATTRIBUTE_READONLY
   #define  FATTR_DIRECTORY        FILE_ATTRIBUTE_DIRECTORY
   #define  FATTR_LABEL            0x0008
   #define  FS_WILDCARD            "*"
   #define  FS_DIR_CMD             "dir"
   #define  FS_PATH_STR            "\\"
   #define  FS_PATH_SEP            '\\'
   #define  FS_ENVP_SEP            ';'
#elif defined (DOS32)
   #define  TXFMODE                ""
   #define  TXWORD                 regs.w
   #define  TXIWRD                 ireg.w
   #define  TXOWRD                 oreg.w
   typedef  unsigned char          BYTE;        // b
   typedef  unsigned long          BOOL;        // f
   typedef  unsigned short         USHORT;      // us
   typedef  unsigned long          ULONG;       // ul
   typedef  long                   LONG;        // l
   typedef  void                  *PVOID;       // p
   #define  LOBYTE(w)              ((BYTE)(          w        & 0xff))
   #define  HIBYTE(w)              ((BYTE)(((USHORT)(w) >> 8) & 0xff))
   #define  NO_ERROR               0
   #define  TRUE                   1
   #define  FALSE                  0
   #define  TXCURTHREAD            ((ULONG) 1)
   #define  TXHTHREAD              ULONG
   #define  TXHFILE                FILE *
   #define  TxRead(hf,b,z,w)       (ULONG) !(*w = fread(b,1,(z),hf))
   #define  TxWrite(hf,b,z,w)      (ULONG) !(*w = fwrite(b,1,(z),hf))
   #define  TxSetSize(hf,s)        0, fseek( hf, (long) (s), SEEK_SET); \
                                   chsize(fileno(hf),(long) (s))
   #define  TxClose(hf)            fclose(hf)
   #define  TxDeleteFile(f)        remove(f)
   #define  TxSleep(msec)          TxBusyWait(msec)
   #define  TxAlloc(num,size)      calloc((long) (num), (size_t) (size))
   #define  TxFreeMem(ptr)         if (ptr != NULL) {free(ptr); ptr = NULL;}
   #define  TxThreadPrioMin()
   #define  TxThreadPrioLow()
   #define  TxThreadPrioStd()
   #define  TxThreadPrioHigh()
   #define  TxThreadPrioMax()
   #define  TxBeginThread(f,s,p)   f(p)
   #define  TxKillThread(ht)
   #define  FATTR_ARCHIVED         _A_ARCH
   #define  FATTR_SYSTEM           _A_SYSTEM
   #define  FATTR_HIDDEN           _A_HIDDEN
   #define  FATTR_READONLY         _A_RDONLY
   #define  FATTR_DIRECTORY        _A_SUBDIR
   #define  FATTR_LABEL            _A_VOLID
   #define  FS_WILDCARD            "*.*"
   #define  FS_DIR_CMD             "dir"
   #define  FS_PATH_STR            "\\"
   #define  FS_PATH_SEP            '\\'
   #define  FS_ENVP_SEP            ';'
   #define  ERROR_FILE_NOT_FOUND    2
   #define  ERROR_PATH_NOT_FOUND    3
   #define  ERROR_ACCESS_DENIED     5
   #define  ERROR_INVALID_DRIVE    15
   #define  ERROR_NO_MORE_FILES    18
   #define  ERROR_WRITE_PROTECT    19
   #define  ERROR_NOT_SUPPORTED    50
   #define  ERROR_FILE_EXISTS      80
#elif defined (LINUX)                           // LINUX native        (WATCOM)
   #define  TXFMODE                "b"
   typedef  unsigned char          BYTE;        // b
   typedef  unsigned long          BOOL;        // f
   typedef  unsigned short         USHORT;      // us
   typedef  unsigned long          ULONG;       // ul
   typedef  long                   LONG;        // l
   typedef  void                  *PVOID;       // p
   #define  LOBYTE(w)              ((BYTE)(          w        & 0xff))
   #define  HIBYTE(w)              ((BYTE)(((USHORT)(w) >> 8) & 0xff))
   #define  NO_ERROR               0
   #define  TRUE                   1
   #define  FALSE                  0
   #define  TXCURTHREAD            ((ULONG) 1)
   #define  TXHTHREAD              ULONG
   #define  TXHFILE                int
   #define  TxRead(hf,b,z,w)       (ULONG) !(*w = read(hf,b,(z)))
   #define  TxWrite(hf,b,z,w)      (ULONG) !(*w = write(hf,b,(z)))
   #define  TxSetSize(hf,s)        0, lseek( hf, (off_t) (s), SEEK_SET); \
                                   chsize((hf),(long) (s))
   #define  TxClose(hf)            close(hf)
   #define  TxDeleteFile(f)        remove(f)
   #define  TxSleep(msec)          TxSleepMsec(msec)
   #define  TxAlloc(num,size)      calloc((long) (num), (size_t) (size))
   #define  TxFreeMem(ptr)         if (ptr != NULL) {free(ptr); ptr = NULL;}
   #define  TxThreadPrioMin()
   #define  TxThreadPrioLow()
   #define  TxThreadPrioStd()
   #define  TxThreadPrioHigh()
   #define  TxThreadPrioMax()
   #define  TxBeginThread(f,s,p)   f(p)
   #define  TxKillThread(ht)
   #define  FATTR_READONLY         0x01
   #define  FATTR_HIDDEN           0x02
   #define  FATTR_SYSTEM           0x04
   #define  FATTR_LABEL            0x08
   #define  FATTR_DIRECTORY        0x10
   #define  FATTR_ARCHIVED         0x20
   #define  FS_WILDCARD            "*"
   #define  FS_DIR_CMD             "ls -l"
   #define  FS_PATH_STR            "/"
   #define  FS_PATH_SEP            '/'
   #define  FS_ENVP_SEP            ':'
   #define  ERROR_FILE_NOT_FOUND           2
   #define  ERROR_PATH_NOT_FOUND           3
   #define  ERROR_TOO_MANY_OPEN_FILES      4    // OUT_OF_HANDLES
   #define  ERROR_ACCESS_DENIED            5
   #define  ERROR_INVALID_HANDLE           6    // INVALID_HANDLE
   #define  ERROR_INVALID_DRIVE           15
   #define  ERROR_NO_MORE_FILES           18
   #define  ERROR_WRITE_PROTECT           19
   #define  ERROR_NOT_READY               21    // DRIVE_NOT_READY
   #define  ERROR_CRC                     23    // DATA_ERROR
   #define  ERROR_SEEK                    25    // SEEK_ERROR
   #define  ERROR_SECTOR_NOT_FOUND        27    // MSG%SECTOR_NOT_FOUND
   #define  ERROR_WRITE_FAULT             29    // WRITE_FAULT
   #define  ERROR_READ_FAULT              30    // READ_FAULT
   #define  ERROR_GEN_FAILURE             31    // GENERAL_FAILURE
   #define  ERROR_SHARING_VIOLATION       32    // SHARING_VIOLATION
   #define  ERROR_LOCK_VIOLATION          33    // LOCK_VIOLATION
   #define  ERROR_WRONG_DISK              34    // INVALID_DISK_CHANGE
   #define  ERROR_NOT_SUPPORTED           50
   #define  ERROR_FILE_EXISTS             80
   #define  ERROR_INVALID_PARAMETER       87    // NET_INCORR_PARAMETER
   #define  ERROR_DEVICE_IN_USE           99    // DEVICE_IN_USE
   #define  ERROR_DISK_CHANGE            107    // DISK_CHANGE
   #define  ERROR_DRIVE_LOCKED           108    // DRIVE_LOCKED
   #define  ERROR_BROKEN_PIPE            109    // BROKEN_PIPE
   #define  ERROR_OPEN_FAILED            110    // ERROR_OPEN_FAILED
   #define  ERROR_DISK_FULL              112    // DISK_FULL
   #define  ERROR_NEGATIVE_SEEK          131    // APPL_DOUBLEFRAMECHAR
   #define  ERROR_SEEK_ON_DEVICE         132    // APPL_ARROWCHAR
   #define  ERROR_DISCARDED              157    // DISCARDED
   #define  ERROR_FILENAME_EXCED_RANGE   206    // NAME_TOO_LONG
   #define  ERROR_INVALID_PATH           253    // INVALID_PATH
#else
   #define  TXFMODE                ""
   #define  TXCURTHREAD         ((ULONG)(*(ULONG  *) _threadid))
   #define  TxAlloc(num,size)   calloc((size_t) (num), (size_t) (size))
   #define  TxFreeMem(ptr)      if (ptr != NULL) {free(ptr); ptr = NULL;}
   #define  TXHTHREAD              TID
   #define  TXHFILE                HFILE
   #define  TxRead(hf,b,z,w)       DosRead(hf,b,(z),w)
   #define  TxWrite(hf,b,z,w)      DosWrite(hf,b,(z),w)
   #define  TxSetSize(hf,s)        DosNewSize(hf,s)
   #define  TxClose(hf)            DosClose(hf)
   #define  TxDeleteFile(f)        remove(f)
   #define  TxSleep(msec)          DosSleep(msec)
   #define  TxThreadPrioMin()      DosSetPrty( PRTYS_THREAD, \
                                               PRTYC_IDLETIME, PRTYD_MAXIMUM, 0)
   #define  TxThreadPrioLow()      DosSetPrty( PRTYS_THREAD, \
                                               PRTYC_REGULAR,  PRTYD_MINIMUM, 0)
   #define  TxThreadPrioStd()      DosSetPrty( PRTYS_THREAD, \
                                               PRTYC_REGULAR,  PRTYD_MAXIMUM, 0)
   #define  TxThreadPrioHigh()     DosSetPrty( PRTYS_THREAD, \
                                       PRTYC_FOREGROUNDSERVER, PRTYD_MINIMUM, 0)
   #define  TxThreadPrioMax()      DosSetPrty( PRTYS_THREAD, \
                                       PRTYC_FOREGROUNDSERVER, PRTYD_MAXIMUM, 0)
   #define  TxBeginThread(f,s,p)   _beginthread(f, NULL, s, p)
   #define  TxKillThread(ht)       DosKillThread(ht)
   #define  FATTR_ARCHIVED         FILE_ARCHIVED
   #define  FATTR_SYSTEM           FILE_SYSTEM
   #define  FATTR_HIDDEN           FILE_HIDDEN
   #define  FATTR_READONLY         FILE_READONLY
   #define  FATTR_DIRECTORY        FILE_DIRECTORY
   #define  FATTR_LABEL            0x0008
   #define  FS_WILDCARD            "*"
   #define  FS_DIR_CMD             "dir"
   #define  FS_PATH_STR            "\\"
   #define  FS_PATH_SEP            '\\'
   #define  FS_ENVP_SEP            ';'
#endif


typedef ULONG          TXHANDLE;

#ifndef SECTORSIZE
#define SECTORSIZE  512
#define SECTORBITS  4096
#endif

// Make sure a value is between a MIN and a MAX value
#define clamp(low, x, high) ((x) < (low) ? (low) : ((x) > (high) ? (high) : (x)))

//- equivalent to M$ 64-bit VCN (and ussually storage compatible with LLONG)
typedef struct xlong                            // eXtreme LONG value (64 bits)
{
   ULONG               lo;                      // low  32 bits
   ULONG               hi;                      // high 32 bits
} XLONG;                                        // end of struct "xlong"

// macro's to convert sector count to hi/lo part of XLONG/VCN (64 bits)
#define TXXLONG_LO(x)    ((ULONG)( (x)    <<  9) )
#define TXXLONG_HI(x)    ((ULONG)( (x)    >> 23) )
#define TXXLONG2SECT(x)  ((ULONG)(((x).lo >>  9) | ((x).hi << 23)))

//- note: LLONG type is equivalent to OS/2 LONGLONG and __i64 types
typedef long long       LLONG;

// macro's to convert between sector count and 64 bits LLONG
#define TXSECT2LLONG(x)  ((LLONG)(((LLONG) (x)) <<  9))
#define TXLLONG2SECT(x)  ((ULONG)(((LLONG) (x)) >>  9))


//- create multiples of 2/4/8/16/32/64/128/256/512 for any given number
#define TXMULTI02(n)    (((n) + 0x001) & ~0x001)
#define TXMULTI04(n)    (((n) + 0x003) & ~0x003)
#define TXMULTI08(n)    (((n) + 0x007) & ~0x007)
#define TXMULTI16(n)    (((n) + 0x00f) & ~0x00f)
#define TXMULTI32(n)    (((n) + 0x01f) & ~0x01f)
#define TXMULTI64(n)    (((n) + 0x03f) & ~0x03f)
#define TXMULT128(n)    (((n) + 0x07f) & ~0x07f)
#define TXMULT256(n)    (((n) + 0x0ff) & ~0x0ff)
#define TXMULT512(n)    (((n) + 0x1ff) & ~0x1ff)


//- Definitions to make a fixed-size bitfield-structure type of 2 or 4 bytes
#if defined (WIN32) || defined (__WATCOMC__)    // force 16-bit on MSC++ and WATCOM
   typedef unsigned short BITFIELD16;
#else
   typedef unsigned       BITFIELD16;
#endif
typedef    unsigned       BITFIELD32;

//- determine if a number is a power of 2 (non-power-of-2 has >= two "1" bits)
#define TxIsPowerOf2(x) (((x) != 0) && (((x) & ((x) - 1)) == 0))


//- TX-BITMAP derived from C-FAQ: use array of ULONG, address individual bits

// size of bit-array items
#define TXBULBITS           32
#define TXBULSIZE(x)  (((x) + TXBULBITS -1) / TXBULBITS)

// dynamic declare x-size TX bit array
#define TXBITALLOC(x)      ((ULONG *) TxAlloc(TXBULSIZE(x),sizeof(ULONG)))

#define TXBITMASK(b)     (1 << ((b) % TXULBITS))
#define TXBITSLOT(b)     (      (b) / TXULBITS)

#define TXBITSET(a,b)    ((a)[TXBITSLOT(b)] |=  TXBITMASK(b))
#define TXBITTEST(a,b)   ((a)[TXBITSLOT(b)] &   TXBITMASK(b))
#define TXBITCLEAR(a,b)  ((a)[TXBITSLOT(b)] &= ~TXBITMASK(b))
#define TXBITFLIP(a,b)   ((a)[TXBITSLOT(b)] ^=  TXBITMASK(b))


#include <txcon.h>                              // TX ANSI console and tracing


typedef enum txsetreset
{
   TX_SET,
   TX_RESET,
   TX_TOGGLE
} TXSETRESET;                                   // end of enum "txsetreset"


// Get operatingsystem version major*minor*micro and text (Win-9x major = 1)
ULONG TxOsVersion                               // RET   number 000000..999999
(
   char               *descr                    // OUT   OS description or NULL
);

// Generic definition for a linked list of strings (like warnings/errors)
typedef struct txstrlist
{
   char               *string;                  // payload
   struct txstrlist   *next;                    // next element
} TXSTRLIST;                                    // end of struct "txstrlist"

// Add string to string-list (allocate memory)
char *TxStrListAdd                              // RET   added string
(
   TXSTRLIST         **list,                    // IN    address of list (head)
   char               *str                      // IN    string to add
);

// Free string-list (free memory)
void  TxStrListFree
(
   TXSTRLIST         **list                     // IN    address of list (head)
);

#pragma pack(1)                                 // byte packing
typedef struct txfs_ebpb                        // ext. boot parameter block
{                                               // at offset 0x0b in bootsector
   USHORT              SectSize;                // bytes per sector (BPB start)
   BYTE                ClustSize;               // sectors per cluster
   USHORT              FatOffset;               // reserved sectors to 1st FAT
   BYTE                NrOfFats;                // nr of FATS        (FAT only)
   USHORT              RootEntries;             // Max std entries \ (FAT only)
   USHORT              Sectors;                 // nr of sectors if <  64K
   BYTE                MediaType;               // mediatype (F8 for HD)
   USHORT              FatSectors;              // nr of sectors/FAT (FAT only)
   USHORT              LogGeoSect;              // nr of sectors/Track
   USHORT              LogGeoHead;              // nr of heads
   ULONG               HiddenSectors;           // Sector-offset from MBR/EBR
   ULONG               BigSectors;              // nr of sectors if >= 64K
} TXFS_EBPB;                                    // last byte is at offset 0x23


#if defined (DOS32)                             // Dos interface defs
   typedef struct txfs_gparm                    // IoCtl structure
   {
      BYTE             special;                 // special functions
      BYTE             devtype;                 // device type
      USHORT           devattr;                 // device attributes
      USHORT           cylinders;               // Nr of cylinders
      BYTE             density;                 // Media density
      TXFS_EBPB        eb;                      // Extended BPB
      BYTE             reserved[32];            // used by WIN9x !!
   } TXFS_GPARM;                                // end of struct "txfs_gparm"


#endif


// Execute standard commands
ULONG TxStdCommand
(
   void                                         // parameters through TXA
);

// Set current thread priority Min/Low/Std/High/maX
char TxSetPriority                              // RET   resulting priority
(
   char                prio                     // IN    priority M/L/S/H/X/Q
);


//--------- TXlib SCRIPT/REXX subcommand definitions and functions -----------

typedef ULONG (* TX_NATIVE_CALLBACK)            // modelled after MultiCmd
(
   char               *cmd,                     // IN    command string (input)
   BOOL                echo,                    // IN    echo   before each cmd
   BOOL                prompt,                  // IN    prompt after  each cmd
   BOOL                quiet                    // IN    screen-off during cmd
);

// Test if script exists, is REXX, return parameter description and full-name
BOOL TxsValidateScript                          // RET   script exists
(
   char               *sname,                   // IN    script name
   BOOL               *rexx,                    // OUT   script is REXX
   char               *pdesc,                   // OUT   param description or NULL
   char               *fname                    // OUT   full scriptname or NULL
);

// Run native TXS script using callback (from parsed RUN cmd if runcmd == NULL)
ULONG TxsNativeRun
(
   char               *name,                    // IN    TXS script to execute
   char               *runcmd,                  // IN    RUN cmd string or NULL
   TX_NATIVE_CALLBACK  subcom                   // IN    subcommand handler
);

#if   defined (DEV32)


typedef ULONG (_System * TX_SUBCOM_CALLBACK)
(
   PRXSTRING           cmd,                     // IN    command string
   PUSHORT             err_flag,                // OUT   ERROR/FAILURE
   PRXSTRING           rv                       // OUT   string retcode
);


// Execute a REXX command script using a subcommand environment
ULONG TxsRexxRun
(
   char               *name,                    // IN    REXX proc to execute
   char               *args,                    // IN    Arguments
   char               *envname,                 // IN    REXX environment name
   TX_SUBCOM_CALLBACK  subcom                   // IN    subcommand handler
);


// Set string variable in the REXX variable pool using printf syntax
ULONG TxsRexxSetnum
(
   char              *name,                     // IN    Name of variable
   ULONG              ulv,                      // IN    value for variable
   char              *fmt                       // IN    format strings
);

// Set a string variable in the REXX variable pool
ULONG TxsRexxSetvar
(
   char              *name,                     // IN    Name of variable
   void              *value,                    // IN    ptr to value buffer
   ULONG              length                    // IN    length of value
);

#endif

//--------- TXlib TxFileTree, recursive tree itterator ---------------------
// Recursive findfile with call-back function to operate on found files
// Modelled after SysFileTree implementation from REXXUTIL
// Major functional changes made:
//
//   - REXX interfaces (stem, variablepool) changed to normal 'C'
//   - Action on each matching file implemented as a call-back function with
//     full "true-filename" and OS-neutral TXFILEFIND structure as parameters
//   - Source and Destination mask use different syntax (+,-,* and ARSHD)
//
//     options  - Any combo of the following:
//                 'B' - Search for files and directories. (default)
//                 'D' - Search for directories only.
//                 'F' - Search for files only.
//                 'S' - Recursively scan subdirectories.
//
// Includes the lower-level TxFindFirstFile() and TxFindNextFile() APIs too


typedef struct ntime                            // NT time value (64 bits)
{
   ULONG               lo;                      // low  32 bits
   ULONG               hi;                      // high 32 bits
} NTIME;                                        // end of struct "ntime"


#define MAX_S_TIME    0xbf80                    // maximum valid time value

typedef struct s_time
{
   BITFIELD16         twosecs : 5;
   BITFIELD16         minutes : 6;
   BITFIELD16         hours   : 5;
} S_TIME;

#define MAX_S_DATE    0x7f9f                    // maximum valid date (Y 2043)
                                                // date 0x0000 is invalid too!

typedef struct s_date
{
   BITFIELD16         day     : 5;
   BITFIELD16         month   : 4;
   BITFIELD16         year    : 7;
} S_DATE;


typedef union s_datim                           // combined date/time int
{
   S_TIME              t;
   S_DATE              d;
   USHORT              u;
} S_DATIM;                                      // end of union "s_datim"


typedef struct txfilefind
{
   time_t              cTime;                   // creation   date-time
   time_t              aTime;                   // last acces date-time
   time_t              wTime;                   // last-write date-time
   ULONG               fAttr;                   // file attributes
   LLONG               fSize;                   // 64-bits filesize
   TXLN                fName;                   // ASCII filename
} TXFILEFIND;                                   // end of struct "txfilefind"


typedef ULONG (* TXTREE_CALLBACK)               // RET   callback result
(
   char               *fullname,                // IN    full filename
   TXFILEFIND         *fileinfo,                // IN    attrib, EA info etc
   void               *calldata                 // INOUT callback data
);



// Find first file matching the specified (wildcard) name
ULONG TxFindFirstFile                           // RET   result
(
   char               *fspec,                   // IN    file specification
   TXHANDLE           *handle,                  // OUT   filefind handle
   TXFILEFIND         *found                    // OUT   found file details
);

// Find next file matching the specified (wildcard) name
ULONG TxFindNextFile                            // RET   result
(
   TXHANDLE            handle,                  // IN    filefind handle
   TXFILEFIND         *found                    // OUT   found file details
);

// Close FindFirst/FindNext session for specified handle
ULONG TxFindClose                               // RET   result
(
   TXHANDLE            handle                   // IN    filefind handle
);

// Simple (example) callback for TxFileTree, listing files date/attr/full-name
ULONG txFileTreeOneFile
(
   char               *fname,                   // IN    full filename
   TXFILEFIND         *finfo,                   // IN    file information
   void               *cdata                    // IN    Callback data
);

// Simple (example) callback for FileTree, listing files/dirs without path
ULONG txFileTreeNoPath
(
   char               *fname,                   // IN    full filename
   TXFILEFIND         *finfo,                   // IN    file information
   void               *cdata                    // IN    Callback data
);

// Split multiple wildcard in BASE and INCLUDE array
char **txWildcard2Filter
(
   char               *base,                    // INOUT base specification
   char               *wildcard,                // IN    wildcard (multi)
   char              **fa,                      // IN    filter array
   int                 size                     // IN    size of filter array
);

// Free filter array memory
char **txFreeFilter                             // RET   NULL
(
   char              **filter,                  // IN    filter array
   int                 size                     // IN    max number of entries
);

// Itterate through files and directories matching filespec and attributes
ULONG TxFileTree
(
   char               *fspec,                   // IN    Filespec to match
   char               *fopts,                   // IN    Find options
   char               *fattr,                   // IN    Select attributes
   char               *mattr,                   // IN    Modify attributes
   char              **exclude,                 // IN    Exclude file-spec
   TXTREE_CALLBACK     callback,                // IN    function to call back
   void               *calldata                 // IN    callback data
);



//--------- TXlib command execution with full output redirection ------------

// Execute an external command, output redirected to TxPrint
ULONG TxExternalCommand                         // RET   command exit-code (RC)
(
   char              *cmd                       // IN    command to execute
);



//--------- 32-bit CRC implementation --------------------------------------
// CRC32 interface; derived from textbook:
//       'C' programmers guide to NETBIOS
//
// Calculate CRC-32 value for supplied data
ULONG TxCrc32
(
   void               *pcDat,                   // IN    pointer to data
   ULONG               ulSize                   // IN    length of data
);

// Calculate sector checksum, using HPFS algorithm and fixed 512 byte size
ULONG TxHpfsCheckSum                            // RET   HPFS style crc
(
   char               *data                     // IN    data sector
);

// Calculate 32-bit CRC value using LVM compatible algorithm and polynom
ULONG TxCalculateLvmCrc
(
   char               *area,                    // IN    data area needing CRC
   ULONG               size                     // IN    size of the data area
);

//--------- FileSystem functions -------------------------------------------

//- Get double MB value, prepare for rounding-down with single digit fraction
#define TXS2MB(secs,bps) ((double) (secs == 0) ? ((double) 0.0) :           \
                                 ( ((double) (secs)                         \
                                 * ((double) (((bps) != 0) ? (bps) : 512))  \
                                 / ((double) 1048576))                      \
                                 - ((double) (secs <= 100) ? 0.0 : 0.05)))

//- Get double MiB value, no special rounding preparation, eXact value
#define TXSMIB(secs,bps) ((double) (secs == 0) ? ((double) 0.0) :           \
                                 ( ((double) (secs)                         \
                                 * ((double) (((bps) != 0) ? (bps) : 512))  \
                                 / ((double) 1048576))))

//- Get double KB value, no special rounding preparation
#define TXS2KB(secs,bps) ((double) ((double) (secs)                         \
                                 * ((double) (((bps) != 0) ? (bps) : 512))  \
                                 / ((double) 1024)))


// Perform a shutdown (if possible) followed by a (COLD) boot of the system
BOOL TxReboot                                   // RET   reboot OK
(
   BOOL                flush                    // IN    Flush buffers/FS-data
);


// Set critical error handling to auto-FAIL ==> application gets a "not-OK" rc
// Calls to this may be nested, autofail is ON until matching fail==false call
void TxFsAutoFailCriticalErrors
(
   BOOL                fail                     // IN    autofail on CritErr
);

// Set (DOS) critical error handling to use TX-handler (MessageBox or AutoFail)
void TxFsSetCritErrHandler
(
   BOOL                install                  // IN    install (or deinstall)
);

#define  TXFSV_HD      0
#define  TXFSV_FLOP    1
#define  TXFSV_LAN     2
#define  TXFSV_CD      4
#define  TXFSV_ALL     7
// Build string with present volumes
int TxFsVolumes                                 // RET   nr of drives listed
(
   ULONG               flags,                   // IN    FLOPPY/LAN/CD select
   char               *vols                     // OUT   Present volumes
);

// Get volumelabel for specified driveletter, or -not-ready- if not accessible
char *TxFsVolumeLabel                           // RET   label or "-not-ready-"
(
   char               *drive,                   // IN    Drive specification
   char               *label                    // INOUT Label string (min 12)
);

// Show volume/filesystem info for all registred drive-letters
void TxFsDrivemap
(
   char               *lead,                    // IN    Leading text string
   ULONG               flags                    // IN    FLOPPY/LAN/CD select
);

// Show volume/filesystem info on a single output line using TxPrint
ULONG TxFsShow
(
   char               *lead,                    // IN    Leading text string
   char               *drive                    // IN    Drive specification
);


// Determine attached fstype, e.g. HPFS for specified drive
BOOL TxFsType                                   // RET   FS type resolved
(
   char               *drive,                   // IN    Drive specification
   char               *fstype,                  // OUT   Attached FS type
   char               *details                  // OUT   details (UNC) or NULL
);


// Determine free and total-available space (sectors) on specified drive
ULONG TxFsSpace
(
   char               *drive,                   // IN    Drive specification
   ULONG              *sfree,                   // OUT   Free sectors
   ULONG              *stotal,                  // OUT   Total sectors
   USHORT             *bpsect                   // OUT   Sectorsize
);

#define TXTN_DIR     0
#define TXTN_FILE   80
// Query type for specified path and resolve truename (canonical)
// Directorynames (for DIR or stripped filename) will ALWAYS end in a '\'
ULONG  TxTrueNameDir                            // RET   0 = Dir, 80 = File;
(                                               //       others are OS errors
   char               *path,                    // IN    path specification
   BOOL                stripfile,               // IN    strip filename part
   char               *true                     // OUT   true filename or NULL
);


// Check path, optionaly create it in a recursive fashion
ULONG TxMakePath                                // RET   result
(
   char               *filename                 // IN    Directory-path
);

// Make an 8-dot-3 compatible copy of supplied path/filename string (no spaces)
ULONG TxMake8dot3                               // RET   length converted name
(
   char               *filename,                // IN    relative/absolute path
   char               *fname8d3                 // OUT   8dot3 compatible buf
);                                              //       with size >= filename

// Function : count number of non-wildcard chars in string
int TxStrWcnt                                   // RET   nr of non-wild chars
(
    char              *s                        // IN    candidate string
);


// wildcard compare of candidate to template string
int TxStrWcmp                                   // RET   negative if no match
(
    char              *cs,                      // IN    candidate string
    char              *ts                       // IN    template string
);


// See TxStrWcmp; case-insensitive
int TxStrWicmp                                  // RET   negative if no match
(
    char              *cs,                      // IN    candidate string
    char              *ts                       // IN    template string
);

// Append translated Unicode string-fragment to an ASCII string
char *TxUnicAppend                              // RET   ptr to ASCII string
(
   USHORT             *uni,                     // IN    Unicode string part
   char               *asc,                     // INOUT ASCII string
   USHORT              len                      // IN    maximum Unicode length
);

// Perform a Unicode to ASCII case-sensitive stringcompare (aka strncmp)
int TxUnicStrncmp                               // RET   compare result
(
   char               *uni,                     // IN    Unicode string
   char               *asc,                     // IN    ASCII   string
   size_t              len                      // IN    max ASCII comp-length
);

// Perform a Unicode to ASCII case-insensitive stringcompare (aka strnicmp)
int TxUnicStrnicmp                              // RET   compare result
(
   char               *uni,                     // IN    Unicode string
   char               *asc,                     // IN    ASCII   string
   size_t              len                      // IN    max ASCII comp-length
);

// Append default filename extension to filename, if none present
char *TxFnameExtension                          // RET   ptr to filename
(
   char               *fname,                   // INOUT filename string
   char               *ext                      // IN    default extention
);

// Remove filename extension from filename, if present
char *TxStripExtension                          // RET   ptr to filename
(
   char               *fname                    // INOUT filename string
);

// Get base-name part from a path+filename string
char *TxGetBaseName                             // RET   ptr to basename
(
   char               *fname                    // IN    path+filename string
);

// Strip basename from path+filename string, leaving the PATH component only
char *TxStripBaseName                           // RET   BaseName location,
(                                               //       or NULL if not there
   char               *fname                    // IN    path+filename string
);                                              // OUT   path only string

// Test if exact (path+) filename is accessible; supports > 2GiB files
BOOL TxFileExists                               // RET   file is accessible
(
   char               *fname                    // IN    filename string
);

// Create empty file with specified path/name, prompt to replace existing
ULONG TxCreateEmptyFile
(
   char               *fname,                   // IN    path and filename
   BOOL                prompt                   // IN    prompt on replace existing
);

// Test if exact (path+) filename is accessible; determine size > 2GiB files
BOOL TxFileSize                                 // RET   file exists
(
   char               *fname,                   // IN    filename string
   LLONG              *size                     // OUT   filesize or NULL
);

// Seek to specified position in open file, platform specific, supports large files
ULONG TxFileSeek
(
   TXHFILE             fh,                      // IN    file handle
   LLONG               offset,                  // IN    seek offset
   int                 whence                   // IN    seek reference
);

// Find file in a path specified by environment variable (like PATH)
char *TxFindByPath                              // RET   ptr to filename
(
   char               *fname,                   // IN    filename string
   char               *path,                    // IN    env-var with path
   char               *found                    // OUT   found file
);


// Find file in current-dir, exe-dir or PATH, and when found, open for read
FILE *TxFindAndOpenFile                         // RET   opened file handle
(
   char               *fname,                   // IN    filename string
   char               *path,                    // IN    env-var with path
   char               *found                    // OUT   found file
);




//--------- CommandParser definitions and functions ---------------------------

#define TXA_BASE      32                        // first valid option ASCII
#define TXA_LAST     'z'                        // last  valid option ASCII
#define TXA_LBASE    123                        // index first 'long' options

#define TXA_O_HELP   '?'                        // remapped from 'help' to '?'
#define TXA_O_QUERY  123                        // predefined 'query'
#define TXA_O_MENU   124                        // predefined 'menu'
#define TXA_O_DEBUG  125                        // predefined 'debug'
#define TXA_O_TEST   126                        // predefined 'test'
#define TXA_O_TRACE  127                        // predefined 'trace'
#define TXA_O_AUTO   128                        // predefined 'auto'
#define TXA_O_ENTRY  129                        // predefined 'entry'
#define TXA_O_COLOR  130                        // predefined 'color'
#define TXA_O_SCHEME 131                        // predefined 'scheme'
#define TXA_O_LINES  132                        // predefined 'lines'
#define TXA_O_INI    133                        // predefined 'ini'
#define TXA_O_CONFIG 134                        // predefined 'config'
#define TXA_O_STYLE  135                        // predefined 'style'
#define TXA_O_KEYB   136                        // predefined 'keyb'
#define TXA_O_MOUSE  137                        // predefined 'mouse'
#define TXA_O_SIMUL  138                        // predefined 'simulate'
#define TXA_O_LIST   139                        // predefined 'list'
#define TXA_O_SCREEN 140                        // predefined 'screen'
#define TXA_O_TEST1  141
#define TXA_O_TEST2  142
#define TXA_O_TEST3  143                        // predefined 'test1' .. 5
#define TXA_O_TEST4  144
#define TXA_O_TEST5  145
#define TXA_O_APP0   146
#define TXA_O_APP1   147                        // application defined
#define TXA_O_APP2   148                        // options/switches
#define TXA_O_APP3   149
#define TXA_O_APP4   150
#define TXA_O_APP5   151

#define TXA_LSIZE     29                        // number of long options
#define TXA_SIZE     120                        // total number of options

#define TXA_O_LEN     31                        // maximum length long name

#define TXA_ARGC      32                        // maximum nr of params

#define TXA_NONE      ((TXA_TYPE)  0)
#define TXA_NUMBER    ((TXA_TYPE)  1)
#define TXA_STRING    ((TXA_TYPE)  2)
#define TXA_NO_VAL    ((TXA_TYPE)  3)

#define TXA_DFUNIT    ((BYTE)    ' ')           // default unit

#define TXA_CUR       ((TXHANDLE)  3)           // special CURRENT  handle
#define TXA_PREV      ((TXHANDLE)  2)           // special PREVIOUS handle
#define TXA_1ST       ((TXHANDLE)  1)           // special FIRST    handle
#define TXA_NUL       ((TXHANDLE)  0)           // special bad TXA  handle

#define TXA_ALL       ((int)       0)           // get ALL arguments
#define TXA_OPT       ((int)  0xffff)           // get ALL args + options

// shorthand macro's for most used option/argument retrieval

#define TxaArgCount()           TxaGetArgc(       TXA_CUR  )
#define TxaArgValue(n)          TxaGetArgv(       TXA_CUR,n)
#define TxaOption(o)            TxaOptionYes(     TXA_CUR,o)
#define TxaOptSet(o)           (TxaGetOption(     TXA_CUR,o) != NULL)
#define TxaOptAsString(o,s,b)   TxaOptionAsString(TXA_CUR,o,s,b)
#define TxaOptStr(o,e,d)        TxaOptionStr(     TXA_CUR,o,e,d)
#define TxaOptNum(o,e,d)        TxaOptionNum(     TXA_CUR,o,e,d)
#define TxaOptValue(o)          TxaGetOption(     TXA_CUR,o)
#define TxaOptCount()           TxaGetOptc(       TXA_CUR  )
#define TxaOptMutEx(c,o,m,e)    TxaMutualX(       TXA_CUR,c,o,m,e)
#define TxaOptSetItem(i)        TxaSetItem(       TXA_CUR,i)
#define TxaOptUnSet(o)         (TxaOptSet(o) && (!TxaOption(o)))

#define TxaExeArgc()            TxaGetArgc(       TXA_1ST  )
#define TxaExeArgv(n)           TxaGetArgv(       TXA_1ST,n)
#define TxaExeSwitch(o)         TxaOptionYes(     TXA_1ST,o)
#define TxaExeSwitchSet(o)     (TxaGetOption(     TXA_1ST,o) != NULL)
#define TxaExeSwAsString(o,s,b) TxaOptionAsString(TXA_1ST,o,s,b)
#define TxaExeSwitchStr(o,e,d)  TxaOptionStr(     TXA_1ST,o,e,d)
#define TxaExeSwitchNum(o,e,d)  TxaOptionNum(     TXA_1ST,o,e,d)
#define TxaExeSwitchValue(o)    TxaGetOption(     TXA_1ST,o)
#define TxaExeOptc()            TxaGetOptc(       TXA_1ST  )
#define TxaExeMutEx(c,o,m,e)    TxaMutualX(       TXA_1ST,c,o,m,e)
#define TxaExeSetItem(i)        TxaSetItem(       TXA_1ST,i)
#define TxaExeSwitchUnSet(o)   (TxaExeSwitchSet(o) && (!TxaExeSwitch(o)))



typedef BYTE TXA_TYPE;                          // option none/number/string

typedef union txa_value
{
   ULONG               number;                  // number value
   char               *string;                  // string value
} TXA_VALUE;                                    // end of union "txa_value"

typedef struct txa_option
{
   TXA_TYPE            type;                    // none/number/string
   BYTE                unit;                    // one char c,m,s etc (on nrs)
   TXA_VALUE           value;
   char               *name;                    // Full option name or NULL
} TXA_OPTION;                                   // end of struct "txa_option"


// Set or query value for a specific LONGNAME value (the LONG option name)
char  *TxaOptionLongName                        // RET   resulting option name
(
   char                opt,                     // IN    TXA_O_ option value
   char               *name                     // IN    long name for option
);                                              //       or NULL to query

// Create new level of command options for a given command string
ULONG TxaParseCommandString                     // RET   result
(
   char               *cmd,                     // IN    command string
   BOOL                freeform,                // IN    free format options
   TXHANDLE           *txh                      // OUT   TXA handle (optional)
);


// Add one item (argument or option) to the specified TXA handle
ULONG TxaSetItem                                // RET   result
(
   TXHANDLE            txh,                     // IN    TXA handle
   char               *item                     // IN    new item
);


// Replace current level of command options for a new command string (Reparse)
ULONG TxaReParseCommand                         // RET   result
(
   char               *cmd                      // IN    new command string
);


// Create new level of command options for a set of argc/argv variables
ULONG TxaParseArgcArgv                          // RET   result
(
   int                 argc,                    // IN    argument count
   char               *argv[],                  // IN    array of arguments
   char               *exename,                 // IN    alternative argv[0]
   BOOL                freeform,                // IN    free format options
   TXHANDLE           *txh                      // OUT   TXA handle (optional)
);

#define  TXAO_QUIET    ((ULONG)0)
#define  TXAO_NORMAL   ((ULONG)1)
#define  TXAO_VERBOSE  ((ULONG)2)
#define  TXAO_EXTREME  ((ULONG)3)
#define  TXAO_MAXIMUM  ((ULONG)4)

// Get verbosity value from Option 'o'
ULONG TxaOutputVerbosity                        // RET   output verbosity value
(
   char                optchar                  // IN    option character
);

#define  TXAE_QUIT     ((int)'q')
#define  TXAE_IGNORE   ((int)'i')
#define  TXAE_CONFIRM  ((int)'c')

// Get errorStrategy value from Option 'e' and batch-mode indicator 'quit'
int TxaErrorStrategy                            // RET   error strategy value
(
   char                optchar,                 // IN    option character
   BOOL                quit                     // IN    quit as default
);

//- define classes each having their own HEX/DEC default in ParseNumber
//- Application can define higher class-numbers from 2^3 upto 2^31
//- A bit that is 'SET' for a class, will result in a HEXadecimal default
#define TX_RADIX_DEC_CLASS       0               // radix-class fixed decimal
#define TX_RADIX_STD_CLASS       1               // radix-class standard
#define TX_RADIX_UN_S_CLASS      2               // radix-class n,s unit
#define TX_RADIX_UNIT_CLASS      4               // radix-class any unit
#define TX_RADIX_APP_CLASS       8               // radix-class, application 1

// Set radix class bits for input like ParseNumber, numeric options; 1 = HEX
void TxSetNumberRadix
(
   ULONG               radixclasses             // IN    radix class bits
);

// Get radix class bits for input like ParseNumber, numeric options; 1 = HEX
ULONG TxGetNumberRadix                          // RET   radix class bits
(
   void
);

// Parse a decimal or hex number value from a string
ULONG TxaParseNumber                            // RET   number value
(
   char               *value,                   // IN    value string with nr
   ULONG               rclass,                  // IN    HEX/DEC radix class
   BYTE               *unit                     // OUT   optional unit char, if
);                                              //       not needed use NULL

// Parse a boolean value from a string (empty string returns TRUE)
BOOL TxaParseBool                               // RET   FALSE: 0 - n f off
(
   char               *param                    // IN    pragma parameter
);

// Terminate use of current or all instances of the TXA abstract-data-type
BOOL TxaDropParsedCommand                       // RET   more instances left
(
   BOOL                whole_stack              // IN    drop all, terminate
);

// Show contents of one or all instances of the TXA abstract-data-type
void TxaShowParsedCommand                       // RET   more instances left
(
   BOOL                whole_stack              // IN    show all levels
);

// Get reference to specified option data, or NULL if option not set
TXA_OPTION *TxaGetOption                        // RET   option ptr or NULL
(
   TXHANDLE            txh,                     // IN    TXA handle
   char                opt                      // IN    option character
);

// Get simple YES/NO status for specified option. Not set is "NO"
BOOL TxaOptionYes                               // RET   option set to YES
(
   TXHANDLE            txh,                     // IN    TXA handle
   char                opt                      // IN    option character
);

// Get string value for specified option, convert to string for any type
char *TxaOptionAsString                         // RET   option String value
(
   TXHANDLE            txh,                     // IN    TXA handle
   char                opt,                     // IN    option character
   int                 size,                    // IN    size of buffer
   char               *str                      // OUT   string buffer, also
);                                              // default when not specified

// Get string value for specified option, use default if no string (and warn!)
char *TxaOptionStr                              // RET   option String value
(
   TXHANDLE            txh,                     // IN    TXA handle
   char                opt,                     // IN    option character
   char               *error,                   // IN    error text or NULL
   char               *deflt                    // IN    default value
);

// Get number value for specified option, use default if no number (and warn!)
ULONG TxaOptionNum                              // RET   option Number value
(
   TXHANDLE            txh,                     // IN    TXA handle
   char                opt,                     // IN    option character
   char               *error,                   // IN    error text or NULL
   ULONG               deflt                    // IN    default value
);

// Build space separated concatenated string of arguments
// Clips output with no warning when buffer too small!
char *TxaGetArgString                           // RET   argument str or NULL
(
   TXHANDLE            txh,                     // IN    TXA handle
   int                 first,                   // IN    first argument to copy
   int                 last,                    // IN    last arg, _ALL or _OPT
   int                 bufsize,                 // IN    size of result buffer
   char               *args                     // OUT   argument string
);

// Return comment part of command, if any
char *TxaGetComment                             // RET   comment  str or NULL
(
   TXHANDLE            txh,                     // IN    TXA handle
   char               *comment                  // OUT   comment string
);

// Get reference to specified argument-string
char *TxaGetArgv                                // RET   ptr to argv string
(
   TXHANDLE            txh,                     // IN    TXA handle
   int                 argn                     // IN    argument number 0..n
);

// Get the number of arguments for specified TXA instance
int TxaGetArgc                                  // RET   argument count
(
   TXHANDLE            txh                      // IN    TXA handle
);

// Get the number of options for specified TXA instance
int TxaGetOptc                                  // RET   option count
(
   TXHANDLE            txh                      // IN    TXA handle
);

// Test for mutual exclusive options and specified condition; handle error msg
BOOL TxaMutualX
(
   TXHANDLE            txh,                     // IN    TXA handle
   BOOL                cond,                    // IN    other  exclusive cond
   char               *opts,                    // IN    mutual exclusive opts
   char               *condmsg,                 // IN    msg when cond TRUE or NULL
   char               *msgtext                  // OUT   message text or NULL
);                                              //       (NULL ==> TxPrint)


//--------- Selection list support (not tied to TXW yet)    (TX-SELIST) ------
//- supports either static (fixed) lists and dynamic ones where the total
//- list can be many times larger than the expanded (visible) part

//- definitions for flags that apply to one item (item->flags)
#define TXSF_MARK_STD                0x01
#define TXSF_MARK_STAR               0x02
#define TXSF_MARKED                  0x03
#define TXSF_P_LISTBOX               0x10       // popup another listbox level
#define TXSF_SEPARATOR               0x20
#define TXSF_AUTOSKIP                0x40       // skip disabled on up/down
#define TXSF_DISABLED                0x80

//- definitions for flags that apply to the whole list (list->flags)
#define TXSL_MULTI_QUICK       0x00000001       // use itemtext for quickselect
                                                // not just the item-index char

//- definitions for Alloc-Status of selist objects (Static/Shared or Dynamic)
//- when static bits for an item are set, TxFreeMem() will be used on them
#define TXS_AS_NOSTATIC              0xff       // all Tx dynamic allocation
#define TXS_AS_LISTRUCT              0x01       // main struct
#define TXS_AS_SELARRAY              0x02       // selection array
#define TXS_AS_PTRARRAY              0x04       // ptr-array to items
#define TXS_AS_SELITEMS              0x08       // item structures
#define TXS_AS_SEL_DESC              0x10       // item description
#define TXS_AS_SEL_TEXT              0x20       // item display text
#define TXS_AS_DSTRINGS  (TXS_AS_SEL_DESC | TXS_AS_SEL_TEXT)
#define TXS_AS_DYNITEMS  (TXS_AS_SELITEMS | TXS_AS_DSTRINGS)
#define TXS_AS_DYNBASED  (TXS_AS_LISTRUCT | TXS_AS_SELARRAY | TXS_AS_PTRARRAY)
#define TXS_AS_STATIC                0x00       // all static, no TxAlloc

typedef struct txs_item
{
   ULONG               value;                   // value, cmd-code, alt helpid
   ULONG               helpid;                  // helpid or 0 to use cmd value
   ULONG               user;                    // user value, list for submenu
   USHORT              index;                   // pos of sel-char 1..x or 0
   BYTE                flags;                   // item flags,  (disabled etc)
   BYTE                reserved;                // select value while sorting
   char               *desc;                    // descriptive help on item
   char               *text;                    // rendered item text
} TXS_ITEM;                                     // end of struct "txs_item"


//- TXSELIST callback function to render new visible list item based on index
typedef ULONG (* TXS_ITEMHANDLER)
(
   TXS_ITEM           *item,                    // INOUT item to handle or NULL
   BYTE               *select,                  // INOUT selection data or NULL
   ULONG               index                    // IN    external index
);


typedef struct txselist
{
   ULONG               astatus;                 // alloc-status bits; 0=static
   ULONG               userinfo;                // user definable info
   ULONG               flags;                   // list specific flags
   ULONG               top;                     // index of 1st visible item
   ULONG               count;                   // actual number of items
   ULONG               asize;                   // size allocated  (items array)
   ULONG               vsize;                   // size of visible part (scroll)
   ULONG               tsize;                   // size for total list (maximum)
   ULONG               selected;                // index or number of selected
   BYTE               *selarray;                // selarray if multi-select
   TXS_ITEMHANDLER     renderNewItem;           // callback when made visible
   TXS_ITEM          **items;                   // array of visible items
} TXSELIST;                                     // end of struct "txselist"

typedef struct txs_menu
{
   ULONG               helpid;                  // help-id for pulldown
   USHORT              index;                   // pos of sel-char 1..x or 0
   BYTE                flags;                   // item flags,  (disabled etc)
   BYTE                ident;                   // identification/select char
   char               *desc;                    // descriptive help on menu
   char               *text;                    // rendered item text
   TXSELIST           *list;                    // list of menu items
} TXS_MENU;                                     // end of struct "txs_menu"

#define TXS_BARSIZE     32
typedef struct txs_menubar
{
   ULONG               count;                   // number of menus
   ULONG               defopen;                 // index of default menu
   TXS_MENU           *menu[TXS_BARSIZE];       // array of menus
} TXS_MENUBAR;                                  // end of struct "txs_menubar"


//- macros to define (menubar/popupmenu/submenu) list items statically

#define TXSitem static TXGitem
#define TXGitem(name,cmd,help,flg,selpos,text,desc)                           \
        TXS_ITEM name = {cmd,help,NULL,selpos,flg,0,desc,text}

#define TXSsubm static TXGsubm
#define TXGsubm(name,help,sub,base,flg,selpos,text,desc)                      \
        TXS_ITEM name = {base,help,(ULONG)sub,selpos,(flg)|TXSF_P_LISTBOX,0,desc,text}

#define TXSlist static TXGlist
#define TXGlist(name,s,v,items)                                               \
        TXSELIST name = {0,0,0,0,s,s,v,s,0,NULL,NULL,(items)}

#define TXSmenu static TXGmenu
#define TXGmenu(name,list,help,flg,selpos,ident,text,desc)                    \
        TXS_MENU name = {help,selpos,flg,ident,desc,text,(list)}

// Get length of longest string in a list
ULONG TxSelGetMaxLength                         // RET   longest string length
(
   TXSELIST           *list                     // IN    selection list
);

// Scroll visible list up by one line (towards lower index)
ULONG TxSelScrollUp                             // RET   scrolling succeeded
(
   TXSELIST           *list                     // INOUT selection list
);

// Scroll visible list down by one line (towards higher index)
ULONG TxSelScrollDown                           // RET   scrolling succeeded
(
   TXSELIST           *list                     // INOUT selection list
);

// Adjust visible list to start at the specified external index
#define TXSEL_TOP   ((ULONG) 0x00000000)        // top of list position
#define TXSEL_END   ((ULONG) 0x7fffffff)        // end of list position
ULONG TxSelSetPosition                          // RET   resulting top index
(
   TXSELIST           *list,                    // INOUT selection list
   ULONG               index                    // IN    top external index
);

// Set selection for specified index, and/or bits if multiple select possible
ULONG TxSelSetSelected                          // RET   resultcode
(
   TXSELIST           *list,                    // INOUT selection list
   ULONG               index,                   // IN    external index
   TXSETRESET          action,                  // IN    set, reset or toggle
   BYTE                mask                     // IN    bits to set/reset
);

// Itterate over selected list items, calback all with specified mask set
ULONG TxSelItterateSelected
(
   TXSELIST           *list,                    // INOUT selection list
   TXS_ITEMHANDLER     selected,                // IN    callback selected
   BYTE                mask                     // IN    selection mask
);


// Set 'selected' to item with specified selection char (at text[index-1])
ULONG TxSelCharSelect                           // RET   index now selected
(
   TXSELIST           *list,                    // INOUT selection list
   char                select                   // IN    selection character
);

// Set 'selected' to item where text starts with 'string' (case-insensitive)
ULONG TxSelStringSelect                         // RET   index now selected
(
   TXSELIST           *list,                    // INOUT selection list
   char               *select,                  // IN    selection string
   int                 length                   // IN    significant length
);

// Get unique quick-select character for specified list index
USHORT TxSelGetCharSelect                       // RET   index for select-char
(                                               //       or 0 if not possible
   TXSELIST           *list,                    // IN    selection list
   ULONG               nr,                      // IN    item to get char for
   int                 offset                   // IN    minimum position 0..n
);

// Set/reset specified flag-bit(s) in menu-item by searching menu-bar structure
ULONG TxSetMenuItemFlag                         // RET   #items found and set
(
   TXS_MENUBAR        *mbar,                    // IN    menubar information
   ULONG               mid,                     // IN    menu item id
   BYTE                mask,                    // IN    bits to set/reset
   BOOL                set                      // IN    set specified bit(s)
);

// Set/reset specified flag-bit(s) in list-item by searching list-structure
ULONG TxSetListItemFlag                         // RET   #items found and set
(
   ULONG               recurse,                 // IN    recursion level (1)
   TXSELIST           *list,                    // IN    list information
   ULONG               mid,                     // IN    value, menu item id
   BYTE                mask,                    // IN    bits to set/reset
   BOOL                set                      // IN    set specified bit(s)
);

// Create a dynamic selection list with specified size
ULONG TxSelCreate
(
   ULONG               vsize,                   // IN    visible size
   ULONG               tsize,                   // IN    total size of list
   ULONG               asize,                   // IN    size for item array
   ULONG               astat,                   // IN    alloc status for items
   BOOL                multi,                   // IN    create multi-select info
   TXS_ITEMHANDLER     render,                  // IN    callback render new
   TXSELIST          **list                     // OUT   selection list
);

// Destroy a dynamic selection list, freeing memory
ULONG txSelDestroy
(
   TXSELIST          **list                     // IN    selection list
);

// Destroy dynamic parts of a selection list-item, freeing memory
ULONG txSelItemDestroy
(
   ULONG               as,                      // IN    allocaton status
   TXS_ITEM          **itemref                  // IN    selection list item
);

// Create static selist from existing Txt list
ULONG TxSelistFromTxt                           // RET   result
(
   char               *txt[],                   // IN    NULL terminated txt
   BOOL                multi,                   // IN    keep multi-select info
   BOOL                share,                   // IN    share data, no alloc
   TXSELIST          **list                     // OUT   selection list
);

// Create selist from existing selist (duplicate)
ULONG TxSelistDuplicate                         // RET   result
(
   TXSELIST           *org,                     // IN    original Selist
   BOOL                multi,                   // IN    keep multi-select info
   BOOL                share,                   // IN    share data, no alloc
   TXSELIST          **list                     // OUT   selection list
);


// Create single-item list with ONE disabled text-item (for 'empty' selections)
TXSELIST *TxSelEmptyList                        // RET   selection list or NULL
(
   char               *text,                    // IN    text for single item
   char               *desc                     // IN    description (footer)
);

//- substring in not-ready drives in list
#define  TXFS_NOTREADY   "-not-ready-"

// Build selection-list with volume/FS info for drive-letters or Linux-devices
TXSELIST *TxFsDriveSelist                       // RET   selection list or NULL
(
   ULONG               flags,                   // IN    include FLOP/LAN/CDROM
   BOOL                flop                     // IN    runtime floppy test
);


#define TXFDS_ADD_DESCRIPTION     0x0001        // add descriptions too

// Build selection-list with File/Directory/Tree information
TXSELIST *TxFileDirSelist                       // RET   selection list or NULL
(                                               //       Tree RC is in userinfo
   char               *fspec,                   // IN    Filespec to match
   char               *fopts,                   // IN    Find options
   char               *fattr,                   // IN    Select attributes
   char              **filter,                  // IN    Excl/Incl file-specs
   ULONG               flags                    // IN    flags - 'descriptions'
);


//--------- Generic utility functions and OS-specific support (TXUTIL) -------

// Remove ANSI control code from a text-string (in place conversion)
char *TxStripAnsiCodes                          // RET   stripped string
(
   char               *text                     // INOUT ANSI text to strip
);

// Replace specfied character by another in whole string, count replacements
int TxRepl                                      // RET   nr of replacements
(
   char              *str,                      // INOUT string to convert
   char               old,                      // IN    old char, to replace
   char               new                       // IN    new char
);

// Strip leading/trailing characters from a string, dest and source can be same
int TxStrip                                     // RET   nr of stripped chars
(
   char               *dest,                    // OUT   destination string
   char               *source,                  // IN    source string
   char                lead,                    // IN    leading  chars to strip
   char                trail                    // IN    trailing chars to strip
);

// Pad/Clip a string to specified length, concatenating or removing characters
int TxPClip                                     // RET   +:padded -:removed
(
   char               *str,                     // INOUT destination string
   int                 size,                    // IN    requested length
   char                pad                      // IN    trailing chars to pad
);

// Test if area consists completely of the empty byte/character ==> is empty
BOOL TxAreaEmpty                                // RET   area is empty
(
   char               *area,                    // IN    area to check
   int                 size,                    // IN    size of area, bytes
   char                empty                    // IN    the empty char
);

#define TXASCII827_STRIP   ((unsigned char) 0x00)
#define TXASCII827_TRANS   ((unsigned char) 0x80)
// Filter 8-bit ASCII, strip to 7-bit, translate, or replace by specified char
void TxAscii827                                 // translate 8-bit to 7-bit
(
   char               *str,                     // INOUT string to convert
   unsigned char       new                      // IN    new char, or 0x00/0x80
);

// Copy string to destination, replacing non printables, clip and terminate it
void TxCopy
(
   char               *dest,                    // OUT   destination string
   char               *source,                  // IN    source string
   int                 len                      // IN    max length incl \0
);

// Find occurence of a string in a memory region (with embedded NUL)
char *TxMemStr                                  // RET   ptr to string or NULL
(
   void               *mem,                     // IN    memory region
   char               *str,                     // IN    string to find
   int                 len                      // IN    length of mem region
);

// Convert C-string to dynamically allocated TXLIB text structure
char  **txString2Text                           // RET   TXLIB text structure
(
   char               *string,                  // IN    null terminated string
   int                *maxlen,                  // INOUT line length
   int                *lines                    // OUT   nr of lines
);

// Free memory allocated for dynamic TXLIB text
void txFreeText
(
   char              **text                     // IN    dyn. allocated text
);

// Convert USHORT value from big to little endian or vice-versa
USHORT txSwapUS                                 // RET   converted value
(
   USHORT              val                      // IN    value
);

// Convert ULONG value from big to little endian or vice-versa
ULONG txSwapUL                                  // RET   converted value
(
   ULONG               val                      // IN    value
);

// Test if string represents a valid hex number, allow 0x prefix and ,unit
BOOL txIsValidHex                               // RET   string is valid hex
(
   char               *num                      // IN    candidate string
);

// Test if string represents a valid decimal nr, allow 0t/0n prefix and ,unit
BOOL txIsValidDec                               // RET   string is not decimal
(
   char               *num                      // IN    candidate string
);

// Test if string starts with a valid MCS prefix 0x 0n 0t 0x (case insensitive)
BOOL txHasMcsPrefix                             // RET   string has MCS prefix
(
   char               *num                      // IN    candidate string
);

// Return 15-character TXLIB version string
char  *txVersionString                          // RET   15-char version string
(
   void
);

// Translate attributes (FAT) to 6-position string format
void txFileAttr2String
(
   ULONG               data,                    // IN    data
   char               *attrib                   // OUT   attribute string[6]
);


// Convert Windows-NT/W2K/XP filetime (64 bit) to compiler time_t value
time_t txWinFileTime2t                          // RET   time_t value
(
   NTIME              *nt,                      // IN    ptr NT time value
   LONG                TimeZone                 // IN    TZ offset to GMT (min)
);

// Convert OS/2 and eCS filetime (2 * USHORT) to compiler time_t value
time_t txOS2FileTime2t                          // RET   time_t value
(
   USHORT             *pd,                      // IN    USHORT OS2 coded date
   USHORT             *pt                       // IN    USHORT OS2 coded time
);


// Format (file) size in 7.1 value + Byte/KiB/MiB/GiB (11); to TxPrint output
void txPrtSize64
(
   char               *text,                    // IN    leading string
   LLONG               data,                    // IN    size data
   char               *trail                    // IN    trailing text
);

// Format (file) size in 7.1 value + Byte/KiB/MiB/GiB (11); append to string
char *txStrSize64                               // RET   resulting string
(
   char               *str,                     // OUT   resulting string
   char               *text,                    // IN    leading string
   LLONG               data,                    // IN    size data
   char               *trail                    // IN    trailing text
);

#if   defined (DOS32)
   //- DOS (32-bit) specific utility functions

// Wait for specified nr of msec (approximation, +/- 32 msec)
void TxBusyWait                                 // RET   waited nr of msec
(
   ULONG               msec                     // IN    nr of msec to wait
);


// Set keyboard mapping using FreeDOS keyb and .kl files
ULONG TxSetNlsKeyboard                          // RET   result
(
   char               *spec,                    // IN    keyb file basename
   char               *cp                       // IN    codepage string or ""
);


// Get number of physical diskette drives from BIOS
USHORT TxPhysDisketteDrives                     // RET   nr of diskette drives
(
   void
);

#define TXX_BIOSD  0x400                        // BIOS data area

#define TXX_EQUIPMENT         0x0010            // equipment list

#define TXX_CRT_MODE          0x0049            // current CRT mode
#define TXX_CRT_COLS          0x004a            // number of columns on screen
#define TXX_CRT_LEN           0x004c            // size of video buffer
#define TXX_CURSOR_POSN       0x0050            // cursor for each page
#define TXX_CURSOR_MODE       0x0060            // cursor shape
#define TXX_PHYS_DISKS        0x0075            // number of physical disks
#define TXX_CRT_ROWS          0x0084            // number of text rows - 1
#define TXX_POINTS            0x0085            // height of character
#define TXX_INFO              0x0087            // miscellaneous info
#define TXX_INFO_3            0x0088            // more miscellaneous info

#define TxxBiosWord(p)     (*((USHORT *)(TXX_BIOSD + p)))
#define TxxBiosChar(p)     (*((char   *)(TXX_BIOSD + p)))

#define TxxClearReg(r)     memset( &(r), 0, sizeof(r))

#define TXDX_VIDEO                 0x10
#define TXDX_VIDEO_SETCSHAPE       0x01
#define TXDX_VIDEO_SETCURSOR       0x02
#define TXDX_VIDEO_GETCURSOR       0x03
#define TXDX_VIDEO_SCROLL_UP       0x06
#define TXDX_VIDEO_SCROLL_DN       0x07
#define TXDX_VIDEO_WRITNCELL       0x09
#define TXDX_VIDEO_SETSTATUS       0x10
#define TxxVideoInt(r,function)    r.h.ah = function; \
                                   int386( TXDX_VIDEO, &r, &r)

#define TXDX_PDISK                 0x13

#define TXDX_KBD                   0x16
#define TXDX_KBD_READ              0x00
#define TXDX_KBD_READY             0x01
#define TXDX_KBD_SHIFTSTATUS       0x02
#define TxxKeyBdInt(r,function)    r.h.ah = function; \
                                   int386( TXDX_KBD, &r, &r)

#define TXDX_DOS                   0x21
#define TXDX_DOS_GETVERSION        0x30
#define TXDX_DOS_FREESPACE         0x36
#define TXDX_DOS_IOCTL             0x44
#define TxxDosInt21(r,function)    r.h.ah = function; \
                                   int386( TXDX_DOS, &r, &r)

#define TXDX_MPLEX                 0x2F
#define TXDX_MPLEX_MSCDEX_INSTALL  0x1500
#define TXDX_MPLEX_MSCDEX_DRCHECK  0x150b
#define TxxMplexInt(r,function)    TXWORD.ax = function; \
                                   int386( TXDX_MPLEX, &r, &r)

#define TXDX_MOUSE                 0x33
#define TXDX_MOUSE_RESET           0x0000
#define TXDX_MOUSE_SHOW            0x0001
#define TXDX_MOUSE_HIDE            0x0002
#define TXDX_MOUSE_STATUS          0x0003
#define TXDX_MOUSE_PRESENT         0xffff
#define TxxMouseInt(r,function)    TXWORD.ax = function; \
                                   int386( TXDX_MOUSE, &r, &r)

#define TXDX_DPMI                  0x31
#define TXDX_DPMI_ALLOC            0x0100
#define TXDX_DPMI_FREEM            0x0101
#define TXDX_DPMI_RMINT            0x0300
#define TXDX_DPMI_RMFAR            0x0301
#define TXDX_DPMI_VERS             0x0400

#define TXDX_REAL_LIMIT  ((void *) 0xfffff)

typedef struct txdx_rminfo                      // DOS extender real-mode data
{
   long             edi;
   long             esi;
   long             ebp;
   long             reserved;
   long             ebx;
   long             edx;
   long             ecx;
   long             eax;
   short            flags;
   short            es;
   short            ds;
   short            fs;
   short            gs;
   short            ip;
   short            cs;
   short            sp;
   short            ss;
} TXDX_RMINFO;                                  // end of struct "tpdx_rminfo"


// Execute a DPMI call, including trace
#define  txDpmiCall(r,s)                                                 \
                    TRACES(("Input  txDpmiCall\n"));                     \
                    TRHEXS( 70, r, sizeof(union  REGS ), "in  REGS");    \
                    TRHEXS( 70, s, sizeof(struct SREGS), "in  SREGS");   \
                    TRACES(("Enter  txDpmiCall\n"));                     \
                    int386x( TXDX_DPMI,r,r,s);                           \
                    TRACES(("Return txDpmiCall\n"));                     \
                    TRHEXS( 70, r, sizeof(union  REGS ), "out REGS");    \
                    TRHEXS( 70, s, sizeof(struct SREGS), "out SREGS");

// get DPMI segment value from DPMI allocated linear address pointer
#define  txDpmiSegment(p)  (short)  (((unsigned long) p) >> 4)

// get DPMI 16:16 pointer from DPMI allocated linear address pointer
#define  txDpmi16ptr16(p)  (void *) (((unsigned long) p) << 12)

// Allocate DPMI compatible memory
void *txDpmiAlloc                               // RET   PM linear address
(
   size_t              nr,                      // IN    number of items
   size_t              size,                    // IN    size per item
   short              *selector                 // OUT   PM selector (for free)
);

// Free DPMI compatible memory
void txDpmiFree
(
   short               selector                 // IN    PM selector
);

// Return DOSEXTENDER version string
char  *txDosExtVersion                          // RET   version string (max 50)
(
   void
);

// Return DOSEXTENDER DPMI info string
char  *txDosExtDpmiInfo                         // RET   DPMI info string
(
   void
);


#elif defined (WIN32)
   //- no Win32 specific stuff yet

#elif defined (LINUX)                           // LINUX native        (WATCOM)

// Sleep for specified nr of msec
void TxSleepMsec
(
   ULONG               msec                     // IN    nr of msec to wait
);

// Translate (Linux) errno value to corresponding DFSee + DOS/OS2/WIN like RC
ULONG TxRcFromErrno                             // RET    TX_ return-code
(
   int                 err                      // IN    error number (errno)
);

#else
   //- OS/2 specific utility functions

typedef ULONG (APIENTRY * TXF_DOSOPENL)         // Large-file open ( > 2Gb)
(                                               // note: APIENTRY essential to
   PSZ                 pszFileName,             // force calling convention
   PHFILE              phf,                     // with all params on stack!
   PULONG              pulAction,
   LONGLONG            cbFile,
   ULONG               ulAttribute,
   ULONG               fsOpenFlags,
   ULONG               fsOpenMode,
   PEAOP2              peaop2
);

typedef ULONG (APIENTRY * TXF_DOSSEEKL)         // Large-file seek ( > 2Gb)
(
   HFILE               hFile,
   LONGLONG            ib,
   ULONG               method,
   PLONGLONG           ibActual
);

typedef struct txf_os2lfapi                     // entrypoint structure
{
   TXF_DOSOPENL        DosOpenLarge;            // open > 2Gb file
   TXF_DOSSEEKL        DosSeekLarge;            // seek > 2Gb file
} TXF_OS2LFAPI;                                 // end of struct "txf_os2lfapi"


// Test if OS2 large-file support (> 2GiB) is available; Fill entrypoints
BOOL TxLargeFileApiOS2                          // RET   large file API's OK
(
   TXF_OS2LFAPI       *entrypoints              // OUT   LF-API entrypoints
);                                              // needed LargeFile API's (3)
#endif

#include <txwin.h>                           // text-mode windowing

#endif
