/******************************************************************************
 * Copyright (c) 1995-2005 palmOne, Inc. or its subsidiaries.
 * All rights reserved.
 *****************************************************************************/
/**
 * @ingroup PIM
 */

/**
 * @file DateDB.h
 *
 * @brief Contains database record type and constants for Calendar application.
 *
 * Calendar application uses a different record format than the old DateBook
 * application due to some feature enhancement and new data fields. This file
 * contains the structure of the record in Calendar DB and can be used to access
 * the database directly. One way to utilize this header file is to combine it
 * with the old DateBook source code so that the record packing and
 * unpacking routines are adjusted for the new structure.
 *
 * Please note that accessing the database directly is generally not recommended
 * and this is offered as a temporary solution for 3rd party developers. The
 * structure might change again in the future.
 *
 * <hr>
 */

#ifndef __TDAPPTMGR_H__
#define __TDAPPTMGR_H__

/************************************************************
 *
 * Appointment Database constants.
 *
 *************************************************************/

/** The app info string resource ID for localized strings to replace the
 *  strings in the application info block depending on the current
 *  system locale state (e.g.: Spanish).
 */
#define LocalizedAppInfoStr         1000
#define LocalizedAppInfoColors      1000

/** max appointments displayable on a day. */
#define apptMaxPerDay       100

/** start time of an untimed appt. */
#define apptNoTime          -1

/** start and end time of a birthday slot */
#define apptBirthdayTime    -2

/** end date of appts that repeat forever */
#define apptNoEndDate       0xffff

#define apptNoAlarm         -1
#define apptDawnOfTime      0
#define apptEndOfTime       0xffffffff

#define apptMaxDisplayableAlarms    10

/** Plus one for the null terminator. */
#define apptMaxLocationLength       21

/** Datebase version that this app can support. */
#define datebookDBVersionNum        1

#ifndef appErrVersionIncompatible
#define appErrVersionIncompatible   (appErrorClass | 1)
#endif

/***********************************************************************
 *  Application Info Block Related Data
 ***********************************************************************/

/** @brief Application Info Block
 *
 * This structure is used to store info applicable to all records
 * in the database, specific to the application, inter-session (like
 * preferences), etc.
 */
typedef struct
{
    /** Bitfield of categories with a different name. */
    UInt16 renamedCategories;

    char   categoryLabels[dmRecNumCategories][dmCategoryLength];

    UInt8  categoryUniqIDs[dmRecNumCategories];

    /**
     * Unique IDs generated by the device are between 0 - 127.
     * Those from the PC are 128 - 255.
     */
    UInt8  lastUniqID;

    /** From the compiler word aligning things. */
    UInt8  reserved1;

    /**
     * Whether category colors were edited since last sync.
     * Least significant bit first.
     */
    UInt16 categoryColorsEdited;

    /**
     * Added as part of the Mullet version of this application,
     * so that we can later support color categories without
     * breaking the conduits.
     */
    UInt8  categoryColors[dmRecNumCategories];

    UInt16 reserved2;

    UInt8  startOfWeek;

    UInt8  reserved;

} ApptAppInfoType;

typedef ApptAppInfoType * ApptAppInfoPtr;

/** Used to give meaning to the AlarmInfoType advance member. */
typedef enum alarmTypes {aauMinutes, aauHours, aauDays} AlarmUnitType;

/**
 * @brief   Record alarm data
 *
 * Alarm advance - the period of time before the appointment that
 * the alarm should sound.
 */
typedef struct
{
    Int8          advance;      /**< Alarm advance (-1 = no alarm) */
    AlarmUnitType advanceUnit;  /**< minutes, hours, days */
} AlarmInfoType;

#ifndef _REPEAT_TYPE_
#define _REPEAT_TYPE_

/**
 * @brief   Repeating record frequency
 *
 * This enum is used to specify the frequency of
 * repeating appointments.
 */
enum repeatTypes {
    repeatNone,
    repeatDaily,
    repeatWeekly,
    repeatMonthlyByDay,
    repeatMonthlyByDate,
    repeatYearly
};
typedef enum repeatTypes RepeatType;


/**
 * @brief   Repeating appointment data.
 *
 * This structure contains information about repeat appointments.
 * The repeatOn member is only used by weekly and monthly-by-day
 * repeating appointments.
 *
 * For weekly the byte is a bit field that contains the days
 * of the week the for which the appointment is scheduled
 * (bit: 0-sun, 1-mon, 2-tue, etc.).
 * For monthly-by-day the byte contains the day for which
 * the appointment is scheduled, (ex: the 3rd friday), and
 * the byte is of type DayOfMonthType.
 */
typedef struct {
    RepeatType      repeatType;        /**< daily, weekly, monthlyByDay, etc. */
    UInt8           reserved1;
    DateType        repeatEndDate;     /**< minus one if forever */
    UInt8           repeatFrequency;   /**< i.e every 2 days if repeatType daily
                                            */
    UInt8           repeatOn;          /**< monthlyByDay & repeatWeekly only */
    UInt8           repeatStartOfWeek; /**< repeatWeekly only */
    UInt8           reserved2;
} RepeatInfoType;

typedef RepeatInfoType * RepeatInfoPtr;

#endif // _REPEAT_TYPE_

/**
 * @brief   Does the appointment repeat on only one day per week.
 *
 * The form (x & (x - 1)) masks out the lowest order bit
 * in x.  (K&R, p. 51)
 * If only one bit is set, which occurs iff the appointment is only
 * once per week, then (x & (x - 1)) == 0.
 *
 * @param R                 A repeat info type struct.
 *
 * @return  True if the appointment repeats only once per week.
 */
#define OnlyRepeatsOncePerWeek(R)  (((R).repeatOn & ((R).repeatOn - 1)) == 0)


/**
 * @brief   Does the appointment repeat on the given day of the week.
 *
 * @param R                 Ptr to a repeat info type struct.
 * @param D                 The day of the week to check.
 *                          For weekly repeating appointments, this
 *                          is a bit field (see RepeatInfoType).
 *                          For monthly-by-day repeating appointments,
 *                          this is a DayOfMonthType.
 *
 * @retval  True if repeat info R repeats on day of week D.
 */
#define RepeatOnDOW(R, D)   ((1 << (D)) & ((R)->repeatOn))


/***********************************************************************
 *  Application Unpacked Record Format Related Data
 ***********************************************************************/

/**
 * @brief   Appointment date and time data.
 */
typedef struct {
    TimeType            startTime;          /**< Time the appointment starts */
    TimeType            endTime;            /**< Time the appointment ends */
    DateType            date;               /**< date of appointment */
} ApptDateTimeType;

/**
 * @brief   Appointment date exception data.
 *
 * This is the structure for a repeating appointment's exceptions
 * list.  The exceptions list is a variable length list of dates
 * that the repeating appointment should not appear on.  The list
 * is in chronological order.
 */
typedef struct {
    UInt16      numExceptions;
    DateType    exception;
} ExceptionsListType;

typedef ExceptionsListType * ExceptionsListPtr;


/***********************************************************************
 *  Application Packed Record Format Related Data
 *
 *  Note: Records are stored in the database in packed format to
 *        conserve space.  When retrieving a record from the database,
 *        it is unpacked into the ApptDBRecordType.  ApptGetRecord()
 *        does the necessary unpacking for you.  When creating a new
 *        record, ApptNewRecord(), or saving a record, ApptChangeRecord(),
 *        the packing is handled for you.
 ***********************************************************************/

/**
 * @brief   Record flags
 *
 * These flags indicate if a packed record contains specific data
 * features, and other info. e.g.: if the alarm flag is set,
 * then the record has an alarm time.  They can also be used to
 * indicate what packed record data to change or retrieve.
 */
 typedef struct {
    unsigned when           :1;     /**< set if when info changed
                                         (ApptChangeRecord) */
    unsigned alarm          :1;     /**< set if record contains alarm info */
    unsigned repeat         :1;     /**< set if record contains repeat info */
    unsigned note           :1;     /**< set if record contains a note */
    unsigned exceptions     :1;     /**< set if record contains exceptions
                                         list */
    unsigned description    :1;     /**< set if record contains a description
                                         (always?) */
    unsigned location       :1;     /**< set if record contains a location */
} ApptDBRecordFlags;

/**
 * This structure describes what has been changed by an edit operation
 * (ApptChangeRecord). This is similar to ApptDBRecordFlags, but includes
 * attributes such as time zone that are stored in blobs, and therefore don't
 * have associated flags in the packed record.
 */
typedef struct {
    unsigned when           :1; /**< set if when info changed
                                     (ApptChangeRecord) */
    unsigned alarm          :1; /**< set if alarm info changed */
    unsigned repeat         :1; /**< set if repeat info changed */
    unsigned note           :1; /**< set if attached note changed */
    unsigned exceptions     :1; /**< set if exceptions list changed */
    unsigned description    :1; /**< set if description changed */
    unsigned location       :1; /**< set if location changed */
    unsigned timeZone       :1; /**< set if time zone changed */
    unsigned meeting        :1; /**< set if meeting info changed */
    unsigned blobs          :1; /**< set if meeting info changed */
} ApptDBRecordChangeFlags;

#define apptMaxTimeZoneNameLen              (UInt16)(100)

/**
 * @ brief Daylight saving time (DST) data format
 *
 * No year member, as the DST date should be consistent
 * from year to year in regards to this "1st Sunday in
 * October" format used here. DST boundary alerts are
 * always set for the current year, or to the following
 * year as needed.
 *
 */
struct DSTDataTag
{
    UInt8 hour;         /**< 1 - 24, usually 1 for 1 a.m. */
    UInt8 dayOrd;       /**< 0 = Sun, 1 = Mon, 2 = Tues, 3 = Weds, 4 = Thurs,
                             5 = Fri, 6 = Sat. */
    UInt8 weekOrd;      /**< 0 = 1st, 1 = 2nd, 2 = 3rd, 3 = 4th, 4 = Last */
    UInt8 month;        /**< 1 - 12 = Jan - Dec */
};

typedef struct DSTDataTag DSTType;

/**
 * @brief Data from the time zone blob.
 * If no time zone blob is present, name will be NULL. Typically only some
 * events will have a time zone blob. If no time zone blob is present, the event
 * is treated as being in the current device time zone.
 */
typedef struct
{
    /** Offset from UTC in minutes. */
    Int16               uTC;

    /** When daylight saving time starts (if observed). */
    DSTType             dSTStart;

    /** When daylight saving time ends (if observed). */
    DSTType             dSTEnd;

    /** 0 = DST not observed, 60 = observed. */
    Int16               dSTAdjustmentInMinutes;

    /** Which country this region is in. */
    CountryType         country;

    /** Whether this location was created by the user. */
    UInt8               custom   : 1,
                        reserved : 7;

    /** Name of location (typically city). Not a pointer! */
    Char                name[apptMaxTimeZoneNameLen + 1];

} ApptTimeZoneType;

#define apptMaxAttendees            30

/** BLOB ID definition: internal blob id will start from 'Bd00'. Creator ids
 *  from 'Bd00 - 'Bd09' is registerd for this purpose */
#define dateTimeZoneBlobId      'Bd00'      /**< For time zone information. */
#define dateMeetingBlobId       'Bd01'      /**< For meeting information. */
#define dateServerIDBlobId      'Bd02'      /**< For server ID string. */

/**
 * Possible values for the meeting status stored within the meeting blob.
 */
typedef enum
{
    unansweredMeeting,   /**< User has not accepted or declined. */
    tentativeMeeting,    /**< User has tentatively accepted. */
    declinedMeeting,     /**< User has declined. Typically not used, as
                              event is deleted. */
    acceptedMeeting,     /**< User has accepted the invitation. */
    cancelledMeeting     /**< Originator has cancelled the meeting. Typically
                              not used, as event is deleted. */
} MeetingStatusType;

/**
 * Possible values for the appointment status stored within the meeting blob. In
 * Outlook, all events have such a status, but we only support this status for
 * meetings. Could theoretically be implemented as an independent blob attached
 * to all events. */
typedef enum
{
    showAsBusy,                 /**< (default) Show as busy to strongly
                                     discourage conflicts. */
    showAsFree,                 /**< Allow others to schedule meetings that
                                     conflict with this one. */
    showAsTentative,            /**< Show as tentative to discourage conflicts.
                                     */
    showAsOutOfOffice           /**< Show as out of office to even more strongly
                                     discourage conflicts. */
} ApptStatusType;

/** Possible values for the role of a given attendee in a meeting. */
typedef enum
{
    originator,       /**< The person who set up the meeting. */
    requiredAttendee, /**< An attendee that the originator feels is required. */
    optionalAttendee  /**< An attendee that the originator feels is not
                           required. */
} AttendeeRoleType;

/**
 * As part of the meeting blob, the following information is provided for each
 * attendee.
 */
typedef struct
{
    /** Originator, required attendee, or optional attendee. */
    AttendeeRoleType        role;

    /** Full name of person, points into packed record. */
    Char *                  name;

    /** Email address of person, points into packed record.*/
    Char *                  email;

    /**
     * This field is internal to the Library and is not PACKED in the record
     * DB. It is the position of the attendee in the Attendee Array and in
     * UnPack routine Lib Calculate the position */
    Int16                   AttendeePos;

} MeetingAttendeeType;

/**
 * Data from the meeting blob. If no meeting blob is present, as is the case for
 * non-meeting events, numAttendees is zero.
 */
typedef struct
{
    /** Accepted, tentative, declined, etc. */
    MeetingStatusType       meetingStatus;

    /** Free, busy, tentative, etc. */
    ApptStatusType          apptStatus;

    /** Number of attendees. */
    UInt16                  numAttendees;

    /** Specifics for each attendee. */
    MeetingAttendeeType attendees[apptMaxAttendees];

} ApptMeetingInfo;

#define apptMaxBlobs                10

/** This is the format of each blob attached to a record. */
typedef struct
{
    UInt32                  creatorID;
    UInt16                  size;      /**< Excludes space used by blob ID and
                                            size. */
    void *                  content;   /**< Remainder of blob is content */
} BlobType;

typedef BlobType * BlobPtr;

/**
 * @brief   Record format
 *
 * This is the unpacked record format as used by the app.  All
 * pointers are either NULL or point to data within the PackedDB
 * record.  All strings are NULL character terminated.
 */
typedef struct
{
    ApptDateTimeType    *when;
    AlarmInfoType       *alarm;
    RepeatInfoType      *repeat;
    ExceptionsListType  *exceptions;
    char                *description;
    char                *note;
    char                *location;
    ApptTimeZoneType    timeZone;            /**< In a blob. */
    ApptMeetingInfo     meetingInfo;         /**< In a blob. */
    UInt16              numBlobs;            /**< Number of blobs, excluding
                                                  those for above data. */
    BlobType            blobs[apptMaxBlobs]; /**< Pointers into packed record.
                                                  */
} ApptDBRecordType;

typedef ApptDBRecordType * ApptDBRecordPtr;

typedef struct
{
    /** Portion within day after adjusting for time zone. */
    TimeType        startTime;

    /** Portion within day after adjusting for time zone. */
    TimeType        endTime;

    UInt16          recordNum;

    /** Spans midnight at start of day. */
    UInt16          startedTheDayBefore:        1;

    /** Spans or ends at midnight end of day. */
    UInt16          continuesOnNextDay:         1;

    /** Part of a daily repeating event. */
    UInt16          repeatsDaily:               1;

    /** Part or first occurrence of repeating event. */
    UInt16          isFirstOccurrence:          1;

    /** Part of last occurrence of repeating event. */
    UInt16          isLastOccurrence:           1;

    /** Pad out to 8 bytes total for fast indexing. */
    UInt16          padding:                    11;

    /** Since 1904, may not be within this day. */
    UInt32          startSeconds;

    /** Ditto */
    UInt32          endSeconds;

    /** UId of the Record */
    UInt32          UId;

} ApptInfoType;
typedef ApptInfoType * ApptInfoPtr;


/************************************************************
 *
 * Appointment database routines.
 *
 *************************************************************/

#ifdef __cplusplus
extern "C" {
#endif


void ECApptDBValidate (DmOpenRef dbP);

/**
 * @brief Create an app info chunk if missing and initialize with defaults.
 *
 * @param dbP               Open database ptr.
 * @return                  0 if successful, error code if not.
 */
extern Err ApptAppInfoInit (DmOpenRef dbP);

/**
 * @brief   Get a record's data from the database and lock the record.
 *
 * @param dbP               Open database ptr.
 * @param index             Index of database record to retrieve.
 * @param r                 Ptr to a record to initialize with the retrieved
 *                              data.
 * @param handleP           Ptr to a handle var to set with the locked record's
 *                          handle for unlocking when finished with the record.
 *
 * @retval                  0 if successful, error code if not.
 */
extern Err ApptGetRecord (DmOpenRef dbP, UInt16 index, ApptDBRecordPtr r,
                          MemHandle * handleP);
/**
 * @brief   Create a new packed record in sorted position in the database.
 *
 * @param dbP               Open database ptr.
 * @param r                 Ptr to a record to copy into the DB.
 * @param category          Category to set the record with.
 * @param secret            True if the record should be marked secret/private.
 * @param index             Ptr to index var to set with the new record's index.
 *
 * @return                  0 if successful, error code if not.
 */
extern Err ApptNewRecord (DmOpenRef dbP, ApptDBRecordPtr r, UInt16 category,
                          Boolean secret, UInt16 *index);

/**
 * @brief   Change a record in the database and reposition it to sorted position
 *          if necessary.
 *
 * @param dbP               Open database ptr.
 * @param index             Ptr to the database index value of the record to
 *                              change.  The value is set if the record is
 *                              moved from sorting.
 * @param r                 Ptr to a record having the changes to save.
 * @param changedFields     The fields to change.
 *
 * @return                  0 if successful, error code if not.
 */
extern Err ApptChangeRecord (DmOpenRef dbP, UInt16 *index, ApptDBRecordPtr r,
                             ApptDBRecordChangeFlags changedFields);

/**
 * @brief   Add an entry to the exceptions list of the specified record.
 *
 * @param dbP               Open database ptr.
 * @param index             Ptr to index var having the index of the record
 *                              to change and to be set with the record's new
 *                              index should it's position be moved due to
 *                              sorting.
 * @param date              The exception date to add.
 *
 * @return                  0 if successful, error code if not.
 */
extern Err ApptAddException (DmOpenRef dbP, UInt16 *index, DateType date);

/**
 * @brief   Find the first appointment record on the given day.
 *
 * @param dbP               Open database ptr.
 * @param date              The date to search for.
 * @param index             Ptr to index var to be set with the found record's
 *                              index.
 * @param repeat            True to find the first repeating event that
 *                              ends on or after the given day, False to find
 *                              the first non-repeating event that starts on or
 *                              after the given day.
 *
 * @return                  True if a record was found.
 */
extern Boolean ApptFindFirst (DmOpenRef dbP, DateType date, UInt16 * indexP,
                              Boolean repeat);

/**
 * @brief   Does a repeating appointment occur on the give day.
 *
 * @param apptRec           The appointment record to check.
 * @param date              The date to check.
 *
 * @return                  True if the repeating appointment occurs on the
 *                              given day.
 */
extern Boolean ApptRepeatsOnDate  (ApptDBRecordPtr apptRec, DateType date);

/**
 * @brief   Get the occurrence of a repeating appointment closest to the
 *          given date, depending on the direction to search.
 *
 * @param apptRec           The appointment record to check.
 * @param dateP             Ptr to the date var to set with the next occurance
 *                              date of the given repeating appointment.
 * @param searchForward     True if the occurance should be sought forward from
 *                              the given date; false if it should be sought
 *                              backward.
 *
 * @retval                  True if there is an occurrence of the appointment
 *                              between the date passed and the appointment's
 *                              end date (if searching forward) or start date
 *                              (if searching backwards).
 */
extern Boolean ApptNextRepeat (ApptDBRecordPtr apptRec, DatePtr dateP,
                               Boolean searchForward);

/**
 * @brief   Does the given appointment occur more than once.
 *
 * This function compares the repeat info and the exception list for
 * an appointment to determine if it has more than one visible (non-excepted)
 * occurence.
 * The decision is based solely on the number of times the appointment
 * repeats versus the number of exceptions.
 *
 * @param apptRec           The appointment record to check.
 *
 * @return                  True if the appointment occurs more than once.
 */
extern Boolean ApptHasMultipleOccurences (ApptDBRecordPtr apptRec);

/**
 * @brief   Post all appointments that have an alarm matching the specified
 *          time to the attention manager.
 *
 * @param inDbR             Open database ptr.
 * @param inAlarmTime       The time in seconds of those alarms to post.
 *
 * @return                  Nothing.
 */
extern void ApptPostTriggeredAlarms (DmOpenRef inDbR, UInt32 inAlarmTime);

/**
 * @brief   Sort the given database.
 *
 * @param dbP               Open database ptr.
 * @return                  Nothing.
 */
extern void ApptSort (DmOpenRef dbP);

/**
 * @brief   Get a list of appointments in the given date range.
 *
 * Events that span midnight and therefore end up partially within
 * this date range are included even if they start on the day
 * before the give date.
 *
 * @param dbP               Open database ptr.
 * @param date              Start date from which to search.
 * @param days              Number a days in search range.
 * @param apptLists         returned: array of handles of the
 *                          appointment list (ApptInfoType).
 * @param counts            returned: array of counts of the
 *                          number of appointments in each list.
 * @param category          The category to seach within.
 *
 * @return                  Nothing.
 */
extern void ApptGetAppointments (DmOpenRef dbP, DateType date, UInt16 days,
                                 MemHandle apptLists [], UInt16 counts [],
                                 UInt16 category);

#ifdef __cplusplus
}
#endif

#endif


