/******************************************************************************
 * Copyright (c) 2005 palmOne, Inc. or its subsidiaries.
 * All rights reserved.
 *****************************************************************************/
/** 
 *@ingroup System
 *
 */
  
/**
 * @file 	palmOneNavigator.h
 * @version 1.0
 * @date 	03/03/2004
 *
 * @brief Contains 5-way navigator-specific definitions.
 *
 * 	  Five-way navigation macros and values for key states are included
 * 	  in this file.
 *
 * <hr>
 */

#ifndef __PALMONENAVIGATOR_H__
#define __PALMONENAVIGATOR_H__

// palmOne includes:
#include <Common/System/palmOneChars.h>

// Handspring includes:
#include <Common/System/HsNavCommon.h>

/**
 * @name 
 *
 */
/*@{*/
#define navFtrCreator           	'fway'		/**< Creator ID for 5-Way navigator. */
#define navFtrVersion           	0		/**< Feature id for 5-Way. */
#define navVersion              	0x00010000	/**< Version for 5-Way. */
/*@}*/

/**
 * @name 
 *
 */
/*@{*/
#define keyBitNavLeft			0x01000000	/**< Key state mask to check the five way navigation LEFT button. */
#define keyBitNavRight			0x02000000	/**< Key state mask to check the five way navigation RIGHT button. */
#define keyBitNavSelect			0x04000000	/**< Key state mask to check the five way navigation SELECT button. */
#define keyBitNavLRS			0x07000000	/**< Key state mask to check the five way navigation LEFT, RIGHT and SELECT buttons. */
/*@}*/

/**
 * @name 
 *
 */
/*@{*/
#define navBitUp			0x0001		/**< Key mask for the five way navigation UP button. */
#define navBitDown			0x0002		/**< Key mask for the five way navigation DOWN button. */
#define navBitLeft			0x0004		/**< Key mask for the five way navigation LEFT button. */
#define navBitRight			0x0008		/**< Key mask for the five way navigation RIGHT button. */
#define navBitSelect			0x0010		/**< Key mask for the five way navigation SELECT button. */
#define navBitsAll			0x001F		/**< Key mask for all five way navigation buttons. */
/*@}*/

/**
 * @name 
 *
 */
/*@{*/
#define navChangeUp			0x0100		/**< Key mask for the five way navigation UP button state change. */
#define navChangeDown			0x0200		/**< Key mask for the five way navigation DOWN button state change. */
#define navChangeLeft			0x0400		/**< Key mask for the five way navigation LEFT button state change. */
#define navChangeRight			0x0800		/**< Key mask for the five way navigation RIGHT button state change. */
#define navChangeSelect			0x1000		/**< Key mask for the five way navigation SELECT button state change. */
#define navChangeBitsAll		0x1F00		/**< Key mask for all five way navigation buttons state change. */
/*@}*/

/**
 * @def IsFiveWayNavPalmEvent(eventP)
 *
 * A macro to use for apps that support navigation using the Palm style 5-way.
 * This macro will let you test whether an event was generated by the 5-way.
 * For applications that treat up and down hard buttons differently than up and
 * down on the 5-way, this macro can be used to determine which action to take
 * if NavKeyPressed returns true for Up or Down.
 *
 * @remark
 * Our 5-way events generate lots of detail about what state changed and what
 * didn't. Events are generated even when a button is released, which is a
 * nice bonus. So, to figure out what happened, you have to look at the bits
 * in the keyCode. Because we only use 10 of the bits in the keyCode, and
 * because we may use the remaining bits at some later date, we
 * mask out these 10 bits before doing the test.
 *
 *  ((eventP)->data.keyDown.keyCode & (navBitsAll | navChangeBitsAll)) != 0
 * will be true if any of the 5 buttons is held down, pressed, or released.
 *
 *  ((eventP)->data.keyDown.keyCode & navChangeBitsAll) != 0
 * will be true if any of the 5 buttons is pressed or released.
 *
 * The situation where these would differ is holding down one or more of the
 * 5 buttons until it generates a repeat event. In that case, none of the
 * navChangeBits will be set.
 *
 * Unfortunately, on the Treo 600, the vchrPageUp/Down is accompanied
 * by a keyCode that has some nav bits set.  Thus, we will not be able to
 * distinguish the HandSpring nav event from the Palm nav event here in such
 * cases, and this macro will return true.  Then, when using NavDirectionPressed,
 * via NavKeyPressed e.g., the nav direction palm and HS macros will return
 * false, as this case matches neither, and the vchrPageUp/Down will incorrectly
 * be returned as not a 5-way nav event for our purposes of supporting non-5-way
 * devices.
 * That can be avoided by ensuring the focus state on HandSpring devices is
 * set to object mode for the given form, thus ensuring vchrRockerUp/Down chrs are
 * sent instead of vcrhPageUp/Down chrs, while also ensuring this continues
 * to work as expected on palmOne devices.
 * 
 * @sample
 * @code if (IsFiveWayNavPalmEvent(eventP)) {
 *		// Handle five way events here
 * } @endcode
 */
#define IsFiveWayNavPalmEvent(eventP)                                                       \
(                                                                                           \
    (   ((eventP)->data.keyDown.chr == vchrPageUp)                                          \
     || ((eventP)->data.keyDown.chr == vchrPageDown)                                        \
     || ((eventP)->data.keyDown.chr == vchrNavChange) )                                     \
&&  (((eventP)->data.keyDown.keyCode & (navBitsAll | navChangeBitsAll)) != 0)               \
)

/*
#define IsFiveWayNavPalmEvent(eventP)                                                       \
(   ((eventP)->data.keyDown.chr == vchrNavChange)                                           \
||                                                                                          \
    (   (   ((eventP)->data.keyDown.chr == vchrPageUp)                                      \
         || ((eventP)->data.keyDown.chr == vchrPageDown) )                                  \
     && (   (((eventP)->data.keyDown.keyCode & navChangeBitsAll) != 0)                      \
         || (((eventP)->data.keyDown.modifiers & autoRepeatKeyMask) != 0)))                 \
)
*/

/*
#define IsFiveWayNavPalmEvent(eventP)                                                       \
(   ((eventP)->data.keyDown.chr == vchrNavChange)                                           \
||  (   ((eventP)->data.keyDown.chr == vchrPageUp)                                          \
     && (((eventP)->data.keyDown.keyCode & (navBitUp | navChangeUp)) != 0) )                \
||  (   ((eventP)->data.keyDown.chr == vchrPageDown)                                        \
     && (((eventP)->data.keyDown.keyCode & (navBitDown | navChangeDown)) != 0) )            \
)
*/

/**
 * @def IsFiveWayNavEvent(eventP)
 *
 * A macro to use for apps that support navigation using the 5-way.
 * This macro will let you test whether an event was generated by the 5-way.
 * For applications that treat up and down hard buttons differently than up and
 * down on the 5-way, this macro can be used to determine which action to take
 * if NavKeyPressed returns true for Up or Down.
 * 
 * @sample
 * @code if (IsFiveWayNavEvent(eventP)) {
 *		// Handle five way events here
 * } @endcode
 */
#define IsFiveWayNavEvent(eventP)														    \
(                                                                                           \
    IsFiveWayNavPalmEvent(eventP)                                                           \
||  TxtCharIsRockerKey((eventP)->data.keyDown.modifiers, (eventP)->data.keyDown.chr)        \
)


/**
 * A macro to use for apps that support navigation using the HandSpring 5-way.
 * By using this macro, we have consistent behavior in all our apps
 * when it comes to how it handles multiple simultaneous button presses.
 * 
 * @remark
 * Checks if it the key down event is of the HandSpring "select" key type.
 *
 * @sample
 * @code if (NavSelectHSPressed(eventP)) {
 *		// Select was pressed
 * } @endcode
 * 
 */
#define NavSelectHSPressed(eventP)														\
(																						\
       (((eventP)->data.keyDown.modifiers & commandKeyMask) != 0)                       \
    && ((eventP)->data.keyDown.chr == vchrRockerCenter)                                 \
)


/**
 * A macro to use for apps that support navigation using the Palm 5-way.
 * By using this macro, we have consistent behavior in all our apps
 * when it comes to how it handles multiple simultaneous button presses.
 * 
 * @remark
 * Checks if it the key down event is of the Palm "select" key type.
 * Note the redundant check of the keyDown.chr.  This is necessary to
 * prevent a false positive, because the modified DAL code for the
 * new 5-way still generates the same keyCodes (it was easier that way),
 * but doesn't generate the same keyDown.chr (new vchrRocker instead).
 * This redundancy avoids a double-press behavior.
 *
 * @sample
 * @code if (NavSelectPalmPressed(eventP)) {
 *		// Select was pressed
 * } @endcode
 * 
 */
#define NavSelectPalmPressed(eventP)													\
(																						\
       (((eventP)->data.keyDown.modifiers & autoRepeatKeyMask) == 0)					\
    && ((eventP)->data.keyDown.chr == vchrNavChange)                                    \
    && (((eventP)->data.keyDown.keyCode & (navBitsAll | navChangeBitsAll)) ==			\
																	 navChangeSelect)   \
)


/**
 * A macro to use for apps that support navigation using the 5-way.
 * By using this macro, we have consistent behavior in all our apps
 * when it comes to how it handles multiple simultaneous button presses.
 * 
 * @remark
 * Only act when the select button is released, and only if none of the
 * direction buttons are down (or being released) at the time, and only if the select
 * button has not been held down long enough to start repeating. By ignoring repeat
 * events for the select button, we ensure that the launcher will be run if the select
 * button is held down. By waiting until the button is released, we ensure that the
 * select action is not taken before the launcher is run.
 *
 * @sample
 * @code if (NavSelectPressed(eventP)) {
 *		// Select was pressed
 * } @endcode
 * 
 */
#define NavSelectPressed(eventP)														\
(																						\
	IsFiveWayNavEvent(eventP)															\
&&  (   NavSelectPalmPressed(eventP)                                                    \
     || NavSelectHSPressed(eventP))                                                     \
)


/**
 * A macro to use for apps that support navigation using the HandSpring 5-way.
 * By using this macro, we have consistent behavior in all our apps
 * when it comes to how it handles multiple simultaneous button presses.
 * You can use this macro even if the device does not have a 5-way controller.
 * 
 * @remark
 * Determine if the key down event indicates the HandSpring nav direction type.
 *
 * @sample
 * @code
 * if (NavDirectionHSPressed(eventP, Left))
 * if (NavDirectionHSPressed(eventP, Right))
 * if (NavDirectionHSPressed(eventP, Up))       - also works without 5-way
 * if (NavDirectionHSPressed(eventP, Down))     - also works without 5-way @endcode
 *
 */
#define NavDirectionHSPressed(eventP, nav)                                              \
(                                                                                       \
        ((vchrRocker ## nav) != vchrRockerCenter)                                       \
     && (   (   (((eventP)->data.keyDown.modifiers & commandKeyMask) != 0)              \
             && ((eventP)->data.keyDown.chr == (vchrRocker ## nav))))                   \
)


/**
 * A macro to use for apps that support navigation using the Palm 5-way.
 * By using this macro, we have consistent behavior in all our apps
 * when it comes to how it handles multiple simultaneous button presses.
 * You can use this macro even if the device does not have a 5-way controller.
 * 
 * @remark
 * Determine if the key down event indicates the palmOne nav direction type.
 *
 * @sample
 * @code
 * if (NavDirectionPalmPressed(eventP, Left))
 * if (NavDirectionPalmPressed(eventP, Right))
 * if (NavDirectionPalmPressed(eventP, Up))     - also works without 5-way
 * if (NavDirectionPalmPressed(eventP, Down))   - also works without 5-way @endcode
 *
 */
#define NavDirectionPalmPressed(eventP, nav)                                            \
(                                                                                       \
    ((eventP)->data.keyDown.modifiers & autoRepeatKeyMask)                              \
        ? (((eventP)->data.keyDown.keyCode & (navBitsAll | navChangeBitsAll)) ==        \
            (navBit ## nav))                                                            \
        : (((eventP)->data.keyDown.keyCode & (navBitsAll | navChangeBitsAll)) ==        \
            (navBit ## nav | navChange ## nav))                                         \
)


/**
 * A macro to use for apps that support navigation using the 5-way.
 * By using this macro, we have consistent behavior in all our apps
 * when it comes to how it handles multiple simultaneous button presses.
 * You can use this macro even if the device does not have a 5-way controller.
 * 
 * @remark
 * Act only when one direction is pressed without any other direction (or select) being
 * down at the time. Repeat if the button is held down, but not if other buttons are
 * pressed.
 *
 * @sample
 * @code
 * if (NavDirectionPressed(eventP, Left))
 * if (NavDirectionPressed(eventP, Right))
 * if (NavDirectionPressed(eventP, Up))		- also works without 5-way
 * if (NavDirectionPressed(eventP, Down))	- also works without 5-way @endcode
 *
 */
#define NavDirectionPressed(eventP, nav)                                                \
(                                                                                       \
    IsFiveWayNavEvent(eventP)                                                           \
        ? (   NavDirectionPalmPressed(eventP, nav)                                      \
           || NavDirectionHSPressed(eventP, nav))                                       \
        : (   ((eventP)->data.keyDown.chr == vchrPageUp && (navBit ## nav) == navBitUp)      \
	 	   || ((eventP)->data.keyDown.chr == vchrPageDown && (navBit ## nav) == navBitDown)) \
)


/**
 * A macro to use for apps that support navigation using the 5-way.
 * By using this macro, we have consistent behavior in all our apps
 * when it comes to how it handles multiple simultaneous button presses.
 * You can use this macro even if the device does not have a 5-way controller.
 * 
 * @sample
 * @code
 * if (NavKeyPressed(eventP, Select))
 * if (NavKeyPressed(eventP, Left))
 * if (NavKeyPressed(eventP, Right))
 * if (NavKeyPressed(eventP, Up))		- also works without 5-way
 * if (NavKeyPressed(eventP, Down))		- also works without 5-way @endcode
 */
#define NavKeyPressed(eventP, nav)                                                      \
(                                                                                       \
    (navBit ## nav == navBitSelect)                                                     \
        ? NavSelectPressed(eventP)                                                      \
        : NavDirectionPressed(eventP, nav)                                              \
)

#endif // __PALMONENAVIGATOR_H__
