/*
 * Decompiled with CFR 0.152.
 */
package de.matthiasmann.twl.utils;

import java.util.Arrays;

public class SparseGrid {
    Node root;
    int numLevels;

    public SparseGrid(int pageSize) {
        this.root = new Node(pageSize);
        this.numLevels = 1;
    }

    public Entry get(int row, int column) {
        if (this.root.size > 0) {
            int levels = this.numLevels;
            Entry e = this.root;
            do {
                Node node;
                int pos;
                if ((pos = (node = e).findPos(row, column, node.size)) == node.size) {
                    return null;
                }
                e = node.children[pos];
            } while (--levels > 0);
            assert (e != null);
            if (e.compare(row, column) == 0) {
                return e;
            }
        }
        return null;
    }

    public void set(int row, int column, Entry entry) {
        entry.row = row;
        entry.column = column;
        if (this.root.size == 0) {
            this.root.insertAt(0, entry);
            this.root.updateRowColumn();
        } else if (!this.root.insert(entry, this.numLevels)) {
            this.splitRoot();
            this.root.insert(entry, this.numLevels);
        }
    }

    public Entry remove(int row, int column) {
        if (this.root.size == 0) {
            return null;
        }
        Entry e = this.root.remove(row, column, this.numLevels);
        if (e != null) {
            this.maybeRemoveRoot();
        }
        return e;
    }

    public void insertRows(int row, int count) {
        if (count > 0 && this.root.size > 0) {
            this.root.insertRows(row, count, this.numLevels);
        }
    }

    public void insertColumns(int column, int count) {
        if (count > 0 && this.root.size > 0) {
            this.root.insertColumns(column, count, this.numLevels);
        }
    }

    public void removeRows(int row, int count) {
        if (count > 0) {
            this.root.removeRows(row, count, this.numLevels);
            this.maybeRemoveRoot();
        }
    }

    public void removeColumns(int column, int count) {
        if (count > 0) {
            this.root.removeColumns(column, count, this.numLevels);
            this.maybeRemoveRoot();
        }
    }

    public void iterate(int startRow, int startColumn, int endRow, int endColumn, GridFunction func) {
        if (this.root.size > 0) {
            int pos;
            Node node;
            int levels = this.numLevels;
            Entry e = this.root;
            do {
                node = e;
                pos = node.findPos(startRow, startColumn, node.size - 1);
                e = node.children[pos];
            } while (--levels > 0);
            assert (e != null);
            if (e.compare(startRow, startColumn) < 0) {
                return;
            }
            do {
                int size = node.size;
                while (pos < size) {
                    e = node.children[pos];
                    if (e.row > endRow) {
                        return;
                    }
                    if (e.column >= startColumn && e.column <= endColumn) {
                        func.apply(e.row, e.column, e);
                    }
                    ++pos;
                }
                pos = 0;
            } while ((node = node.next) != null);
        }
    }

    public boolean isEmpty() {
        return this.root.size == 0;
    }

    public void clear() {
        Arrays.fill(this.root.children, null);
        this.root.size = 0;
        this.numLevels = 1;
    }

    private void maybeRemoveRoot() {
        while (this.numLevels > 1 && this.root.size == 1) {
            this.root = (Node)this.root.children[0];
            this.root.prev = null;
            this.root.next = null;
            --this.numLevels;
        }
        if (this.root.size == 0) {
            this.numLevels = 1;
        }
    }

    private void splitRoot() {
        Node newNode = this.root.split();
        Node newRoot = new Node(this.root.children.length);
        newRoot.children[0] = this.root;
        newRoot.children[1] = newNode;
        newRoot.size = 2;
        this.root = newRoot;
        ++this.numLevels;
    }

    public static class Entry {
        int row;
        int column;

        int compare(int row, int column) {
            int diff = this.row - row;
            if (diff == 0) {
                diff = this.column - column;
            }
            return diff;
        }
    }

    static class Node
    extends Entry {
        final Entry[] children;
        int size;
        Node next;
        Node prev;

        public Node(int size) {
            this.children = new Entry[size];
        }

        boolean insert(Entry e, int levels) {
            if (--levels == 0) {
                return this.insertLeaf(e);
            }
            while (true) {
                int pos = this.findPos(e.row, e.column, this.size - 1);
                assert (pos < this.size);
                Node node = (Node)this.children[pos];
                if (node.insert(e, levels)) break;
                if (this.isFull()) {
                    return false;
                }
                Node node2 = node.split();
                this.insertAt(pos + 1, node2);
            }
            this.updateRowColumn();
            return true;
        }

        boolean insertLeaf(Entry e) {
            int pos = this.findPos(e.row, e.column, this.size);
            if (pos < this.size) {
                Entry c = this.children[pos];
                assert (c.getClass() != Node.class);
                int cmp = c.compare(e.row, e.column);
                if (cmp == 0) {
                    this.children[pos] = e;
                    return true;
                }
                assert (cmp > 0);
            }
            if (this.isFull()) {
                return false;
            }
            this.insertAt(pos, e);
            return true;
        }

        Entry remove(int row, int column, int levels) {
            if (--levels == 0) {
                return this.removeLeaf(row, column);
            }
            int pos = this.findPos(row, column, this.size - 1);
            assert (pos < this.size);
            Node node = (Node)this.children[pos];
            Entry e = node.remove(row, column, levels);
            if (e != null) {
                if (node.size == 0) {
                    this.removeNodeAt(pos);
                } else if (node.isBelowHalf()) {
                    this.tryMerge(pos);
                }
                this.updateRowColumn();
            }
            return e;
        }

        Entry removeLeaf(int row, int column) {
            int pos = this.findPos(row, column, this.size);
            if (pos == this.size) {
                return null;
            }
            Entry c = this.children[pos];
            assert (c.getClass() != Node.class);
            int cmp = c.compare(row, column);
            if (cmp == 0) {
                this.removeAt(pos);
                if (pos == this.size && this.size > 0) {
                    this.updateRowColumn();
                }
                return c;
            }
            return null;
        }

        int findPos(int row, int column, int high) {
            int low = 0;
            while (low < high) {
                int mid = low + high >>> 1;
                Entry e = this.children[mid];
                int cmp = e.compare(row, column);
                if (cmp > 0) {
                    high = mid;
                    continue;
                }
                if (cmp < 0) {
                    low = mid + 1;
                    continue;
                }
                return mid;
            }
            return low;
        }

        void insertRows(int row, int count, int levels) {
            if (--levels > 0) {
                int i = this.size;
                while (i-- > 0) {
                    Node n = (Node)this.children[i];
                    if (n.row >= row) {
                        n.insertRows(row, count, levels);
                        continue;
                    }
                    break;
                }
            } else {
                int i = this.size;
                while (i-- > 0) {
                    Entry e = this.children[i];
                    if (e.row >= row) {
                        e.row += count;
                        continue;
                    }
                    break;
                }
            }
            this.updateRowColumn();
        }

        void insertColumns(int column, int count, int levels) {
            if (--levels > 0) {
                for (int i = 0; i < this.size; ++i) {
                    Node n = (Node)this.children[i];
                    n.insertColumns(column, count, levels);
                }
            } else {
                for (int i = 0; i < this.size; ++i) {
                    Entry e = this.children[i];
                    if (e.column < column) continue;
                    e.column += count;
                }
            }
            this.updateRowColumn();
        }

        boolean removeRows(int row, int count, int levels) {
            if (--levels > 0) {
                boolean needsMerging = false;
                int i = this.size;
                while (i-- > 0) {
                    Node n = (Node)this.children[i];
                    if (n.row < row) break;
                    if (n.removeRows(row, count, levels)) {
                        this.removeNodeAt(i);
                        continue;
                    }
                    needsMerging |= n.isBelowHalf();
                }
                if (needsMerging && this.size > 1) {
                    this.tryMerge();
                }
            } else {
                int i = this.size;
                while (i-- > 0) {
                    Entry e = this.children[i];
                    if (e.row < row) break;
                    e.row -= count;
                    if (e.row >= row) continue;
                    this.removeAt(i);
                }
            }
            if (this.size == 0) {
                return true;
            }
            this.updateRowColumn();
            return false;
        }

        boolean removeColumns(int column, int count, int levels) {
            if (--levels > 0) {
                boolean needsMerging = false;
                int i = this.size;
                while (i-- > 0) {
                    Node n = (Node)this.children[i];
                    if (n.removeColumns(column, count, levels)) {
                        this.removeNodeAt(i);
                        continue;
                    }
                    needsMerging |= n.isBelowHalf();
                }
                if (needsMerging && this.size > 1) {
                    this.tryMerge();
                }
            } else {
                int i = this.size;
                while (i-- > 0) {
                    Entry e = this.children[i];
                    if (e.column < column) continue;
                    e.column -= count;
                    if (e.column >= column) continue;
                    this.removeAt(i);
                }
            }
            if (this.size == 0) {
                return true;
            }
            this.updateRowColumn();
            return false;
        }

        void insertAt(int idx, Entry what) {
            System.arraycopy(this.children, idx, this.children, idx + 1, this.size - idx);
            this.children[idx] = what;
            if (idx == this.size++) {
                this.updateRowColumn();
            }
        }

        void removeAt(int idx) {
            --this.size;
            System.arraycopy(this.children, idx + 1, this.children, idx, this.size - idx);
            this.children[this.size] = null;
        }

        void removeNodeAt(int idx) {
            Node n = (Node)this.children[idx];
            if (n.next != null) {
                n.next.prev = n.prev;
            }
            if (n.prev != null) {
                n.prev.next = n.next;
            }
            n.next = null;
            n.prev = null;
            this.removeAt(idx);
        }

        void tryMerge() {
            if (this.size == 2) {
                this.tryMerge2(0);
            } else {
                int i = this.size - 1;
                while (i-- > 1) {
                    if (!this.tryMerge3(i)) continue;
                    --i;
                }
            }
        }

        void tryMerge(int pos) {
            switch (this.size) {
                case 0: 
                case 1: {
                    break;
                }
                case 2: {
                    this.tryMerge2(0);
                    break;
                }
                default: {
                    if (pos + 1 == this.size) {
                        this.tryMerge3(pos - 1);
                        break;
                    }
                    if (pos == 0) {
                        this.tryMerge3(1);
                        break;
                    }
                    this.tryMerge3(pos);
                }
            }
        }

        private void tryMerge2(int pos) {
            Node n1 = (Node)this.children[pos];
            Node n2 = (Node)this.children[pos + 1];
            if (n1.isBelowHalf() || n2.isBelowHalf()) {
                int sumSize = n1.size + n2.size;
                if (sumSize < this.children.length) {
                    System.arraycopy(n2.children, 0, n1.children, n1.size, n2.size);
                    n1.size = sumSize;
                    n1.updateRowColumn();
                    this.removeNodeAt(pos + 1);
                } else {
                    Object[] temp = this.collect2(sumSize, n1, n2);
                    this.distribute2(temp, n1, n2);
                }
            }
        }

        private boolean tryMerge3(int pos) {
            Node n0 = (Node)this.children[pos - 1];
            Node n1 = (Node)this.children[pos];
            Node n2 = (Node)this.children[pos + 1];
            if (n0.isBelowHalf() || n1.isBelowHalf() || n2.isBelowHalf()) {
                int sumSize = n0.size + n1.size + n2.size;
                if (sumSize < this.children.length) {
                    System.arraycopy(n1.children, 0, n0.children, n0.size, n1.size);
                    System.arraycopy(n2.children, 0, n0.children, n0.size + n1.size, n2.size);
                    n0.size = sumSize;
                    n0.updateRowColumn();
                    this.removeNodeAt(pos + 1);
                    this.removeNodeAt(pos);
                    return true;
                }
                Object[] temp = this.collect3(sumSize, n0, n1, n2);
                if (sumSize < 2 * this.children.length) {
                    this.distribute2(temp, n0, n1);
                    this.removeNodeAt(pos + 1);
                } else {
                    this.distribute3(temp, n0, n1, n2);
                }
            }
            return false;
        }

        private Object[] collect2(int sumSize, Node n0, Node n1) {
            Object[] temp = new Object[sumSize];
            System.arraycopy(n0.children, 0, temp, 0, n0.size);
            System.arraycopy(n1.children, 0, temp, n0.size, n1.size);
            return temp;
        }

        private Object[] collect3(int sumSize, Node n0, Node n1, Node n2) {
            Object[] temp = new Object[sumSize];
            System.arraycopy(n0.children, 0, temp, 0, n0.size);
            System.arraycopy(n1.children, 0, temp, n0.size, n1.size);
            System.arraycopy(n2.children, 0, temp, n0.size + n1.size, n2.size);
            return temp;
        }

        private void distribute2(Object[] src, Node n0, Node n1) {
            int sumSize = src.length;
            n0.size = sumSize / 2;
            n1.size = sumSize - n0.size;
            System.arraycopy(src, 0, n0.children, 0, n0.size);
            System.arraycopy(src, n0.size, n1.children, 0, n1.size);
            n0.updateRowColumn();
            n1.updateRowColumn();
        }

        private void distribute3(Object[] src, Node n0, Node n1, Node n2) {
            int sumSize = src.length;
            n0.size = sumSize / 3;
            n1.size = (sumSize - n0.size) / 2;
            n2.size = sumSize - (n0.size + n1.size);
            System.arraycopy(src, 0, n0.children, 0, n0.size);
            System.arraycopy(src, n0.size, n1.children, 0, n1.size);
            System.arraycopy(src, n0.size + n1.size, n2.children, 0, n2.size);
            n0.updateRowColumn();
            n1.updateRowColumn();
            n2.updateRowColumn();
        }

        boolean isFull() {
            return this.size == this.children.length;
        }

        boolean isBelowHalf() {
            return this.size * 2 < this.children.length;
        }

        Node split() {
            Node newNode = new Node(this.children.length);
            int size1 = this.size / 2;
            int size2 = this.size - size1;
            System.arraycopy(this.children, size1, newNode.children, 0, size2);
            Arrays.fill(this.children, size1, this.size, null);
            newNode.size = size2;
            newNode.updateRowColumn();
            newNode.prev = this;
            newNode.next = this.next;
            this.size = size1;
            this.updateRowColumn();
            this.next = newNode;
            if (newNode.next != null) {
                newNode.next.prev = newNode;
            }
            return newNode;
        }

        void updateRowColumn() {
            Entry e = this.children[this.size - 1];
            this.row = e.row;
            this.column = e.column;
        }
    }

    public static interface GridFunction {
        public void apply(int var1, int var2, Entry var3);
    }
}

