/*
 * PTUI stands for "Portable Terminal UI". It is an ANSI C library that
 * provides simple terminal-handling routines that can operate on Linux,
 * Windows and DOS.
 *
 * Copyright (C) 2013-2020 Mateusz Viste
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */


#define _XOPEN_SOURCE_EXTENDED

#include <locale.h>
#include <ncursesw/curses.h>
#include <stdio.h> /* this one contains the NULL definition */
#include <string.h>

#include "ptui.h"  /* include self for control */


/* mouse-related global variables */
static int lastclick_btn = -1;
static unsigned short lastclick_x, lastclick_y;


static attr_t getorcreatecolor(int col) {
  static attr_t DOSPALETTE[256] = {0};
  static int lastcolid = 0;
  /* if color doesn't exist yet, create it */
  if (DOSPALETTE[col] == 0) {
    unsigned long DOSCOLORS[16] = { COLOR_BLACK, COLOR_BLUE, COLOR_GREEN, COLOR_CYAN, COLOR_RED, COLOR_MAGENTA, COLOR_YELLOW, COLOR_WHITE, COLOR_BLACK, COLOR_BLUE, COLOR_GREEN, COLOR_CYAN, COLOR_RED,   COLOR_MAGENTA, COLOR_YELLOW, COLOR_WHITE };
    if (col & 0x80) {         /* bright background */
      init_pair(++lastcolid, DOSCOLORS[col >> 4], DOSCOLORS[col & 0xf]);
      DOSPALETTE[col] = COLOR_PAIR(lastcolid) | WA_BOLD | WA_REVERSE;
    } else if (col & 0x08) {   /* bright foreground */
      init_pair(++lastcolid, DOSCOLORS[col & 0x0f], DOSCOLORS[col >> 4]);
      DOSPALETTE[col] = COLOR_PAIR(lastcolid) | A_BOLD;
    } else {                  /* no bright nothing */
      init_pair(++lastcolid, DOSCOLORS[col & 0x0f], DOSCOLORS[col >> 4]);
      /* init_pair(col+1, COLOR_BLUE, COLOR_MAGENTA); */
      DOSPALETTE[col] = COLOR_PAIR(lastcolid) | A_NORMAL;
    }
  }

  return(DOSPALETTE[col]);
}


/* returns 0 on monochrome terminals, 1 on color terminals */
int ptui_hascolor(void) {
  if (has_colors() == TRUE) return(1);
  return(0);
}


/* inits the UI subsystem */
int ptui_init(int flags) {
  setlocale(LC_ALL, "");
  if (initscr() == NULL) return(-1); /* returns a ptr to stdscr on success */
  start_color();
  raw();
  noecho();
  keypad(stdscr, TRUE); /* capture arrow keys */
  timeout(100); /* getch blocks for 50ms max */
  set_escdelay(50); /* ESC should wait for 50ms max */
  nonl(); /* allow ncurses to detect KEY_ENTER */
  /* enable MOUSE? */
  if (flags & PTUI_ENABLE_MOUSE) {
    mousemask(BUTTON1_RELEASED, NULL);
  }
  return(0);
}


void ptui_close(void) {
  endwin();
}


int ptui_getrowcount(void) {
  return(getmaxy(stdscr));
}


int ptui_getcolcount(void) {
  return(getmaxx(stdscr));
}


void ptui_cls(void) {
  int x, y;
  int maxrows, maxcols;
  clear();
  getmaxyx(stdscr, maxrows, maxcols);
  attron(0);
  for (y = 0; y < maxrows; y++) {
    for (x = 0; x < maxcols; x++) {
      mvaddch(y, x, ' ');
    }
  }
  attroff(0);
  /* bkgd(COLOR_PAIR(colattr)); */
  move(0,0);
  ptui_refresh();
}


void ptui_puts(const char *str) {
  puts(str);
}


void ptui_locate(int x, int y) {
  move(y, x);
  ptui_refresh();
}


void ptui_putchar(int wchar, int attr, int x, int y) {
  int oldx, oldy;
  cchar_t t;

  memset(&t, 0, sizeof(t));

  /* remember cursor position to restore it afterwards */
  getyx(stdscr, oldy, oldx);

  t.attr = getorcreatecolor(attr);
  t.chars[0] = wchar;
  mvadd_wch(y, x, &t);

  /* restore cursor to its initial location */
  move(oldy, oldx);
}


void ptui_putchar_rep(int wchar, int attr, int x, int y, int r) {
  int oldx, oldy;
  cchar_t t;

  memset(&t, 0, sizeof(t));

  /* remember cursor position to restore it afterwards */
  getyx(stdscr, oldy, oldx);

  t.attr = getorcreatecolor(attr);
  t.chars[0] = wchar;
  while (r--) mvadd_wch(y, x++, &t);

  /* restore cursor to its initial location */
  move(oldy, oldx);
}


int ptui_getmouse(unsigned int *x, unsigned *y) {
  int r = lastclick_btn;
  if (lastclick_btn < 0) return(-1);
  *x = lastclick_x;
  *y = lastclick_y;
  lastclick_btn = -1;
  return(r);
}


int ptui_getkey(void) {
  int res;

  for (;;) {
    res = getch();
    if (res == KEY_MOUSE) {
      MEVENT event;
      if (getmouse(&event) == OK) {
        if (event.bstate & BUTTON1_RELEASED) {
          lastclick_btn = 0;
          lastclick_x = event.x;
          lastclick_y = event.y;
          return(PTUI_MOUSE);
        }
      }
      continue; /* ignore invalid mouse events */
    }
    if (res != ERR) break;          /* ERR means "no input available yet" */
  }

  /* either ESC or ALT+some key */
  if (res == 27) {
    res = getch();
    if (res == ERR) return(27);
    /* else this is an ALT+something combination */
    switch (res) {
      case 'f': return(0x121);
      case 'h': return(0x123);
      case 'j': return(0x124);
      case 's': return(0x11F);
      case 'u': return(0x116);
    }
  }

  switch (res) {
    case KEY_LEFT:      return(0x14B);
    case KEY_RIGHT:     return(0x14D);
    case KEY_UP:        return(0x148);
    case KEY_DOWN:      return(0x150);
    case KEY_BACKSPACE: return(8);
    case KEY_ENTER:     return(13);
    case KEY_PPAGE:     return(0x149);  /* PAGEUP */
    case KEY_NPAGE:     return(0x151);  /* PGDOWN */
    case KEY_HOME:      return(0x147);
    case KEY_END:       return(0x14F);
    case KEY_DC:        return(0x153);  /* DEL */
    case KEY_F(1):      return(0x13b);
    case KEY_F(2):      return(0x13c);
    case KEY_F(3):      return(0x13d);
    case KEY_F(4):      return(0x13e);
    case KEY_F(5):      return(0x13f);
    case KEY_F(6):      return(0x140);
    case KEY_F(7):      return(0x141);
    case KEY_F(8):      return(0x142);
    case KEY_F(9):      return(0x143);
    case KEY_F(10):     return(0x144);
    default:            return(res); /* return the scancode as-is otherwise */
  }
}


int ptui_kbhit(void) {
  int tmp;
  timeout(0);
  tmp = getch();
  timeout(100);
  if (tmp == ERR) return(0);
  ungetch(tmp);
  return(1);
}


void ptui_mouseshow(int status) {
}


void ptui_cursor_show(void) {
  curs_set(1);
}


void ptui_cursor_hide(void) {
  curs_set(0);
}


void ptui_refresh(void) {
  refresh();
}
