//**************************************************************************
//*                     This file is part of the                           *
//*                MMC - Mpxplay Multimedia Commander                      *
//*                   The source code of MMC is                            *
//*        (C) copyright 1998-2021 by PDSoft (Attila Padar)                *
//*                http://mpxplay.sourceforge.net                          *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//function: ffmpeg subtitle handling

//#define MPXPLAY_USE_DEBUGF 1
#define DISPQT_DEBUGOUT_ERROR    stderr
#define DISPQT_DEBUGOUT_SUBTITLE stdout

#include "moc_video_qt.h"

#define INFFMPG_MUTEX_SUBTITLE_TIMEOUT  20

#define INFFMPG_SUBTITLE_DURATION_MUL  (1200)              // msec to AV_TIME_BASE, +20% duration for subtitles
#define INFFMPG_SUBTITLE_DISPLAY_GAP   (AV_TIME_BASE / 10) // min 0.1 sec gap between subtitles
#define INFFMPG_SUBTITLE_DISPLAY_MIN   (AV_TIME_BASE    )  // min 1 sec subtitle displaying (even if the next one is coming sooner)
#define INFFMPG_SUBTITLE_DISPLAY_HOLD  (AV_TIME_BASE * 2)  // hold for (min) 2 secs the subtitle displaying (even if the duration is less)
#define INFFMPG_SUBTITLE_DISPLAY_MAX   (AV_TIME_BASE * 10) // max 10 secs subtitle displaying

#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG

static unsigned int video_worksubt_bitmap_convert(ffmpegvideo_callback_list_s *cbelem, AVCodecContext *subtitle_codec_ctx, AVSubtitle *subtitle_frame)
{
	ffmpegvideo_subtitle_info_s *subtitle_infos = cbelem->subtitle_infos;
	unsigned int i, lines = 0;

	if(!subtitle_frame->num_rects || !subtitle_frame->rects)
		return lines;

	for(i = 0; ((i < subtitle_frame->num_rects) && (lines < INFFMPEG_ASS_DECODED_MAX_LINES)); i++)
	{
		AVSubtitleRect *rect_src = subtitle_frame->rects[i];
		if(!rect_src)
			break;
		if((rect_src->type != SUBTITLE_BITMAP) || (rect_src->w <= 0) || (rect_src->h <= 0) || !rect_src->data[0] || (rect_src->linesize[0] <= 0))
			continue;
		if(subtitle_codec_ctx->pix_fmt > AV_PIX_FMT_NONE)
		{
			AVFrame subtitle_input_avframe;
			pds_memset(&subtitle_input_avframe, 0, sizeof(subtitle_input_avframe));
			subtitle_input_avframe.format = (int)subtitle_codec_ctx->pix_fmt;
			subtitle_input_avframe.width = rect_src->w;
			subtitle_input_avframe.height = rect_src->h;
			pds_memcpy(&subtitle_input_avframe.data[0], &rect_src->data[0], sizeof(rect_src->data));
			pds_memcpy(&subtitle_input_avframe.linesize[0], &rect_src->linesize[0], sizeof(rect_src->linesize));
			if(mpxplay_dispqt_videotrans_ffmpeg_swsctx_pixfmt_conv(&cbelem->subtitle_bitmap_cv_infos, &subtitle_input_avframe, DISPQT_VIDEO_SUBTITLE_BITMAP_AVFORMAT))
			{
				struct ffmpegvideo_subtitle_data_s *subt_data = &subtitle_infos->subtitle_datas[lines];
				AVFrame *output_frame = cbelem->subtitle_bitmap_cv_infos.ffmpeg_sws_frame;
				const int linesize = output_frame->linesize[0];
				const int image_size = linesize * rect_src->h;
				subt_data->subtitle_line_data = (char *)av_malloc(image_size);
				if(!subt_data->subtitle_line_data)
					break;
				pds_memcpy(subt_data->subtitle_line_data, output_frame->data[0], image_size);
				subt_data->subtitle_line_info = linesize;
				subt_data->subtitle_bitmap_w = rect_src->w;
				subt_data->subtitle_bitmap_h = rect_src->h;
				subtitle_infos->subtitle_type = DISPQT_SUBTITLE_TYPE_BITMAP;
				lines++;
				mpxplay_debugf(DISPQT_DEBUGOUT_SUBTITLE, "worksubt_bitmap_convert fmt:%d w:%d h:%d len:%d",
						(int)subtitle_codec_ctx->pix_fmt, subtitle_input_avframe.width, subtitle_input_avframe.height, output_frame->linesize[0]);
			}
			else
			{
				mpxplay_debugf(DISPQT_DEBUGOUT_ERROR, "video_worksubt_bitmap_convert FAILED!");
			}
		}
		else if((rect_src->nb_colors > 0) && rect_src->data[1]) // color mapped data (DVD subtitles)
		{
			struct ffmpegvideo_subtitle_data_s *subt_data = &subtitle_infos->subtitle_datas[lines];
			const int nb_colors = rect_src->nb_colors;
			const int in_linesize = rect_src->linesize[0];
			const int out_image_size = in_linesize * rect_src->h * sizeof(mpxp_uint32_t);
			uint8_t *in_subtitle_data = rect_src->data[0];
			uint8_t *in_color_data = rect_src->data[1];
			uint8_t *out_rgba_data = (uint8_t *)av_malloc(out_image_size);
			if(!out_rgba_data)
				break;
			subt_data->subtitle_line_data = (char *)out_rgba_data;
			subt_data->subtitle_line_info = in_linesize * sizeof(mpxp_uint32_t);
			subt_data->subtitle_bitmap_w = rect_src->w;
			subt_data->subtitle_bitmap_h = rect_src->h;
			subtitle_infos->subtitle_type = DISPQT_SUBTITLE_TYPE_BITMAP;
			lines++;
			for(int y = 0; y < rect_src->h; y++)
			{
				for(int x = 0; x < in_linesize; x++)
				{
					const int d = *in_subtitle_data++;
					if(d < nb_colors)
					{
						uint8_t *in_cd = &in_color_data[d << 2];
						out_rgba_data[0] = out_rgba_data[1] = out_rgba_data[2] = ((in_cd[0] > 0)  || (in_cd[1] > 0) || (in_cd[2] > 0))? 0 : 255; // FIXME: inverted monochrome color always?
						out_rgba_data[3] = in_cd[3]; // alpha channel
					}
					out_rgba_data += 4;
				}
			}
			mpxplay_debugf(DISPQT_DEBUGOUT_SUBTITLE, "worksubt_bitmap_convert DVD sub w:%d h:%d len:%d nbc:%d",
														rect_src->w, rect_src->h, in_linesize, nb_colors);
		}
		else
		{
			mpxplay_debugf(DISPQT_DEBUGOUT_ERROR, "video_worksubt_bitmap_convert no suitable conversion has found!");
		}
	}

	subtitle_infos->subtitle_nb_lines = lines;

	return lines;
}

static unsigned int video_worksubt_ffmpeg_subtitle_prepare(ffmpegvideo_callback_list_s *cbelem, AVCodecContext *subtitle_codec_ctx, AVSubtitle *subtitle_frame)
{
	ffmpegvideo_subtitle_info_s *sub_lines = cbelem->subtitle_infos;
	unsigned int i, j, lines = 0, textlen, flags, comma_count, textcoding_type_source, textbuf_size;
	AVSubtitleRect *rect_src;
	char *text, *next, *sb, strtmp[512];

	if(!sub_lines || !subtitle_frame->num_rects || !subtitle_frame->rects)
		return lines;

	pds_memset(sub_lines, 0, sizeof(*sub_lines));

	rect_src = subtitle_frame->rects[0]; // TODO: multiply rectangles are supported at bitmap subtitles only (do we need it at texts?)
	if(!rect_src)
		return lines;

	switch(rect_src->type){
		case SUBTITLE_BITMAP:
			return video_worksubt_bitmap_convert(cbelem, subtitle_codec_ctx, subtitle_frame);
		case SUBTITLE_TEXT:
			text = rect_src->text;
			mpxplay_debugf(DISPQT_DEBUGOUT_SUBTITLE,"SUBTITLE_TEXT: %s", text);
			break;
		case SUBTITLE_ASS:
			text = rect_src->ass;
			mpxplay_debugf(DISPQT_DEBUGOUT_SUBTITLE,"SUBTITLE_ASS: %s", text);
			comma_count = (pds_strlicmp(text, (char *)"Dialogue:") == 0)? 9 : 8;
			do{
				next = pds_strchr(text, ',');
				if(!next)
					break;
				text = next + 1;
			}while(--comma_count);
			break;
		default: return lines;
	}

	flags = 0;
	do{
		sub_lines->subtitle_datas[lines].subtitle_line_info = flags;

		// search for end of line (multiply lines are merged in one subtitle, separated with newlines)
		next = NULL;
		textlen = 0;

		if(rect_src->type == SUBTITLE_ASS){
			next = text;
			while((next = pds_strchr(next, '\\'))){
				if((next[1] == 'N') || (next[1] == 'n')){
					textlen = next - text;
					next += 2;
					break;
				}
				next++;
			}
		}

		if(!next){
			if((next = pds_strstr(text, (char *)"\r\n")) || (next = pds_strstr(text, (char *)"\n\r"))){
				textlen = next - text;
				next += 2;
			}
			else if((next = pds_strchr(text, '\n'))){
				textlen = next - text;
				next++;
			}
		}

		if(!textlen){ // last line in this subtitle
			textlen = pds_strlen(text);
			if(!textlen)
				break;
		}
		if(textlen >= sizeof(strtmp))
			textlen = sizeof(strtmp) - 1;
		pds_strncpy(strtmp, text, textlen);
		strtmp[textlen] = 0;
		switch(rect_src->type){
			case SUBTITLE_TEXT: // TODO: not tested
				while((sb = pds_strchr(strtmp,'<'))){
					int ctrllen = 0;
					if(sb[2] == '>'){
						if(sb[1] == 'i'){  // "<i>"
							funcbit_enable(sub_lines->subtitle_datas[lines].subtitle_line_info, INFFMPEG_ASS_DECODED_FLAG_ITALIC); // TODO: flag changes inside line is not handled (only at begin and end)
							funcbit_enable(flags, INFFMPEG_ASS_DECODED_FLAG_ITALIC);
						}
						ctrllen = 3;
					}else if(sb[1] == '/' && sb[3] == '>'){
						if(sb[2] == 'i')  // "</i>"
							funcbit_disable(flags, INFFMPEG_ASS_DECODED_FLAG_ITALIC);
						ctrllen = 4;
					}

					if(ctrllen){
						pds_strcpy(sb, sb + ctrllen);
						textlen = pds_strlen(strtmp);
					}
				}
				break;
			case SUBTITLE_ASS:
				while((sb = pds_strchr(strtmp,'{'))){
					char *se;
					if(sb[1] == '\\' && sb[2] =='i')
						if(sb[3] == '1'){
							funcbit_enable(sub_lines->subtitle_datas[lines].subtitle_line_info, INFFMPEG_ASS_DECODED_FLAG_ITALIC);  // TODO: flag changes inside line is not handled (only at begin and end)
							funcbit_enable(flags, INFFMPEG_ASS_DECODED_FLAG_ITALIC);
						}else if(sb[3] == '0')
							funcbit_disable(flags, INFFMPEG_ASS_DECODED_FLAG_ITALIC);
					se = pds_strchr(sb, '}');
					if(se){
						if(se[1])
							pds_strcpy(sb, &se[1]);
						else // end of line
							sb[0] = 0;
					}
					textlen = pds_strlen(strtmp);
				}
				break;
		}
		textcoding_type_source = (cbelem->subtitle_pktlist_elem->flags >> MPXPLAY_PACKETLISTFLAG_TEXTTYPE_BITPOS) & MPXPLAY_PACKETLISTFLAG_TEXTTYPE_MASK;
		textbuf_size = textlen * 3; // max required space for char to UTF8
		sub_lines->subtitle_datas[lines].subtitle_line_data = (char *)av_malloc(textbuf_size + 4);
		mpxplay_playlist_textconv_by_texttypes(MPXPLAY_TEXTCONV_TYPES_PUT(textcoding_type_source ,MPXPLAY_TEXTCONV_TYPE_MPXNATIVE), strtmp, textlen, sub_lines->subtitle_datas[lines].subtitle_line_data, textbuf_size + 2);
		lines++;
		text = next;
	}while(text && (lines < INFFMPEG_ASS_DECODED_MAX_LINES));

	if(lines)
	{
		sub_lines->subtitle_type = DISPQT_SUBTITLE_TYPE_TEXT;
		sub_lines->subtitle_nb_lines = lines;
	}

	return lines;
}

void mpxplay_video_ffmpeg_subtitleinfo_clear(ffmpegvideo_subtitle_info_s *sub_lines)
{
	int i;

	if(!sub_lines || !sub_lines->subtitle_nb_lines)
		return;

	for(i = 0; i < sub_lines->subtitle_nb_lines; i++)
	{
		struct ffmpegvideo_subtitle_data_s *subt_data = &sub_lines->subtitle_datas[i];
		subt_data->subtitle_line_info = subt_data->subtitle_bitmap_w = subt_data->subtitle_bitmap_h = 0;
		subt_data->subtitle_line_info = 0;
		av_freep(&subt_data->subtitle_line_data);
	}
	sub_lines->subtitle_nb_lines = 0;
}

void mpxplay_video_ffmpeg_subtitleinfo_copy(ffmpegvideo_subtitle_info_s *dest_lines, ffmpegvideo_subtitle_info_s *sub_lines)
{
	int i, j;
	if(!dest_lines || !sub_lines)
		return;
	dest_lines->subtitle_type = sub_lines->subtitle_type;
	dest_lines->subtitle_nb_lines = sub_lines->subtitle_nb_lines;
	for(i = 0; i < sub_lines->subtitle_nb_lines; i++)
	{
		struct ffmpegvideo_subtitle_data_s *subt_src_data = &sub_lines->subtitle_datas[i];
		struct ffmpegvideo_subtitle_data_s *subt_dest_data = &dest_lines->subtitle_datas[i];
		unsigned int len = (sub_lines->subtitle_type == DISPQT_SUBTITLE_TYPE_TEXT)? pds_strlen(subt_src_data->subtitle_line_data) : (subt_src_data->subtitle_bitmap_h * subt_src_data->subtitle_line_info);
		subt_dest_data->subtitle_line_info = subt_dest_data->subtitle_bitmap_w = subt_dest_data->subtitle_bitmap_h = 0;
		subt_dest_data->subtitle_line_data = NULL;
		if(len)
		{
			if(sub_lines->subtitle_type == DISPQT_SUBTITLE_TYPE_TEXT)
				len++;
			subt_dest_data->subtitle_line_data = (char *)av_mallocz(len);
			if(subt_dest_data->subtitle_line_data)
			{
				pds_memcpy(subt_dest_data->subtitle_line_data, subt_src_data->subtitle_line_data, len);
				subt_dest_data->subtitle_line_info = subt_src_data->subtitle_line_info;
				if(sub_lines->subtitle_type == DISPQT_SUBTITLE_TYPE_BITMAP)
				{
					subt_dest_data->subtitle_bitmap_w = subt_src_data->subtitle_bitmap_w;
					subt_dest_data->subtitle_bitmap_h = subt_src_data->subtitle_bitmap_h;
				}
			}
		}
	}
}

static void video_worksubt_subtitle_load_next_frame(ffmpegvideo_callback_list_s *cbelem)
{
	int streamtype_index = MPXPLAY_STREAMTYPEINDEX_SUBTITLE;
	int retval_subtitle = cbelem->infile_callback(cbelem->infile_passdata, MPXPLAY_INFILE_CBKCTRL_GET_PACKETLIST_ELEM, (mpxp_ptrsize_t)&cbelem->subtitle_pktlist_elem, streamtype_index, INFFMPG_MUTEX_SUBTITLE_TIMEOUT);
	if((retval_subtitle >= 0) && cbelem->subtitle_pktlist_elem && (cbelem->subtitle_pktlist_elem->timestamp_us > 0) && (cbelem->subtitle_end_timestamp >= cbelem->subtitle_pktlist_elem->timestamp_us))
	{
		cbelem->subtitle_end_timestamp = cbelem->subtitle_pktlist_elem->timestamp_us - INFFMPG_SUBTITLE_DISPLAY_GAP;
		if((cbelem->subtitle_end_timestamp - cbelem->subtitle_begin_timestamp) < INFFMPG_SUBTITLE_DISPLAY_MIN)
			cbelem->subtitle_end_timestamp = cbelem->subtitle_begin_timestamp + INFFMPG_SUBTITLE_DISPLAY_MIN;
	}
	if((retval_subtitle >= 0) && cbelem->subtitle_pktlist_elem)
	{
		mpxplay_debugf(DISPQT_DEBUGOUT_SUBTITLE, "subtitle_load_next_frame r:%d ts:%lld", retval_subtitle, ((cbelem->subtitle_pktlist_elem)? cbelem->subtitle_pktlist_elem->timestamp_us : 0LL));
	}
}

static void video_worksubt_subtitle_decode(ffmpegvideo_callback_list_s *cbelem, mpxp_int64_t timestamp_curr_us)
{
	if(cbelem->subtitle_end_timestamp)
	{
		if(timestamp_curr_us >= cbelem->subtitle_end_timestamp)
		{
			mpxplay_video_ffmpeg_subtitleinfo_clear(cbelem->subtitle_infos);
			cbelem->subtitle_end_timestamp = 0;
		}
	}

	if(!cbelem->subtitle_end_timestamp && cbelem->subtitle_pktlist_elem)
	{
		AVCodecContext *subtitle_codec_ctx = (AVCodecContext *)cbelem->subtitle_pktlist_elem->codec_ctx;
		AVPacket *subtitle_packet = (AVPacket *)(cbelem->subtitle_pktlist_elem->frame_pkt);

		if(!cbelem->subtitle_infos)
			cbelem->subtitle_infos = (ffmpegvideo_subtitle_info_s *)av_mallocz(sizeof(ffmpegvideo_subtitle_info_s));
		else
			mpxplay_video_ffmpeg_subtitleinfo_clear(cbelem->subtitle_infos);

		if(subtitle_codec_ctx && subtitle_packet)
		{
			int got_frame = 0, retval;
			AVSubtitle subtitle_frame;

			pds_memset(&subtitle_frame, 0, sizeof(subtitle_frame));

			retval = avcodec_decode_subtitle2(subtitle_codec_ctx, &subtitle_frame, &got_frame, subtitle_packet);

			if((retval >= 0) && (got_frame > 0))
			{
				if(video_worksubt_ffmpeg_subtitle_prepare(cbelem, subtitle_codec_ctx, &subtitle_frame))
				{
					mpxp_int64_t duration;
					mpxplay_debugf(DISPQT_DEBUGOUT_SUBTITLE, "subtitle_decode ts:%lld pts:%llu s:%d e:%d d:%llu %s", cbelem->subtitle_pktlist_elem->timestamp_us, subtitle_packet->pts,
						subtitle_frame.start_display_time, subtitle_frame.end_display_time, subtitle_packet->duration, cbelem->subtitle_infos->subtitle_datas->subtitle_line_data);
					cbelem->subtitle_begin_timestamp = cbelem->subtitle_pktlist_elem->timestamp_us;
					duration = subtitle_packet->duration * INFFMPG_SUBTITLE_DURATION_MUL;
					if(duration < INFFMPG_SUBTITLE_DISPLAY_HOLD)
						duration = INFFMPG_SUBTITLE_DISPLAY_HOLD;
					else if(duration > INFFMPG_SUBTITLE_DISPLAY_MAX)
						duration = INFFMPG_SUBTITLE_DISPLAY_MAX;
					cbelem->subtitle_end_timestamp = cbelem->subtitle_begin_timestamp + duration;
				}
			}
			else
			{
				mpxplay_debugf(DISPQT_DEBUGOUT_SUBTITLE, "avcodec_decode_subtitle2 has failed %d %d ctx:%8.8X", retval, got_frame, (mpxp_ptrsize_t)subtitle_codec_ctx);
			}
			avsubtitle_free(&subtitle_frame);
		}

		mpxplay_ffmpgdec_queuelistelem_clear(&cbelem->subtitle_pktlist_elem);
	}
}

ffmpegvideo_subtitle_info_s *mpxplay_decoderworker_subtitle_get(ffmpegvideo_callback_list_s *cbelem, mpxp_int64_t timestamp_curr_us, bool resetvideo)
{
	ffmpegvideo_subtitle_info_s *subtitle = NULL;

	if(cbelem->subtitle_pktlist_elem)
	{   // TODO: maybe it could be merged with GET_PACKETLIST_ELEM to reduce callback calls
		if(cbelem->infile_callback(cbelem->infile_passdata, MPXPLAY_INFILE_CBKCTRL_SET_PACKETQUEUE_CLEAR, MPXPLAY_STREAMTYPEINDEX_SUBTITLE, MPXPLAY_PACKETLISTFLAG_CLEARBUF, INFFMPG_MUTEX_SUBTITLE_TIMEOUT) == 1)
		{
			resetvideo = true;
		}
	}
	if(resetvideo)
	{
		mpxplay_decoderworker_subtitle_reset(cbelem);
	}
	if(!cbelem->subtitle_pktlist_elem)
	{
		video_worksubt_subtitle_load_next_frame(cbelem);
	}

	video_worksubt_subtitle_decode(cbelem, timestamp_curr_us);

	if((timestamp_curr_us >= cbelem->subtitle_begin_timestamp) && (timestamp_curr_us < cbelem->subtitle_end_timestamp))
	{
		subtitle = cbelem->subtitle_infos;
	}
	else if(timestamp_curr_us > (cbelem->subtitle_begin_timestamp + DISPQT_FFMPGVIDEO_SUBTITLE_MAX_VIDEO_DIFF_US))
	{
		cbelem->subtitle_begin_timestamp = cbelem->subtitle_end_timestamp = 0;
	}

	mpxplay_debugf(DISPQT_DEBUGOUT_SUBTITLE, "subtitle_get %s sb:%lld se:%lld ts:%lld", ((subtitle)? "SHOW" : ""), cbelem->subtitle_begin_timestamp, cbelem->subtitle_end_timestamp, timestamp_curr_us);

	return subtitle;
}

void mpxplay_decoderworker_subtitle_reset(ffmpegvideo_callback_list_s *cbelem)
{
	mpxplay_ffmpgdec_queuelistelem_clear(&cbelem->subtitle_pktlist_elem);
	mpxplay_video_ffmpeg_subtitleinfo_clear(cbelem->subtitle_infos);
	cbelem->subtitle_begin_timestamp = cbelem->subtitle_end_timestamp = 0;
}

#endif // MPXPLAY_LINK_ORIGINAL_FFMPEG
