//**************************************************************************
//*                     This file is part of the                           *
//*                MMC - Mpxplay Multimedia Commander                      *
//*                   The source code of MMC is                            *
//*        (C) copyright 1998-2020 by PDSoft (Attila Padar)                *
//*                http://mpxplay.sourceforge.net                          *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//function: video widget handling

//#define MPXPLAY_USE_DEBUGF 1
#define DISPQT_DEBUG_ERROR stderr
#define DISPQT_DEBUGOUT_TRACE NULL // stdout
#define DISPQT_DEBUGOUT_FFMPEG stdout
#define DISPQT_DEBUGOUT_RENDER stdout
#define DISPQT_DEBUGOUT_SUBTITLE NULL // stdout
#define DISPQT_DEBUGOUT_CROP NULL // stdout
#define DISPQT_DEBUGOUT_ACROP NULL // stdout

#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG

#include "moc_video_qt.h"
#include "moc_mainwindow.h"
#include "display/display.h"

extern struct dispqt_video_filter_format_s mpxplay_videoworkdec_videowidget_format;

static void dispqt_videowidget_frameforclear_init(struct dispqt_video_surface_info_s *video_surface_infos);
static void dispqt_videowidget_frameforclear_close(struct dispqt_video_surface_info_s *video_surface_infos);
static void dispqt_videowidget_ffmpeg_subtitle_close(struct dispqt_video_surface_info_s *video_surface_infos);

FFMpegVideoWidget::FFMpegVideoWidget(MainWindow *main_window, QWidget *parent) : QWidget(parent)
{
	struct dispqt_video_surface_info_s *video_surface_infos = &this->video_surface_infos;
	mpxplay_debugf(DISPQT_DEBUGOUT_FFMPEG,"FFMpegVideoWidget INIT");
	this->main_window = main_window;
	this->parent_widget = parent;
	this->video_renderer = NULL;

	pds_memset(&this->video_source_infos, 0, sizeof(this->video_source_infos));
	pds_memset(video_surface_infos, 0, sizeof(*video_surface_infos));
	video_surface_infos->sws_dst_width = video_surface_infos->window_size_x = video_surface_infos->windowed_size_x = this->width();
	video_surface_infos->sws_dst_height = video_surface_infos->window_size_y = video_surface_infos->windowed_size_y = this->height();
	video_surface_infos->window_pos_x = QWidget::mapToGlobal(this->pos()).x();
	video_surface_infos->window_pos_y = QWidget::mapToGlobal(this->pos()).y();
	video_surface_infos->video_zoom_globalpos_x = video_surface_infos->window_size_x / 2;
	video_surface_infos->video_zoom_globalpos_y = video_surface_infos->window_size_y / 2;
	video_surface_infos->video_render_infos = &mpxplay_video_render_infos;
	video_surface_infos->video_source_infos = &this->video_source_infos;
	video_surface_infos->gui_config = main_window->gui_config;
	dispqt_videowidget_frameforclear_init(video_surface_infos);
	pds_threads_mutex_new(&video_surface_infos->mutex_render);

	this->videowidget_fullscreen_display_select();
	this->videowidget_renderer_init();

	this->video_sw_transform = new FFMpegVideoSoftwareTransform(main_window, &this->video_source_infos, video_surface_infos);

	connect(this, SIGNAL(signal_ffmpeg_swctx_resize(int,int,bool)), this, SLOT(videowidget_ffmpeg_swctx_resize(int,int,bool)));
	connect(this, SIGNAL(signal_ffmpeg_swctx_move(int,int)), this, SLOT(videowidget_ffmpeg_swctx_move(int,int)));
	connect(this, SIGNAL(signal_ffmpeg_swctx_zoom(int,int,int)), this, SLOT(videowidget_ffmpeg_swctx_zoom(int,int,int)));
	connect(this, SIGNAL(signal_ffmpeg_filtered_trans_refresh()), this, SLOT(videowidget_ffmpeg_filtered_trans_refresh()));
	connect(this, SIGNAL(signal_ffmpeg_filtered_eq_refresh()), this, SLOT(videowidget_ffmpeg_filtered_eq_refresh()));

	this->video_decoder_worker = new FFMpegVideoDecoderWorker(main_window, video_surface_infos);
	this->video_decoder_worker->decoderworker_video_surface_refresh(DISPQT_VIDEOSURFACE_REFRESHTYPE_CLEAR); // !!! initial window CLEAR
}

FFMpegVideoWidget::~FFMpegVideoWidget()
{
	struct dispqt_video_surface_info_s *video_surface_infos = &this->video_surface_infos;
	const int mutex_error = PDS_THREADS_MUTEX_LOCK(&video_surface_infos->mutex_render, DISPQT_VIDEO_MUTEX_TIMEOUT);
	this->mutex_widget.tryLock(DISPQT_VIDEO_MUTEX_TIMEOUT);
	if(this->video_decoder_worker){
		FFMpegVideoDecoderWorker *decoder_worker = this->video_decoder_worker;
		this->video_decoder_worker = NULL;
		delete decoder_worker;
	}
	if(this->video_sw_transform)
	{
		FFMpegVideoSoftwareTransform *swt = this->video_sw_transform;
		this->video_sw_transform = NULL;
		delete swt;
	}
	this->videowidget_renderer_close();
	dispqt_videowidget_ffmpeg_subtitle_close(video_surface_infos);
	dispqt_videowidget_frameforclear_close(video_surface_infos);
	if(mutex_error == MPXPLAY_ERROR_OK)
		PDS_THREADS_MUTEX_UNLOCK(&video_surface_infos->mutex_render);
	pds_threads_mutex_del(&video_surface_infos->mutex_render);
	this->mutex_widget.unlock();
}

//----------------------------------------------------------------------------------------------------------------------

static void dispqt_videowidget_video_output_errormsg_with_reopen(void)
{
	funcbit_enable(mpxplay_signal_events, MPXPLAY_SIGNALTYPE_GUIREBUILD);
	display_textwin_openwindow_errormsg_confirm_ok((char *)"Selected video output initialization has failed!\nSwitching back to QWidget mode!", (void *)mpxplay_dispqt_main_close, NULL);
}

static void dispqt_videowidget_video_output_reopen_gui(void)
{
	mpxplay_timer_addfunc((void *)dispqt_videowidget_video_output_errormsg_with_reopen, NULL, MPXPLAY_TIMERTYPE_SIGNAL, MPXPLAY_SIGNALTYPE_GUIREADY);
}

//----------------------------------------------------------------------------------------------------------------------

static mpxp_bool_t dispqt_videowidget_render_function_check_videodecoder_inputformat_empty(const GUID format_guid, int av_pixfmt, int video_width, int video_height)
{
	return FALSE;
}

static mpxp_uint32_t dispqt_videowidget_render_function_get_frame_delay_empty(void *render_private_data)
{
	return 0;
}

static mpxp_bool_t dispqt_videowidget_poolbufelem_validity_check_empty(void *avframe)
{
	return FALSE;
}

static mpxp_bool_t dispqt_videowidget_render_function_is_framebuffer_support_empty(void)
{
	return FALSE;
}

static int dispqt_videowidget_render_function_get_frame_buffer_empty(void *av_ctx, void *av_destframe, void *av_srcframe, int flags)
{
	return -1;
}

static void dispqt_videowidget_render_function_frame_buffer_consolidate_empty(void)
{
}

static mpxp_bool_t dispqt_videowidget_render_function_copy_to_pool_buffer_empty(void *dest_avframe, void *src_avframe)
{
	return FALSE;
}

static void dispqt_videowidget_render_function_poolbufelem_unmap_empty(void *av_frame)
{
}

static void videowidget_renderer_reset(void)
{
	//mpxplay_video_render_infos.hwdevice_avpixfmt = AV_PIX_FMT_NONE; // FIXME: collision with static D3D init (workaround: added setting for all renderers)
	mpxplay_video_render_infos.renderer_private_data = NULL;
	mpxplay_video_render_infos.render_function_check_videodecoder_inputformat = dispqt_videowidget_render_function_check_videodecoder_inputformat_empty;
	mpxplay_video_render_infos.render_function_get_frame_delay = dispqt_videowidget_render_function_get_frame_delay_empty;
	mpxplay_video_render_infos.render_function_poolbufelem_validity_check = dispqt_videowidget_poolbufelem_validity_check_empty;
	mpxplay_video_render_infos.render_function_is_framebuffer_support = dispqt_videowidget_render_function_is_framebuffer_support_empty;
	mpxplay_video_render_infos.render_function_get_frame_buffer = dispqt_videowidget_render_function_get_frame_buffer_empty;
	mpxplay_video_render_infos.render_function_frame_buffer_consolidate = dispqt_videowidget_render_function_frame_buffer_consolidate_empty;
	mpxplay_video_render_infos.render_function_copy_to_pool_buffer = dispqt_videowidget_render_function_copy_to_pool_buffer_empty;
	mpxplay_video_render_infos.render_function_poolbufelem_unmap = dispqt_videowidget_render_function_poolbufelem_unmap_empty;
	mpxplay_video_render_infos.renderer_capabilities = 0;
}

//----------------------------------------------------------------------------------------------------------------------

mpxp_bool_t mpxplay_dispqt_videowidget_videorender_static_init(void)
{
	mpxp_bool_t success = FALSE;
#ifdef MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK
#if !defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_GLOBALLOCK)
	int mutex_error;
#endif

	if(!mpxplay_video_render_infos.d3d_device_mutex)
		pds_threads_mutex_new(&mpxplay_video_render_infos.d3d_device_mutex);
#if !defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_GLOBALLOCK)
	mutex_error = PDS_THREADS_MUTEX_LOCK(&mpxplay_video_render_infos.d3d_device_mutex, DISPQT_VIDEO_MUTEX_TIMEOUT);
#endif
#endif

	mpxplay_video_render_infos.hwdevice_avpixfmt = AV_PIX_FMT_NONE;
	mpxplay_video_render_infos.render_function_check_videodecoder_inputformat = dispqt_videowidget_render_function_check_videodecoder_inputformat_empty;

	switch(mmc_config.video_renderer_type)
	{
#if MPXPLAY_DISPQT_ENABLE_RENDER_D3D11
		case DISPQT_VIDEORENDERTYPE_D3D11:
			success = mpxplay_dispqt_videorender_d3d11_static_init();
			break;
#endif
#if MPXPLAY_DISPQT_ENABLE_RENDER_D3D9
		case DISPQT_VIDEORENDERTYPE_D3D9:
			success = mpxplay_dispqt_videorender_d3d9_static_init();
			break;
#endif
		default: break;
	}

#if defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK) && !defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_GLOBALLOCK)
	if(mutex_error == MPXPLAY_ERROR_OK)
		PDS_THREADS_MUTEX_UNLOCK(&mpxplay_video_render_infos.d3d_device_mutex);
#endif

	return success;
}

void mpxplay_dispqt_videowidget_videorender_static_close(void)
{
#ifdef MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK
	const int mutex_error = PDS_THREADS_MUTEX_LOCK(&mpxplay_video_render_infos.d3d_device_mutex, DISPQT_VIDEO_MUTEX_TIMEOUT);
#endif
	mpxplay_video_render_infos.d3d_render_handler = NULL;
#if MPXPLAY_DISPQT_ENABLE_RENDER_D3D11
	mpxplay_dispqt_videorender_d3d11_static_close();
#endif
#if MPXPLAY_DISPQT_ENABLE_RENDER_D3D9
	mpxplay_dispqt_videorender_d3d9_static_close();
#endif
#ifdef MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK
	if(mpxplay_video_render_infos.d3d_device_mutex)
	{
		void *mutex = mpxplay_video_render_infos.d3d_device_mutex;
		mpxplay_video_render_infos.d3d_device_mutex = NULL;
		if(mutex_error == MPXPLAY_ERROR_OK)
			PDS_THREADS_MUTEX_UNLOCK(&mutex);
		pds_threads_mutex_del(&mutex);
	}
#endif
	pds_memset(&mpxplay_video_render_infos, 0, sizeof(mpxplay_video_render_infos));
	mpxplay_video_render_infos.hwdevice_avpixfmt = AV_PIX_FMT_NONE;
}

void mpxplay_dispqt_videowidget_videorender_mutex_lock(void)
{
#if defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK) && !defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_GLOBALLOCK)
	PDS_THREADS_MUTEX_LOCK(&mpxplay_video_render_infos.d3d_device_mutex, DISPQT_VIDEO_MUTEX_TIMEOUT);
#endif
}

void mpxplay_dispqt_videowidget_videorender_mutex_unlock(void)
{
#if defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK) && !defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_GLOBALLOCK)
	PDS_THREADS_MUTEX_UNLOCK(&mpxplay_video_render_infos.d3d_device_mutex);
#endif
}

#ifdef DISPQT_VIDEO_DECODER_USE_GPU_FLUSH
void mpxplay_dispqt_videowidget_videorender_flush(mpxp_bool_t wait_for_completition)
{
#if defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK) && !defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_GLOBALLOCK)
	const int mutex_error = PDS_THREADS_MUTEX_LOCK(&mpxplay_video_render_infos.d3d_device_mutex, DISPQT_VIDEO_MUTEX_TIMEOUT);
#endif
	switch(mmc_config.video_renderer_type)
	{
#if MPXPLAY_DISPQT_ENABLE_RENDER_D3D11
		case DISPQT_VIDEORENDERTYPE_D3D11:
			mpxplay_dispqt_videorender_d3d11_flush(wait_for_completition);
			break;
#endif
		default: break;
	}
#if defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK) && !defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_GLOBALLOCK)
	if(mutex_error == MPXPLAY_ERROR_OK)
		PDS_THREADS_MUTEX_UNLOCK(&mpxplay_video_render_infos.d3d_device_mutex);
#endif
}
#endif

mpxp_bool_t mpxplay_dispqt_videowidget_videorender_refer_d3ddevice(struct mpxplay_video_d3d_device_ref_s *d3d_device)
{
#if defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK) && !defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_GLOBALLOCK)
	const int mutex_error = PDS_THREADS_MUTEX_LOCK(&mpxplay_video_render_infos.d3d_device_mutex, DISPQT_VIDEO_MUTEX_TIMEOUT);
#endif
	mpxp_bool_t success = TRUE;

	switch(mmc_config.video_renderer_type)
	{
#if MPXPLAY_DISPQT_ENABLE_RENDER_D3D11
		case DISPQT_VIDEORENDERTYPE_D3D11:
			success = mpxplay_dispqt_videorender_d3d11_refer_d3ddevice(d3d_device);
			break;
#endif
#if MPXPLAY_DISPQT_ENABLE_RENDER_D3D9
		case DISPQT_VIDEORENDERTYPE_D3D9:
			success = TRUE;
			break;
#endif
		default: break;
	}

#if defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK) && !defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_GLOBALLOCK)
	if(mutex_error == MPXPLAY_ERROR_OK)
		PDS_THREADS_MUTEX_UNLOCK(&mpxplay_video_render_infos.d3d_device_mutex);
#endif

	return success;
}

#ifdef DISPQT_VIDEO_DECODER_DETECT_D3D_FAILURE
mpxp_bool_t mpxplay_dispqt_videowidget_videorender_detect_device_context_failure(void)
{
#if defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK) && !defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_GLOBALLOCK)
	const int mutex_error = PDS_THREADS_MUTEX_LOCK(&mpxplay_video_render_infos.d3d_device_mutex, DISPQT_VIDEO_MUTEX_TIMEOUT);
#endif
	mpxp_bool_t result = FALSE;

	switch(mmc_config.video_renderer_type)
	{
#if MPXPLAY_DISPQT_ENABLE_RENDER_D3D11
		case DISPQT_VIDEORENDERTYPE_D3D11:
			result = mpxplay_dispqt_videorender_d3d11_detect_device_context_failure();
			break;
#endif
		default: break;
	}

#if defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK) && !defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_GLOBALLOCK)
	if(mutex_error == MPXPLAY_ERROR_OK)
		PDS_THREADS_MUTEX_UNLOCK(&mpxplay_video_render_infos.d3d_device_mutex);
#endif

	return result;
}
#endif

// --------------------------------------------------------------------------------------------------------------------

static void dispqt_videowidget_frameforclear_init(struct dispqt_video_surface_info_s *video_surface_infos)
{
	AVFrame *avframe = &video_surface_infos->ffmpeg_empty_frame;
	mpxp_uint32_t i, *pixel;
	avframe->data[0] = (uint8_t *)av_malloc(128 * 64 * 4);
	if(!avframe->data[0])
		return;
	pixel = (mpxp_uint32_t *)avframe->data[0];
	for(i = 0; i < (128 * 64); i++)
		*pixel++ = 0xFF000000;
	avframe->width = 128;
	avframe->height = 64;
	avframe->format = DISPQT_VIDEO_QIMAGE_OUTPUT_AVFORMAT;
	avframe->linesize[0] = avframe->width * 4;
	avframe->colorspace = AVCOL_SPC_BT709;
	avframe->color_primaries = AVCOL_PRI_BT709;
	avframe->color_trc = AVCOL_TRC_BT709;
}

static void dispqt_videowidget_frameforclear_set(struct dispqt_video_surface_info_s *video_surface_infos)
{
	AVFrame *avframe = &video_surface_infos->ffmpeg_empty_frame;
	video_surface_infos->sws_crop_x = video_surface_infos->sws_crop_y = 0;
	video_surface_infos->sws_crop_w = avframe->width;
	video_surface_infos->sws_crop_h = avframe->height;
	video_surface_infos->sws_dst_pos_x = video_surface_infos->sws_dst_pos_y = 0;
	video_surface_infos->sws_dst_width = video_surface_infos->window_size_x;
	video_surface_infos->sws_dst_height = video_surface_infos->window_size_y;
	video_surface_infos->ffmpeg_output_frame = avframe;
}

static void dispqt_videowidget_frameforclear_close(struct dispqt_video_surface_info_s *video_surface_infos)
{
	AVFrame *avframe = &video_surface_infos->ffmpeg_empty_frame;
	if(avframe->data[0])
		av_free(avframe->data[0]);
	pds_memset(avframe, 0, sizeof(*avframe));
}

// --------------------------------------------------------------------------------------------------------------------

void FFMpegVideoWidget::videowidget_renderer_init(void)
{
	struct mmc_dispqt_config_s *gcfg = main_window->gui_config;

	this->videowidget_renderer_close();

#ifdef MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK
	if(!mpxplay_video_render_infos.d3d_device_mutex)
		pds_threads_mutex_new(&mpxplay_video_render_infos.d3d_device_mutex);
#endif

	switch(gcfg->video_renderer_type)
	{
#if MPXPLAY_DISPQT_ENABLE_RENDER_QPAINT
		case DISPQT_VIDEORENDERTYPE_QPAINT: this->video_renderer = new MMCDispQtVideoRendererQPaint(main_window, this); break;
#endif
#if MPXPLAY_DISPQT_ENABLE_RENDER_D3D11
		case DISPQT_VIDEORENDERTYPE_D3D11: this->video_renderer = new MMCDispQtVideoRendererD3D11(main_window, this); break;
#endif
#if MPXPLAY_DISPQT_ENABLE_RENDER_D3D9
		case DISPQT_VIDEORENDERTYPE_D3D9: this->video_renderer = new MMCDispQtVideoRendererD3D9(main_window, this); break;
#endif
#if MPXPLAY_DISPQT_ENABLE_RENDER_VULKAN
		case DISPQT_VIDEORENDERTYPE_VULKAN: this->video_renderer = new MMCDispQtVideoRendererVulkan(main_window, this); break;
#endif
#if MPXPLAY_DISPQT_ENABLE_RENDER_GDI
		case DISPQT_VIDEORENDERTYPE_GDI: this->video_renderer = new MMCDispQtVideoRendererGDI(main_window, this); break;
#endif
		default: this->video_renderer = NULL;
	}

	this->video_surface_infos.render_base = this->video_renderer;

	if(!this->video_renderer || !this->video_renderer->videorenderer_init(&this->video_surface_infos))
	{
		this->videowidget_renderer_close();
		gcfg->video_renderer_type = DISPQT_VIDEORENDERTYPE_QPAINT;
		mpxplay_dispqt_mainthread_callback_init((void *)dispqt_videowidget_video_output_reopen_gui, NULL, 0);
		return;
	}

	gcfg->selected_videorenderer_type = gcfg->video_renderer_type;

	this->video_renderer->videorenderer_get_surface_attribs(&this->video_surface_infos, this->video_source_infos.ffmpeg_decoded_frame);

	mpxplay_videoworkdec_videowidget_format.width = this->video_surface_infos.window_size_x;
	mpxplay_videoworkdec_videowidget_format.height = this->video_surface_infos.window_size_y;
	mpxplay_videoworkdec_videowidget_format.pixelformat = DISPQT_VIDEO_QIMAGE_OUTPUT_AVFORMAT; // format for videowall

	if(this->video_surface_infos.video_render_infos->renderer_capabilities & DISPQT_VIDEORENDERER_CAPFLAG_WIDGET)
	{
		setAttribute(Qt::WA_NativeWindow, false);
		setAttribute(Qt::WA_PaintOnScreen, false);
		setAttribute(Qt::WA_UpdatesDisabled, false);
	}
	else // native painter
	{
		setAttribute(Qt::WA_NativeWindow, true);
		setAttribute(Qt::WA_PaintOnScreen, true);
		setAttribute(Qt::WA_UpdatesDisabled, true);
	}
}

void FFMpegVideoWidget::videowidget_renderer_close(void)
{
	this->main_window->gui_config->selected_videorenderer_type = DISPQT_VIDEORENDERTYPE_MAX;

	videowidget_renderer_reset();

	if(this->video_renderer)
	{
		MMCDispQtVideoRendererBase *renderer = this->video_renderer;
		this->video_renderer = NULL;
		delete renderer;
	}
}

void FFMpegVideoWidget::videowidget_fullscreen_display_select(void)
{
	const int screen_number = this->main_window->video_get_fullscreen_display_number();
	const QRect selectedFullScreenGeometry = QApplication::desktop()->screenGeometry(screen_number);

	if(screen_number >= 0)
		setGeometry(selectedFullScreenGeometry);

	this->video_surface_infos.fullscreen_size_x = selectedFullScreenGeometry.width();
	this->video_surface_infos.fullscreen_size_y = selectedFullScreenGeometry.height();
}

void FFMpegVideoWidget::videowidget_fullscreen_set(mpxp_bool_t full)
{
	struct dispqt_video_surface_info_s *video_surface_infos = &this->video_surface_infos;
	const int mutex_error = PDS_THREADS_MUTEX_LOCK(&video_surface_infos->mutex_render, DISPQT_VIDEO_MUTEX_TIMEOUT);
	video_surface_infos->is_fullscreen = full;
	if(full)
		this->videowidget_fullscreen_display_select();
	if(this->video_renderer)
		this->video_renderer->videorenderer_set_fullscreen(full);
	video_surface_infos->videoout_surface_recalc = TRUE;
	if(mutex_error == MPXPLAY_ERROR_OK)
		PDS_THREADS_MUTEX_UNLOCK(&video_surface_infos->mutex_render);
}

void FFMpegVideoWidget::videowidget_screen_clear(void)
{
	struct dispqt_video_surface_info_s *video_surface_infos = &this->video_surface_infos;
	const int mutex_error = PDS_THREADS_MUTEX_LOCK(&video_surface_infos->mutex_render, DISPQT_VIDEO_MUTEX_TIMEOUT);
	struct dispqt_video_source_info_s *video_source_infos = video_surface_infos->video_source_infos;
	//av_frame_free(&video_source_infos->ffmpeg_decoded_frame); //TODO: check
	video_surface_infos->ffmpeg_output_frame = NULL;
	video_surface_infos->videoout_surface_clearpic = true;
	this->videowidget_ffmpeg_filtered_surface_update(true);
	if(mutex_error == MPXPLAY_ERROR_OK)
		PDS_THREADS_MUTEX_UNLOCK(&video_surface_infos->mutex_render);
}

void FFMpegVideoWidget::videowidget_vmixer_set(unsigned int u_id, int value)
{
	if(this->video_renderer)
		this->video_renderer->videorenderer_set_mixer(u_id, value);
}

// --------------------------------------------------------------------------------------------------------------------

void FFMpegVideoWidget::videowidget_ffmpeg_swctx_resize(int window_width, int window_height, bool do_repaint)
{
	struct dispqt_video_surface_info_s *video_surface_infos = &this->video_surface_infos;
	video_surface_infos->do_resize_repaint = do_repaint;
	video_surface_infos->update_resize = true;
	video_surface_infos->videoout_surface_recalc = true; // FIXME: messy videoout_surface_recalc usage
	this->video_decoder_worker->decoderworker_video_surface_refresh(DISPQT_VIDEOSURFACE_REFRESHTYPE_UPDATE);
}

void FFMpegVideoWidget::videowidget_ffmpeg_swctx_move(int move_x, int move_y)
{
	int widget_global_pos_x = 0, widget_global_pos_y = 0;

	if(!this->video_surface_infos.is_fullscreen)
	{
		widget_global_pos_x = QWidget::mapToGlobal(this->pos()).x();
		widget_global_pos_y = QWidget::mapToGlobal(this->pos()).y();
	}

	if(mpxplay_dispqt_videotrans_videooutput_surface_move_calculate(&this->video_source_infos, &this->video_surface_infos, this->main_window->gui_config, widget_global_pos_x, widget_global_pos_y, move_x, move_y))
	{
		if( funcbit_test(this->video_surface_infos.scheduler_flags, (DISPQT_FFMPGVIDEOCALLBACKFLAG_STILLPICT | DISPQT_FFMPGVIDEOCALLBACKFLAG_RF_PAUSEDPLAY))
		 && PDS_THREADS_MUTEX_LOCK(&this->video_surface_infos.mutex_render, DISPQT_VIDEO_RENDER_MUTEX_TIMEOUT / 4) == MPXPLAY_ERROR_OK)
		{
			this->videowidget_ffmpeg_filtered_surface_update(false);
			PDS_THREADS_MUTEX_UNLOCK(&this->video_surface_infos.mutex_render);
		}
		else
		{
			videowidget_ffmpeg_swctx_resize(this->width(), this->height(), true);
			this->video_surface_infos.videoout_surface_recalc = false; // FIXME: messy videoout_surface_recalc usage (to keep surface_move_calculate result)
		}
	}
}

void FFMpegVideoWidget::videowidget_ffmpeg_swctx_zoom(int globalpos_x, int globalpos_y, int direction)
{
	if(mpxplay_dispqt_videotrans_videooutput_surface_zoom_calculate(&this->video_source_infos, &this->video_surface_infos, this->main_window->gui_config, globalpos_x, globalpos_y, direction))
	{
		if( funcbit_test(this->video_surface_infos.scheduler_flags, (DISPQT_FFMPGVIDEOCALLBACKFLAG_STILLPICT | DISPQT_FFMPGVIDEOCALLBACKFLAG_RF_PAUSEDPLAY))
		 && PDS_THREADS_MUTEX_LOCK(&this->video_surface_infos.mutex_render, DISPQT_VIDEO_RENDER_MUTEX_TIMEOUT / 4) == MPXPLAY_ERROR_OK)
		{
			this->videowidget_ffmpeg_filtered_surface_update(false);
			PDS_THREADS_MUTEX_UNLOCK(&this->video_surface_infos.mutex_render);
		}
		else
		{
			videowidget_ffmpeg_swctx_resize(this->width(), this->height(), true);
		}
	}
	emit this->main_window->signal_video_mouse_cursor_show(true);
}

//-------------------------------------------------------------------------------------------------------------------------------------
void FFMpegVideoWidget::resizeEvent(QResizeEvent *e)
{
	struct dispqt_video_surface_info_s *video_surface_infos = &this->video_surface_infos;
	bool is_widget_renderer = (video_surface_infos->video_render_infos->renderer_capabilities & DISPQT_VIDEORENDERER_CAPFLAG_WIDGET);
	if(!video_surface_infos->is_fullscreen)
	{
		const int new_window_height = e->size().height();
		video_surface_infos->windowed_size_x = e->size().width();;
		if(new_window_height >= 4) // it can be zero at fullscreen switch off (because window of video widget is hide by shrink to zero in fs), we keep the old value (else d3d11 swapchain fails)
			video_surface_infos->windowed_size_y = new_window_height;
	}
	this->videowidget_ffmpeg_swctx_resize(video_surface_infos->windowed_size_x, video_surface_infos->windowed_size_y, !is_widget_renderer);
	if(is_widget_renderer)
		QWidget::resizeEvent(e);
}

void FFMpegVideoWidget::moveEvent(QMoveEvent *e)
{
	struct dispqt_video_surface_info_s *video_surface_infos = &this->video_surface_infos;
	video_surface_infos->window_pos_x = QWidget::mapToGlobal(this->pos()).x();
	video_surface_infos->window_pos_y = QWidget::mapToGlobal(this->pos()).y();
	QWidget::moveEvent(e);
}

void FFMpegVideoWidget::paintEvent(QPaintEvent *event)
{
	struct dispqt_video_surface_info_s *video_surface_infos = &this->video_surface_infos;
	if(this->video_surface_infos.video_render_infos->renderer_capabilities & DISPQT_VIDEORENDERER_CAPFLAG_WIDGET)
	{
		QPainter p;
		p.begin(this);
		this->video_renderer->videorenderer_videoframe_render(&p, video_surface_infos, video_surface_infos->sub_title_infos);
		p.end();
	}
	else
	{
		this->videowidget_ffmpeg_swctx_resize(this->width(), this->height(), true); // FIXME: delayed refresh needed for proper window update?
		if(this->updatesEnabled())
			this->setUpdatesEnabled(false);
	}
	//mpxplay_debugf(DISPQT_DEBUGOUT_RENDER, "paintEvent cp:%d ofr:%d", video_surface_infos->videoout_surface_clearpic, ((video_surface_infos->ffmpeg_output_frame)? 1 : 0));
}

QPaintEngine* FFMpegVideoWidget::paintEngine() const
{
	if(this->video_surface_infos.video_render_infos->renderer_capabilities & DISPQT_VIDEORENDERER_CAPFLAG_WIDGET)
		return QWidget::paintEngine();
	return NULL;
}

//-------------------------------------------------------------------------------------------------------------------------------------
// common video surface functions

void dispqt_videowidget_ffmpeg_subtitle_store(struct dispqt_video_surface_info_s *video_surface_infos, ffmpegvideo_subtitle_info_s *subtitle_infos)
{
	if(video_surface_infos->sub_title_infos)
	{
		mpxplay_video_ffmpeg_subtitleinfo_clear(video_surface_infos->sub_title_infos);
	}
	if(subtitle_infos)
	{
		if(!video_surface_infos->sub_title_infos)
			video_surface_infos->sub_title_infos = (ffmpegvideo_subtitle_info_s *)av_mallocz(sizeof(ffmpegvideo_subtitle_info_s));
		mpxplay_video_ffmpeg_subtitleinfo_copy(video_surface_infos->sub_title_infos, subtitle_infos);
	}
}

void dispqt_videowidget_ffmpeg_subtitle_close(struct dispqt_video_surface_info_s *video_surface_infos)
{
	if(video_surface_infos->sub_title_infos)
	{
		mpxplay_video_ffmpeg_subtitleinfo_clear(video_surface_infos->sub_title_infos);
		av_freep(&video_surface_infos->sub_title_infos);
	}
}

//-------------------------------------------------------------------------------------------------------------------------------------
void FFMpegVideoWidget::videowidget_ffmpeg_filtered_trans_refresh(void)
{
	this->video_surface_infos.update_filter_trans = true;
	this->video_decoder_worker->decoderworker_video_surface_refresh(DISPQT_VIDEOSURFACE_REFRESHTYPE_UPDATE);
}

void FFMpegVideoWidget::videowidget_ffmpeg_filtered_eq_refresh(void)
{
	this->video_surface_infos.update_filter_eq = true;
	this->video_decoder_worker->decoderworker_video_surface_refresh(DISPQT_VIDEOSURFACE_REFRESHTYPE_UPDATE);
}

static void dispqt_videowidget_surface_update_preprocess(struct dispqt_video_surface_info_s *video_surface_infos)
{
	AVFrame *output_frame = video_surface_infos->ffmpeg_output_frame;
	mpxplay_debugf(DISPQT_DEBUGOUT_TRACE, "dispqt_videowidget_surface_update_preprocess BEGIN");
	if(!output_frame)
	{
		dispqt_videowidget_frameforclear_set(video_surface_infos);
		output_frame = video_surface_infos->ffmpeg_output_frame;
	}
	if((video_surface_infos->video_outframe_last_width != output_frame->width) || (video_surface_infos->video_outframe_last_height != output_frame->height))
	{   // video has changed
		video_surface_infos->video_outframe_last_width = output_frame->width;
		video_surface_infos->video_outframe_last_height = output_frame->height;
		mpxplay_dispqt_videotrans_autocrop_clear(video_surface_infos->video_source_infos);
		video_surface_infos->videoout_surface_recalc = true;
		mpxplay_dispqt_videotrans_videoframe_hw_rotate_check(video_surface_infos, output_frame);
		mpxplay_debugf(DISPQT_DEBUGOUT_ACROP, "dispqt_videowidget_surface_update_preprocess RES CHANGE");
	}
	else if(video_surface_infos->update_filter_trans) // FIXME: videowidget_ffmpeg_frame_display -> videowidget_ffmpeg_handle_refresh sometimes is not executed
	{   // rotation has changed (manually)
		mpxplay_dispqt_videotrans_videoframe_hw_rotate_check(video_surface_infos, output_frame);
	}
	mpxplay_dispqt_videotrans_crop_change_check(video_surface_infos->video_source_infos, video_surface_infos);
	video_surface_infos->update_filter_trans = false;
	if(video_surface_infos->videoout_surface_recalc)
	{
		mpxplay_dispqt_videotrans_videooutput_surface_window_calculate(video_surface_infos->video_source_infos, video_surface_infos, video_surface_infos->gui_config);
	}
	mpxplay_debugf(DISPQT_DEBUGOUT_TRACE, "dispqt_videowidget_surface_update_preprocess END");
}

static void dispqt_videowidget_surface_update_postprocess(struct dispqt_video_surface_info_s *video_surface_infos)
{
	if(!funcbit_test(video_surface_infos->video_render_infos->renderer_capabilities, DISPQT_VIDEORENDERER_CAPFLAG_WIDGET))
		video_surface_infos->videoout_surface_clearpic = false;
	video_surface_infos->videoout_surface_reset = false;
	video_surface_infos->videoout_input_change = false;
	video_surface_infos->videoout_inputframe_change = false;
	funcbit_disable(video_surface_infos->scheduler_flags, DISPQT_FFMPGVIDEOCALLBACKFLAGS_ONESHOT);
}

void FFMpegVideoWidget::videowidget_ffmpeg_filtered_surface_update(bool high_priority)
{
	struct dispqt_video_surface_info_s *video_surface_infos = &this->video_surface_infos;

	dispqt_videowidget_surface_update_preprocess(video_surface_infos);

	if(video_surface_infos->video_render_infos->renderer_capabilities & DISPQT_VIDEORENDERER_CAPFLAG_WIDGET)
	{
		if(high_priority)
			this->repaint(); // immediate paint
		else
			this->update();
	}
	else
	{
		this->video_renderer->videorenderer_videoframe_render(video_surface_infos, video_surface_infos->sub_title_infos);
	}

	dispqt_videowidget_surface_update_postprocess(video_surface_infos);
}

//-------------------------------------------------------------------------------------------------------------------------------------

bool FFMpegVideoWidget::videowidget_ffmpeg_handle_refresh(unsigned int refresh_type, bool doRepaint)
{
	struct dispqt_video_source_info_s *video_source_infos = &this->video_source_infos;
	struct dispqt_video_surface_info_s *video_surface_infos = &this->video_surface_infos;

	if((refresh_type != DISPQT_VIDEOSURFACE_REFRESHTYPE_NONE) && (refresh_type != DISPQT_VIDEOSURFACE_REFRESHTYPE_CLOSE))
	{
		if(video_surface_infos->update_resize)
		{
			this->video_sw_transform->videowidget_ffmpeg_filtered_crop_update(false);
			doRepaint |= mpxplay_dispqt_videotrans_videooutput_surface_window_calculate(&this->video_source_infos, &this->video_surface_infos, this->main_window->gui_config);
			doRepaint |= video_surface_infos->do_resize_repaint;
		}
		if(!doRepaint)
		{
			if(video_surface_infos->update_filter_eq)
			{
				this->video_sw_transform->videowidget_ffmpeg_filtered_eq_update();
				this->videowidget_ffmpeg_filtered_surface_update(false);
			}
			else if(video_surface_infos->update_filter_trans || (refresh_type == DISPQT_VIDEOSURFACE_REFRESHTYPE_UPDATE))
			{
				doRepaint = true;
			}
		}
	}

	if(refresh_type != DISPQT_VIDEOSURFACE_REFRESHTYPE_NONE)
	{
		video_surface_infos->update_resize = false;
		video_surface_infos->update_filter_trans = false;
		video_surface_infos->update_filter_eq = false;
	}
	return doRepaint;
}

//------------------------------------------------------------------------------------------------------------------------

bool FFMpegVideoWidget::videowidget_ffmpeg_config_callback(unsigned int cfgcmd, mpxp_ptrsize_t cbk_func, mpxp_ptrsize_t passdata)
{
	bool is_video_open = false;
	mpxplay_debugf(DISPQT_DEBUGOUT_FFMPEG, "videowidget_ffmpeg_config_callback BEGIN %8.8X %8.8X %8.8X", cfgcmd, (int)cbk_func, (int)passdata);
	const bool mutex_locked = this->mutex_widget.tryLock(DISPQT_VIDEO_MUTEX_TIMEOUT);
	const int mutex_error = PDS_THREADS_MUTEX_LOCK(&this->video_surface_infos.mutex_render, DISPQT_VIDEO_MUTEX_TIMEOUT);
	mpxplay_debugf(DISPQT_DEBUGOUT_FFMPEG, "videowidget_ffmpeg_config_callback LOCK %8.8X %8.8X %8.8X lock:%d err:%d", cfgcmd, (int)cbk_func, (int)passdata, (int)mutex_locked, mutex_error);
	if(this->video_decoder_worker)
		is_video_open = this->video_decoder_worker->videothread_ffmpeg_config_callback(cfgcmd, cbk_func, passdata);
	if(mutex_error == MPXPLAY_ERROR_OK)
		PDS_THREADS_MUTEX_UNLOCK(&this->video_surface_infos.mutex_render);
	if(mutex_locked)
		this->mutex_widget.unlock();
	mpxplay_debugf(DISPQT_DEBUGOUT_FFMPEG, "videowidget_ffmpeg_config_callback END %8.8X %8.8X %8.8X lock:%d", cfgcmd, (int)cbk_func, (int)passdata, (int)mutex_locked);
	return is_video_open;
}

void FFMpegVideoWidget::videowidget_ffmpeg_callback_call(int globalpos_x, int globalpos_y, unsigned long command, mpxp_ptrsize_t arg1, mpxp_ptrsize_t arg2)
{
	mpxplay_debugf(DISPQT_DEBUGOUT_FFMPEG, "videowidget_ffmpeg_callback_call BEGIN");
	const bool mutex_locked = this->mutex_widget.tryLock(DISPQT_VIDEO_MUTEX_TIMEOUT);
	if(this->video_decoder_worker)
		this->video_decoder_worker->videothread_ffmpeg_callback_call(globalpos_x, globalpos_y, command, arg1, arg2);
	if(mutex_locked)
		this->mutex_widget.unlock();
	mpxplay_debugf(DISPQT_DEBUGOUT_FFMPEG, "videowidget_ffmpeg_callback_call END");
}

bool FFMpegVideoWidget::videowidget_ffmpeg_schedulerthread_suspend(void)
{
	return this->video_decoder_worker->decoderworker_video_scheduler_control(DISPQT_VIDEOSCHEDULER_CONTROL_REQ_SUSPEND);
}

void FFMpegVideoWidget::videowidget_ffmpeg_schedulerthread_resume(void)
{
	this->video_decoder_worker->decoderworker_video_scheduler_control(DISPQT_VIDEOSCHEDULER_CONTROL_RUN);
}

static bool dispqt_videowidget_surface_info_update(struct dispqt_video_surface_info_s *video_surface_infos, unsigned int flags, AVFrame *videoout_frame, ffmpegvideo_subtitle_info_s *subtitle_infos, unsigned int refresh_type)
{
	struct dispqt_video_source_info_s *video_source_infos = video_surface_infos->video_source_infos;
	bool doRepaint = false;

	if(video_surface_infos->gui_config->video_renderer_type != video_surface_infos->gui_config->selected_videorenderer_type)
		return doRepaint;

	video_surface_infos->scheduler_flags = flags;

	if(refresh_type == DISPQT_VIDEOSURFACE_REFRESHTYPE_UPDATE) // means redraw
	{
		doRepaint = true;
	}
	else if(refresh_type == DISPQT_VIDEOSURFACE_REFRESHTYPE_RESET) // means seeking
	{
		av_frame_free(&video_source_infos->ffmpeg_decoded_frame);
		mpxplay_dispqt_videotrans_autocrop_reset(video_source_infos);
		video_surface_infos->videoout_surface_reset = true;
		mpxplay_debugf(DISPQT_DEBUGOUT_ACROP, "dispqt_videowidget_surface_info_update RESET");
	}
	else if(refresh_type == DISPQT_VIDEOSURFACE_REFRESHTYPE_CLEAR) // means clear window
	{
		video_surface_infos->videoout_surface_recalc = true;
		video_surface_infos->videoout_surface_clearpic = true;
		doRepaint = true;
		mpxplay_debugf(DISPQT_DEBUGOUT_FFMPEG, "dispqt_videowidget_surface_info_update CLEAR");
	}
	else if(refresh_type == DISPQT_VIDEOSURFACE_REFRESHTYPE_CLOSE) // means close rendering (stop playing)
	{
		video_surface_infos->ffmpeg_output_frame = NULL;
		av_frame_free(&video_source_infos->ffmpeg_decoded_frame);
		mpxplay_dispqt_videotrans_ffmpeg_swsctx_close(video_surface_infos->video_filter_infos);
		dispqt_videowidget_ffmpeg_subtitle_close(video_surface_infos);
		video_surface_infos->videoout_surface_recalc = true;
		video_surface_infos->videoout_surface_clearpic = true;
		doRepaint = true;
		mpxplay_debugf(DISPQT_DEBUGOUT_FFMPEG, "dispqt_videowidget_surface_info_update CLOSE");
	}

	if(flags & DISPQT_FFMPGVIDEOCALLBACKFLAG_FIRSTFRAME) // means new file
	{
		video_surface_infos->video_outframe_last_width = 0;
		video_surface_infos->video_outframe_last_height = 0;
		mpxplay_dispqt_videotrans_autocrop_clear(video_source_infos);
		video_surface_infos->render_base->renderbase_subtitle_reset();
		video_surface_infos->videoout_input_change = true;
		video_surface_infos->videoout_surface_reset = true;
		mpxplay_debugf(DISPQT_DEBUGOUT_ACROP, "dispqt_videowidget_surface_info_update FIRSTFRAME");
	}

	if(videoout_frame) // there is a new decoded video frame
	{
		mpxplay_debugf(DISPQT_DEBUGOUT_TRACE, "dispqt_videowidget_surface_info_update videoout_frame BEGIN");
		if(video_surface_infos->sub_title_infos)
		{
			mpxplay_video_ffmpeg_subtitleinfo_clear(video_surface_infos->sub_title_infos);
		}
		if(subtitle_infos)
		{
			if(!video_surface_infos->sub_title_infos)
				video_surface_infos->sub_title_infos = (ffmpegvideo_subtitle_info_s *)av_mallocz(sizeof(ffmpegvideo_subtitle_info_s));
			mpxplay_video_ffmpeg_subtitleinfo_copy(video_surface_infos->sub_title_infos, subtitle_infos);
		}
		if((video_source_infos->video_aspect_ratio.num != videoout_frame->sample_aspect_ratio.num) || (video_source_infos->video_aspect_ratio.den != videoout_frame->sample_aspect_ratio.den))
		{
			video_source_infos->video_aspect_ratio.num = videoout_frame->sample_aspect_ratio.num;
			video_source_infos->video_aspect_ratio.den = videoout_frame->sample_aspect_ratio.den;
			video_surface_infos->videoout_surface_recalc = true;
		}

		mpxplay_debugf(DISPQT_DEBUGOUT_TRACE, "dispqt_videowidget_surface_info_update videoout_frame FREE %8.8X", (mpxp_ptrsize_t)video_source_infos->ffmpeg_decoded_frame);
		av_frame_free(&video_source_infos->ffmpeg_decoded_frame);
		video_source_infos->ffmpeg_decoded_frame = videoout_frame;
		video_surface_infos->render_base->videorenderer_get_surface_attribs(video_surface_infos, videoout_frame);
		video_surface_infos->video_filter_infos->ffmpeg_filtered_frame->format = video_surface_infos->videoout_surface_pix_fmt; // aka this->video_sw_transform->videowidget_ffmpeg_videoctx_reconfig()
		video_surface_infos->videoout_inputframe_change = true;
		doRepaint = true;
	}

	mpxplay_debugf(DISPQT_DEBUGOUT_TRACE, "dispqt_videowidget_surface_info_update END dr:%d", (int)doRepaint);

	return doRepaint;
}

void FFMpegVideoWidget::videowidget_ffmpeg_frame_display(int video_index, unsigned int flags, AVFrame *videoout_frame, ffmpegvideo_subtitle_info_s *subtitle_infos, unsigned int refresh_type)
{
	struct dispqt_video_surface_info_s *video_surface_infos = &this->video_surface_infos;
#ifdef MPXPLAY_USE_DEBUGF
	mpxp_int64_t sendframe_begin = pds_gettimem(), sendframe_trans;
#endif
	bool doRepaint = false;

	mpxplay_debugf(DISPQT_DEBUGOUT_RENDER, "frame_display BEGIN rt:%d", refresh_type);

	if( funcbit_test(mpxplay_signal_events, MPXPLAY_SIGNALTYPE_GUIREBUILD)
	 || this->main_window->gui_config->video_renderer_type != this->main_window->gui_config->selected_videorenderer_type)
	{
		mpxplay_debugf(DISPQT_DEBUGOUT_RENDER, "frame_display RET");
		return;
	}

	doRepaint = dispqt_videowidget_surface_info_update(video_surface_infos, flags, videoout_frame, subtitle_infos, refresh_type);

	if(refresh_type == DISPQT_VIDEOSURFACE_REFRESHTYPE_CLOSE) // means close rendering
	{
		this->video_sw_transform->videowidget_ffmpeg_videoctx_clear();
		this->video_sw_transform->videowidget_ffmpeg_swsctx_close();
	}

	doRepaint = this->videowidget_ffmpeg_handle_refresh(refresh_type, doRepaint);

	if(doRepaint)
	{
		this->video_sw_transform->videowidget_ffmpeg_filtered_trans_update();
#ifdef MPXPLAY_USE_DEBUGF
		sendframe_trans = pds_gettimem();
#endif
		this->videowidget_ffmpeg_filtered_surface_update(true);
	}

	mpxplay_videoworkdec_videowidget_format.width = video_surface_infos->window_size_x;
	mpxplay_videoworkdec_videowidget_format.height = video_surface_infos->window_size_y;

	mpxplay_debugf(DISPQT_DEBUGOUT_RENDER, "frame_display END do:%d dur:%2d tr:%2d up:%d", doRepaint, (int)(pds_gettimem() - sendframe_begin),
			(int)(sendframe_trans - sendframe_begin), (int)(pds_gettimem() - sendframe_trans));
}

mpxp_bool_t mpxplay_dispqt_videowidget_direct_frame_display(dispqt_video_surface_info_s *video_surface_infos, unsigned int flags, AVFrame *videoout_frame, ffmpegvideo_subtitle_info_s *subtitle_infos, unsigned int refresh_type)
{
#ifdef MPXPLAY_USE_DEBUGF
	mpxp_int64_t sendframe_begin = pds_gettimem(), sendframe_mutex, sendframe_trans;
#endif
	bool doRepaint;

	mpxplay_debugf(DISPQT_DEBUGOUT_RENDER, "direct_frame_display BEGIN rt:%d", refresh_type);

	if(funcbit_test(mpxplay_signal_events, MPXPLAY_SIGNALTYPE_GUIREBUILD))
		return FALSE;
	if(video_surface_infos->gui_config->video_renderer_type != video_surface_infos->gui_config->selected_videorenderer_type)
		return FALSE;

	if(PDS_THREADS_MUTEX_LOCK(&video_surface_infos->mutex_render, DISPQT_VIDEO_RENDER_MUTEX_TIMEOUT) != MPXPLAY_ERROR_OK)
	{
		mpxplay_debugf(DISPQT_DEBUGOUT_RENDER, "direct_frame_display LOCK ERROR");
		return FALSE;
	}

#ifdef MPXPLAY_USE_DEBUGF
	sendframe_mutex = pds_gettimem();
#endif

	doRepaint = dispqt_videowidget_surface_info_update(video_surface_infos, flags, videoout_frame, subtitle_infos, refresh_type);

	if(doRepaint)
	{
		AVFrame *output_frame = video_surface_infos->video_source_infos->ffmpeg_decoded_frame;
		if(output_frame && (output_frame->width > 0) && (output_frame->format != mpxplay_video_render_infos.hwdevice_avpixfmt) && !mpxplay_video_render_infos.render_function_poolbufelem_validity_check(output_frame))
		{
			if(video_surface_infos->videoout_surface_pix_fmt != (AVPixelFormat)output_frame->format)
			{
				if(mpxplay_dispqt_videotrans_ffmpeg_swsctx_pixfmt_conv(video_surface_infos->video_filter_infos, output_frame, video_surface_infos->videoout_surface_pix_fmt))
				{
					output_frame = video_surface_infos->video_filter_infos->ffmpeg_sws_frame;
				}
			}
			if((video_surface_infos->gui_config->video_crop_type == DISPQT_VIDEOSCRCROPTYPE_AUTO) && (output_frame->height <= video_surface_infos->gui_config->video_autocrop_limit))
			{
				if(mpxplay_dispqt_videotrans_autocrop_process(output_frame->data[0], output_frame->linesize[0], output_frame->format,
						output_frame->width, output_frame->height, output_frame->linesize[0] / output_frame->width,
						video_surface_infos->video_source_infos, &video_surface_infos->videoout_surface_clearpic))
				{
					video_surface_infos->videoout_surface_recalc = true;
				}
			}
		}
		video_surface_infos->ffmpeg_output_frame = output_frame;

		dispqt_videowidget_surface_update_preprocess(video_surface_infos);

#ifdef MPXPLAY_USE_DEBUGF
		sendframe_trans = pds_gettimem();
#endif

		video_surface_infos->render_base->videorenderer_videoframe_render(video_surface_infos, subtitle_infos);

		dispqt_videowidget_surface_update_postprocess(video_surface_infos);
	}

	mpxplay_videoworkdec_videowidget_format.width = video_surface_infos->window_size_x;
	mpxplay_videoworkdec_videowidget_format.height = video_surface_infos->window_size_y;

	mpxplay_debugf(DISPQT_DEBUGOUT_RENDER, "direct_frame END dur:%2d mut:%d tr:%2d up:%d", (int)(pds_gettimem() - sendframe_begin),
			(int)(sendframe_mutex - sendframe_begin), (int)(sendframe_trans - sendframe_mutex), (int)(pds_gettimem() - sendframe_trans));

	PDS_THREADS_MUTEX_UNLOCK(&video_surface_infos->mutex_render);

	return TRUE;
}

#endif // MPXPLAY_LINK_ORIGINAL_FFMPEG
