//**************************************************************************
//*                     This file is part of the                           *
//*                MMC - Mpxplay Multimedia Commander                      *
//*                   The source code of MMC is                            *
//*        (C) copyright 1998-2022 by PDSoft (Attila Padar)                *
//*                http://mpxplay.sourceforge.net                          *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//function: Direct3D 9 render output definitions

#ifndef VIDEO_RENDER_DX9_H
#define VIDEO_RENDER_DX9_H

#include <d3d9.h>
#include <d3dx9effect.h>
#include <dxva2api.h>
#include "video_render_comm.h"

#define DISPQT_D3D9_USE_SHADER_VS 0

#define DISPQT_D3D9_SWAPCHAN_BUFFERS    2

#define DISPQT_D3D9_DEINTERLACE_SWITCHOFF_COUNT 30

#define DISPQT_RENDER_D3D9_VERTEX_POSITION_SCALE DISPQT_RENDER_VERTEX_POSITION_BASE_SCALE

#define DISPQT_D3D9_FVF_FORMAT (D3DFVF_XYZ | D3DFVF_TEX1) // d3d9 format descriptor of d3d_vertex_t

enum {DISPQT_D3D9_PIXSHADERCONST_VIDEOINFOS = 0, DISPQT_D3D9_PIXSHADERCONST_COLORMIXER, DISPQT_D3D9_PIXSHADERCONST_COLORMIXHUE, DISPQT_D3D9_PIXSHADERCONST_NUM};

typedef LPDIRECT3D9 (WINAPI* PFN_DIRECT3D_CREATE9FUNC)(UINT SDKVersion);
typedef HRESULT WINAPI pCreateDeviceManager9(UINT *, IDirect3DDeviceManager9 **);

typedef struct dispqt_renderer_d3d9_static_info_s{
	HMODULE            d3d9_dll_hnd;       // dll handler for D3D9.DLL
	HMODULE            dxva_dll_hnd;       // dll handler for DXVA2.DLL
	IDirect3D9        *d3d_device_ptr;     // IDirect3D9
	IDirect3DDevice9  *d3d_device_ctx;     // IDirect3DDevice9 current/latest
	pCreateDeviceManager9 *createDeviceManager;  // pointer to DXVA2CreateDirect3DDeviceManager9 function
	IDirect3DDeviceManager9 *d3d_device_manager; // IDirect3DDeviceManager9 handler
	mpxp_bool_t        reopen_devicemanager;
}dispqt_renderer_d3d9_static_info_s;

typedef HRESULT (WINAPI * PFN_D3D9_COMPILE_SHADER)(
	LPCSTR pSrcData,
	UINT srcDataLen,
	const D3DXMACRO *pDefines,
	LPD3DXINCLUDE pInclude,
	LPCSTR pFunctionName,
	LPCSTR pProfile,
	DWORD Flags,
	LPD3DXBUFFER *ppShader,
	LPD3DXBUFFER *ppErrorMsgs,
	LPD3DXCONSTANTTABLE *ppConstantTable);

typedef struct dispqt_render_d3d9_api_s{
	HMODULE                   d3dcompiler_dll_hnd;           // handler for D3DCOMPILER_4x.DLL
	PFN_D3D9_COMPILE_SHADER   d3d_shader_compiler_func;      // function pointer to D3DXCompileShader
}dispqt_render_d3d9_api_s;

typedef struct dispqt_d3d9_deinterlace_data_s
{
	int deinterlace_switch_status;                           // is deinterlace currently enabled (in ps)?
	int deinterlace_switchoff_counter;                       // switch off deinterlace after some time (keep it for a while)
}dispqt_d3d9_deinterlace_data_s;

typedef struct dispqt_render_d3d9_input_texture_data_s{
	unsigned int               texture_id;                   // DISPQT_EVR_INPUTTEXUREID_
	int                        input_width, input_height;    // resolution of selected input texture (d3d_input_texture_2d, hwdec or pool)
	int                        video_width, video_height;    // reference video resolution, used to create d3d_input_surface_2d (without alignment)
	int                        texture_width, texture_height;// resolution of the created d3d_input_texture_2d (with alignment at video frames) (at subtitles: video and texture resolutions are always equal)
	int                        av_pixelformat;               // pixel format of input video (can be AV_PIX_FMT_DXVA_VLD or match with avframe pixfmt)
	AVColorTransferCharacteristic av_colortrc;               // color transfer characteristic of the input video
	enum AVColorRange          av_colorrange;                // color range of input video (MPEG/FULL)
	AVColorSpace               av_colorspace;                // color space of input video
	AVColorPrimaries           av_colorprimaries;            // color primaries of the input video
#if DISPQT_D3D9_USE_SHADER_VS
	IDirect3DVertexShader9    *d3d_vertex_shader_hnd;        // handler of compiled vertex shader code           // because both of them uses flat/constant vertices (later vertex method of video frame will be different)
#endif
	IDirect3DVertexBuffer9    *d3d_vertex_buffer;            // transformation data buffer of vertex shader
	IDirect3DPixelShader9     *d3d_pixel_shader_hnd;         // handler of compiled pixel shader code
	IDirect3DTexture9         *d3d_input_texture_2d;         // created input texture for subtitles
	IDirect3DSurface9         *d3d_input_surface_2d;         // created input surface for sw decoded frames
	struct dispqt_evr_format_desc_s *pixel_format_infos;     // informations of selected input pixelformat
	struct dispqt_evr_format_desc_s custom_pixel_format_info;// informations of a non standard/base texture format
	struct dispqt_evr_input_viewport_s texture_crop;         // last set texture crop (input of vertex)
	struct dispqt_evr_input_viewport_s texture_viewport;     // last set viewport values (output rectangle)
//	struct dispqt_evr_deinterlace_data_s deinterlace_datas;  // datas related to the deinterlacing
}dispqt_render_d3d9_input_texture_data_s;

typedef struct dispqt_render_d3d9_hwdec_data_s{
	int       hwdec_tex_width, hwdec_tex_height;         // real resolution of hwdec input texture
	D3DFORMAT hwdec_tex_format;                          // d3d pixelformat of hwdec input texture
}dispqt_render_d3d9_hwdec_data_s;

typedef struct dispqt_render_d3d9_textdown_data_s{
	IDirect3DSurface9  *d3d_surface_video_download;     // used to copy video texture from gpu to main memory (for autocrop)
	int down_tex_width, down_tex_height;                // rectangles of the allocated texture
	D3DFORMAT down_text_format;                         // d3d format of the allocated texture
	mpxp_uint64_t last_process_time_ms;                 // to gain the autocrop processing (>=50fps can block the GPU)
}dispqt_render_d3d9_textdown_data_s;

typedef struct dispqt_render_d3d9_data_s{
	HWND                   parent_window_handler;
	IDirect3D9             *d3d_device;
	IDirect3DDevice9       *d3d_device_context;
	IDirect3DSurface9      *d3d_hwdec_surface_ptr;        // pointer to the received hw decoded frame surface
	IDirect3DTexture9      *d3d_render_target_texture;    // render target texture
	IDirect3DSurface9      *d3d_render_target_surface;    // current working/backbuffer surface of d3d_render_target_texture (temp)
	IDirect3DSurface9      *d3d_screen_target_surface;    // GetRenderTarget of device surface
	mpxp_uint32_t          clear_counter;
	int                    subtitle_rendermode;
	mpxp_bool_t            update_vmixer_data;
	mpxp_bool_t            is_transparent_gui_used;
	mpxp_int32_t           vmixer_values[DISPQT_VIDEO_VMIXERTYPE_MAX];
	struct dispqt_render_d3d9_api_s d3d_api_funcs;
	struct dispqt_d3d9_deinterlace_data_s deinterlace_datas;
	struct dispqt_render_d3d9_hwdec_data_s hwdec_datas;
	struct dispqt_render_d3d9_textdown_data_s textdown_datas;
	struct dispqt_render_d3d9_input_texture_data_s input_texture_datas[DISPQT_EVR_INPUTTEXUREID_NUM];
	D3DCAPS9               d3d_device_caps;
	D3DPRESENT_PARAMETERS  d3d_present_params;
	struct dispqt_render_poolbufelem_s decoder_pool_datas[DISPQT_RENDER_POOLBUF_SIZE_MAX];
}dispqt_render_d3d9_data_s;

// ====================================================================================================================
// structures passed to the pixel shader

typedef struct dispqt_d3d9_video_info_s {
	FLOAT EnableDeinterlace;
	FLOAT VideoHeight;
    FLOAT dummy[2];
} dispqt_d3d9_video_info_s;

// ====================================================================================================================
// pixel shader code blocks for Shader Model 2.0 (D3D9)

static const char *dispqt_d3d9_pscode_COMMON_datas = "\
sampler samplerStateHnd : register(s0);\n\
\n";

static const char *dispqt_d3d9_pscode_CONSTANTDATA0_color_mixer = "\
float4 videoinfos : register(c0);\n\
\n\
#define EnableDeinterlace      (videoinfos[0])\n\
#define VideoHeight            (videoinfos[1])\n\
\n\
float4 cmixvals : register(c1);\n\
\n\
#define ColorMixerBrightness   (cmixvals[0])\n\
#define ColorMixerContrast     (cmixvals[1])\n\
#define ColorMixerSaturation   (cmixvals[2])\n\
#define ColorMixerHueVal       (cmixvals[3])\n\
\n\
float4x4 ColorMixerHueMatrix : register(c2);\n\
\n";

static const char *dispqt_d3d9_pscode_mainfunc_BEGIN ="\
float4 main(float2 Tex : TEXCOORD0) : COLOR\n\
{\n\
    float4 rgba;\n";

static const char *dispqt_d3d9_pscode_mainfunc_GET_RGB_packed = "\
    rgba = tex2D(samplerStateHnd, Tex);\n";

static const char *dispqt_d3d9_pscode_GET_RGB_packed_deinterlace_blend = "\
    if(EnableDeinterlace == 0.0f)\n\
    {\n\
        rgba = tex2D(samplerStateHnd, Tex);\n\
    }\n\
    else\n\
    {\n\
        rgba = tex2D(samplerStateHnd, Tex);\n\
        float2 h = float2(0, 1 / VideoHeight);\n\
        float4 c1 = tex2D(samplerStateHnd, Tex - h);\n\
        float4 c2 = tex2D(samplerStateHnd, Tex + h);\n\
        rgba = (rgba * 2 + c1 + c2) / 4;\n\
    }\n\
\n";

// ====================================================================================================================
// supported color formats and their attributes (sorted by native-support fall-through)

static struct dispqt_evr_format_desc_s dispqt_render_d3d9_supported_pixel_formats[] = { // TODO: check other unique pixel formats, not handled by dispqt_evr_input_select_color_format() generally
 { AV_PIX_FMT_BGR565LE,   D3DFMT_R5G6B5,                   5, 0, 1, 0, 0, 1, { D3DFMT_R5G6B5 } },
 { AV_PIX_FMT_RGB0,       D3DFMT_X8B8G8R8,                 8, 0, 1, 0, 0, 1, { D3DFMT_X8R8G8B8 } },
 { AV_PIX_FMT_BGR0,       D3DFMT_X8R8G8B8,                 8, 0, 1, 0, 0, 1, { D3DFMT_X8R8G8B8 } },
 { AV_PIX_FMT_RGBA,       D3DFMT_A8B8G8R8,                 8, 0, 1, 0, 0, 1, { D3DFMT_A8B8G8R8 } },
 { AV_PIX_FMT_BGRA,       D3DFMT_A8R8G8B8,                 8, 0, 1, 0, 0, 1, { D3DFMT_A8R8G8B8 } },
 { AV_PIX_FMT_YUYV422,    D3DFMT_YUY2,                     8, 0, 0, 1, 1, 1, { D3DFMT_YUY2 } },
 { AV_PIX_FMT_YUV420P,    MAKEFOURCC('Y','V','1','2'),     8, 0, 0, 1, 1, 3, { D3DFMT_L8, D3DFMT_A8, D3DFMT_A8 } },
 { AV_PIX_FMT_NV24,       MAKEFOURCC('N','V','1','2'),     8, 0, 0, 0, 0, 2, { D3DFMT_L8, D3DFMT_A8L8 } }, // Y, U and V sizes are equal
 { AV_PIX_FMT_NV42,       MAKEFOURCC('N','V','1','2'),     8, 0, 0, 0, 0, 2, { D3DFMT_L8, D3DFMT_A8L8 } }, // Y, U and V sizes are equal
 { AV_PIX_FMT_NV12,       MAKEFOURCC('N','V','1','2'),     8, 0, 0, 1, 1, 2, { D3DFMT_L8, D3DFMT_A8L8 } }, // if NV12 works, stop the 8 bit search (pixfmt replacement) here
 { AV_PIX_FMT_RGBA64BE,   D3DFMT_A16B16G16R16,            16, 0, 1, 0, 0, 1, { D3DFMT_A16B16G16R16 } },
 { AV_PIX_FMT_P016LE,     MAKEFOURCC('P','0','1','0'),    16, 0, 0, 1, 1, 2, { D3DFMT_L16, D3DFMT_G16R16 } },
 { AV_PIX_FMT_P010LE,     MAKEFOURCC('P','0','1','0'),    10, 0, 0, 1, 1, 2, { D3DFMT_L16, D3DFMT_G16R16 } },
 { AV_PIX_FMT_NV21,       MAKEFOURCC('N','V','1','2'),     8, 0, 0, 1, 1, 2, { D3DFMT_L8, D3DFMT_A8L8 } },
 { AV_PIX_FMT_NV12,       MAKEFOURCC('N','V','1','2'),     8, 0, 0, 1, 1, 2, { D3DFMT_L8, D3DFMT_A8L8 } },
 { AV_PIX_FMT_YUYV422,    D3DFMT_YUY2,                     8, 0, 0, 1, 1, 1, { D3DFMT_YUY2 } },
 { AV_PIX_FMT_NONE,       D3DFMT_UNKNOWN,                  0, 0, 0, 0, 0, 0, {} }
};

#endif // VIDEO_RENDER_DX9_H
