//**************************************************************************
//*                     This file is part of the                           *
//*                MMC - Mpxplay Multimedia Commander                      *
//*                   The source code of MMC is                            *
//*        (C) copyright 1998-2022 by PDSoft (Attila Padar)                *
//*                http://mpxplay.sourceforge.net                          *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//function: common functions for render output

//#define MPXPLAY_USE_DEBUGF 1
#define DISPQT_DEBUG_ERROR stderr
#define DISPQT_DEBUG_WARNING stdout // stderr
#define DISPQT_DEBUGOUT_TEXTURE NULL // stdout
#define DISPQT_DEBUGOUT_POOL stdout
#define DISPQT_DEBUGOUT_LUMINANCE NULL // stdout

#include "moc_config.h"
#include "moc_video_qt.h"
#include "moc_mainwindow.h"
#include "video_render_comm.h"
#include <libavutil/mastering_display_metadata.h>
#include <dxgiformat.h> // TODO: remove

extern "C" {
 // FFmpeg colorspace.c and imgutils_init.c functions (compiled, but it's not the part of the public API)
 extern int  ff_image_copy_plane_uc_from_x86(uint8_t *dst, ptrdiff_t dst_linesize, const uint8_t *src, ptrdiff_t src_linesize, ptrdiff_t bytewidth, int height);
 extern void ff_matrix_invert_3x3(const double in[3][3], double out[3][3]);
 extern void ff_matrix_mul_3x3(double dst[3][3], const double src1[3][3], const double src2[3][3]);
}

typedef struct dispqt_evr_colormatrix_light_level_s
{
	FLOAT black_level, light_range, color_range, achromacy;
}dispqt_evr_colormatrix_light_level_s;

enum DispQtEvrWhitepointIds { WP_D65, WP_C, WP_DCI, WP_E, WP_NB};

struct dispst_evr_color_primariy_s {
	const enum DispQtEvrWhitepointIds wp_id;
	const struct PrimaryCoefficients primary_coefs;
};

static const FLOAT dispqt_evr_colormatrix_whitepoints[4 * 4] = {
    1.f, 0.f, 0.f, 0.f,
    0.f, 1.f, 0.f, 0.f,
    0.f, 0.f, 1.f, 0.f,
    0.f, 0.f, 0.f, 1.f,
};

// color matrixes without range adjustment
// based on http://www.videoq.com/Downloads/VQYT/VideoQ_Presents_Video_Standards_Part6_Color_Spaces_Formats_Levels_Ranges.PDF
static const FLOAT dispqt_evr_colormatrix_BT601_YUV_to_RGBA[4*4] = {
	1.0f,     0.000f,    1.402f, 0.0f,
	1.0f,  -0.34414f, -0.71414f, 0.0f,
	1.0f,     1.772f,    0.000f, 0.0f,
	0.0f,       0.0f,      0.0f, 1.0f
};

static const FLOAT dispqt_evr_colormatrix_BT709_YUV_to_RGBA[4*4] = {
	1.0f,     0.000f,   1.5748f, 0.0f,
	1.0f,  -0.18732f, -0.46812f, 0.0f,
 	1.0f,    1.8556f,    0.000f, 0.0f,
	0.0f,       0.0f,      0.0f, 1.0f
};

static const FLOAT dispqt_evr_colormatrix_BT2020_YUV_to_RGBA[4*4] = {
	1.0f,     0.0f,   1.4746, 0.0f,
	1.0f, -0.16455, -0.57135, 0.0f, // VideoQ doc
	//1.0f, -0.1115678, -0.38737742, 0.0f, // other source (lighter red colors)
	1.0f,   1.8814,     0.0f, 0.0f,
	0.0f,      0.f,      0.f, 1.0f,
};

// light & range levels for unknown, 8, 10 or 12 bit inputs (bits / 2 - 3)
static const struct dispqt_evr_colormatrix_light_level_s dispqt_evr_colormatrix_light_levels[4] = {
	{16.0f/256.0f,  219.0f/256.0f,  224.0f/256.0f, 128.0f/256.0f},  {16.0f/255.0f, 219.0f/255.0f, 224.0f/255.0f, 128.0f/255.0f},
	{64.0f/1023.0f, 876.0f/1023.0f, 896.0f/1023.0f, 512.0f/1023.0f}, {256.0f/4095.0f, 3504.0f/4095.0f, 3584.0f/4095.0f, 2048.0f/4095.0f},
};

static const struct WhitepointCoefficients dispqt_evr_whitepoint_coefficients[WP_NB] = {
 { 0.3127, 0.3290 }, { 0.3100, 0.3160 }, { 0.3140, 0.3510 }, { 1/3.0f, 1/3.0f }
};

static const struct dispst_evr_color_primariy_s dispqt_evr_standard_color_primaries[AVCOL_PRI_SMPTE432 + 2] = { // FIXME: AVColorPrimaries dependant field
 {},                                                       // AVCOL_PRI_RESERVED0
 { WP_D65, { 0.640, 0.330, 0.300, 0.600, 0.150, 0.060} },  // AVCOL_PRI_BT709
 {},                                                       // AVCOL_PRI_UNSPECIFIED
 {},                                                       // AVCOL_PRI_RESERVED
 { WP_C,   { 0.670, 0.330, 0.210, 0.710, 0.140, 0.080 } }, // AVCOL_PRI_BT470M
 { WP_D65, { 0.640, 0.330, 0.290, 0.600, 0.150, 0.060 } }, // AVCOL_PRI_BT470BG
 { WP_D65, { 0.630, 0.340, 0.310, 0.595, 0.155, 0.070 } }, // AVCOL_PRI_SMPTE170M
 { WP_D65, { 0.630, 0.340, 0.310, 0.595, 0.155, 0.070 } }, // AVCOL_PRI_SMPTE240M
 { WP_C,   { 0.681, 0.319, 0.243, 0.692, 0.145, 0.049 } }, // AVCOL_PRI_FILM
 { WP_D65, { 0.708, 0.292, 0.170, 0.797, 0.131, 0.046 } }, // AVCOL_PRI_BT2020
 { WP_E,   { 0.735, 0.265, 0.274, 0.718, 0.167, 0.009 } }, // AVCOL_PRI_SMPTE428
 { WP_DCI, { 0.680, 0.320, 0.265, 0.690, 0.150, 0.060 } }, // AVCOL_PRI_SMPTE431
 { WP_D65, { 0.680, 0.320, 0.265, 0.690, 0.150, 0.060 } }, // AVCOL_PRI_SMPTE432
 { WP_D65, { 0.630, 0.340, 0.295, 0.605, 0.155, 0.077 } }, // AVCOL_PRI_JEDEC_P22 (it's the elem 13. here, not the 22., corrected in dispqt_d3d_shader_pixel_get_colorprimaries_matrix)
};

//------------------------------------------------------------------------------------------------------------
// calculate color primaries matrix (for pixel shader)

/*
 * RGB to XYZ conversion from FFmpeg colorspace.c (moved to here due to v5.1 API change)
 * see e.g. http://www.brucelindbloom.com/index.html?Eqn_RGB_XYZ_Matrix.html
 */
static void dispqt_evr_colorcv_primaries_fill_rgb2xyz_table(const struct PrimaryCoefficients *coeffs,
                           const struct WhitepointCoefficients *wp,
                           double rgb2xyz[3][3])
{
    double i[3][3], sr, sg, sb, zw;

    rgb2xyz[0][0] = coeffs->xr / coeffs->yr;
    rgb2xyz[0][1] = coeffs->xg / coeffs->yg;
    rgb2xyz[0][2] = coeffs->xb / coeffs->yb;
    rgb2xyz[1][0] = rgb2xyz[1][1] = rgb2xyz[1][2] = 1.0;
    rgb2xyz[2][0] = (1.0 - coeffs->xr - coeffs->yr) / coeffs->yr;
    rgb2xyz[2][1] = (1.0 - coeffs->xg - coeffs->yg) / coeffs->yg;
    rgb2xyz[2][2] = (1.0 - coeffs->xb - coeffs->yb) / coeffs->yb;
    ff_matrix_invert_3x3(rgb2xyz, i);
    zw = 1.0 - wp->xw - wp->yw;
    sr = i[0][0] * wp->xw + i[0][1] * wp->yw + i[0][2] * zw;
    sg = i[1][0] * wp->xw + i[1][1] * wp->yw + i[1][2] * zw;
    sb = i[2][0] * wp->xw + i[2][1] * wp->yw + i[2][2] * zw;
    rgb2xyz[0][0] *= sr; rgb2xyz[0][1] *= sg; rgb2xyz[0][2] *= sb;
    rgb2xyz[1][0] *= sr; rgb2xyz[1][1] *= sg; rgb2xyz[1][2] *= sb;
    rgb2xyz[2][0] *= sr; rgb2xyz[2][1] *= sg; rgb2xyz[2][2] *= sb;
}

// calculate whitepoint matrix for chromatic adaptation based on FFmpeg colorspace video filter
// (see http://www.brucelindbloom.com/index.html?Eqn_ChromAdapt.html)
static void dispqt_evr_colorcv_primaries_fill_whitepoint_conv_table(double out[3][3], const struct WhitepointCoefficients *wp_src, const struct WhitepointCoefficients *wp_dst)
{
	static const double ma[3][3] = { // Bradford method
		{  0.8951,  0.2664, -0.1614 },
		{ -0.7502,  1.7135,  0.0367 },
		{  0.0389, -0.0685,  1.0296 }
	};
	double zw_src = 1.0 - wp_src->xw - wp_src->yw;
	double zw_dst = 1.0 - wp_dst->xw - wp_dst->yw;
	double mai[3][3], fac[3][3], tmp[3][3];
	double rs, gs, bs, rd, gd, bd;

	ff_matrix_invert_3x3(ma, mai);
	rs = ma[0][0] * wp_src->xw + ma[0][1] * wp_src->yw + ma[0][2] * zw_src;
	gs = ma[1][0] * wp_src->xw + ma[1][1] * wp_src->yw + ma[1][2] * zw_src;
	bs = ma[2][0] * wp_src->xw + ma[2][1] * wp_src->yw + ma[2][2] * zw_src;
	rd = ma[0][0] * wp_dst->xw + ma[0][1] * wp_dst->yw + ma[0][2] * zw_dst;
	gd = ma[1][0] * wp_dst->xw + ma[1][1] * wp_dst->yw + ma[1][2] * zw_dst;
	bd = ma[2][0] * wp_dst->xw + ma[2][1] * wp_dst->yw + ma[2][2] * zw_dst;
	fac[0][0] = rd / rs;
	fac[1][1] = gd / gs;
	fac[2][2] = bd / bs;
	fac[0][1] = fac[0][2] = fac[1][0] = fac[1][2] = fac[2][0] = fac[2][1] = 0.0;
	ff_matrix_mul_3x3(tmp, ma, fac);
	ff_matrix_mul_3x3(out, tmp, mai);
}

static void dispqt_render_common_shader_pixel_get_colorprimaries_matrix(FLOAT Primaries[4*4], enum AVColorPrimaries src, enum AVColorPrimaries dst,
		const struct WhitepointCoefficients *custom_whitepoints_src, const struct PrimaryCoefficients *custom_primary_coefs_src)
{
	const struct dispst_evr_color_primariy_s *cp_src = &dispqt_evr_standard_color_primaries[(src > AVCOL_PRI_SMPTE432)? (AVCOL_PRI_SMPTE432 + 1) : src];
	const struct dispst_evr_color_primariy_s *cp_dst = &dispqt_evr_standard_color_primaries[dst];
	const struct WhitepointCoefficients *wp_src = (custom_whitepoints_src)? custom_whitepoints_src : &dispqt_evr_whitepoint_coefficients[cp_src->wp_id];
	const struct WhitepointCoefficients *wp_dst = &dispqt_evr_whitepoint_coefficients[cp_dst->wp_id];
	const struct PrimaryCoefficients *primary_coefs_src = (custom_primary_coefs_src)? custom_primary_coefs_src : &cp_src->primary_coefs;
	double rgb2xyz[3][3], xyz2rgb[3][3], out_rgb[3][3];

	mpxplay_debugf(DISPQT_DEBUGOUT_LUMINANCE,"get_colorprimaries_matrix src:%d dst:%d cust:%d wp: %1.4f,%1.4f cp: %1.3f, %1.3f, %1.3f, %1.3f, %1.3f, %1.3f",
			(int)src, (int)dst, ((custom_whitepoints_src)? 1 : 0),
			wp_src->xw, wp_src->yw, primary_coefs_src->xr, primary_coefs_src->yr, primary_coefs_src->xg, primary_coefs_src->yg,
			primary_coefs_src->xb, primary_coefs_src->yb);

	// src[RGB] -> src[XYZ] conv table
	dispqt_evr_colorcv_primaries_fill_rgb2xyz_table(primary_coefs_src, wp_src, rgb2xyz);

	// dst[XYZ] -> dst[RGB] conv table
	dispqt_evr_colorcv_primaries_fill_rgb2xyz_table(&cp_dst->primary_coefs, &dispqt_evr_whitepoint_coefficients[cp_dst->wp_id], xyz2rgb);
	ff_matrix_invert_3x3(xyz2rgb, xyz2rgb);

	// src[RGB] -> src[XYZ] -> dst[XYZ] -> dst[RGB]
	if (pds_memcmp(wp_src, wp_dst, sizeof(struct WhitepointCoefficients)) != 0)
	{   // TODO: this part is not completely tested
		double wpconv[3][3], tmp[3][3];
		dispqt_evr_colorcv_primaries_fill_whitepoint_conv_table(wpconv, wp_src, wp_dst);
		ff_matrix_mul_3x3(tmp, rgb2xyz, wpconv);
		ff_matrix_mul_3x3(out_rgb, tmp, xyz2rgb);
		mpxplay_debugf(DISPQT_DEBUGOUT_LUMINANCE,"whitepoint_conv!");
	}
	else
	{
		ff_matrix_mul_3x3(out_rgb, rgb2xyz, xyz2rgb);
	}

	for (unsigned int i = 0; i < 3; i++) {
		for (unsigned int j = 0; j < 3; j++)
			Primaries[i * 4 + j] = out_rgb[i][j];
		Primaries[i * 4 + 3] = 0;
	}
	for (unsigned int j = 0; j < 4; j++)
		Primaries[3 * 4 + j] = (j == 3)? 1.0f : 0.0f;
}

//------------------------------------------------------------------------------------------------------------
// pixel shader

void mpxplay_dispqt_render_common_shader_pixel_constant_colortransform_calculate(
		struct dispqt_evr_color_transform_matrixes_s *color_transform_data, const struct dispqt_evr_format_desc_s *pixform_infos,
		AVColorSpace av_colorspace, enum AVColorRange av_colorrange, enum AVColorPrimaries av_colorprimaries, int texture_height,
		const struct WhitepointCoefficients *custom_whitepoints_src, const struct PrimaryCoefficients *custom_primary_coefs_src)
{
	pds_memcpy((void *)&color_transform_data->WhitePointMatrix[0], (void *)&dispqt_evr_colormatrix_whitepoints[0], sizeof(color_transform_data->WhitePointMatrix));

	if(pixform_infos->is_rgb_format)
	{
		pds_memcpy((void *)&color_transform_data->ColorspaceMatrix[0], (void *)&dispqt_evr_colormatrix_whitepoints[0], sizeof(color_transform_data->ColorspaceMatrix));
	}
	else
	{
		int level_index = pixform_infos->bits_per_channel / 2 - 3; // unknown, 8, 10, 12 bits (0, 1, 2, 3)
		if((level_index < 0) || (level_index > 3))
			level_index = 0;
		const struct dispqt_evr_colormatrix_light_level_s *light_level = &dispqt_evr_colormatrix_light_levels[level_index];
		const FLOAT *colorMatrixSelect;

		switch(av_colorspace){
			case AVCOL_SPC_BT709:
				colorMatrixSelect = dispqt_evr_colormatrix_BT709_YUV_to_RGBA;
				break;
			case AVCOL_SPC_BT2020_NCL:
			case AVCOL_SPC_BT2020_CL:
				colorMatrixSelect = dispqt_evr_colormatrix_BT2020_YUV_to_RGBA;
				break;
			case AVCOL_SPC_BT470BG:
			case AVCOL_SPC_SMPTE170M:
			case AVCOL_SPC_SMPTE240M:
				colorMatrixSelect = dispqt_evr_colormatrix_BT601_YUV_to_RGBA;
				break;
			default:
				if(texture_height > 576)
					colorMatrixSelect = dispqt_evr_colormatrix_BT709_YUV_to_RGBA;
				else
					colorMatrixSelect = dispqt_evr_colormatrix_BT601_YUV_to_RGBA;
				break;
		}

		pds_memcpy((void *)&color_transform_data->ColorspaceMatrix[0], (void *)colorMatrixSelect, sizeof(color_transform_data->ColorspaceMatrix));

		if( (pixform_infos->av_pixfmt != AV_PIX_FMT_Y210LE) // TODO: this format is not tested (probably it does auto range expansion)
		 && ((av_colorrange == AVCOL_RANGE_MPEG) || ((!pixform_infos->is_rgb_format) && (av_colorrange != AVCOL_RANGE_JPEG)))
		){  // expand output range MPEG to FULL
			for(int i = 0; i < 3; i++)
			{
				color_transform_data->ColorspaceMatrix[i*4] /= light_level->light_range;
				color_transform_data->ColorspaceMatrix[i*4 + 1] /= light_level->color_range;
				color_transform_data->ColorspaceMatrix[i*4 + 2] /= light_level->color_range;
			}
			color_transform_data->WhitePointMatrix[0*4 + 3] = -light_level->black_level;
		}

		color_transform_data->WhitePointMatrix[1*4 + 3] = -light_level->achromacy;
		color_transform_data->WhitePointMatrix[2*4 + 3] = -light_level->achromacy;
	}

	pds_memset((void *)&color_transform_data->PrimariesMatrix[0], 0, sizeof(color_transform_data->PrimariesMatrix));
	if((av_colorprimaries >= AVCOL_PRI_BT470M) && (av_colorprimaries <= AVCOL_PRI_SMPTE432))
		dispqt_render_common_shader_pixel_get_colorprimaries_matrix(&color_transform_data->PrimariesMatrix[0], av_colorprimaries, AVCOL_PRI_BT709, custom_whitepoints_src, custom_primary_coefs_src);

	for(int i = 0; i < DISPQT_EVR_NB_MAX_PLANES; i++)
		color_transform_data->pixdata_shift_mul[i] = (FLOAT)(1 << pixform_infos->data_shift[i]);

	mpxplay_debugf(DISPQT_DEBUGOUT_LUMINANCE,"colortransform_calculate cps:%d clr:%d cpr:%d cust_wp:%d cust_pcs:%d",
		(int)av_colorspace, (int)av_colorrange, (int)av_colorprimaries, ((custom_whitepoints_src)? 1 : 0), ((custom_primary_coefs_src)? 1 : 0));
}

void mpxplay_dispqt_render_common_shader_pixel_assemble_trc(char *pscode_str_body, char *pscode_str_mainfunc,
		struct mmc_dispqt_config_s *gcfg, int av_colorprimaries, int av_colortrc)
{
	if((av_colorprimaries >= AVCOL_PRI_BT470M) && (av_colorprimaries <= AVCOL_PRI_SMPTE432))
		pds_strcat(pscode_str_body, (char *)dispqt_d3d_pscode_colorcv_primaries_transform);
	else
		pds_strcat(pscode_str_body, (char *)dispqt_d3d_pscode_colorcv_primaries_empty);

	switch(av_colortrc)
	{
		case AVCOL_TRC_BT709:
			pds_strcat(pscode_str_mainfunc, (char *)dispqt_d3d_pscode_mainfunc_colorcv_primaries);
			pds_strcat(pscode_str_mainfunc, (char *)dispqt_d3d_pscode_mainfunc_colorcv_BT709_to_SRGB);
			break;
//			case AVCOL_TRC_GAMMA28: // FIXME: produces too dark picture
//				pds_strcat(pscode_str_mainfunc, (char *)dispqt_d3d_pscode_mainfunc_colorcv_primaries);
//				pds_strcat(pscode_str_mainfunc, (char *)dispqt_d3d_pscode_mainfunc_colorcv_BT470BG_to_SRGB);
//				break;
		case AVCOL_TRC_SMPTE2084:
			if(funcbit_test(gcfg->video_hdr_control, DISPQT_CONFIG_VIDEO_HDR_TO_SDR_CONV))
			{
				pds_strcat(pscode_str_body, (char *)dispqt_d3d_pscode_hdr_tonemap);
				pds_strcat(pscode_str_body, (char *)dispqt_d3d_pscode_colorcv_SMPTE_ST2084_to_linear);
				pds_strcat(pscode_str_body, (char *)dispqt_d3d_pscode_colorcv_linear_to_SRGB);
				pds_strcat(pscode_str_mainfunc, (char *)dispqt_d3d_pscode_mainfunc_colorcv_SMPTE2084_to_SRGB);
			}
			break;
		case AVCOL_TRC_ARIB_STD_B67:
			if(funcbit_test(gcfg->video_hdr_control, DISPQT_CONFIG_VIDEO_HDR_TO_SDR_CONV))
			{
				pds_strcat(pscode_str_body, (char *)dispqt_d3d_pscode_hdr_tonemap);
				pds_strcat(pscode_str_body, (char *)dispqt_d3d_pscode_colorcv_HLG_to_linear);
				pds_strcat(pscode_str_body, (char *)dispqt_d3d_pscode_colorcv_linear_to_SRGB);
				pds_strcat(pscode_str_mainfunc, (char *)dispqt_d3d_pscode_mainfunc_colorcv_HLG_to_SRGB);
			}
			break;
		default:
			pds_strcat(pscode_str_mainfunc, (char *)dispqt_d3d_pscode_mainfunc_colorcv_primaries);
			break;
	}
}

//------------------------------------------------------------------------------------------------------------
// update luminance and color matrix by dynamic HDR metadata (if exists int the stream)

void mpxplay_dispqt_render_common_shader_pixel_dynamic_hdr_reset(struct dispqt_render_evr_dynamic_hdr_data_s *dynamic_hdr_datas, struct mmc_dispqt_config_s *gcfg)
{
	pds_memset(&dynamic_hdr_datas->custom_wp_coefs, 0, sizeof(dynamic_hdr_datas->custom_wp_coefs));
	pds_memcpy(&dynamic_hdr_datas->custom_pri_coefs, 0, sizeof(dynamic_hdr_datas->custom_pri_coefs));
	dynamic_hdr_datas->is_custom_primaries_sent = FALSE;
	dynamic_hdr_datas->hdr_input_luminance_value = 0;
	dynamic_hdr_datas->last_hdr_control_setting = gcfg->video_hdr_control;
}

bool mpxplay_dispqt_render_common_shader_pixel_dynamic_hdr_update(struct dispqt_render_evr_dynamic_hdr_data_s *dynamic_hdr_datas,
		AVFrame *video_avframe, struct dispqt_video_surface_info_s *video_surface_infos, struct mmc_dispqt_config_s *gcfg)
{
	int new_input_luminance_value = 0, output_luminance_value = 100; // SRGB luminance value
	struct dispqt_evr_dynamic_runtime_data_s dynamic_runtime_datas;
	bool new_custom_primaries_received = false;
	AVFrameSideData *sd;

	if((gcfg->video_hdr_control & (DISPQT_CONFIG_VIDEO_HDR_TO_SDR_CONV | DISPQT_CONFIG_VIDEO_HDR_DYN_LUMINANCE)) == (DISPQT_CONFIG_VIDEO_HDR_TO_SDR_CONV | DISPQT_CONFIG_VIDEO_HDR_DYN_LUMINANCE))
	{
		sd = av_frame_get_side_data(video_avframe, AV_FRAME_DATA_CONTENT_LIGHT_LEVEL);
		if(sd && sd->data && (sd->size == sizeof(struct AVContentLightMetadata)))
		{
			AVContentLightMetadata *clm = (AVContentLightMetadata *)sd->data;
			new_input_luminance_value = clm->MaxCLL;
			//mpxplay_debugf(DISPQT_DEBUGOUT_LUMINANCE,"AVContentLightMetadata: %d", new_input_luminance_value);
		}
	}
	if( ((gcfg->video_hdr_control & (DISPQT_CONFIG_VIDEO_HDR_TO_SDR_CONV | DISPQT_CONFIG_VIDEO_HDR_DYN_COLORS)) == (DISPQT_CONFIG_VIDEO_HDR_TO_SDR_CONV | DISPQT_CONFIG_VIDEO_HDR_DYN_COLORS))
	 || (((gcfg->video_hdr_control & (DISPQT_CONFIG_VIDEO_HDR_TO_SDR_CONV | DISPQT_CONFIG_VIDEO_HDR_DYN_LUMINANCE)) == (DISPQT_CONFIG_VIDEO_HDR_TO_SDR_CONV | DISPQT_CONFIG_VIDEO_HDR_DYN_LUMINANCE)) && !new_input_luminance_value)
	){
		sd = av_frame_get_side_data(video_avframe, AV_FRAME_DATA_MASTERING_DISPLAY_METADATA);
		if(sd && sd->data && (sd->size == sizeof(struct AVMasteringDisplayMetadata)))
		{
			AVMasteringDisplayMetadata *metadata = (AVMasteringDisplayMetadata *)sd->data;
			if(metadata->has_luminance && funcbit_test(gcfg->video_hdr_control, DISPQT_CONFIG_VIDEO_HDR_DYN_LUMINANCE) && !new_input_luminance_value)
			{
				new_input_luminance_value = (int)av_q2d(metadata->max_luminance);
				//mpxplay_debugf(DISPQT_DEBUGOUT_LUMINANCE,"AVMasteringDisplayMetadata: %d", new_input_luminance_value);
			}
			if(metadata->has_primaries && funcbit_test(gcfg->video_hdr_control, DISPQT_CONFIG_VIDEO_HDR_DYN_COLORS))
			{
				struct WhitepointCoefficients custom_wp_coefs;
				struct PrimaryCoefficients custom_pri_coefs;
				custom_wp_coefs.xw = av_q2d(metadata->white_point[0]);
				custom_wp_coefs.yw = av_q2d(metadata->white_point[1]);
				custom_pri_coefs.xr = av_q2d(metadata->display_primaries[0][0]);
				custom_pri_coefs.yr = av_q2d(metadata->display_primaries[0][1]);
				custom_pri_coefs.xg = av_q2d(metadata->display_primaries[1][0]);
				custom_pri_coefs.yg = av_q2d(metadata->display_primaries[1][1]);
				custom_pri_coefs.xb = av_q2d(metadata->display_primaries[2][0]);
				custom_pri_coefs.yb = av_q2d(metadata->display_primaries[2][1]);
				if(pds_memcmp(&dynamic_hdr_datas->custom_wp_coefs, &custom_wp_coefs, sizeof(custom_wp_coefs)) != 0)
				{
					pds_memcpy(&dynamic_hdr_datas->custom_wp_coefs, &custom_wp_coefs, sizeof(custom_wp_coefs));
					new_custom_primaries_received = true;
				}
				if(pds_memcmp(&dynamic_hdr_datas->custom_pri_coefs, &custom_pri_coefs, sizeof(custom_pri_coefs)) != 0)
				{
					pds_memcpy(&dynamic_hdr_datas->custom_pri_coefs, &custom_pri_coefs, sizeof(custom_pri_coefs));
					new_custom_primaries_received = true;
				}
				if(new_custom_primaries_received)
				{
					mpxplay_debugf(DISPQT_DEBUGOUT_LUMINANCE,"AVMasteringDisplayMetadata PRIMARIES wp: %1.4f,%1.4f cp: %1.3f, %1.3f, %1.3f, %1.3f, %1.3f, %1.3f",
						custom_wp_coefs.xw, custom_wp_coefs.yw, custom_pri_coefs.xr, custom_pri_coefs.yr, custom_pri_coefs.xg, custom_pri_coefs.yg,
						custom_pri_coefs.xb, custom_pri_coefs.yb);
				}
			}
		}
	}

    if(!new_input_luminance_value)
    {
		switch(video_avframe->color_trc)
		{
			case AVCOL_TRC_SMPTE2084: new_input_luminance_value = 10000; break;
			case AVCOL_TRC_ARIB_STD_B67: new_input_luminance_value = 1000; break;
			default: new_input_luminance_value = 100;
		}
		//mpxplay_debugf(DISPQT_DEBUGOUT_LUMINANCE,"Luminance default: %d", new_input_luminance_value);
    }

	if(dynamic_hdr_datas->hdr_input_luminance_value != new_input_luminance_value)
	{
		if((gcfg->video_hdr_control & DISPQT_CONFIG_VIDEO_HDR_DYN_LUM_SLOW) && !video_surface_infos->videoout_input_change && !video_surface_infos->videoout_surface_reset)
			new_input_luminance_value = (dynamic_hdr_datas->hdr_input_luminance_value + new_input_luminance_value) / 2;

		dynamic_hdr_datas->hdr_input_luminance_value = new_input_luminance_value;
	}

	return new_custom_primaries_received;
}

//------------------------------------------------------------------------------------------------------------
// update color changes for pixel shader
void mpxplay_dispqt_render_common_shader_pixel_colormixer_configure(struct dispqt_evr_color_mixer_s *color_mixdata_dst, mpxp_int32_t vmixer_values[DISPQT_VIDEO_VMIXERTYPE_MAX])
{
	pds_memset(color_mixdata_dst, 0, sizeof(*color_mixdata_dst));

	if(vmixer_values[DISPQT_VIDEO_VMIXERTYPE_BRIGHTNESS])
		color_mixdata_dst->colormixer_values[DISPQT_VIDEO_VMIXERTYPE_BRIGHTNESS] = (FLOAT)vmixer_values[DISPQT_VIDEO_VMIXERTYPE_BRIGHTNESS] / (1.5 * DISPQT_EVR_COLORCHANGE_VALUE_SCALE);

	for(int i = DISPQT_VIDEO_VMIXERTYPE_CONTRAST; i < DISPQT_VIDEO_VMIXERTYPE_HUE; i++)
		color_mixdata_dst->colormixer_values[i] = (vmixer_values[i])? ((FLOAT)vmixer_values[i] / DISPQT_EVR_COLORCHANGE_VALUE_SCALE + 1.0f) : 1.0f;

	if(vmixer_values[DISPQT_VIDEO_VMIXERTYPE_HUE] != 0)
	{
		color_mixdata_dst->colormixer_values[DISPQT_VIDEO_VMIXERTYPE_HUE] = (FLOAT)vmixer_values[DISPQT_VIDEO_VMIXERTYPE_HUE] / DISPQT_EVR_COLORCHANGE_VALUE_SCALE;
		const double radians_hue = color_mixdata_dst->colormixer_values[DISPQT_VIDEO_VMIXERTYPE_HUE] * M_PI;
		const double sqrtThirdSinHue = sqrt(1.0/3.0) * sin(radians_hue);
		const double cosHue = cos(radians_hue), cos1mThird = (1.0 - cosHue) / 3.0;
		FLOAT (*hue_matrix)[4] = &color_mixdata_dst->colormixer_hue_matrix[0];
		hue_matrix[0][0] = hue_matrix[1][1] = hue_matrix[2][2] = cos1mThird + cosHue;
		hue_matrix[0][1] = hue_matrix[1][2] = hue_matrix[2][0] = cos1mThird - sqrtThirdSinHue;
		hue_matrix[0][2] = hue_matrix[1][0] = hue_matrix[2][1] = cos1mThird + sqrtThirdSinHue;
	}
}

//============================================================================================================

void mpxplay_dispqt_render_common_vertex_set_input_texture_crop_and_rotate(void *vertex_destptr,
	const struct dispqt_evr_input_viewport_s *crop_datas, int input_texture_width, int input_texture_height,
	float vertex_position_scale)
{
	FLOAT pos_x0, pos_y0, pos_x1, pos_y1;
	int crop_pos_x, crop_pos_y, crop_width, crop_height;
	struct d3d_vertex_t *vertex;

	vertex = (struct d3d_vertex_t *)vertex_destptr;

	pds_memcpy(vertex, (void *)&vertices_full_texture_default[0], sizeof(vertices_full_texture_default));

#if 0 // unused, not required yet
	if(vertex_position_scale != DISPQT_RENDER_VERTEX_POSITION_BASE_SCALE)
	{
		for(int i = 0; i < DISPQT_EVR_D3D_NUMVERTICES_DEFAULT; i++)
		{
			vertex[i].position.x *= vertex_position_scale;
			if(vertex[i].position.x > vertex_position_scale)
				vertex[i].position.x = vertex_position_scale;
			else if(vertex[i].position.x < -vertex_position_scale)
				vertex[i].position.x = -vertex_position_scale;
			vertex[i].position.y *= vertex_position_scale;
			if(vertex[i].position.y > vertex_position_scale)
				vertex[i].position.y = vertex_position_scale;
			else if(vertex[i].position.y < -vertex_position_scale)
				vertex[i].position.y = -vertex_position_scale;
		}
	}
#endif

	if((crop_datas->rotation_type == DISPQT_VIDEO_ROTATIONTYPE_ROTL90) || (crop_datas->rotation_type == DISPQT_VIDEO_ROTATIONTYPE_ROTR90))
	{   // rectangles are rotated by mpxplay_dispqt_videotrans_videooutput_surface_window_calculate, so rotate back for the texture
		crop_pos_x = crop_datas->pos_y; crop_pos_y = crop_datas->pos_x; crop_width = crop_datas->height; crop_height = crop_datas->width;
	}
	else
	{
		crop_pos_x = crop_datas->pos_x; crop_pos_y = crop_datas->pos_y; crop_width = crop_datas->width; crop_height = crop_datas->height;
	}

	pos_x0 = ((FLOAT)crop_pos_x / (FLOAT)input_texture_width);
	pos_y0 = ((FLOAT)crop_pos_y / (FLOAT)input_texture_height);

	crop_pos_x += crop_width;
	pos_x1 = (crop_pos_x == input_texture_width)? 1.0 : ((FLOAT)(crop_pos_x - 1) / (FLOAT)input_texture_width);
	crop_pos_y += crop_height;
	pos_y1 = (crop_height == input_texture_height)? 1.0 : ((FLOAT)(crop_pos_y - 1) / (FLOAT)input_texture_height);

	switch(crop_datas->rotation_type)
	{
		case DISPQT_VIDEO_ROTATIONTYPE_ROTL90:
			vertex[0].texture.x = pos_x0; vertex[0].texture.y = pos_y0;
			vertex[1].texture.x = vertex[4].texture.x = pos_x1; vertex[1].texture.y = vertex[4].texture.y = pos_y0;
			vertex[2].texture.x = vertex[3].texture.x = pos_x0; vertex[2].texture.y = vertex[3].texture.y = pos_y1;
			vertex[5].texture.x = pos_x1; vertex[5].texture.y = pos_y1;
			break;
		case DISPQT_VIDEO_ROTATIONTYPE_ROTR90:
			vertex[0].texture.x = pos_x1; vertex[0].texture.y = pos_y1;
			vertex[1].texture.x = vertex[4].texture.x = pos_x0; vertex[1].texture.y = vertex[4].texture.y = pos_y1;
			vertex[2].texture.x = vertex[3].texture.x = pos_x1; vertex[2].texture.y = vertex[3].texture.y = pos_y0;
			vertex[5].texture.x = pos_x0; vertex[5].texture.y = pos_y0;
			break;
		case DISPQT_VIDEO_ROTATIONTYPE_ROT180:
			vertex[0].texture.x = pos_x0; vertex[0].texture.y = pos_y0;
			vertex[1].texture.x = vertex[4].texture.x = pos_x0; vertex[1].texture.y = vertex[4].texture.y = pos_y1;
			vertex[2].texture.x = vertex[3].texture.x = pos_x1; vertex[2].texture.y = vertex[3].texture.y = pos_y0;
			vertex[5].texture.x = pos_x1; vertex[5].texture.y = pos_y1;
			break;
		default:
			vertex[0].texture.x = pos_x0; vertex[0].texture.y = pos_y1;
			vertex[1].texture.x = vertex[4].texture.x = pos_x0; vertex[1].texture.y = vertex[4].texture.y = pos_y0;
			vertex[2].texture.x = vertex[3].texture.x = pos_x1; vertex[2].texture.y = vertex[3].texture.y = pos_y1;
			vertex[5].texture.x = pos_x1; vertex[5].texture.y = pos_y0;
	}
}

//============================================================================================================

// select format descriptor by DXGI_FORMAT (from the dispqt_evr_supported_pixel_formats)
struct dispqt_evr_format_desc_s *mpxplay_dispqt_render_common_select_dxgi_format(struct dispqt_evr_format_desc_s *format_descs, int srch_dxgi)
{
	struct dispqt_evr_format_desc_s *selected_format = NULL;

	do
	{
		if(format_descs->d3d_texture_format == srch_dxgi)
		{
			if(format_descs->is_native_support & DISPQT_EVR_SUPPORTFLAG_INPUT)
			{
				selected_format = format_descs;
			}
			else // TODO: copy input texture into a local one in dispqt_evr_input_hwdec_texture_process()
			{
				mpxplay_debugf(DISPQT_DEBUG_ERROR, "select_dxgi_format FAIL dxgi:%d", (int)format_descs->d3d_texture_format);
			}
			break;
		}
		format_descs++;
	}while(format_descs->av_pixfmt != AV_PIX_FMT_NONE);

	return selected_format;
}

// select format descriptor by AVPixelformat (from the dispqt_evr_supported_pixel_formats), walk to the nearest supported native format (difference will be converted by sws-scale)
struct dispqt_evr_format_desc_s *mpxplay_dispqt_render_common_input_select_native_color_format(struct dispqt_evr_format_desc_s *format_descs,
		struct dispqt_evr_format_desc_s *format_info_store, int srch_pixfmt, mpxp_bool_t search_next)
{
	struct dispqt_evr_format_desc_s *selected_format = NULL;
	bool search_next_natively_supported = false;

	do
	{
		if(search_next_natively_supported && (format_descs->is_native_support & DISPQT_EVR_SUPPORTFLAG_INPUT))
		{
			selected_format = format_descs;
			if(format_info_store)
				pds_memcpy(format_info_store, selected_format, sizeof(*format_info_store));
			mpxplay_debugf(DISPQT_DEBUG_WARNING, "select_native_color_format SKIP pix: %d dxgi:%d", format_descs->av_pixfmt, (int)format_descs->d3d_texture_format);
			break;
		}
		else if(format_descs->av_pixfmt == srch_pixfmt)
		{
			if(format_descs->is_native_support & DISPQT_EVR_SUPPORTFLAG_INPUT)
			{
				selected_format = format_descs;
				if(format_info_store)
					pds_memcpy(format_info_store, selected_format, sizeof(*format_info_store));
				break;
			}
			if(!search_next)
			{
				break;
			}
			search_next_natively_supported = true; // walk to the nearest supported format in the support-format-list
			mpxplay_debugf(DISPQT_DEBUG_WARNING, "select_native_color_format FAIL pix: %d dxgi:%d", format_descs->av_pixfmt, (int)format_descs->d3d_texture_format);
		}
		format_descs++;
	}while(format_descs->av_pixfmt != AV_PIX_FMT_NONE);

	return selected_format;
}

// prepare format descriptor by AVPixelformat (select from the list, or create a custom one by the av_pix_fmt_desc_get infos)
struct dispqt_evr_format_desc_s *mpxplay_dispqt_render_common_input_select_color_format(struct dispqt_evr_format_desc_s *format_descs,
		struct dispqt_evr_format_desc_s *format_info_store, int srch_pixfmt, mpxp_bool_t select_intermediate_pixfmt)
{
	struct dispqt_evr_format_desc_s *selected_format = mpxplay_dispqt_render_common_input_select_native_color_format(format_descs, format_info_store, srch_pixfmt, TRUE);

	if(!selected_format)
	{
		const AVPixFmtDescriptor *desc = av_pix_fmt_desc_get((enum AVPixelFormat)srch_pixfmt);
		if(!desc
			|| desc->comp[0].offset || desc->comp[1].offset || desc->comp[2].offset // this packed format is not supported by native types (dispqt_evr_supported_pixel_formats[])
			|| (desc->nb_components < 3)                                            // only 3-plane custom color formats are handled natively
			|| (desc->flags & (AV_PIX_FMT_FLAG_BE | AV_PIX_FMT_FLAG_PAL | AV_PIX_FMT_FLAG_BITSTREAM | AV_PIX_FMT_FLAG_HWACCEL)) // BE is not supported natively
		){
			// select a natively supported pixeltype, sws-scale will convert to it
			if(select_intermediate_pixfmt)
			{
				if(!desc || (desc->flags & AV_PIX_FMT_FLAG_RGB))
					srch_pixfmt = (desc && (desc->comp[0].depth > 8))? AV_PIX_FMT_RGBA64LE : AV_PIX_FMT_RGBA;
				else
					srch_pixfmt = (desc->comp[0].depth > 8)? AV_PIX_FMT_P010LE : AV_PIX_FMT_NV12;
				selected_format = mpxplay_dispqt_render_common_input_select_native_color_format(format_descs, format_info_store, srch_pixfmt, TRUE);
			}
			return selected_format;
		}

		pds_memset(format_info_store, 0, sizeof(*format_info_store));
		format_info_store->av_pixfmt = srch_pixfmt;
		format_info_store->d3d_texture_format = DXGI_FORMAT_UNKNOWN;
		format_info_store->bits_per_channel = desc->comp[0].depth;
		format_info_store->is_rgb_format = (desc->flags & AV_PIX_FMT_FLAG_RGB)? 1 : 0;
		format_info_store->align_mask_x = desc->log2_chroma_w;
		format_info_store->align_mask_y = desc->log2_chroma_h;
		format_info_store->nb_components = min(desc->nb_components, DISPQT_EVR_NB_MAX_PLANES);
		for(int i = 0; i < format_info_store->nb_components; i++)
		{
			unsigned int plane_pos = desc->comp[i].plane;
			if(plane_pos >= format_info_store->nb_components) // AVPixFmtDescriptor error, should not happen
				return selected_format;
			format_info_store->plane_pos[i] = plane_pos;
			if(desc->flags & AV_PIX_FMT_FLAG_FLOAT)
				format_info_store->shader_view_formats[i] = (desc->comp[plane_pos].depth > 16)? DXGI_FORMAT_R32_FLOAT : DXGI_FORMAT_R16_FLOAT; // TODO: these formats are not tested by dispqt_evr_d3d11_native_format_support_check()
			else
			{
				bool is_16_bit = ((desc->comp[plane_pos].depth + desc->comp[plane_pos].shift) > 8)? true : false;
				format_info_store->shader_view_formats[i] = (is_16_bit)? DXGI_FORMAT_R16_UNORM : DXGI_FORMAT_R8_UNORM;
				format_info_store->data_shift[i] = ((is_16_bit)? 16 : 8) - desc->comp[plane_pos].depth - desc->comp[plane_pos].shift;
				if(format_info_store->data_shift[i] >= 8)
					format_info_store->data_shift[i] = 0;
			}
		}
		selected_format = format_info_store;
	}

	return selected_format;
}

struct dispqt_evr_format_desc_s *mpxplay_dispqt_render_common_select_best_subtitle_format(struct dispqt_evr_format_desc_s *format_descs)
{
	const int subtitle_pixfmts[] = {DISPQT_VIDEO_SUBTITLE_QIMAGE_AVFORMAT, AV_PIX_FMT_RGBA, AV_PIX_FMT_BGR0, AV_PIX_FMT_RGB0};
	const int nb_subtitle_fmts = sizeof(subtitle_pixfmts) / sizeof(subtitle_pixfmts[0]);
	struct dispqt_evr_format_desc_s *selected_format = NULL;
	int i = 0;

	do {
		selected_format = mpxplay_dispqt_render_common_input_select_native_color_format(format_descs, NULL, subtitle_pixfmts[i], FALSE);
		if(selected_format)
			break;
	}while(++i < nb_subtitle_fmts);

	return selected_format;
}

//============================================================================================================

mpxp_bool_t mpxplay_dispqt_render_common_deinterlace_condition(AVFrame *video_avframe, struct mmc_dispqt_config_s *gcfg)
{
	if(!video_avframe || !gcfg || (gcfg->video_deinterlace_type == DISPQT_VIDEO_DEINTERLACETYPE_DISABLE))
		return FALSE;
	if(funcbit_test(video_avframe->flags, DISPQT_AVFRAME_FLAG_STILLPIC))
		return FALSE;
	if(!video_avframe->interlaced_frame && ((mmc_config.video_deinterlace_type == DISPQT_VIDEO_DEINTERLACETYPE_AUTO) || (video_avframe->width > DISPQT_VIDEOFILTER_PROGRESSIVE_DEINTERLACE_MAX_WIDTH)))
		return FALSE;
	return TRUE;
}

mpxp_bool_t mpxplay_dispqt_render_common_deinterdup_condition(AVFrame *video_avframe, struct mmc_dispqt_config_s *gcfg)
{
	if(!funcbit_test(gcfg->selected_videoplayer_control, MPXPLAY_CONFIG_VIDEOPLAYERCONTROL_DUPLICATE_INTERLACED_FRAMES))
		return FALSE;
	if(video_avframe->width > DISPQT_VIDEOFILTER_DEINTERLACE_HWDUPLICATEDFPS_MAX_WIDTH)
		return FALSE;
	return TRUE;
}

//============================================================================================================
// copy video frame to gpu mapped memory (NV12, P010, YV12)
static void dispqt_videorender_common_videoframecopy_NV12(
		struct dispqt_render_mapped_subresource_s *texture_mapres,
		struct dispqt_evr_format_desc_s *format_datas,
		uint8_t *indata_ptrs[AV_NUM_DATA_POINTERS], int indata_linesizes[AV_NUM_DATA_POINTERS], unsigned int indata_lines)
{
	const int av_pixfmt = format_datas->av_pixfmt;

	for(int i = 0; i < format_datas->nb_components; i++)
	{
		register const unsigned int linelen_in = indata_linesizes[i];
		register const unsigned int linelen_out = ((i == 0) || (av_pixfmt != AV_PIX_FMT_YUV420P))? texture_mapres->vidmem_linesize : (texture_mapres->vidmem_linesize / 2);
		register const unsigned int copy_len = min(linelen_in, linelen_out);
		register const unsigned int frame_index = ((i == 0) || (av_pixfmt != AV_PIX_FMT_YUV420P))? i : ((i == 1)? 2 : 1); // swap UV
		register uint8_t *indataptr = indata_ptrs[frame_index];
		register const unsigned int linenum_in = (i == 0)? indata_lines : (indata_lines / 2);
		register const unsigned int linenum_out = (i == 0)? texture_mapres->vidmem_linenum : (texture_mapres->vidmem_linenum / 2);
		register unsigned int copy_linecount = min(linenum_in, linenum_out);
		register uint8_t *outdataptr = (uint8_t *)texture_mapres->vidmem_beginptr;
		if(!copy_len || !indataptr || !outdataptr || !copy_linecount)
			return;
		if(i > 0)
			outdataptr += texture_mapres->vidmem_linesize * texture_mapres->vidmem_linenum;
		if(i > 1)
			outdataptr += linelen_out * linenum_out; // skip V
		if(ff_image_copy_plane_uc_from_x86(outdataptr, linelen_out, indataptr, linelen_in, copy_len, copy_linecount) != 0)
		{
			do
			{
				pds_memcpy(outdataptr, indataptr, copy_len);
				outdataptr += linelen_out; indataptr += linelen_in;
			}while(--copy_linecount);
		}
	}
}

// copy video frame to gpu mapped memory (packed 1 plane, like RGB and YUY2)
void mpxplay_dispqt_videorender_common_videoframecopy_PACKED(
		struct dispqt_render_mapped_subresource_s *texture_mapres,
		int texture_id,
		uint8_t *indata_ptr, unsigned int indata_linesize, unsigned int indata_lines)
{
	register const unsigned int linelen_in = indata_linesize;
	register const unsigned int linelen_out = texture_mapres->vidmem_linesize;
	register const unsigned int copy_len = min(linelen_in, linelen_out);
	register uint8_t *indataptr = indata_ptr;
	register uint8_t *outdataptr = (uint8_t *)texture_mapres->vidmem_beginptr;
	register unsigned int copy_linecount = min(indata_lines, texture_mapres->vidmem_linenum);
	if(!copy_len || !indataptr || !outdataptr || !copy_linecount)
		return;
	if((texture_id != DISPQT_EVR_INPUTTEXUREID_VIDEO) || (ff_image_copy_plane_uc_from_x86(outdataptr, linelen_out, indataptr, linelen_in, copy_len, copy_linecount) != 0))
	{
		do
		{
			pds_memcpy(outdataptr, indataptr, copy_len);
			outdataptr += linelen_out; indataptr += linelen_in;
		}while(--copy_linecount);
	}
}

// copy video frame to gpu mapped memory (multiply planes on multiply textures) (D3D11 / YUV420)
static void dispqt_videorender_common_videoframecopy_PLANAR(
		struct dispqt_render_mapped_subresource_s texture_mapres[DISPQT_EVR_NB_MAX_PLANES],
		struct dispqt_evr_format_desc_s *format_datas,
		uint8_t *indata_ptrs[AV_NUM_DATA_POINTERS], int indata_linesizes[AV_NUM_DATA_POINTERS], unsigned int indata_lines)
{
	for(int i = 0; i < format_datas->nb_components; i++)
	{
		const unsigned int plane_pos = format_datas->plane_pos[i];
		register const unsigned int linelen_in = indata_linesizes[plane_pos];
		register const unsigned int linelen_out = texture_mapres[i].vidmem_linesize;
		register const unsigned int copy_len = min(linelen_in, linelen_out);
		register uint8_t *indataptr = indata_ptrs[plane_pos];
		register uint8_t *outdataptr = (uint8_t *)texture_mapres[i].vidmem_beginptr;
		register const unsigned int linecount_in = (i > 0)? (indata_lines >> format_datas->align_mask_y) : indata_lines;
		register unsigned int copy_linecount = min(linecount_in, texture_mapres[i].vidmem_linenum);
		if(!copy_len || !indataptr || !outdataptr || !copy_linecount)
			return;
		if(ff_image_copy_plane_uc_from_x86(outdataptr, linelen_out, indataptr, linelen_in, copy_len, copy_linecount) != 0)
		{
			do
			{
				pds_memcpy(outdataptr, indataptr, copy_len);
				outdataptr += linelen_out; indataptr += linelen_in;
			}while(--copy_linecount);
		}
	}
}

mpxp_bool_t mpxplay_dispqt_videorender_common_videoframe_copy(
		struct dispqt_render_mapped_subresource_s texture_mapres[DISPQT_EVR_NB_MAX_PLANES],
		struct dispqt_evr_format_desc_s *format_datas,
		uint8_t *indata_ptrs[AV_NUM_DATA_POINTERS], int indata_linesizes[AV_NUM_DATA_POINTERS], unsigned int indata_lines)
{
	mpxp_bool_t success = FALSE;
	switch(format_datas->av_pixfmt)
	{
		case AV_PIX_FMT_NV12:
		case AV_PIX_FMT_NV21:
		case AV_PIX_FMT_NV24:
		case AV_PIX_FMT_NV42:
		case AV_PIX_FMT_P010LE:
		case AV_PIX_FMT_P016LE:
		case AV_PIX_FMT_YUV420P:
			if(format_datas->d3d_texture_format > 0)
			{
				dispqt_videorender_common_videoframecopy_NV12(texture_mapres,
						format_datas, indata_ptrs, indata_linesizes, indata_lines);
				success = TRUE;
				break;
			} // hack for planar pixel formats, at using multiply textures/views (generated/assembled in mpxplay_dispqt_render_common_input_select_color_format -> d3d_texture_format is DXGI_FORMAT_UNKNOWN)
			// @suppress("No break at end of case")
		default:
			if(format_datas->nb_components == 1)
			{
				mpxplay_dispqt_videorender_common_videoframecopy_PACKED(&texture_mapres[0],
						DISPQT_EVR_INPUTTEXUREID_VIDEO, indata_ptrs[0], indata_linesizes[0], indata_lines);
				success = TRUE;
			}
			else // for D3D11 only
			{
				dispqt_videorender_common_videoframecopy_PLANAR(texture_mapres, format_datas, indata_ptrs, indata_linesizes, indata_lines);
			}
			break;
	}

	return success;
}

//============================================================================================================
// common pool buffer functions (to pre-upload and store decoded textures in a pool buffer)

static mpxp_bool_t dispqt_videorender_common_poolbuf_is_frame_buffer_support(void)
{
	return (mpxplay_video_render_infos.d3d_render_handler)? TRUE : FALSE;
}

static bool dispqt_evr_poolbufelem_alloc(struct dispqt_render_poolbufelem_s *bufelem, AVCodecContext *avctx, AVFrame *avframe_dest, AVFrame *avframe_src, int flags)
{
	int tex_width = avframe_dest->width, tex_height = avframe_dest->height;
	struct dispqt_evr_format_desc_s *format_datas;
	int allocated_pixfmt = bufelem->bufpoolelem_format_desc.av_pixfmt;
	bool success;

	if(!mpxplay_video_render_infos.render_function_poolbuf_pixform_get || !mpxplay_video_render_infos.render_function_poolbuf_texture_alloc)
		return false;
	format_datas = (struct dispqt_evr_format_desc_s *)mpxplay_video_render_infos.render_function_poolbuf_pixform_get(&bufelem->bufpoolelem_format_desc, avframe_src);
	if(!format_datas)
	{
		mpxplay_debugf(DISPQT_DEBUGOUT_POOL, "POOL format_datas not found %d", frame->format);
		return false;
	}

	if(bufelem->poolbufelem_texture_2d[0] && (bufelem->videoframe_width == avframe_dest->width) && (bufelem->videoframe_height == avframe_dest->height))
	{   // if the required pixfmt cannot be set natively, we don't clear/rebuild the surface/texture, swsctx conversion will be used
		if((format_datas->av_pixfmt == allocated_pixfmt) && (format_datas->av_pixfmt != avframe_dest->format))
		{
			mpxplay_debugf(DISPQT_DEBUGOUT_POOL, "POOL texture reuse %d", bufelem->elem_id);
			return true;
		}
	}

	bufelem->bufpoolelem_struct_id = 0;
	if(mpxplay_video_render_infos.render_function_poolbuf_texture_free)
		mpxplay_video_render_infos.render_function_poolbuf_texture_free(&bufelem->poolbufelem_texture_2d[0], &bufelem->poolbufelem_shader_resource_views[0]);

	tex_width = tex_width & ~format_datas->align_mask_x;
	tex_width = (tex_width + 63) & ~63; // rounding up for SSE copy
	tex_height = tex_height & ~format_datas->align_mask_y;

	success = mpxplay_video_render_infos.render_function_poolbuf_texture_alloc(format_datas, tex_width,
				tex_height, &bufelem->poolbufelem_texture_2d[0], &bufelem->poolbufelem_shader_resource_views[0],
				&bufelem->poolbufelem_texture_mappedResource[0]);
	if(success)
	{
		if(!bufelem->poolbufelem_texture_mappedResource[0].vidmem_linenum) // shouldn't be
			bufelem->poolbufelem_texture_mappedResource[0].vidmem_linenum = bufelem->allocated_height;
		bufelem->bufpoolelem_struct_id = DISPQT_RENDER_BUFPOOLELEM_STRUCT_ID;
		bufelem->videoframe_width = avframe_dest->width;
		bufelem->videoframe_height = avframe_dest->height;
		bufelem->allocated_width = tex_width;
		bufelem->allocated_height = tex_height;
	}

	mpxplay_debugf(DISPQT_DEBUGOUT_POOL, "POOL ALLOC s:%d id:%d w:%d h:%d f:%d", (int)success, bufelem->elem_id, tex_width, tex_height, format_datas->av_pixfmt);

	return success;
}

static void dispqt_evr_poolbufelem_texture_unmap(struct dispqt_render_poolbufelem_s *bufelem)
{
	for(int i = 0; i < DISPQT_EVR_NB_MAX_PLANES; i++)
	{
		struct dispqt_render_mapped_subresource_s *mapres = &bufelem->poolbufelem_texture_mappedResource[i];
		if(mpxplay_video_render_infos.render_function_poolbuf_texture_unmap)
			mpxplay_video_render_infos.render_function_poolbuf_texture_unmap(bufelem->poolbufelem_texture_2d[i], mapres);
	}
}

static void dispqt_evr_poolbufelem_derefer(void *opaque, uint8_t *data)
{
	struct dispqt_render_poolbufelem_s *bufelem = (struct dispqt_render_poolbufelem_s *)opaque;
	if(bufelem)
	{
#if defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK) && !defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_GLOBALLOCK)
		const int mutex_error = PDS_THREADS_MUTEX_LOCK(&mpxplay_video_render_infos.d3d_device_mutex, DISPQT_VIDEO_MUTEX_TIMEOUT);
#endif
		if((bufelem->bufpoolelem_struct_id == DISPQT_RENDER_BUFPOOLELEM_STRUCT_ID) && (bufelem->bufpoolelem_status != DISPQT_RENDER_BUFPOOLELEM_STATUS_FREE))
		{
			dispqt_evr_poolbufelem_texture_unmap(bufelem);
			bufelem->bufpoolelem_status = DISPQT_RENDER_BUFPOOLELEM_STATUS_FREE;
			mpxplay_debugf(DISPQT_DEBUGOUT_POOL, "POOL poolbufelem_derefer %d", bufelem->elem_id);
		}
#if defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK) && !defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_GLOBALLOCK)
		if(mutex_error == MPXPLAY_ERROR_OK)
			PDS_THREADS_MUTEX_UNLOCK(&mpxplay_video_render_infos.d3d_device_mutex);
#endif
	}
}

static void dispqt_videorender_common_poolbuf_consolidate(void)
{
	struct dispqt_render_poolbufelem_s *poolbufptr;
#if defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK) && !defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_GLOBALLOCK)
	int mutex_error;
#endif

	if(!mpxplay_video_render_infos.render_function_poolbuf_bufptr_get)
		return;

#if defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK) && !defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_GLOBALLOCK)
	mutex_error = PDS_THREADS_MUTEX_LOCK(&mpxplay_video_render_infos.d3d_device_mutex, DISPQT_VIDEO_MUTEX_TIMEOUT);
	if(mutex_error != MPXPLAY_ERROR_OK)
		return;
#endif

	poolbufptr = (struct dispqt_render_poolbufelem_s *)mpxplay_video_render_infos.render_function_poolbuf_bufptr_get();
	if(poolbufptr)
	{
		for(int i = 0; i < DISPQT_RENDER_POOLBUF_SIZE_MAX; i++)
		{
			struct dispqt_render_poolbufelem_s *bufelem = &poolbufptr[i];
			if((bufelem->bufpoolelem_status == DISPQT_RENDER_BUFPOOLELEM_STATUS_FREE) || (bufelem->bufpoolelem_status == DISPQT_RENDER_BUFPOOLELEM_STATUS_INVALIDATED))
			{
				dispqt_evr_poolbufelem_texture_unmap(bufelem);
				if(mpxplay_video_render_infos.render_function_poolbuf_texture_free)
					mpxplay_video_render_infos.render_function_poolbuf_texture_free(&bufelem->poolbufelem_texture_2d[0], &bufelem->poolbufelem_shader_resource_views[0]);
			}
		}
	}

#if defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK) && !defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_GLOBALLOCK)
	PDS_THREADS_MUTEX_UNLOCK(&mpxplay_video_render_infos.d3d_device_mutex);
#endif
}

static int dispqt_videorender_common_poolbuf_get_frame_buffer(void *av_ctx, void *av_destframe, void *av_srcframe, int flags)
{
	AVCodecContext *avctx = (AVCodecContext *)av_ctx;
	AVFrame *avframe_dest = (AVFrame *)av_destframe;
	AVFrame *avframe_src = (AVFrame *)av_srcframe;
	struct dispqt_render_poolbufelem_s *bufelem = NULL;
	struct dispqt_evr_format_desc_s *format_datas = NULL;
	struct dispqt_render_poolbufelem_s *poolbufptr;
	int retval = 0;
#if defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK) && !defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_GLOBALLOCK)
	int mutex_error = -1;
#endif
	bool buffer_ok = false;
#ifdef MPXPLAY_USE_DEBUGF
		mpxp_int64_t getbuf_begin = pds_gettimem(), getbuf_mutex, getbuf_map, getbuf_endmap;
#endif

	if(!avctx || !avframe_dest || ((avframe_dest->width < 8) && (avframe_dest->height < 8)) || !(flags & MPXPLAY_VIDEO_RENDERER_FRAME_POOL_ELEMFLAG_WRITEONLY)
	 || !mpxplay_video_render_infos.render_function_poolbuf_bufptr_get || !mpxplay_video_render_infos.render_function_poolbuf_texture_map)
	{
		mpxplay_debugf(DISPQT_DEBUG_WARNING, "POOL buffer uses old method");
		goto err_out_buffer;
	}

#if defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK) && !defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_GLOBALLOCK)
	mutex_error = PDS_THREADS_MUTEX_LOCK(&mpxplay_video_render_infos.d3d_device_mutex, DISPQT_VIDEO_MUTEX_TIMEOUT);
#endif

	poolbufptr = (struct dispqt_render_poolbufelem_s *)mpxplay_video_render_infos.render_function_poolbuf_bufptr_get();

#ifdef MPXPLAY_USE_DEBUGF
	getbuf_mutex = pds_gettimem();
#endif

	if(poolbufptr)
	{
		int i = 0;
		for(; i < DISPQT_RENDER_POOLBUF_SIZE_MAX; i++)
		{
			struct dispqt_render_poolbufelem_s *bufp = &poolbufptr[i];
			if(bufp->poolbufelem_texture_mappedResource[0].vidmem_beginptr || (bufp->bufpoolelem_status != DISPQT_RENDER_BUFPOOLELEM_STATUS_FREE))
				continue;
			bufelem = bufp;
			bufelem->bufpoolelem_status = DISPQT_RENDER_BUFPOOLELEM_STATUS_PROCESSING;
			if( !bufelem->poolbufelem_texture_2d[0] || (bufelem->videoframe_width != avframe_dest->width)
			 || (bufelem->videoframe_height != avframe_dest->height) || (bufelem->bufpoolelem_format_desc.av_pixfmt != avframe_dest->format)
			){
				bufelem->elem_id = i;
				if(!dispqt_evr_poolbufelem_alloc(bufelem, avctx, avframe_dest, avframe_src, flags))
					break;
			}
			buffer_ok = true;
			break;
		}
		if(!buffer_ok)
		{
			mpxplay_debugf(DISPQT_DEBUG_ERROR, "POOL buffer empty elem search failed %d", i);
		}
	}

	if(buffer_ok)
	{
		format_datas = &bufelem->bufpoolelem_format_desc;
		pds_memset(&avframe_dest->data[0], 0, sizeof(avframe_dest->data));
		pds_memset(&avframe_dest->linesize[0], 0, sizeof(avframe_dest->linesize));
		pds_memset(&avframe_dest->buf[0], 0, sizeof(avframe_dest->buf));

#ifdef MPXPLAY_USE_DEBUGF
		getbuf_map = pds_gettimem();
#endif
		switch(format_datas->av_pixfmt)
		{
			case AV_PIX_FMT_NV12:
			case AV_PIX_FMT_NV21:
			case AV_PIX_FMT_NV24:
			case AV_PIX_FMT_NV42:
			case AV_PIX_FMT_P010LE:
			case AV_PIX_FMT_P016LE:
			case AV_PIX_FMT_YUV420P:
				if(format_datas->d3d_texture_format > 0)
				{
					if(mpxplay_video_render_infos.render_function_poolbuf_texture_map(bufelem->poolbufelem_texture_2d[0], &bufelem->poolbufelem_texture_mappedResource[0]))
					{
						struct dispqt_render_mapped_subresource_s *mapres = &bufelem->poolbufelem_texture_mappedResource[0];
						register const unsigned int linelen_out = mapres->vidmem_linesize;
						register uint8_t *outdataptr = (uint8_t *)mapres->vidmem_beginptr;
						const unsigned int linenum_out = mapres->vidmem_linenum;
						avframe_dest->data[0] = outdataptr;
						avframe_dest->data[1] = outdataptr + linelen_out * linenum_out;
						avframe_dest->linesize[0] = avframe_dest->linesize[1] = linelen_out;
						if(format_datas->av_pixfmt == AV_PIX_FMT_YUV420P)
						{
							register const unsigned int linelen_uv = linelen_out / 2;
							register const unsigned int height_uv = linenum_out / 2;
							avframe_dest->data[2] = avframe_dest->data[1] + linelen_uv * height_uv;
							avframe_dest->linesize[1] = avframe_dest->linesize[2] = linelen_uv;
						}
					}
					else
					{
						buffer_ok = false;
						mpxplay_debugf(DISPQT_DEBUG_ERROR, "POOL buffer elem MAP1 error at %d", bufelem->elem_id);
					}
					break;
				} // hack for planar pixel formats, at using multiply textures/views (D3D11)
				// @suppress("No break at end of case")
			default:
				for(int i = 0; i < format_datas->nb_components; i++)
				{
					struct dispqt_render_mapped_subresource_s *mapres = &bufelem->poolbufelem_texture_mappedResource[i];
					if(mpxplay_video_render_infos.render_function_poolbuf_texture_map(bufelem->poolbufelem_texture_2d[i], mapres))
					{
						if(!mapres->vidmem_linenum)
							mapres->vidmem_linenum = bufelem->allocated_height;
					}
					else
					{
						buffer_ok = false;
						mpxplay_debugf(DISPQT_DEBUG_ERROR, "POOL buffer elem MAP2 error at %d", bufelem->elem_id);
						break;
					}
				}
				if(!buffer_ok)
				{
					dispqt_evr_poolbufelem_texture_unmap(bufelem);
					break;
				}
				for(int i = 0; i < format_datas->nb_components; i++)
				{
					struct dispqt_render_mapped_subresource_s *mapres = &bufelem->poolbufelem_texture_mappedResource[i];
					avframe_dest->data[format_datas->plane_pos[i]] = (uint8_t *)mapres->vidmem_beginptr;
					avframe_dest->linesize[format_datas->plane_pos[i]] = mapres->vidmem_linesize;
				}
				break;
		}
#ifdef MPXPLAY_USE_DEBUGF
		getbuf_endmap = pds_gettimem();
#endif
	}

	if(buffer_ok)
	{
		avframe_dest->buf[0] = av_buffer_create((uint8_t *)bufelem, sizeof(*bufelem), dispqt_evr_poolbufelem_derefer, (void *)bufelem, 0);
		if(!avframe_dest->buf[0])
		{
			buffer_ok = false;
			mpxplay_debugf(DISPQT_DEBUG_ERROR, "POOL buffer elem av_buffer_create error at %d", bufelem->elem_id);
		}
		else
		{
			avframe_dest->format = format_datas->av_pixfmt;
			mpxplay_debugf(DISPQT_DEBUGOUT_POOL, "POOL buffer elem %d prepared nbc:%d (dur:%2d mutex:%d map:%d)", bufelem->elem_id, format_datas->nb_components,
			 (int)(pds_gettimem() - getbuf_begin), (int)(getbuf_mutex - getbuf_begin), (int)(getbuf_endmap - getbuf_map));
		}
	}

err_out_buffer:
	if(!buffer_ok)
	{
		if(bufelem)
		{
			dispqt_evr_poolbufelem_texture_unmap(bufelem);
			bufelem->bufpoolelem_status = DISPQT_RENDER_BUFPOOLELEM_STATUS_FREE;
			mpxplay_debugf(DISPQT_DEBUG_ERROR, "POOL buffer elem create error at %d", bufelem->elem_id);
		}
		if(!(flags & MPXPLAY_VIDEO_RENDERER_FRAME_POOL_ELEMFLAG_WRITEONLY) && avctx && avframe_dest)
			retval = avcodec_default_get_buffer2(avctx, avframe_dest, flags);
		else
			retval = -1;
	}

#if defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK) && !defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_GLOBALLOCK)
	if(mutex_error == MPXPLAY_ERROR_OK)
		PDS_THREADS_MUTEX_UNLOCK(&mpxplay_video_render_infos.d3d_device_mutex);
#endif

	return retval;
}

static mpxp_bool_t dispqt_videorender_common_poolbuf_elem_validity_check(void *avframe)
{
	struct dispqt_render_poolbufelem_s *bufelem;
	AVFrame *av_frame = (AVFrame *)avframe;
	if(!av_frame || !av_frame->buf[0] || !av_frame->buf[0]->data || (av_frame->buf[0]->size != sizeof(*bufelem)))
	{
//		mpxplay_debugf(DISPQT_DEBUG_ERROR, "POOL buffer invalid frame buf:%d d:%d size:%d pix:%d", ((av_frame->buf[0])? 1 : 0),
//				((av_frame->buf[0] && av_frame->buf[0]->data)? 1 : 0), ((av_frame->buf[0])? av_frame->buf[0]->size : 0), av_frame->format);
		return FALSE;
	}
	bufelem = (struct dispqt_render_poolbufelem_s *)av_frame->buf[0]->data;
	if(bufelem->bufpoolelem_struct_id != DISPQT_RENDER_BUFPOOLELEM_STRUCT_ID)
	{
		mpxplay_debugf(DISPQT_DEBUG_ERROR, "POOL buffer invalid frame (MMC id)");
		return FALSE;
	}
	return TRUE;
}

static mpxp_bool_t dispqt_videorender_common_poolbuf_copy_to_pool_buffer(void *dst_avframe, void *src_avframe)
{
	AVFrame *dest_frame = (AVFrame *)dst_avframe, *src_frame = (AVFrame *)src_avframe;
	struct dispqt_render_poolbufelem_s *bufelem;
	mpxp_bool_t success = FALSE;
	int mutex_error;

	if(!src_frame)
		return success;

	mutex_error = PDS_THREADS_MUTEX_LOCK(&mpxplay_video_render_infos.d3d_close_mutex, DISPQT_VIDEO_MUTEX_TIMEOUT);
	if(mutex_error != MPXPLAY_ERROR_OK)
		return success;

	if(!dispqt_videorender_common_poolbuf_elem_validity_check((void *)dest_frame))
		goto err_out_copy;

	bufelem = (struct dispqt_render_poolbufelem_s *)dest_frame->buf[0]->data;

	if(bufelem->bufpoolelem_status != DISPQT_RENDER_BUFPOOLELEM_STATUS_PROCESSING)
		goto err_out_copy;

	mpxplay_dispqt_videorender_common_videoframe_copy(bufelem->poolbufelem_texture_mappedResource,
		&bufelem->bufpoolelem_format_desc, src_frame->data, src_frame->linesize, src_frame->height);

	success = TRUE;

err_out_copy:

	if(mutex_error == MPXPLAY_ERROR_OK)
		PDS_THREADS_MUTEX_UNLOCK(&mpxplay_video_render_infos.d3d_close_mutex);

	return success;
}

static void dispqt_videorender_common_poolbuf_elem_unmap(void *avframe)
{
	AVFrame *av_frame = (AVFrame *)avframe;
#if defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK) && !defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_GLOBALLOCK)
	const int mutex_error = PDS_THREADS_MUTEX_LOCK(&mpxplay_video_render_infos.d3d_device_mutex, DISPQT_VIDEO_MUTEX_TIMEOUT);
	if(mutex_error != MPXPLAY_ERROR_OK)
		return;
#endif
	if(dispqt_videorender_common_poolbuf_elem_validity_check((void *)av_frame))
	{
		struct dispqt_render_poolbufelem_s *bufelem = (struct dispqt_render_poolbufelem_s *)av_frame->buf[0]->data;
		dispqt_evr_poolbufelem_texture_unmap(bufelem);
		if(bufelem->bufpoolelem_status == DISPQT_RENDER_BUFPOOLELEM_STATUS_PROCESSING)
			bufelem->bufpoolelem_status = DISPQT_RENDER_BUFPOOLELEM_STATUS_DONE;
	}
#if defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK) && !defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_GLOBALLOCK)
	PDS_THREADS_MUTEX_UNLOCK(&mpxplay_video_render_infos.d3d_device_mutex);
#endif
}

void mpxplay_dispqt_videorender_common_poolbuf_functions_assign(void)
{
	mpxplay_video_render_infos.render_function_poolbufelem_validity_check = dispqt_videorender_common_poolbuf_elem_validity_check;
	mpxplay_video_render_infos.render_function_is_framebuffer_support     = dispqt_videorender_common_poolbuf_is_frame_buffer_support;
	mpxplay_video_render_infos.render_function_get_frame_buffer           = dispqt_videorender_common_poolbuf_get_frame_buffer;
	mpxplay_video_render_infos.render_function_frame_buffer_consolidate   = dispqt_videorender_common_poolbuf_consolidate;
	mpxplay_video_render_infos.render_function_copy_to_pool_buffer        = dispqt_videorender_common_poolbuf_copy_to_pool_buffer;
	mpxplay_video_render_infos.render_function_poolbufelem_unmap          = dispqt_videorender_common_poolbuf_elem_unmap;
}

void mpxplay_dispqt_videorender_common_poolbuf_elems_close(struct dispqt_render_poolbufelem_s *bufelem_first)
{
	for(int i = 0; i < DISPQT_RENDER_POOLBUF_SIZE_MAX; i++)
	{
		struct dispqt_render_poolbufelem_s *bufelem = &bufelem_first[i];
		if(bufelem->bufpoolelem_status != DISPQT_RENDER_BUFPOOLELEM_STATUS_FREE)
			bufelem->bufpoolelem_status = DISPQT_RENDER_BUFPOOLELEM_STATUS_INVALIDATED;
	}
	for(int i = 0; i < DISPQT_RENDER_POOLBUF_SIZE_MAX; i++)
	{
		struct dispqt_render_poolbufelem_s *bufelem = &bufelem_first[i];
		dispqt_evr_poolbufelem_texture_unmap(bufelem);
		if(mpxplay_video_render_infos.render_function_poolbuf_texture_free)
			mpxplay_video_render_infos.render_function_poolbuf_texture_free(&bufelem->poolbufelem_texture_2d[0], &bufelem->poolbufelem_shader_resource_views[0]);
	}
}
