//**************************************************************************
//*                     This file is part of the                           *
//*                MMC - Mpxplay Multimedia Commander                      *
//*                   The source code of MMC is                            *
//*        (C) copyright 1998-2022 by PDSoft (Attila Padar)                *
//*                http://mpxplay.sourceforge.net                          *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************

#ifndef MOC_VIDEO_RENDER_H
#define MOC_VIDEO_RENDER_H

#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG
 
#include <QWidget>
#include <QtWidgets>

#include "moc_config.h"

#ifdef __cplusplus
extern "C" {
#endif
#include <libavcodec/avcodec.h>
#ifdef __cplusplus
}
#endif // __cplusplus

QT_BEGIN_NAMESPACE
class MainWindow;
class MMCDispQtVideoRendererBase;
QT_END_NAMESPACE

#define DISPQT_VIDEO_RENDER_MUTEX_TIMEOUT     200

#define DISPQT_VIDEO_QIMAGE_OUTPUT_AVFORMAT AV_PIX_FMT_BGRA        // little endian ARGB
#define DISPQT_VIDEO_QIMAGE_OUTPUT_QFORMAT  QImage::Format_RGB32

// format(s) of subtitle_memory_render_image
#define DISPQT_VIDEO_SUBTITLE_QIMAGE_AVFORMAT AV_PIX_FMT_BGRA
#define DISPQT_VIDEO_SUBTITLE_QIMAGE_QFORMAT  QImage::Format_ARGB32_Premultiplied

#define DISPQT_VIDEO_SUBTITLE_BITMAP_AVFORMAT DISPQT_VIDEO_SUBTITLE_QIMAGE_AVFORMAT

#define DISPQT_VIDEO_FILTERPROCESS_AVFORMAT AV_PIX_FMT_YUV420P // primary pixfmt for video processing

#define DISPQT_VIDEO_AUTOCROP_USE_RGB32 1  // QWidget and Vulkan outputs (FIXME: the autocrop doesn't work properly in YUV / D3D output)

#if DISPQT_VIDEO_AUTOCROP_USE_RGB32
#define DISPQT_VIDEO_AUTOCROP_AVFORMAT AV_PIX_FMT_BGRA
#else
#define DISPQT_VIDEO_AUTOCROP_AVFORMAT DISPQT_VIDEO_FILTERPROCESS_AVFORMAT
#endif

typedef struct dispqt_video_source_info_s{
	AVFrame *ffmpeg_decoded_frame;     // came from the decoder (decoded, sw deinterlaced frame), in DISPQT_VIDEO_FILTERPROCESS_AVFORMAT, or in native decoder output
	AVRational video_aspect_ratio;     // to detect aspect ratio changes (recalc surface)
	int sws_srccrop_x, sws_srccrop_y, sws_srccrop_width, sws_srccrop_height; // crop of the source (manual or auto)
	unsigned long last_video_crop_type, last_video_autocrop_limit, ffmpeg_swsframe_cropreset;
	int sws_autoxcrop_framecounter;
}dispqt_video_source_info_s;

typedef struct dispqt_video_filter_format_s{
	int width;
	int height;
	enum AVPixelFormat pixelformat;
}dispqt_video_filter_format_s;

typedef struct dispqt_video_filter_info_s{
	AVFrame *ffmpeg_sws_frame;         // for autocrop and for rgb32 (qwidget) output if picture filtering is disabled
	AVFrame *ffmpeg_transformed_frame; // picture transformed (pad, rotation, etc.) frame
	AVFrame *ffmpeg_filtered_frame;    // picture filtered (brightness, contrast, etc.) frame
	mpxp_bool_t ffmpeg_swsframe_hasdata;      // ffmpeg_sws_frame has valid data
	struct SwsContext *sws_ctx;               // context of frame conversion
	struct dispqt_video_filter_format_s input_format_cfg;  // sws is initialized with this source format infos (came from the ffmpeg_decoded_frame)
	struct dispqt_video_filter_format_s output_format_cfg; // sws is initialized to this output format (sws output frame may has different pixelformat, like YUVJ to YUV change)
}dispqt_video_filter_info_s;

#define DISPQT_SUBTITLE_TYPE_TEXT       0 // subtitle in text format
#define DISPQT_SUBTITLE_TYPE_BITMAP     1 // subtitle in bitmap (ARGB) format

#define INFFMPEG_ASS_DECODED_MAX_LINES    8
#define INFFMPEG_ASS_DECODED_FLAG_ITALIC (1 << 0)

typedef struct ffmpegvideo_subtitle_data_s
{
	char *subtitle_line_data;          // ARGB bitmap data OR utf-8 text subtitle strings (depending on subtitle_type)
	unsigned int subtitle_line_info;   // bitmap picture linesize OR control/attrib flags for text subtitle displaying/format (eg. italic) (depending on subtitle_type)
	int subtitle_bitmap_w, subtitle_bitmap_h; // resolution of bitmap subtitle_line_data
}ffmpegvideo_subtitle_data_s;

typedef struct ffmpegvideo_subtitle_info_s
{
	unsigned int subtitle_type;                           // DISPQT_SUBTITLE_TYPE_
	unsigned int subtitle_nb_lines;                       // number of subtitle text/bitmap lines
	struct ffmpegvideo_subtitle_data_s subtitle_datas[INFFMPEG_ASS_DECODED_MAX_LINES];
}ffmpegvideo_subtitle_info_s;

// video renderer capability flags
#define DISPQT_VIDEORENDERER_CAPFLAG_WIDGET        (1 << 0)  // it's a widget renderer (else a native painter)
#define DISPQT_VIDEORENDERER_CAPFLAG_DIRECTRENDER  (1 << 1)  // can call directly mpxplay_dispqt_videowidget_frame_display
#define DISPQT_VIDEORENDERER_CAPFLAG_PREPRESENT    (1 << 2)  // renderer has multiply image buffer for render and it can call present before render
#define DISPQT_VIDEORENDERER_CAPFLAG_DEINTERLACE   (1 << 4)  // renderer has deinterlace function
#define DISPQT_VIDEORENDERER_CAPFLAG_DEINTERDUP    (1 << 5)  // renderer has deinterlace frame duplication function
#define DISPQT_VIDEORENDERER_CAPFLAG_ROTATE        (1 << 6)  // renderer has video frame rotation function
#define DISPQT_VIDEORENDERER_CAPFLAG_COLORMIXER    (1 << 8)  // renderer has color change function (brightness, contrast, saturation, hue)
#define DISPQT_VIDEORENDERER_CAPENABLE_DEINTERDUP  (1 << 16) // renderer deinterlace frame duplication function is enabled
#define DISPQT_VIDEORENDERER_CAPFLAGS_PREPROCESS   (DISPQT_VIDEORENDERER_CAPFLAG_DEINTERLACE)
#define DISPQT_VIDEORENDERER_CAPFLAGS_TRANSFORM    (DISPQT_VIDEORENDERER_CAPFLAG_ROTATE) // sw transform currently contains rotate only
#define DISPQT_VIDEORENDERER_CAPFLAGS_POSTPROCESS  (DISPQT_VIDEORENDERER_CAPFLAG_COLORMIXER)

typedef struct dispqt_video_surface_info_s{
	int sws_crop_x, sws_crop_y, sws_crop_w, sws_crop_h;              // video source attribs (crop of the output (zoom / move))
	int sws_dst_width, sws_dst_height, sws_dst_pos_x, sws_dst_pos_y; // video frame (stretch) in video window
	int window_pos_x, window_pos_y, window_size_x, window_size_y;    // video window/widget attribs (positions are global)
	int windowed_size_x, windowed_size_y;                            // size of window in non-fullscreen mode
	int fullscreen_size_x, fullscreen_size_y;                        // resolution of display (for fullscreen window calculation)
	int last_video_zoom_min_val, last_video_zoom_ti_val;
	int video_zoom_globalpos_x, video_zoom_globalpos_y;              // global display pos for zooming (using the mouse pos as the center of zoom)
	bool is_fullscreen;
	AVFrame *ffmpeg_output_frame;                                    // points to ffmpeg_sws_frame or ffmpeg_filtered_frame after preprocess
	int video_outframe_last_width, video_outframe_last_height;       // last displayed video frame width / height (to detect resolution changes -> recalc surface)
	mpxp_bool_t videoout_surface_recalc;               // re-calculate surface rectangles (videowidget_ffmpeg_swsctx_calculate)
	mpxp_bool_t videoout_surface_clearpic;             // clear output surface before or instead of rendering (eg: at autocrop, or at new file)
	mpxp_bool_t videoout_surface_reset;                // reset surface (eg. clear interlace refs at seeking)
	mpxp_bool_t videoout_input_change;                 // new input video file
	mpxp_bool_t videoout_inputframe_change;            // new input video frame (not a reused repaint, like at color mixer change or zoom)
	mpxp_bool_t update_resize, do_resize_repaint;
	mpxp_bool_t update_filter_trans, update_filter_eq; // basically they belong to the filter
	unsigned long rotation_type;                       // DISPQT_VIDEO_ROTATIONTYPE_ (real output rotation type)
	unsigned long scheduler_flags;                     // DISPQT_FFMPGVIDEOCALLBACKFLAG_RF_ came from scheduler through frame_display()
	enum AVPixelFormat videoout_surface_pix_fmt;       // the renderer output pixel format (eg. AV_PIX_FMT_BGRA for QWidget)
	mpxplay_video_renderer_info_s *video_render_infos; // mpxplay_video_render_infos
	ffmpegvideo_subtitle_info_s *sub_title_infos;      // stored subtitle datas
	dispqt_video_source_info_s *video_source_infos;    // pointet to the video source infos
	dispqt_video_filter_info_s *video_filter_infos;    // pointer to the video filter infos
	struct mmc_dispqt_config_s *gui_config;            // pointer to mainwindow's gui config
	MMCDispQtVideoRendererBase *render_base;           // pointer to render base
	void *mutex_render;                                // mutex for rendering
	AVFrame ffmpeg_empty_frame;                        // empty frame to clear output surface
}dispqt_video_surface_info_s;

#define DISPQT_RENDERBASE_SUBTITLE_CTRLFLAG_FULLSCREEN    (1 << 0) // subtitle rendering is on full window surface (else into small subpicture) (QWidget)
#define DISPQT_RENDERBASE_SUBTITLE_CTRLFLAG_DRAWEVERY     (1 << 1) // draw subtitle at every call (don't check differences for update) (QWidget)
#define DISPQT_RENDERBASE_SUBTITLE_CTRLFLAG_CLEARBORDERS  (1 << 2) // clear black borders at subtitle change (QWidget)
#define DISPQT_RENDERBASE_SUBTITLE_CTRLFLAG_BLACKBORDERS  (1 << 3) // draw black borders (always) at the case of subtitles (GDI)
#define DISPQT_RENDERBASE_SUBTITLE_CTRLFLAG_SCALETOVIDEO  (1 << 4) // scale font size to video height (not to the screen height) (Vulkan puts subtitle into the video frame)
#define DISPQT_RENDERBASE_SUBTITLE_CTRLFLAG_RENDERTOVIDEO (1 << 5) // render into the video frame (RGB32 outputs only!)
#define DISPQT_RENDERBASE_SUBTITLE_CTRLFLAG_QIMAGERENDER  (1 << 16) // render into a memory field (non QWidget) (internal flag) (from externally it indicated by the different videorenderer_videoframe_render calls)

// return values/bits of renderbase_subtitle_qpaint
#define DISPQT_RENDERBASE_SUBPAINT_RETVAL_NOOP         0  // no any sub-picture activity
#define DISPQT_RENDERBASE_SUBPAINT_RETVAL_CLEAR        1  // clear sub-picture (no subtitle after a subtitle rendering)
#define DISPQT_RENDERBASE_SUBPAINT_RETVAL_OLDRENDER    2  // subtitle content has not changed, using previous rendering
#define DISPQT_RENDERBASE_SUBPAINT_RETVAL_NEWRENDER    4  // a new sub-picture has rendered (content has changed)
#define DISPQT_RENDERBASE_SUBPAINT_RETVAL_RECTCHANGE   8  // resolution of sub-picture has changed
#define DISPQT_RENDERBASE_SUBPAINT_RETVAL_POSCHANGE   16  // position of sub-picture has changed

class MMCDispQtVideoRendererBase : public QObject
{
Q_OBJECT
public:
	MMCDispQtVideoRendererBase(MainWindow *mainwindow = 0, QWidget *parent = 0);
	void renderbase_subtitle_reset(void);

	virtual bool videorenderer_init(struct dispqt_video_surface_info_s *video_surface_infos) {return true;}
	virtual void videorenderer_get_surface_attribs(struct dispqt_video_surface_info_s *video_surface_infos, AVFrame *decoded_frame)
	{ video_surface_infos->videoout_surface_pix_fmt = DISPQT_VIDEO_QIMAGE_OUTPUT_AVFORMAT; video_surface_infos->video_render_infos->renderer_capabilities = 0;}
	virtual void videorenderer_set_mixer(unsigned int u_id, int value) {}
	virtual void videorenderer_set_fullscreen(mpxp_bool_t full) {}

	virtual void videorenderer_videoframe_render(struct dispqt_video_surface_info_s *video_surface_infos, ffmpegvideo_subtitle_info_s *subtitle_infos) {}
	virtual void videorenderer_videoframe_render(QPainter *p, struct dispqt_video_surface_info_s *video_surface_infos, ffmpegvideo_subtitle_info_s *subtitle_infos) {}

protected:
	int renderbase_subtitle_qpaint(struct dispqt_video_surface_info_s *video_surface_infos, ffmpegvideo_subtitle_info_s *subtitle_infos, unsigned int control_flags);
	int renderbase_subtitle_qpaint(struct dispqt_video_surface_info_s *video_surface_infos, ffmpegvideo_subtitle_info_s *subtitle_infos, QPainter *painter, unsigned int control_flags);
	uint8_t *get_subtitle_image_memory_ptr(void){const QImage* im = &this->subtitle_memory_render_image; return (uint8_t *)im->bits();}
	int get_subtitle_image_memory_linesize(void){return this->subtitle_memory_render_image.bytesPerLine();}
	int get_subtitle_pic_width(){ return this->subtitle_memory_render_image.width(); }
	int get_subtitle_pic_height(){ return this->subtitle_memory_render_image.height(); }
	int get_subtitle_pic_globalpos_x() { return this->subtitle_pic_globalpos_x; }
	int get_subtitle_pic_globalpos_y() { return this->subtitle_pic_globalpos_y; }

	MainWindow *main_window;
	QWidget *parent_widget;

private:
	bool renderbase_subtitle_font_init(struct dispqt_video_surface_info_s *video_surface_infos, unsigned int control_flags);
	int  renderbase_subtitle_subpicture_calc(struct dispqt_video_surface_info_s *video_surface_infos, ffmpegvideo_subtitle_info_s *subtitle_infos, unsigned int control_flags);
	int  subtitle_screen_scale, subtitle_bitmap_scale;
	int  subtitle_font_pointsize, subtitle_font_char_pixelsize;
	int  subtitle_pic_globalpos_x, subtitle_pic_globalpos_y;
	int  subtitle_pic_width, subtitle_pic_heigth;
	int  subtitle_pic_renderpos_y;
	bool was_subtitle_rendered;
	unsigned int subtitle_line_flags;
	unsigned int subtitle_line_pixellens[INFFMPEG_ASS_DECODED_MAX_LINES];    // length of the subtitle lines in pixels
	unsigned int last_subtitle_nb_lines;
	int  last_bitmap_width, last_bitmap_height;
	char last_line_texts[INFFMPEG_ASS_DECODED_MAX_LINES][256];
	char last_font_text_name[128];
	QFont font_subtitle;
	QImage subtitle_memory_render_image;
};

#if MPXPLAY_DISPQT_ENABLE_RENDER_QPAINT
class MMCDispQtVideoRendererQPaint : public MMCDispQtVideoRendererBase
{
Q_OBJECT
public:
	MMCDispQtVideoRendererQPaint(MainWindow *mainwindow = 0, QWidget *parent = 0);

	void videorenderer_get_surface_attribs(struct dispqt_video_surface_info_s *video_surface_infos, AVFrame *decoded_frame);
	void videorenderer_videoframe_render(QPainter *p, struct dispqt_video_surface_info_s *video_surface_infos, ffmpegvideo_subtitle_info_s *subtitle_infos);
};
#endif // MPXPLAY_DISPQT_ENABLE_RENDER_QPAINT

#if MPXPLAY_DISPQT_ENABLE_RENDER_D3D11
class MMCDispQtVideoRendererD3D11 : public MMCDispQtVideoRendererBase
{
Q_OBJECT
public:
	MMCDispQtVideoRendererD3D11(MainWindow *mainwindow = 0, QWidget *parent = 0);
	~MMCDispQtVideoRendererD3D11();

	bool videorenderer_init(struct dispqt_video_surface_info_s *video_surface_infos);
	void videorenderer_get_surface_attribs(struct dispqt_video_surface_info_s *video_surface_infos, AVFrame *decoded_frame);
	void videorenderer_set_mixer(unsigned int u_id, int value);
	void videorenderer_set_fullscreen(mpxp_bool_t full);
	void videorenderer_videoframe_render(struct dispqt_video_surface_info_s *video_surface_infos, ffmpegvideo_subtitle_info_s *subtitle_infos);

private:
	bool videorenderer_d3d11_textures_load(struct dispqt_video_surface_info_s *video_surface_infos, ffmpegvideo_subtitle_info_s *subtitle_infos);
	void videorenderer_d3d11_texture_subtitle_load(void);
	void videorenderer_d3d11_textures_render(struct dispqt_video_surface_info_s *video_surface_infos, ffmpegvideo_subtitle_info_s *subtitle_infos);
	void videorenderer_d3d11_textures_present(struct dispqt_video_surface_info_s *video_surface_infos, ffmpegvideo_subtitle_info_s *subtitle_infos, bool texture_load_ok);
	bool videorenderer_d3d11_deinterlace_2ndpass(struct dispqt_video_surface_info_s *video_surface_infos, ffmpegvideo_subtitle_info_s *subtitle_infos);
	void *private_data;
	QWidget *parent_widget;
};

extern mpxp_bool_t mpxplay_dispqt_videorender_d3d11_static_init(void);
extern void mpxplay_dispqt_videorender_d3d11_static_close(void);
extern void mpxplay_dispqt_videorender_d3d11_flush(mpxp_bool_t wait_for_completition);
extern mpxp_bool_t mpxplay_dispqt_videorender_d3d11_refer_d3ddevice(struct mpxplay_video_d3d_device_ref_s *d3d_device);
extern mpxp_bool_t mpxplay_dispqt_videorender_d3d11_detect_device_context_failure(void);

#endif // MPXPLAY_DISPQT_ENABLE_RENDER_D3D11

#if MPXPLAY_DISPQT_ENABLE_RENDER_D3D9
class MMCDispQtVideoRendererD3D9 : public MMCDispQtVideoRendererBase
{
Q_OBJECT
public:
	MMCDispQtVideoRendererD3D9(MainWindow *mainwindow = 0, QWidget *parent = 0);
	~MMCDispQtVideoRendererD3D9();

	bool videorenderer_init(struct dispqt_video_surface_info_s *video_surface_infos);
	void videorenderer_get_surface_attribs(struct dispqt_video_surface_info_s *video_surface_infos, AVFrame *decoded_frame);
	void videorenderer_set_mixer(unsigned int u_id, int value);
	void videorenderer_set_fullscreen(mpxp_bool_t full);
	void videorenderer_videoframe_render(struct dispqt_video_surface_info_s *video_surface_infos, ffmpegvideo_subtitle_info_s *subtitle_infos);

private:
	bool videorenderer_d3d9_textures_load(struct dispqt_video_surface_info_s *video_surface_infos, ffmpegvideo_subtitle_info_s *subtitle_infos);
	void videorenderer_d3d9_texture_subtitle_load(struct dispqt_video_surface_info_s *video_surface_infos, ffmpegvideo_subtitle_info_s *subtitle_infos);
	bool videorenderer_d3d9_textures_render(struct dispqt_video_surface_info_s *video_surface_infos);
	void *private_data;
};

extern mpxp_bool_t mpxplay_dispqt_videorender_d3d9_static_init(void);
extern void mpxplay_dispqt_videorender_d3d9_static_close(void);

#endif // MPXPLAY_DISPQT_ENABLE_RENDER_D3D9

#if MPXPLAY_DISPQT_ENABLE_RENDER_VULKAN

class MMCDispQtVideoRendererVulkan : public MMCDispQtVideoRendererBase
{
Q_OBJECT
public:
	MMCDispQtVideoRendererVulkan(MainWindow *mainwindow = 0, QWidget *parent = 0);
	~MMCDispQtVideoRendererVulkan();

	bool videorenderer_init(struct dispqt_video_surface_info_s *video_surface_infos);
	void videorenderer_get_surface_attribs(struct dispqt_video_surface_info_s *video_surface_infos, AVFrame *decoded_frame);
	void videorenderer_videoframe_render(struct dispqt_video_surface_info_s *video_surface_infos, ffmpegvideo_subtitle_info_s *subtitle_infos);

private:
	bool videorenderer_vulkan_fill_textures(struct dispqt_video_surface_info_s *video_surface_infos, ffmpegvideo_subtitle_info_s *subtitle_infos);
	void *private_data;
};
#endif // MPXPLAY_DISPQT_ENABLE_RENDER_VULKAN

#if MPXPLAY_DISPQT_ENABLE_RENDER_GDI
class MMCDispQtVideoRendererGDI : public MMCDispQtVideoRendererBase
{
Q_OBJECT
public:
	MMCDispQtVideoRendererGDI(MainWindow *mainwindow = 0, QWidget *parent = 0);

	void videorenderer_videoframe_render(struct dispqt_video_surface_info_s *video_surface_infos, ffmpegvideo_subtitle_info_s *subtitle_infos);

};
#endif // MPXPLAY_DISPQT_ENABLE_RENDER_GDI

#endif // MPXPLAY_LINK_ORIGINAL_FFMPEG

#endif // MOC_VIDEO_RENDER_H
