//**************************************************************************
//*                     This file is part of the                           *
//*                MMC - Mpxplay Multimedia Commander                      *
//*                   The source code of MMC is                            *
//*        (C) copyright 1998-2020 by PDSoft (Attila Padar)                *
//*                http://mpxplay.sourceforge.net                          *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//function: initialize, load, save gui config

//#define MPXPLAY_USE_DEBUGF 1
#define DISPQT_DEBUG_OUTPUT stdout

#include "mpxplay.h"
#include "control/control.h"
#include "control/cntfuncs.h"
#include "moc_config.h"
#include "moc_mainwindow.h"

extern "C" {
 extern unsigned long mpxplay_config_videoplayer_type;
 extern unsigned long mpxplay_config_video_audiovisualization_type;
 extern char mpxplay_extrafiletypes_config[MPXINI_MAX_CHARDATA_LEN];
#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG
 extern unsigned long mpxplay_config_dvbepg_pastevents_hours;
 extern char mpxplay_config_video_preffered_audiotypes[MPXINI_MAX_CHARDATA_LEN];
 extern char mpxplay_config_video_preferred_language[MPXPLAY_STREAMTYPEINDEX_PLAYNUM][MPXINI_MAX_CHARDATA_LEN];
 extern unsigned int mpxplay_config_video_extstream_loadtype[MPXPLAY_STREAMTYPEINDEX_EXTSTREAMNUM];
 extern char mpxplay_config_video_extstream_subdirs[MPXPLAY_STREAMTYPEINDEX_EXTSTREAMNUM][MPXINI_MAX_CHARDATA_LEN];
#endif
}

struct mmc_dispqt_config_s mmc_config;
char config_video_file_record_save_directory[MAX_PATHNAMELEN];
char config_video_file_record_type_name[MPXINI_MAX_CHARDATA_LEN];

static const mpxini_var_s mpxpgui_ini[]={
{"GuiLayoutType",  &mmc_config.gui_layout_type, ARG_SAVE },
{"GuiBckgrndType", &mmc_config.gui_background_type, ARG_SAVE },
{"GuiBckgrndEfct", &mmc_config.gui_background_effect, ARG_SAVE },
{"GuiBckgrndAlpha",&mmc_config.gui_background_color_alpha, ARG_SAVE },
{"MainWinControl", &mmc_config.mainwin_control,(ARG_HEX | ARG_SAVE) },
{"MainWinState",   &mmc_config.mainwin_state,  ARG_SAVE },
{"MainWinPosX",    &mmc_config.mainwin_pos_x,  (ARG_SIGNED|ARG_SAVE) },
{"MainWinPosY",    &mmc_config.mainwin_pos_y,  (ARG_SIGNED|ARG_SAVE) },
{"MainWinSizeX",   &mmc_config.mainwin_size_x, ARG_SAVE },
{"MainWinSizeY",   &mmc_config.mainwin_size_y, ARG_SAVE },
{"MainWinOpGen",   &mmc_config.mainwin_opacity_general, ARG_SAVE },
{"MainWinOpIna",   &mmc_config.mainwin_opacity_inactive, ARG_SAVE },
{"MainWinOpTra",   &mmc_config.mainwin_opacity_transparent, ARG_SAVE },
{"MainWinOpVSh",   &mmc_config.mainwin_opacity_onvideo_show, ARG_SAVE },
{"MainWinOpVHi",   &mmc_config.mainwin_opacity_onvideo_hide, ARG_SAVE },
{"MainWinOpVSKT",  &mmc_config.mainwin_opacity_showkeep_time, ARG_SAVE },
{"MainWinOpVFIT",  &mmc_config.mainwin_opacity_fadein_time, ARG_SAVE },
{"MainWinOpVFOT",  &mmc_config.mainwin_opacity_fadeout_time, ARG_SAVE },
{"EditorControl",  &mmc_config.editor_flags, (ARG_HEX | ARG_SAVE) },
{"EditorSizeY",    &mmc_config.editor_size_y, ARG_SAVE },
{"FontEditor",     &mmc_config.font_text_name[DISPQT_FONTTEXTID_EDITOR],  (ARG_PACHAR|ARG_SAVE) },
{"FontMainwin",    &mmc_config.font_text_name[DISPQT_FONTTEXTID_MAINWIN], (ARG_PACHAR|ARG_SAVE) },
{"FontDialog",     &mmc_config.font_text_name[DISPQT_FONTTEXTID_DIALOG],  (ARG_PACHAR|ARG_SAVE) },
{"FontSubtitle",   &mmc_config.font_text_name[DISPQT_FONTTEXTID_SUBTITLE],(ARG_PACHAR|ARG_SAVE) },
{"ColorWinBase",       &mmc_config.color_window_text_name[DISPQT_COLORTEXTID_WINDOW_BASE],        (ARG_PACHAR|ARG_SAVE) },
{"ColorWinTransp",     &mmc_config.color_window_text_name[DISPQT_COLORTEXTID_WINDOW_TRANSPARENT], (ARG_PACHAR|ARG_SAVE) },
#ifdef DISPQT_COLORTEXTID_CONFIGURE_TEXTCOLORS
{"ColorWinTextBase",   &mmc_config.color_window_text_name[DISPQT_COLORTEXTID_WINDOW_TEXT_BASE],   (ARG_PACHAR|ARG_SAVE) },
{"ColorWinTextTransp", &mmc_config.color_window_text_name[DISPQT_COLORTEXTID_WINDOW_TEXT_TRANSP], (ARG_PACHAR|ARG_SAVE) },
#endif
{"ColorEditFontbase",  &mmc_config.color_text_name[DISPQT_COLORTEXTID_EDITOR_FONT],  (ARG_PACHAR|ARG_SAVE) },
{"ColorEditBackGr1",   &mmc_config.color_text_name[DISPQT_COLORTEXTID_EDITOR_BACK1], (ARG_PACHAR|ARG_SAVE) },
{"ColorEditBackGr2",   &mmc_config.color_text_name[DISPQT_COLORTEXTID_EDITOR_BACK2], (ARG_PACHAR|ARG_SAVE) },
{"ColorEditCursor",    &mmc_config.color_text_name[DISPQT_COLORTEXTID_EDITOR_CURSOR], (ARG_PACHAR|ARG_SAVE) },
{"ColorEditMark",      &mmc_config.color_text_name[DISPQT_COLORTEXTID_EDITOR_MARK],   (ARG_PACHAR|ARG_SAVE) },
{"ColorEditCurrSong",  &mmc_config.color_text_name[DISPQT_COLORTEXTID_EDITOR_CURRSONG], (ARG_PACHAR|ARG_SAVE) },
{"ColorEditNextSong",  &mmc_config.color_text_name[DISPQT_COLORTEXTID_EDITOR_NEXTSONG], (ARG_PACHAR|ARG_SAVE) },
{"ColorEditPlayed",    &mmc_config.color_text_name[DISPQT_COLORTEXTID_EDITOR_PLAYED],   (ARG_PACHAR|ARG_SAVE) },
{"ColorEditNotPlayed", &mmc_config.color_text_name[DISPQT_COLORTEXTID_EDITOR_NOTPLAYED], (ARG_PACHAR|ARG_SAVE) },
{"TransparentControl", &mmc_config.transparent_control, (ARG_HEX | ARG_SAVE) },
{"VideoCrtrlbarSize",  &mmc_config.videocontrolbar_size_percent, ARG_SAVE },
{"VideoSeekPreVSize",  &mmc_config.videoseekpreview_size_percent, ARG_SAVE },
{"VideoSeekPreVAspr",  &mmc_config.videoseekpreview_aspect_ratio_type, ARG_SAVE },
{"VideoSeekPreVTime",  &mmc_config.videoseekpreview_refresh_interval_ms, ARG_SAVE },
{"VideoControl",       &mmc_config.video_control, (ARG_HEX | ARG_SAVE) },
{"VideoRendererType",  &mmc_config.video_renderer_type, ARG_SAVE },
{"VideoARFullscreen"  ,&mmc_config.video_fullscreen_ar_type, ARG_SAVE },
{"VideoARWindowed",    &mmc_config.video_windowed_ar_type, ARG_SAVE },
{"VideoARCustom",      &mmc_config.video_ar_custom[0], (ARG_CHAR|ARG_SAVE) },
{"VideoCropType",      &mmc_config.video_crop_type, ARG_SAVE },
{"VideoCropCustom",    &mmc_config.video_crop_custom[0], (ARG_CHAR|ARG_SAVE) },
{"VideoAutocropLimit", &mmc_config.video_autocrop_limit, ARG_SAVE },
{"VideoZoomType",      &mmc_config.video_zoom_type, ARG_SAVE },
{"VideoZoomCustom",    &mmc_config.video_zoom_custom[0], (ARG_CHAR|ARG_SAVE) },
{"VideoZoomValue",     &mmc_config.video_zoom_value, ARG_SAVE },
{"VideoDeinterlace",   &mmc_config.video_deinterlace_type, ARG_SAVE },
{"VideoRotation",      &mmc_config.video_rotation_type, ARG_SAVE },
{"VideoHDRControl",    &mmc_config.video_hdr_control, (ARG_HEX | ARG_SAVE) },
{"VideoScreenName",    &mmc_config.video_screen_select_name[0], (ARG_CHAR|ARG_SAVE) },
{"EPGDialogWinPosX",   &mmc_config.epgdialogwin_pos_x,  (ARG_SIGNED|ARG_SAVE) },
{"EPGDialogWinPosY",   &mmc_config.epgdialogwin_pos_y,  (ARG_SIGNED|ARG_SAVE) },
{"EPGDialogWinSizeX",  &mmc_config.epgdialogwin_size_x, ARG_SAVE },
{"EPGDialogWinSizeY",  &mmc_config.epgdialogwin_size_y, ARG_SAVE },
{"EPGPastEventsHours", &mpxplay_config_dvbepg_pastevents_hours, ARG_SAVE },
{"VideoBrightness",    &mmc_config.video_mixer_values[DISPQT_VIDEO_VMIXERTYPE_BRIGHTNESS], ARG_SAVE },
{"VideoContrast",      &mmc_config.video_mixer_values[DISPQT_VIDEO_VMIXERTYPE_CONTRAST], ARG_SAVE },
{"VideoSaturation",    &mmc_config.video_mixer_values[DISPQT_VIDEO_VMIXERTYPE_SATURATION], ARG_SAVE },
{"VideoHue",           &mmc_config.video_mixer_values[DISPQT_VIDEO_VMIXERTYPE_HUE], ARG_SAVE },
{"AudioVisualization", &mpxplay_config_video_audiovisualization_type, ARG_SAVE },

// placed here, due to mpxplay.ini's limited string data length
{"FileTypesExt",       &mpxplay_extrafiletypes_config[0], (ARG_CHAR|ARG_SAVE) },
#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG
{"PreferAudioTypes",   &mpxplay_config_video_preffered_audiotypes[0], (ARG_CHAR|ARG_SAVE) },
{"AudioLanguage",      &mpxplay_config_video_preferred_language[MPXPLAY_STREAMTYPEINDEX_AUDIO][0], (ARG_CHAR | ARG_SAVE)},
{"SubtitleLange",      &mpxplay_config_video_preferred_language[MPXPLAY_STREAMTYPEINDEX_SUBTITLE][0], (ARG_CHAR | ARG_SAVE)},
{"VideoLanguage",      &mpxplay_config_video_preferred_language[MPXPLAY_STREAMTYPEINDEX_VIDEO][0], (ARG_CHAR | ARG_SAVE)},
{"ExtStreamAudioLoad",    &mpxplay_config_video_extstream_loadtype[MPXPLAY_STREAMTYPEINDEX_AUDIO], ARG_SAVE},
{"ExtStreamSubtitleLoad", &mpxplay_config_video_extstream_loadtype[MPXPLAY_STREAMTYPEINDEX_SUBTITLE], ARG_SAVE},
{"ExtStreamAudioDirs",    &mpxplay_config_video_extstream_subdirs[MPXPLAY_STREAMTYPEINDEX_AUDIO][0], (ARG_CHAR | ARG_SAVE)},
{"ExtStreamSubtitleDirs", &mpxplay_config_video_extstream_subdirs[MPXPLAY_STREAMTYPEINDEX_SUBTITLE][0], (ARG_CHAR | ARG_SAVE)},
{"RecordFileSaveDir",  &config_video_file_record_save_directory[0], (ARG_CHAR | ARG_SAVE)},
{"RecordFileTypeName", &config_video_file_record_type_name[0], (ARG_CHAR | ARG_SAVE)},
#endif
{NULL, NULL, 0}
};

#define MPXPGUI_INI_CONFIG_LINES (sizeof(mpxpgui_ini) / sizeof(mpxini_var_s))

static const char *config_colortext_window_default[DISPQT_COLORTEXTID_WINDOW_MAX] = {
 "#ececec", // DISPQT_COLORTEXTID_WINDOW_BASE
 "#002d4b" // DISPQT_COLORTEXTID_WINDOW_TRANSPARENT  (dark cyan-blue)
 //"#082853" // DISPQT_COLORTEXTID_WINDOW_TRANSPARENT (matching with Win11 base blue)
#ifdef DISPQT_COLORTEXTID_CONFIGURE_TEXTCOLORS
,"#000000", // DISPQT_COLORTEXTID_WINDOW_TEXT_BASE
 "#ffffff"  // DISPQT_COLORTEXTID_WINDOW_TEXT_TRANSP
#endif
};

static const char *config_colortext_editortable_default[DISPQT_COLORTEXTID_EDITOR_MAX] = {
 "#aaffff", // DISPQT_COLORTEXTID_EDITOR_FONT : light-cyan
 "#002d4b", // DISPQT_COLORTEXTID_EDITOR_BACK1: dark cyan-blue
 "#002d4b", // DISPQT_COLORTEXTID_EDITOR_BACK2: dark cyan-blue
 "#808080", // DISPQT_COLORTEXTID_EDITOR_CURSOR: gray
 "#ffff00", // DISPQT_COLORTEXTID_EDITOR_MARK:  yellow
 "#00ffff", // DISPQT_COLORTEXTID_EDITOR_CURRSONG: cyan
 "#00ff00", // DISPQT_COLORTEXTID_EDITOR_NEXTSONG: green
 "#c8c8c8", // DISPQT_COLORTEXTID_EDITOR_PLAYED: light-gray
 "#ffffff"  // DISPQT_COLORTEXTID_EDITOR_NOTPLAYED: white
};

static const char *config_video_extstream_audio_subdirs_default = ".;." PDS_DIRECTORY_SEPARATOR_STR "audio;." PDS_DIRECTORY_SEPARATOR_STR "fandub";
static const char *config_video_extstream_subtitle_subdirs_default = ".;." PDS_DIRECTORY_SEPARATOR_STR "subtitles;." PDS_DIRECTORY_SEPARATOR_STR "subs;." PDS_DIRECTORY_SEPARATOR_STR "sub";

void mpxplay_dispqt_configini_init(void)
{
	struct mmc_dispqt_config_s *gcfg = &mmc_config;

	memset((void *)&mmc_config, 0, sizeof(mmc_config));

	gcfg->mainwin_pos_x = -1;
	gcfg->mainwin_pos_y = -1;
	gcfg->mainwin_size_x = DISPQT_MAINWIN_INITSIZE_X;
	gcfg->mainwin_size_y = -1;
	gcfg->editor_size_y  = 100;

	mpxplay_dispqt_configini_reset_configurables();
}

static void mpxplay_dispqt_configini_reset_video_surface_settings(void)
{
	struct mmc_dispqt_config_s *gcfg = &mmc_config;

	gcfg->video_fullscreen_ar_type = DISPQT_CONFIG_VIDEOFULLSCREENARTYPE_DEFAULT;
	gcfg->video_crop_type = DISPQT_CONFIG_VIDEOSRCCROPTYPE_DEFAULT;
	gcfg->video_zoom_type = DISPQT_CONFIG_VIDEOZOOMTYPE_DEFAULT;
	gcfg->video_zoom_value = DISPQT_CONFIG_VIDEOZOOM_CENTER;
	gcfg->video_deinterlace_type = DISPQT_VIDEO_DEINTERLACETYPE_AUTO;
	gcfg->video_rotation_type = DISPQT_VIDEO_ROTATIONTYPE_AUTO;

	for(int i = 0; i < DISPQT_VIDEO_VMIXERTYPE_MAX; i++)
		gcfg->video_mixer_values[i] = DISPQT_VMIXER_VALUE_CENTER;

	mpxplay_config_video_audiovisualization_type = DISPQT_CONFIG_AUDIOVISUALIZATIONTYPE_DEFAULT;

	pds_strcpy(config_video_file_record_type_name, (char *)"Matroska (*.mkv)");
}

static void mpxplay_dispqt_configini_reset_color_settings(void)
{
	const char **default_colortext;
	char **color_text_name;

	color_text_name = &(mmc_config.color_window_text_name[0]);
	default_colortext = &config_colortext_window_default[0];
	for(int i = 0; i < DISPQT_COLORTEXTID_WINDOW_MAX; i++, color_text_name++, default_colortext++){
		int len = pds_strlen((char *)*default_colortext);
		*color_text_name = new char[len + 4];
		memcpy(*color_text_name, *default_colortext, (len + 1));
	}

	color_text_name = &(mmc_config.color_text_name[0]);
	default_colortext = &config_colortext_editortable_default[0];
	for(int i = 0; i < DISPQT_COLORTEXTID_EDITOR_MAX; i++, color_text_name++, default_colortext++){
		int len = pds_strlen((char *)*default_colortext);
		*color_text_name = new char[len + 4];
		memcpy(*color_text_name, *default_colortext, (len + 1));
	}
}

// later can be called from config.cpp to reset configurable values
void mpxplay_dispqt_configini_reset_configurables(void)
{
	const unsigned int windows_version = pds_mswin_getver();
	struct mmc_dispqt_config_s *gcfg = &mmc_config;
	bool isTranslucentEnabled = false;

	if(windows_version >= MPXPLAY_MSWIN_VERSIONID_VISTA)
	{
		bool isHighContrastThemeUsed = false;
		isTranslucentEnabled = mpxplay_dispqt_mainwin_check_translucent_available(&isHighContrastThemeUsed) && !isHighContrastThemeUsed;
	}

	mpxplay_dispqt_configini_reset_video_surface_settings();
	mpxplay_dispqt_configini_reset_color_settings();

	gcfg->gui_layout_type = DISPQT_GUILAYOUT_BLUE_TRANSPARENT;
	gcfg->gui_background_type = DISPQT_GUIBKGTYPE_BLUR;
	gcfg->gui_background_effect = DISPQT_GUIBKGEFCT_COLORIZED;
	gcfg->gui_background_color_alpha = DISPQT_CONFIG_MAINWIN_COLOR_ALPHA_DEFAULT;
	if(isTranslucentEnabled) {
//		if(windows_version >= MPXPLAY_MSWIN_VERSIONID_WIN11)
//			gcfg->gui_background_effect = DISPQT_GUIBKGEFCT_GLASS_PNG; // TODO: fine tune it for useful default
//		else
		if(windows_version < MPXPLAY_MSWIN_VERSIONID_WIN80) // WinVista / Win7 with Aero enabled
			gcfg->gui_background_effect = DISPQT_GUIBKGEFCT_NONE;
	}

	gcfg->mainwin_control = DISPQT_CONFIG_MAINWINCTRLS_DEFAULT;
	gcfg->mainwin_opacity_general = DISPQT_CONFIG_MAINWIN_OPACITY_GENERAL_DEFAULT;
	gcfg->mainwin_opacity_transparent = DISPQT_CONFIG_MAINWIN_OPACITY_TRANSPARENT_DEFAULT;
	gcfg->mainwin_opacity_inactive = DISPQT_CONFIG_MAINWIN_OPACITY_INACTIVE_DEFAULT;
	gcfg->mainwin_opacity_onvideo_show = (windows_version < MPXPLAY_MSWIN_VERSIONID_WIN80)? DISPQT_CONFIG_MAINWIN_OPACITY_ONVIDEO_SHOW_DEF_WIN7 : DISPQT_CONFIG_MAINWIN_OPACITY_ONVIDEO_SHOW_DEF_WIN10;
	gcfg->mainwin_opacity_onvideo_hide = DISPQT_CONFIG_MAINWIN_OPACITY_ONVIDEO_HIDE_DEF;
	gcfg->mainwin_opacity_showkeep_time = DISPQT_CONFIG_MAINWIN_OPACITY_SHOWKEEP_TIMEMS_DEFAULT;
	gcfg->mainwin_opacity_fadein_time = DISPQT_CONFIG_MAINWIN_OPACITY_FADEIN_TIMEMS_DEFAULT;
	gcfg->mainwin_opacity_fadeout_time = DISPQT_CONFIG_MAINWIN_OPACITY_FADEOUT_TIMEMS_DEFAULT;

	gcfg->editor_flags = DISPQT_CONFIG_EDITORFLAGS_DEFAULT;
	if((windows_version >= MPXPLAY_MSWIN_VERSIONID_WIN11) && isTranslucentEnabled) {
		funcbit_enable(gcfg->editor_flags, DISPQT_CONFIG_EDITORFLAG_TRANSPARENT_INWINDOW); // with GUIBKGEFCT_GLASS
	}
	gcfg->transparent_control = DISPQT_CONFIG_TRANSCTRLS_DEFAULT;
	gcfg->video_control = DISPQT_CONFIG_VIDEOCTRLS_DEFAULT;
	if((windows_version < MPXPLAY_MSWIN_VERSIONID_WIN11) && (!isTranslucentEnabled || (windows_version >= MPXPLAY_MSWIN_VERSIONID_WIN80))) {
		gcfg->videocontrolbar_size_percent = DISPQT_CONFIG_VIDEO_CONTROLBAR_SIZE_DEF_FULL;
		funcbit_enable(gcfg->video_control, DISPQT_CONFIG_VIDEOCTRL_CONTROLBAR_HEADERLESS);
	} else {
		gcfg->videocontrolbar_size_percent = DISPQT_CONFIG_VIDEO_CONTROLBAR_SIZE_DEF_HALF; // normal control bar for Win7 Aero and Win11
	}
	gcfg->videoseekpreview_size_percent = DISPQT_CONFIG_VIDEO_SEEKPREVIEW_SIZE_DEFAULT;
	gcfg->videoseekpreview_refresh_interval_ms = DISPQT_CONFIG_VIDEO_SEEKPREVIEW_INTERVAL_DEFAULT;
	gcfg->videoseekpreview_aspect_ratio_type = DISPQT_CONFIG_VIDEO_SEEKPREVIEW_AR_DEFAULT;

	gcfg->video_renderer_type = (windows_version >= MPXPLAY_MSWIN_VERSIONID_VISTA)? DISPQT_CONFIG_VIDEORENDERTYPE_DEFAULT : DISPQT_CONFIG_VIDEORENDERTYPE_DEF_XP;
	gcfg->video_windowed_ar_type = DISPQT_CONFIG_VIDEOWINDOWEDARTYPE_DEFAULT;
	gcfg->video_hdr_control = DISPQT_CONFIG_VIDEO_HDR_CONTROL_DEFAULT;
	gcfg->video_autocrop_limit = 1440;
	gcfg->video_screen_select_name[0] = 0;

	gcfg->epgdialogwin_pos_x = gcfg->epgdialogwin_pos_y = gcfg->epgdialogwin_size_x = gcfg->epgdialogwin_size_y = -1;

	pds_strcpy(gcfg->video_ar_custom, (char *)DISPQT_CONFIG_VIDEOARCUSTOM_DEFAULTSTR);
	pds_strcpy(gcfg->video_crop_custom, (char *)DISPQT_CONFIG_VIDEOCROPCUSTOM_DEFAULTSTR);
	pds_strcpy(gcfg->video_zoom_custom, (char *)DISPQT_CONFIG_VIDEOZOOMCUSTOM_DEFAULTSTR);
	mpxplay_config_video_extstream_loadtype[MPXPLAY_STREAMTYPEINDEX_AUDIO] = MPXPLAY_EXTERNALSTREAM_LOADTYPE_SEARCH;
	mpxplay_config_video_extstream_loadtype[MPXPLAY_STREAMTYPEINDEX_SUBTITLE] = MPXPLAY_EXTERNALSTREAM_LOADTYPE_USE;
	pds_strcpy(&mpxplay_config_video_extstream_subdirs[MPXPLAY_STREAMTYPEINDEX_AUDIO][0], (char *)config_video_extstream_audio_subdirs_default);
	pds_strcpy(&mpxplay_config_video_extstream_subdirs[MPXPLAY_STREAMTYPEINDEX_SUBTITLE][0], (char *)config_video_extstream_subtitle_subdirs_default);
	if(gcfg->font_text_name[DISPQT_FONTTEXTID_EDITOR])
	{
		pds_free(gcfg->font_text_name[DISPQT_FONTTEXTID_EDITOR]);
		gcfg->font_text_name[DISPQT_FONTTEXTID_EDITOR] = NULL;
	}
	if(gcfg->font_text_name[DISPQT_FONTTEXTID_SUBTITLE])
	{
		pds_free(gcfg->font_text_name[DISPQT_FONTTEXTID_SUBTITLE]);
		gcfg->font_text_name[DISPQT_FONTTEXTID_SUBTITLE] = NULL;
	}
}

void mpxplay_dispqt_configini_set_default_fonts(void)
{
	struct mmc_dispqt_config_s *gcfg = &mmc_config;
#ifdef MPXPLAY_WIN32
	if(!gcfg->font_text_name[DISPQT_FONTTEXTID_EDITOR]){
		char *fonttext_editor = (char *)"MS Sans Serif,8,-1,5,75,0,0,0,0,0";
		unsigned int len = pds_strlen(fonttext_editor) + 1;
		gcfg->font_text_name[DISPQT_FONTTEXTID_EDITOR] = (char *)pds_malloc(len);
		if(gcfg->font_text_name[DISPQT_FONTTEXTID_EDITOR])
			pds_memcpy(gcfg->font_text_name[DISPQT_FONTTEXTID_EDITOR], fonttext_editor, len);
	}
#endif
	if(!gcfg->font_text_name[DISPQT_FONTTEXTID_SUBTITLE]){
		char *fonttext_subtitle = (char *)"Arial,36,-1,5,75,0,0,0,0,0";
		unsigned int len = pds_strlen(fonttext_subtitle) + 1;
		gcfg->font_text_name[DISPQT_FONTTEXTID_SUBTITLE] = (char *)pds_malloc(len);
		if(gcfg->font_text_name[DISPQT_FONTTEXTID_SUBTITLE])
			pds_memcpy(gcfg->font_text_name[DISPQT_FONTTEXTID_SUBTITLE], fonttext_subtitle, len);
	}
}

static void mpxplay_dispqt_config_check(void)
{
	const unsigned int windows_version = pds_mswin_getver();
	struct mmc_dispqt_config_s *gcfg = &mmc_config;

	mpxplay_dispqt_configini_set_default_fonts();

	if(gcfg->gui_layout_type >= DISPQT_GUILAYOUT_MAX)
		gcfg->gui_layout_type = DISPQT_GUILAYOUT_TRANSPARENT_TYPE;
	else if((gcfg->gui_layout_type < DISPQT_GUILAYOUT_TRANSPARENT_TYPE) && (gcfg->gui_layout_type >= DISPQT_GUILAYOUT_CLASSIC_MAX))
		gcfg->gui_layout_type = DISPQT_GUILAYOUT_CLASSIC_MAX - 1;
	if(gcfg->gui_background_type >= DISPQT_GUIBKGTYPE_MAX)
		gcfg->gui_background_type = DISPQT_GUIBKGTYPE_BLUR;
	if(gcfg->gui_background_effect >= DISPQT_GUIBKGEFCT_MAX)
		gcfg->gui_background_effect = ((windows_version >= MPXPLAY_MSWIN_VERSIONID_VISTA) && (windows_version < MPXPLAY_MSWIN_VERSIONID_WIN80))? DISPQT_GUIBKGEFCT_NONE : DISPQT_GUIBKGEFCT_COLORIZED;
	if(gcfg->gui_background_color_alpha > DISPQT_CONFIG_MAINWIN_COLOR_ALPHA_MAX)
		gcfg->gui_background_color_alpha = DISPQT_CONFIG_MAINWIN_COLOR_ALPHA_MAX;

	if(gcfg->mainwin_opacity_general < DISPQT_CONFIG_MAINWIN_OPACITY_GENERAL_MIN)
		gcfg->mainwin_opacity_general = DISPQT_CONFIG_MAINWIN_OPACITY_GENERAL_MIN;
	else if(gcfg->mainwin_opacity_general > DISPQT_CONFIG_MAINWIN_OPACITY_GENERAL_MAX)
		gcfg->mainwin_opacity_general = DISPQT_CONFIG_MAINWIN_OPACITY_GENERAL_MAX;
	if(gcfg->mainwin_opacity_inactive < DISPQT_CONFIG_MAINWIN_OPACITY_GENERAL_MIN)
		gcfg->mainwin_opacity_inactive = DISPQT_CONFIG_MAINWIN_OPACITY_GENERAL_MIN;
	else if(gcfg->mainwin_opacity_inactive > DISPQT_CONFIG_MAINWIN_OPACITY_GENERAL_MAX)
		gcfg->mainwin_opacity_inactive = DISPQT_CONFIG_MAINWIN_OPACITY_GENERAL_MAX;
	if(gcfg->mainwin_opacity_transparent < DISPQT_CONFIG_MAINWIN_OPACITY_TRANSPARENT_MIN)
		gcfg->mainwin_opacity_transparent = DISPQT_CONFIG_MAINWIN_OPACITY_TRANSPARENT_MIN;
	else if(gcfg->mainwin_opacity_transparent > DISPQT_CONFIG_MAINWIN_OPACITY_TRANSPARENT_MAX)
		gcfg->mainwin_opacity_transparent = DISPQT_CONFIG_MAINWIN_OPACITY_TRANSPARENT_MAX;
	if(gcfg->mainwin_opacity_onvideo_show <= gcfg->mainwin_opacity_onvideo_hide)
		gcfg->mainwin_opacity_onvideo_show = gcfg->mainwin_opacity_onvideo_hide + 1;
	if(gcfg->mainwin_opacity_onvideo_show < DISPQT_CONFIG_MAINWIN_OPACITY_ONVIDEO_SHOW_MIN)
		gcfg->mainwin_opacity_onvideo_show = DISPQT_CONFIG_MAINWIN_OPACITY_ONVIDEO_SHOW_MIN;
	else if(gcfg->mainwin_opacity_onvideo_show > DISPQT_CONFIG_MAINWIN_OPACITY_ONVIDEO_SHOW_MAX)
		gcfg->mainwin_opacity_onvideo_show = DISPQT_CONFIG_MAINWIN_OPACITY_ONVIDEO_SHOW_MAX;
	if(gcfg->mainwin_opacity_onvideo_hide < DISPQT_CONFIG_MAINWIN_OPACITY_ONVIDEO_HIDE_MIN)
		gcfg->mainwin_opacity_onvideo_hide = DISPQT_CONFIG_MAINWIN_OPACITY_ONVIDEO_HIDE_MIN;
	else if(gcfg->mainwin_opacity_onvideo_hide > DISPQT_CONFIG_MAINWIN_OPACITY_ONVIDEO_HIDE_MAX)
		gcfg->mainwin_opacity_onvideo_hide = DISPQT_CONFIG_MAINWIN_OPACITY_ONVIDEO_HIDE_MAX;
	if(gcfg->mainwin_opacity_showkeep_time < DISPQT_CONFIG_MAINWIN_OPACITY_SHOWKEEP_TIMEMS_MIN)
		gcfg->mainwin_opacity_showkeep_time = DISPQT_CONFIG_MAINWIN_OPACITY_SHOWKEEP_TIMEMS_MIN;
	else if(gcfg->mainwin_opacity_showkeep_time > DISPQT_CONFIG_MAINWIN_OPACITY_SHOWKEEP_TIMEMS_MAX)
		gcfg->mainwin_opacity_showkeep_time = DISPQT_CONFIG_MAINWIN_OPACITY_SHOWKEEP_TIMEMS_MAX;
	if(gcfg->mainwin_opacity_fadein_time < DISPQT_CONFIG_MAINWIN_OPACITY_FADEIN_TIMEMS_MIN)
		gcfg->mainwin_opacity_fadein_time = DISPQT_CONFIG_MAINWIN_OPACITY_FADEIN_TIMEMS_MIN;
	else if(gcfg->mainwin_opacity_fadein_time > DISPQT_CONFIG_MAINWIN_OPACITY_FADEIN_TIMEMS_MAX)
		gcfg->mainwin_opacity_fadein_time = DISPQT_CONFIG_MAINWIN_OPACITY_FADEIN_TIMEMS_MAX;
	if(gcfg->mainwin_opacity_fadein_time < DISPQT_CONFIG_MAINWIN_OPACITY_FADEOUT_TIMEMS_MIN)
		gcfg->mainwin_opacity_fadein_time = DISPQT_CONFIG_MAINWIN_OPACITY_FADEOUT_TIMEMS_MIN;
	else if(gcfg->mainwin_opacity_fadein_time > DISPQT_CONFIG_MAINWIN_OPACITY_FADEOUT_TIMEMS_MAX)
		gcfg->mainwin_opacity_fadein_time = DISPQT_CONFIG_MAINWIN_OPACITY_FADEOUT_TIMEMS_MAX;
	if(!(gcfg->transparent_control & DISPQT_CONFIG_TRANSCTRL_DESKTOPTRANS_RESTORE))
		funcbit_disable(gcfg->transparent_control, DISPQT_CONFIG_TRANSCTRL_DESKTOPTRANS_ENABLED);

	if(gcfg->videocontrolbar_size_percent < DISPQT_CONFIG_VIDEO_CONTROLBAR_SIZE_MIN)
		gcfg->videocontrolbar_size_percent = DISPQT_CONFIG_VIDEO_CONTROLBAR_SIZE_MIN;
	else if(gcfg->videocontrolbar_size_percent > DISPQT_CONFIG_VIDEO_CONTROLBAR_SIZE_MAX)
		gcfg->videocontrolbar_size_percent = DISPQT_CONFIG_VIDEO_CONTROLBAR_SIZE_MAX;

	if(gcfg->videoseekpreview_size_percent < DISPQT_CONFIG_VIDEO_SEEKPREVIEW_SIZE_MIN)
		gcfg->videoseekpreview_size_percent = DISPQT_CONFIG_VIDEO_SEEKPREVIEW_SIZE_MIN;
	else if(gcfg->videoseekpreview_size_percent > DISPQT_CONFIG_VIDEO_SEEKPREVIEW_SIZE_MAX)
		gcfg->videoseekpreview_size_percent = DISPQT_CONFIG_VIDEO_SEEKPREVIEW_SIZE_MAX;
	if(gcfg->videoseekpreview_refresh_interval_ms > DISPQT_CONFIG_VIDEO_SEEKPREVIEW_INTERVAL_MAX)
		gcfg->videoseekpreview_refresh_interval_ms = DISPQT_CONFIG_VIDEO_SEEKPREVIEW_INTERVAL_MAX;
	if(gcfg->videoseekpreview_aspect_ratio_type >= DISPQT_VIDEOFULLSCREENARTYPE_MAX)
		gcfg->videoseekpreview_aspect_ratio_type = DISPQT_CONFIG_VIDEO_SEEKPREVIEW_AR_DEFAULT;

	if(mpxplay_config_dvbepg_pastevents_hours < DISPQT_CONFIG_DVBEPG_PASTEVENTS_HOURS_MIN)
		mpxplay_config_dvbepg_pastevents_hours = DISPQT_CONFIG_DVBEPG_PASTEVENTS_HOURS_MIN;
	else if(mpxplay_config_dvbepg_pastevents_hours > DISPQT_CONFIG_DVBEPG_PASTEVENTS_HOURS_MAX)
		mpxplay_config_dvbepg_pastevents_hours = DISPQT_CONFIG_DVBEPG_PASTEVENTS_HOURS_MAX;

	if(gcfg->video_renderer_type >= DISPQT_VIDEORENDERTYPE_MAX)
		gcfg->video_renderer_type = (windows_version >= MPXPLAY_MSWIN_VERSIONID_VISTA)? DISPQT_CONFIG_VIDEORENDERTYPE_DEFAULT : DISPQT_CONFIG_VIDEORENDERTYPE_DEF_XP;

	if(gcfg->video_fullscreen_ar_type >= DISPQT_VIDEOFULLSCREENARTYPE_MAX)
		gcfg->video_fullscreen_ar_type = DISPQT_CONFIG_VIDEOFULLSCREENARTYPE_DEFAULT;
	if(gcfg->video_windowed_ar_type >= DISPQT_VIDEOWINDOWEDARTYPE_MAX)
		gcfg->video_windowed_ar_type = DISPQT_CONFIG_VIDEOWINDOWEDARTYPE_DEFAULT;
	if(gcfg->video_crop_type >= DISPQT_VIDEOSCRCROPTYPE_MAX)
		gcfg->video_crop_type = DISPQT_CONFIG_VIDEOSRCCROPTYPE_DEFAULT;
	if(gcfg->video_zoom_type >= DISPQT_VIDEOFULLSCREENZOOMTYPE_MAX)
		gcfg->video_zoom_type = DISPQT_CONFIG_VIDEOZOOMTYPE_DEFAULT;
	if(gcfg->video_zoom_value < DISPQT_CONFIG_VIDEOZOOM_MIN)
		gcfg->video_zoom_value = DISPQT_CONFIG_VIDEOZOOM_MIN;
	else if(gcfg->video_zoom_value > DISPQT_CONFIG_VIDEOZOOM_MAX)
		gcfg->video_zoom_value = DISPQT_CONFIG_VIDEOZOOM_MAX;
	if(gcfg->video_deinterlace_type >= DISPQT_VIDEO_DEINTERLACETYPE_MAX)
		gcfg->video_deinterlace_type = DISPQT_VIDEO_DEINTERLACETYPE_AUTO;
	if(gcfg->video_rotation_type >= DISPQT_VIDEO_ROTATIONTYPE_MAX)
		gcfg->video_rotation_type = DISPQT_VIDEO_ROTATIONTYPE_AUTO;

	for(int i = 0; i < DISPQT_VIDEO_VMIXERTYPE_MAX; i++)
	{
		if(gcfg->video_mixer_values[i] < DISPQT_VMIXER_VALUE_MIN)
			gcfg->video_mixer_values[i] = DISPQT_VMIXER_VALUE_MIN;
		else if(gcfg->video_mixer_values[i] > DISPQT_VMIXER_VALUE_MAX)
			gcfg->video_mixer_values[i] = DISPQT_VMIXER_VALUE_MAX;
	}

	if(mpxplay_config_video_audiovisualization_type >= DISPQT_AUDIOVISUALIZATIONTYPE_MAX)
		mpxplay_config_video_audiovisualization_type = DISPQT_CONFIG_AUDIOVISUALIZATIONTYPE_DEFAULT;

	gcfg->selected_videoplayer_type = mpxplay_config_videoplayer_type;
	if(gcfg->selected_videoplayer_type == MPXPLAY_VIDEOPLAYERTYPE_NONE)
		funcbit_disable(gcfg->video_control, DISPQT_CONFIG_VIDEOCTRL_ENABLE_VIDEO);
	else
		funcbit_enable(gcfg->video_control, DISPQT_CONFIG_VIDEOCTRL_ENABLE_VIDEO);

	for(int i = 0; i < MPXPLAY_STREAMTYPEINDEX_EXTSTREAMNUM; i++)
		if(mpxplay_config_video_extstream_loadtype[i] >= MPXPLAY_EXTERNALSTREAM_LOADTYPE_MAX)
			mpxplay_config_video_extstream_loadtype[i] = MPXPLAY_EXTERNALSTREAM_LOADTYPE_SEARCH;

	mpxplay_control_extrafiletypes_slice();
}

struct mmc_dispqt_config_s *mpxplay_dispqt_configini_get(void)
{
	return &mmc_config;
}

void mpxplay_dispqt_configini_load_from_file(void)
{
	FILE *filehand = NULL;
	char *memfield = NULL;
	int linecount;
	mpxini_part_t pt;
	mpxini_line_t mpxgui_config_lines[MPXPGUI_INI_CONFIG_LINES + 8];

	filehand = mpxplay_control_configfile_open((char *)"MPXP_GUI.INI", "r", NULL, NULL);
	if(!filehand)
		goto err_out_load;
	memset(mpxgui_config_lines, 0, sizeof(mpxgui_config_lines));
	linecount = mpxplay_control_configfile_load(filehand, &memfield, mpxgui_config_lines, (MPXPGUI_INI_CONFIG_LINES + 6), 0);
	if(linecount <= 0)
		goto err_out_load;
	pt.filepos = 0;
	pt.partbegin_linenum = 0;
	pt.partlinenum = linecount;
	mpxplay_control_general_loadini(mpxgui_config_lines, &pt, (mpxini_var_s *)mpxpgui_ini);

err_out_load:
	mpxplay_dispqt_config_check();
	if(filehand)
		fclose(filehand);
	if(memfield)
		free(memfield);
}

static void dispqt_configini_save_correct_values(void)
{
	struct mmc_dispqt_config_s *gcfg = &mmc_config;
	mpxplay_config_videoplayer_type = gcfg->selected_videoplayer_type;
	if(gcfg->selected_videoplayer_type == MPXPLAY_VIDEOPLAYERTYPE_NONE)
		funcbit_disable(gcfg->video_control, DISPQT_CONFIG_VIDEOCTRL_ENABLE_VIDEO);
	else
		funcbit_enable(gcfg->video_control, DISPQT_CONFIG_VIDEOCTRL_ENABLE_VIDEO);
}

void mpxplay_dispqt_configini_save_to_file(void)
{
	FILE *filehand;
	mpxini_part_t pt;
	mpxini_line_t mpxgui_config_lines[MPXPGUI_INI_CONFIG_LINES + 4];

	dispqt_configini_save_correct_values();

	filehand = mpxplay_control_configfile_open((char *)"MPXP_GUI.INI", "w", NULL, NULL);
	if(!filehand) {
		mpxplay_debugf(DISPQT_DEBUG_OUTPUT, "configsave file open failed!");
		return;
	}
	memset(mpxgui_config_lines, 0, sizeof(mpxgui_config_lines));
	const mpxini_var_s *varp = &mpxpgui_ini[0];
	mpxini_line_t *cfg = &mpxgui_config_lines[0];
	for(int i = 0; i < MPXPGUI_INI_CONFIG_LINES; i++, varp++, cfg++){
		cfg->storepoint = (void *)varp;
		cfg->varnamep = (char *)varp->name;
		cfg->valuep = (char *)varp->c;
	}
	pt.filepos = 0;
	pt.partbegin_linenum = 0;
	pt.partlinenum = MPXPGUI_INI_CONFIG_LINES;
	mpxplay_control_general_saveini(mpxgui_config_lines, &pt, filehand);
	fclose(filehand);
}
