//**************************************************************************
//*                     This file is part of the                           *
//*                 Mpxplay/MMC - multimedia player.                       *
//*                  The source code of Mpxplay is                         *
//*        (C) copyright 1998-2020 by PDSoft (Attila Padar)                *
//*                http://mpxplay.sourceforge.net                          *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//function: FFMPEG stream selector functions

#include "ffmpgdec.h"

#ifdef MPXPLAY_LINK_INFILE_FF_MPEG

#include "mpxplay.h"
#include <control/control.h>
#ifdef MPXPLAY_GUI_QT
#include <disp_qt/disp_qt.h>
#endif

extern unsigned int playcontrol;
extern int mpxplay_control_startup_programid_select;
extern unsigned long mpxplay_config_videoplayer_control, mpxplay_config_dvbepg_control_flags;
extern char mpxplay_config_video_preferred_language[MPXPLAY_STREAMTYPEINDEX_PLAYNUM][MPXINI_MAX_CHARDATA_LEN];
extern char *mpxplay_config_video_preffered_audiotype_ptrs[MPXINI_MAX_EXTRA_FILETYPES];
extern unsigned int mpxplay_config_video_extstream_loadtype[MPXPLAY_STREAMTYPEINDEX_EXTSTREAMNUM];

static void in_ffmpstrm_all_programs_videostreams_update(struct ffmpg_demuxer_data_s *ffmpi);

enum AVMediaType in_ffmpstrm_streamtypeindex_to_avmediatype(unsigned int streamtype_index)
{
	enum AVMediaType ff_mediatype;
	switch(streamtype_index){
		case MPXPLAY_STREAMTYPEINDEX_AUDIO: ff_mediatype = AVMEDIA_TYPE_AUDIO; break;
		case MPXPLAY_STREAMTYPEINDEX_SUBTITLE: ff_mediatype = AVMEDIA_TYPE_SUBTITLE; break;
		default: ff_mediatype = AVMEDIA_TYPE_VIDEO; break;
	}
	return ff_mediatype;
}

mpxp_bool_t in_ffmpgdec_content_has_programs(struct ffmpg_demuxer_data_s *ffmpi)
{
	if(ffmpi->fctx && ffmpi->fctx->programs && (ffmpi->fctx->nb_programs > 1))
		return TRUE;
	return FALSE;
}

// check that AVStream / AVCodecParameters are exists and valid (stream is parsed properly) */
mpxp_bool_t mpxplay_ffmpstrm_is_avstream_parsed(AVStream *st, unsigned int stream_type_idx)
{
	AVCodecParameters *codecpar;

	if(!st || !st->av_class || (st->index < 0) || (st->index >= INFFMPG_MAX_STREAMS))
		return FALSE;

	codecpar = st->codecpar;
	if(!codecpar)
		return FALSE;

	switch(stream_type_idx)
	{
		case MPXPLAY_STREAMTYPEINDEX_AUDIO:
			if(codecpar->codec_type != AVMEDIA_TYPE_AUDIO)
				return FALSE;
			if((codecpar->codec_id <= AV_CODEC_ID_FIRST_AUDIO) || (codecpar->codec_id >= AV_CODEC_ID_FIRST_SUBTITLE))
				return FALSE;
			if(codecpar->width || codecpar->height)
				return FALSE;
			if(codecpar->sample_rate < 16)
				return FALSE;
#if MPXPLAY_USE_FFMPEG_V7X
			if(codecpar->ch_layout.nb_channels <= 0)
#else
			if((codecpar->ch_layout.nb_channels <= 0) && (codecpar->channels <= 0))
#endif
				return FALSE;
			break;
		case MPXPLAY_STREAMTYPEINDEX_VIDEO:
			if(codecpar->codec_type != AVMEDIA_TYPE_VIDEO)
				return FALSE;
			if((codecpar->codec_id <= AV_CODEC_ID_NONE) || (codecpar->codec_id >= AV_CODEC_ID_FIRST_AUDIO))
				return FALSE;
			if(codecpar->sample_rate != 0)
				return FALSE;
#if MPXPLAY_USE_FFMPEG_V7X
			if(codecpar->ch_layout.nb_channels > 0)
#else
			if((codecpar->ch_layout.nb_channels > 0) || (codecpar->channels > 0))
#endif
				return FALSE;
			break;
	}

	return TRUE;
}

// clear base stream infos for re-loading/parsing (mpegts live streams can be parsed incorrectly)
static void in_ffmpstrm_invalidate_ffstream(AVStream *st)
{
	st->id = 0;
	st->codecpar->codec_id = AV_CODEC_ID_NONE;
}

// check that AVCodecContext is exists, valid and open */
mpxp_bool_t mpxplay_ffmpstrm_is_avctx_valid(void *codec_ctx, unsigned int stream_type_idx)
{
	AVCodecContext *avctx = (AVCodecContext *)codec_ctx;

	if(!avctx || !avctx->av_class || !avctx->codec)
		return FALSE;

	switch(stream_type_idx)
	{
		case MPXPLAY_STREAMTYPEINDEX_AUDIO:
			if((avctx->codec_type != AVMEDIA_TYPE_AUDIO) || (avctx->codec_id <= AV_CODEC_ID_FIRST_AUDIO) || (avctx->codec_id >= AV_CODEC_ID_FIRST_SUBTITLE))
				return FALSE;
			if(avctx->width || avctx->height || avctx->coded_width || avctx->coded_height || avctx->hw_device_ctx || (avctx->sample_rate < 16))
				return FALSE;
			break;
		case MPXPLAY_STREAMTYPEINDEX_VIDEO:
			if((avctx->codec_type != AVMEDIA_TYPE_VIDEO) || (avctx->codec_id <= AV_CODEC_ID_NONE) || (avctx->codec_id >= AV_CODEC_ID_FIRST_AUDIO))
				return FALSE;
			if(avctx->pix_fmt >= AV_PIX_FMT_NB)
				return FALSE;
			if((avctx->sample_fmt != AV_SAMPLE_FMT_NONE) || (avctx->sample_rate != 0))
				return FALSE;
#if MPXPLAY_USE_FFMPEG_V7X
			if(avctx->ch_layout.nb_channels > 0)
#else
			if((avctx->ch_layout.nb_channels > 0) || (avctx->channels > 0))
#endif
				return FALSE;
			break;
	}

	if(!avcodec_is_open(avctx))
		return FALSE;

	return TRUE;
}

// search first video stream for the specified program (requires avformat_find_stream_info() before call this)
AVStream *in_ffmpstrm_find_videostream_and_decoder(AVFormatContext *fctx, unsigned int stream_type, int program_number_select, AVCodec **codec_av_new)
{
	const enum AVMediaType required_avtype = in_ffmpstrm_streamtypeindex_to_avmediatype(stream_type);
	AVStream *stream = NULL;
	int i;

	for(i = 0; i < fctx->nb_streams; i++) // search for the first stream
	{
		stream = fctx->streams[i];
		if(stream && stream->codecpar && (stream->codecpar->codec_type == required_avtype))
		{   // search for the program (in a multiplexed content, like recorded DVB mux)
			if(program_number_select >= 0)
			{
				AVProgram *program = av_find_program_from_stream(fctx, NULL, stream->index);
				if(!program || (program->id != program_number_select))
					continue; // to keep a video stream, if program has not found
			}
			break;
		}
		stream = NULL;
	}

	if(!stream)
		return stream;

	*codec_av_new = (AVCodec *)avcodec_find_decoder(stream->codecpar->codec_id);
	if(!*codec_av_new)
		stream = NULL;

	return stream;
}

// search for a new program, if the current has lost
mpxp_bool_t in_ffmpstrm_skip_program_auto(struct ffmpg_demuxer_data_s *ffmpi, int prognum_current_id)
{
	mpxp_bool_t use_next = FALSE;
	int i;

	if(!in_ffmpgdec_content_has_programs(ffmpi))
	{
		return FALSE;
	}

	for(i = 0; i < ffmpi->fctx->nb_programs; i++)
	{
		int prog_id = ffmpi->fctx->programs[i]->id;
		//mpxplay_debugf(MPXPLAY_DEBUG_PRGCHG, "in_ffmpstrm_skip_program_auto i:%d c:%d id:%d", i, prognum_current_id, prog_id);
		if(prog_id == prognum_current_id)
		{
			if(funcbit_test(playcontrol, PLAYC_NOAUTOPRGSKIP))
			{
				return FALSE;
			}
			use_next = TRUE;
		}
		else if(use_next)
		{
			ffmpi->select_new_program_number = prog_id;
			break;
		}
	}

	if(ffmpi->select_new_program_number < 0)
	{
		ffmpi->select_new_program_number = ffmpi->fctx->programs[0]->id;
	}
	mpxplay_debugf(MPXPLAY_DEBUG_PRGCHG, "in_ffmpstrm_skip_program_auto END cid:%d nid:%d", prognum_current_id, ffmpi->select_new_program_number);
	return TRUE;
}

// manual select stream for the specified stream-type (audio, video, subtitle) by select_stream_number and program_number_current, count number of streams for this type
int in_ffmpstrm_streams_count(struct ffmpg_demuxer_data_s *ffmpi, int streamtype_index, int *select_stream_number, int *selected_ff_streamindex, AVCodec **codec_av_new)
{
	int max_stindex = MPXPLAY_STREAMTYPEINDEX_AUDIO, ff_stindex, new_stream_number = INFFMPG_STREAM_INDEX_DISABLE, new_ff_streamindex = INFFMPG_STREAM_INDEX_DISABLE;
	int program_number_save = ffmpi->program_number_current, found = 0, retry = ffmpi->fctx->nb_programs + 2, retval = 0;
	enum AVMediaType ff_mediatype;

	ffmpi->nb_streams[streamtype_index] = 0;

#ifdef MPXPLAY_GUI_QT
	if(mpxplay_dispqt_ffmpegvideo_configcbk_check(MPXPLAY_INFILE_CBKCFG_SRVFFMV_OPEN_CALLBACK) == 0)
		max_stindex = MPXPLAY_STREAMTYPEINDEX_SUBTITLE;
#endif
	if(streamtype_index > max_stindex)
		return -1;

	ff_mediatype = in_ffmpstrm_streamtypeindex_to_avmediatype(streamtype_index);

	do{
		for(ff_stindex = 0; (ff_stindex < ffmpi->fctx->nb_streams) && (ff_stindex < INFFMPG_MAX_STREAMS); ff_stindex++)
		{
			AVStream *st = ffmpi->fctx->streams[ff_stindex];
			enum AVMediaType codec_type;
			//mpxplay_debugf(MPXPLAY_DEBUG_STRMCHG, "in_ffmpstrm_streams_count i:%d st:%8.8X cp:%8.8X", streamtype_index, (mpxp_uint32_t)st, (st)? (mpxp_uint32_t)st->codecpar : 0);
			if(!st || !st->codecpar)
			{
				retval = INFFMPG_ERROR_STREAM_NOTPARSED;
				mpxplay_debugf(MPXPLAY_DEBUG_STRMCHG, "in_ffmpstrm_streams_count STFAIL st:%8.8X cp:%8.8X", (mpxp_ptrsize_t)st, (st)? (mpxp_ptrsize_t)st->codecpar : 0);
				continue;
			}
			codec_type = st->codecpar->codec_type;
			if(codec_type != ff_mediatype)
			{
				//mpxplay_debugf(MPXPLAY_DEBUG_STRMCHG, "in_ffmpstrm_streams_count CTFAIL ct:%d mt:%d sti:%d", (mpxp_uint32_t)codec_type, (mpxp_uint32_t)ff_mediatype, (mpxp_uint32_t)streamtype_index);
				continue;
			}
			// check that the audio stream parsing is correct
			if( !funcbit_test(ffmpi->flags, INFFMPG_FLAG_LOADHEADONLY) && (streamtype_index == MPXPLAY_STREAMTYPEINDEX_AUDIO)
			 && !mpxplay_ffmpstrm_is_avstream_parsed(st, MPXPLAY_STREAMTYPEINDEX_AUDIO)
			){
				mpxplay_debugf(MPXPLAY_DEBUG_WARNING, "in_ffmpstrm_streams_count AUDIO BAD ch:%d sr:%d id:%d", st->codecpar->ch_layout.nb_channels, st->codecpar->sample_rate, (int)st->codecpar->codec_id);
				in_ffmpstrm_invalidate_ffstream(st);
				retval = INFFMPG_ERROR_STREAM_NOTPARSED;
				continue;
			}
			// check that the stream belongs to a dvb program
			if(ffmpi->program_number_current >= 0)
			{
				AVProgram *program = av_find_program_from_stream(ffmpi->fctx, NULL, ff_stindex);
				if(!program || (program->id != ffmpi->program_number_current))
					continue;
				if(streamtype_index == MPXPLAY_STREAMTYPEINDEX_AUDIO)
				{
					mpxplay_debugf(MPXPLAY_DEBUG_STRMCHG, "in_ffmpstrm_streams_count PROGRAM cp:%d pr:%8.8X pid:%d ffi:%d sti:%d", ffmpi->program_number_current, program, ((program)? program->id : 0), ff_stindex, streamtype_index);
				}
			}
			if(select_stream_number)
			{
				if(!found || ((int)ffmpi->nb_streams[streamtype_index] <= *select_stream_number))
				{
					enum AVCodecID codec_id = (st->codecpar)? st->codecpar->codec_id : AV_CODEC_ID_NONE;
					AVCodec *codecavnew;
					if(codec_id == AV_CODEC_ID_NONE)
					{
						retval = INFFMPG_ERROR_STREAM_NOTPARSED;
						mpxplay_debugf(MPXPLAY_DEBUG_STRMCHG, "in_ffmpstrm_streams_count CIFAIL ci:%8.8X stc:%8.8X",
								(mpxp_uint32_t)codec_id, (mpxp_ptrsize_t)st->codecpar);
#if (LIBAVFORMAT_VERSION_MAJOR < 59) && (LIBAVFORMAT_VERSION_MINOR <= 45)
						st->request_probe = 5;
#endif
						continue;
					}
					codecavnew = (AVCodec *)avcodec_find_decoder(codec_id);
					if(codecavnew)
					{
						if(codec_av_new)
							*codec_av_new = codecavnew;
						new_ff_streamindex = ff_stindex;
						new_stream_number = ffmpi->nb_streams[streamtype_index];
						found = 1;
					}
				}
			}
			else
			{
				found = 1;
			}
			ffmpi->nb_streams[streamtype_index]++;
		}

#ifdef INFFMP_SUPPORT_DVB_AUDIO
		if(found || (ffmpi->program_number_current < 0) || (streamtype_index == MPXPLAY_STREAMTYPEINDEX_VIDEO) || (streamtype_index == MPXPLAY_STREAMTYPEINDEX_SUBTITLE))
#else
		if(found || (ffmpi->program_number_current < 0) || (streamtype_index == MPXPLAY_STREAMTYPEINDEX_SUBTITLE))
#endif
			break;
		if(!(--retry))
		{
			if(!funcbit_test(playcontrol, PLAYC_NOAUTOPRGSKIP))
				ffmpi->program_number_current = INFFMPG_INVALID_STREAM_INDEX; // program is invalid / lost
			break;
		}
		program_number_save = ffmpi->program_number_current;
		if(in_ffmpstrm_skip_program_auto(ffmpi, program_number_save))  // program is invalid / lost (and not the selected one, and auto skip is enabled)
		{
			ffmpi->program_number_current = ffmpi->select_new_program_number;
			funcbit_disable(ffmpi->flags, INFFMPG_FLAG_NOPRGIDCHGSEND); // update this auto program change in playlist editor too
		}
		ffmpi->select_new_program_number = INFFMPG_INVALID_STREAM_INDEX;
		if(ffmpi->program_number_current > 0)
			funcbit_enable(ffmpi->flags, INFFMPG_FLAG_DEMUX_BUFREWIND);
	}while(TRUE);

	if(select_stream_number && (new_stream_number >= 0))
		*select_stream_number = new_stream_number;
	if(selected_ff_streamindex && (new_ff_streamindex >= 0))
		*selected_ff_streamindex = new_ff_streamindex;

	if(ffmpi->fctx->nb_programs > 0) // verify program selection (on 1 program element too)
	{
		if((ffmpi->program_number_current < 0) && (new_ff_streamindex >= 0)) // valid stream is selected without program
		{
			AVProgram *program = av_find_program_from_stream(ffmpi->fctx, NULL, new_ff_streamindex);
			if(program)  // try to correct program number to selected stream
				ffmpi->program_number_current = program->id;
			mpxplay_debugf(MPXPLAY_DEBUG_PRGCHG, "in_ffmpstrm_streams_count new_ff_streamindex pold:%d pnew:%d nfs:%d", program_number_save, ffmpi->program_number_current, new_ff_streamindex);
		}

		if(!funcbit_test(ffmpi->flags, INFFMPG_FLAG_LOADHEADONLY) && in_ffmpgdec_content_has_programs(ffmpi) && (ffmpi->program_number_current >= 0) && codec_av_new) // codec_av_new -> called from in_ffmpgdec_initialize_stream
		{
			mpxplay_control_startup_programid_select = ffmpi->program_number_current;
			mpxplay_debugf(MPXPLAY_DEBUG_TEMP, "mpxplay_control_startup_programid_select %d", mpxplay_control_startup_programid_select);
		}
	}

//	if((streamtype_index == MPXPLAY_STREAMTYPEINDEX_AUDIO) || (streamtype_index == MPXPLAY_STREAMTYPEINDEX_VIDEO))
//	{
//		mpxplay_debugf(MPXPLAY_DEBUG_STRMCHG, "in_ffmpstrm_streams_count END sti:%d found:%d nbs:%d", streamtype_index, found, ffmpi->fctx->nb_streams);
//	}

	if(found)
		retval = found;

	return retval;
}

// give a priority value using the stream codec name, comparing with the preffered audiotypes list (lower number means higher priority)
static int inffmpstrm_get_audio_priority(AVStream *st)
{
	int priority = MPXINI_MAX_EXTRA_FILETYPES, i;
	char *codec_name;
	if(!st || !st->codecpar || (st->codecpar->codec_id == AV_CODEC_ID_NONE))
	{
		mpxplay_debugf(MPXPLAY_DEBUG_STRMPRIO, "inffmpg_get_audio_priority NO ST");
		return priority;
	}
	codec_name = (char *)avcodec_get_name(st->codecpar->codec_id); // FIXME: never returns NULL by description
	if(!codec_name)
	{
		mpxplay_debugf(MPXPLAY_DEBUG_STRMPRIO, "inffmpg_get_audio_priority NO NAME");
		return priority;
	}
	i = 0;
	do
	{
		char *audiotype_ptr = mpxplay_config_video_preffered_audiotype_ptrs[i];
		if(!audiotype_ptr)
			break;
		if(pds_filename_wildcard_cmp(codec_name, audiotype_ptr))
		{
			priority = i;
			mpxplay_debugf(MPXPLAY_DEBUG_STRMPRIO, "inffmpg_get_audio_priority i:%d p:%d a:\"%s\" cn:\"%s\"", i, priority, audiotype_ptr, codec_name);
			break;
		}
		i++;
	}while(i < MPXINI_MAX_EXTRA_FILETYPES);
	return priority;
}

// select stream for the specified stream-type (audio, video, subtitle) by language code and preferred audio format
static int inffmpstrm_primary_stream_select(struct ffmpg_demuxer_data_s *ffmpi, int streamtype_index, mpxp_bool_t forced_stream_only)
{
	int best_stream_num = INFFMPG_STREAM_INDEX_DISABLE, first_forced_stream_num = INFFMPG_STREAM_INDEX_DISABLE;
	int stream_ff_idx = INFFMPG_INVALID_STREAM_INDEX, selected_stream_num = INFFMPG_STREAM_INDEX_DISABLE;
	int i, sc = 0, stream_number = 0, best_audio_priority = MPXINI_MAX_EXTRA_FILETYPES, selected_audio_priority = MPXINI_MAX_EXTRA_FILETYPES + 1;

	// load preferred external stream
	if((streamtype_index < MPXPLAY_STREAMTYPEINDEX_EXTSTREAMNUM) && (mpxplay_config_video_extstream_loadtype[streamtype_index] == MPXPLAY_EXTERNALSTREAM_LOADTYPE_PREFER))
	{
		selected_stream_num = in_ffmpfile_filelist_search_by_language(&ffmpi->external_file_infos[streamtype_index], mpxplay_config_video_preferred_language[streamtype_index]);
	}
	// load internal stream
	if((selected_stream_num < INFFMPG_STREAM_INDEX_VALID) && ((streamtype_index != MPXPLAY_STREAMTYPEINDEX_SUBTITLE) || !funcbit_test(mpxplay_config_videoplayer_control, MPXPLAY_CONFIG_VIDEOPLAYERCONTROL_SUBTITLES_EMBEDDED_DISABLE)))
	{
		sc = in_ffmpstrm_streams_count(ffmpi, streamtype_index, &stream_number, &stream_ff_idx, NULL);
	}
	// no internal stream has found
	if(sc <= 0)
	{
		if(!funcbit_test(ffmpi->flags, INFFMPG_FLAG_IS_LIVESTREAM))
		{   // try to load external stream for local files
			if((streamtype_index < MPXPLAY_STREAMTYPEINDEX_EXTSTREAMNUM) && (mpxplay_config_video_extstream_loadtype[streamtype_index] == MPXPLAY_EXTERNALSTREAM_LOADTYPE_USE))
			{
				selected_stream_num = in_ffmpfile_filelist_search_by_language(&ffmpi->external_file_infos[streamtype_index], mpxplay_config_video_preferred_language[streamtype_index]);
			}
		}
		else // search (wait for) a new stream at live streams
		{
			selected_stream_num = INFFMPG_STREAM_INDEX_SEARCH;
		}
		goto err_out_primary_select;
	}

	if(streamtype_index == MPXPLAY_STREAMTYPEINDEX_AUDIO)
	{
		mpxplay_debugf(MPXPLAY_DEBUG_STRMPRIO, "inffmpstrm_primary_stream_select BEGIN sti:%d nbs:%d sc:%d pr:%d sn:%d ffid:%d",
				streamtype_index, ffmpi->nb_streams[streamtype_index], sc, ffmpi->program_number_current, stream_number, stream_ff_idx);
	}

	stream_number = 0;
	for(i = 0; i < ffmpi->nb_streams[streamtype_index]; i++)
	{
		stream_ff_idx = INFFMPG_INVALID_STREAM_INDEX;
		sc = in_ffmpstrm_streams_count(ffmpi, streamtype_index, &stream_number, &stream_ff_idx, NULL);
		if((sc > 0) && (stream_ff_idx >= INFFMPG_STREAM_INDEX_VALID) && (stream_ff_idx < ffmpi->fctx->nb_streams))
		{
			AVStream *st = ffmpi->fctx->streams[stream_ff_idx];
			if(st)
			{
				int audio_priority_stream = MPXINI_MAX_EXTRA_FILETYPES;
				if(best_stream_num < INFFMPG_STREAM_INDEX_VALID)
				{
					best_stream_num = stream_number;
				}
				if(streamtype_index == MPXPLAY_STREAMTYPEINDEX_AUDIO)
				{
					audio_priority_stream = inffmpstrm_get_audio_priority(st);
					mpxplay_debugf(MPXPLAY_DEBUG_STRMPRIO, "in_ffmpgdec_stream_select_default PRIO i:%d sn:%d p:%d", i, stream_number, audio_priority_stream);
					if(audio_priority_stream < best_audio_priority) // we've found a better audio format (by the preferred audiotypes list)
					{
						best_audio_priority = audio_priority_stream;
						best_stream_num = stream_number;
					}
				}
				if( ((streamtype_index == MPXPLAY_STREAMTYPEINDEX_AUDIO) && (!!funcbit_test(mpxplay_config_videoplayer_control, MPXPLAY_CONFIG_VIDEOPLAYERCONTROL_AUDIOSTREAM_PREF_DESCRIPTOR) == !!(st->disposition & AV_DISPOSITION_VISUAL_IMPAIRED))) // stream 'descriptor' flag matches with the preferred one
				 || ((streamtype_index != MPXPLAY_STREAMTYPEINDEX_AUDIO) && (!forced_stream_only || (!funcbit_test(mpxplay_config_videoplayer_control, MPXPLAY_CONFIG_VIDEOPLAYERCONTROL_STREAMS_FORCED_DISABLE) && (st->disposition & (AV_DISPOSITION_DEFAULT | AV_DISPOSITION_FORCED)))))
				){
					AVDictionaryEntry *t = av_dict_get((const AVDictionary *)st->metadata, "language", NULL, 0);
					if(t && t->value && (pds_stricmp(t->value, mpxplay_config_video_preferred_language[streamtype_index]) == 0)) // found language for this streamtype
					{
						if(streamtype_index == MPXPLAY_STREAMTYPEINDEX_AUDIO)
						{
							mpxplay_debugf(MPXPLAY_DEBUG_STRMPRIO, "in_ffmpgdec_stream_select_default LANG sn:%d ssn:%d bsn:%d sap:%d ap:%d", stream_number, selected_stream_num, best_stream_num, selected_audio_priority, audio_priority_stream);
						}
						if( (selected_stream_num < INFFMPG_STREAM_INDEX_VALID)  // (audio) language not found yet
						 || (audio_priority_stream < selected_audio_priority)   // or we've found a better audio format for this language
						 || ((streamtype_index == MPXPLAY_STREAMTYPEINDEX_SUBTITLE) && !forced_stream_only && !(st->disposition & (AV_DISPOSITION_DEFAULT | AV_DISPOSITION_FORCED))) // or non-forced subtitle found
						){
							selected_stream_num = stream_number;
							selected_audio_priority = audio_priority_stream;
						}
						if(streamtype_index == MPXPLAY_STREAMTYPEINDEX_VIDEO) // we don't search further the language at video streams, just using the first one
						{
							break;
						}
					}
					if( !funcbit_test(mpxplay_config_videoplayer_control, MPXPLAY_CONFIG_VIDEOPLAYERCONTROL_STREAMS_FORCED_DISABLE)
					 && ((st->disposition & AV_DISPOSITION_FORCED) || ((st->disposition & AV_DISPOSITION_DEFAULT) && !mpxplay_config_video_preferred_language[streamtype_index][0])) // forced stream OR default stream without language config
					){
						first_forced_stream_num = stream_number;
					}
				}
			}
			stream_number++;
		}
	}

	if((selected_stream_num < INFFMPG_STREAM_INDEX_VALID) && ((streamtype_index != MPXPLAY_STREAMTYPEINDEX_SUBTITLE) || !forced_stream_only)) // language not found at all
	{
		if((streamtype_index < MPXPLAY_STREAMTYPEINDEX_EXTSTREAMNUM) && (mpxplay_config_video_extstream_loadtype[streamtype_index] == MPXPLAY_EXTERNALSTREAM_LOADTYPE_USE) && mpxplay_config_video_preferred_language[streamtype_index][0])
		{
			selected_stream_num = in_ffmpfile_filelist_search_by_language(&ffmpi->external_file_infos[streamtype_index], mpxplay_config_video_preferred_language[streamtype_index]); // try to load external stream
		}
		if(selected_stream_num < INFFMPG_STREAM_INDEX_VALID)
		{
			selected_stream_num = (streamtype_index == MPXPLAY_STREAMTYPEINDEX_SUBTITLE)? first_forced_stream_num : best_stream_num; // use the first found stream (or the highest-priority audio stream)
		}
		if(streamtype_index == MPXPLAY_STREAMTYPEINDEX_AUDIO)
		{
			mpxplay_debugf(MPXPLAY_DEBUG_STRMPRIO, "inffmpstrm_primary_stream_select FINAL sti:%d sn:%d", streamtype_index, selected_stream_num);
		}
	}

err_out_primary_select:
	mpxplay_debugf(MPXPLAY_DEBUG_STRMCHG, "inffmpstrm_primary_stream_select END sti:%d nbs:%d sn:%d ffi:%d", streamtype_index, ffmpi->fctx->nb_streams, selected_stream_num, stream_ff_idx);

	return selected_stream_num;
}

static int inffmpstrm_stream_language_cmp(struct ffmpg_demuxer_data_s *ffmpi, int streamtype_index, int stream_number, char *language)
{
	int sn = stream_number, stream_ff_idx = INFFMPG_INVALID_STREAM_INDEX;
	AVDictionaryEntry *t;
	AVStream *st;

	if(in_ffmpstrm_streams_count(ffmpi, streamtype_index, &sn, &stream_ff_idx, NULL) <= 0)
	{
		return -1;
	}
	if((stream_ff_idx < 0) || (stream_ff_idx >= ffmpi->fctx->nb_streams))
	{
		return -1;
	}
	st = ffmpi->fctx->streams[stream_ff_idx];
	if(!st)
	{
		return -1;
	}
	t = av_dict_get((const AVDictionary *)st->metadata, "language", NULL, 0);
	if(!t || !t->value)
	{
		return INFFMPG_ERROR_STREAM_LANGCODE_MISSING;
	}
	return pds_stricmp(t->value, language);
}

// select primary audio, video and subtitle streams (by language code and preferred audio format)
void in_ffmpstrm_primary_streams_select(struct ffmpg_demuxer_data_s *ffmpi)
{
	int i;

	for(i = 0; i <= ffmpi->streamtype_limit; i++)
	{
		mpxp_bool_t forced_stream_only = FALSE;
		ffmpi->select_new_stream_number[i] = INFFMPG_INVALID_STREAM_INDEX;
		ffmpi->selected_ff_stream_index[i] = INFFMPG_INVALID_STREAM_INDEX;
		ffmpi->nb_frame_mismatches[i] = 0;
		switch(i)
		{
			case MPXPLAY_STREAMTYPEINDEX_SUBTITLE:
				if(ffmpi->select_new_stream_number[MPXPLAY_STREAMTYPEINDEX_AUDIO] >= 0)
				{
					int retval = inffmpstrm_stream_language_cmp(ffmpi, MPXPLAY_STREAMTYPEINDEX_AUDIO, ffmpi->select_new_stream_number[MPXPLAY_STREAMTYPEINDEX_AUDIO], mpxplay_config_video_preferred_language[MPXPLAY_STREAMTYPEINDEX_SUBTITLE]);
					if(retval == 0) // audio and subtitle languages match
					{
						forced_stream_only = TRUE; // TODO: default/forced flag is used at subtitle only (audio format priority and language code are preferred)
					}
				}
				// @suppress("No break at end of case")
			case MPXPLAY_STREAMTYPEINDEX_AUDIO:
			case MPXPLAY_STREAMTYPEINDEX_VIDEO:
				ffmpi->select_new_stream_number[i] = inffmpstrm_primary_stream_select(ffmpi, i, forced_stream_only);
				break;
		}
	}
}

// check primary audio and video streams, if they are lost, then select a new stream
void in_ffmpstrm_primary_streams_check(struct ffmpg_demuxer_data_s *ffmpi)
{
	int i, streamtype_limit = min(ffmpi->streamtype_limit, MPXPLAY_STREAMTYPEINDEX_VIDEO);

	for(i = 0; i <= streamtype_limit; i++)
	{
		int select_new_stream = ffmpi->select_new_stream_number[i];
		if(((select_new_stream == INFFMPG_STREAM_INDEX_LOST) || (select_new_stream == INFFMPG_STREAM_INDEX_SEARCH)) && (++ffmpi->nb_frame_mismatches[i] > (INFFMPG_MAX_FRAME_MISMACTHES * 2)))
		{
			ffmpi->nb_frame_mismatches[i] = 0;
			//funcbit_enable(ffmpi->fctx->ctx_flags, AVFMTCTX_NOHEADER);
			ffmpi->select_new_stream_number[i] = inffmpstrm_primary_stream_select(ffmpi, i, FALSE);
		}
	}
}

// open stream related codec (partial open at video streams)
static int in_ffmpstrm_stream_codec_open(struct ffmpg_demuxer_data_s *ffmpi, int streamtype_index, AVStream *selected_av_st, AVCodec **codec_av_new_p, AVCodecContext **codec_ctx_new_p)
{
	AVCodecContext *codec_ctx_new;
	AVCodec *codec_av_new;
	int retval = AVERROR(ENOMEM);

	if(!selected_av_st || !selected_av_st->codecpar || !codec_av_new_p || !codec_ctx_new_p)
		return retval;

	if(!funcbit_test(ffmpi->flags, INFFMPG_FLAG_LOADHEADONLY) && (streamtype_index == MPXPLAY_STREAMTYPEINDEX_AUDIO))
	{
		// check that the audio stream parsing is correct
		if(!mpxplay_ffmpstrm_is_avstream_parsed(selected_av_st, MPXPLAY_STREAMTYPEINDEX_AUDIO))
		{
			mpxplay_debugf(MPXPLAY_DEBUG_WARNING, "in_ffmpstrm_stream_codec_open AUDIO BAD ch:%d sr:%d id:%d", selected_av_st->codecpar->ch_layout.nb_channels,
					selected_av_st->codecpar->sample_rate, selected_av_st->codecpar->codec_id);
			in_ffmpstrm_invalidate_ffstream(selected_av_st);
			return INFFMPG_ERROR_STREAM_NOTPARSED;
		}
	}
	else if(streamtype_index == MPXPLAY_STREAMTYPEINDEX_VIDEO)
	{
		mpxplay_in_ffmpgdec_hwdec_override_avcodec(selected_av_st->codecpar, codec_av_new_p);
		if((selected_av_st->sample_aspect_ratio.num > 0) && (selected_av_st->sample_aspect_ratio.den > 0))
		{   // FIXME: FFmpeg bug?
			selected_av_st->codecpar->sample_aspect_ratio = selected_av_st->sample_aspect_ratio;
		}
	}


	codec_av_new = *codec_av_new_p;
	codec_ctx_new = avcodec_alloc_context3(codec_av_new);
	if(!codec_ctx_new)
		goto err_out_codecopen;

	retval = 0;

	switch(streamtype_index)
	{
		case MPXPLAY_STREAMTYPEINDEX_VIDEO:
		case MPXPLAY_STREAMTYPEINDEX_SEEKPREVIEW:
			// video codec ctx is opened from video_worker -> in_ffmpgdec_infile_callback (for multithread safety)
			break;
		case MPXPLAY_STREAMTYPEINDEX_AUDIO:
			codec_ctx_new->request_sample_fmt = AV_SAMPLE_FMT_FLT;
#ifndef INFFMP_USE_DEMUXER_AUINIT
			if(funcbit_test(ffmpi->flags, INFFMPG_FLAG_IS_LIVESTREAM))
				break;
#endif
		// @suppress("No break at end of case")
		default:
			retval = avcodec_parameters_to_context(codec_ctx_new, selected_av_st->codecpar);
			if(retval >= 0)
			{
				retval = avcodec_open2(codec_ctx_new, codec_av_new, NULL);
				if(retval >= 0)
				{
					if(codec_ctx_new->codec_type == AVMEDIA_TYPE_SUBTITLE)
						codec_ctx_new->sub_charenc_mode = FF_SUB_CHARENC_MODE_IGNORE;
				}
			}
	}

	if(retval >= 0)
	{
		*codec_av_new_p = codec_av_new;
		*codec_ctx_new_p = codec_ctx_new;
		return retval;
	}

err_out_codecopen:
	if(codec_ctx_new)
		avcodec_free_context(&codec_ctx_new);
	return retval;
}

int in_ffmpstrm_initialize_one_stream(struct ffmpg_demuxer_data_s *ffmpi, int streamtype_index, int select_stream_number)
{
	int stream_number = select_stream_number, selected_ff_streamindex = 0, retval = 0, stream_count;
	AVCodecContext *codec_ctx_new = NULL;
	AVStream *selected_av_st = NULL;
	AVCodec *codec_av_new = NULL;
	unsigned int packet_flags;

	mpxplay_debugf(MPXPLAY_DEBUG_STRMCHG, "in_ffmpstrm_initialize_one_stream BEGIN sti:%d ssn:%d", streamtype_index, select_stream_number);

	if(ffmpi->selected_stream_number[streamtype_index] >= INFFMPG_STREAM_INDEX_EXTERNALFILE_BEGIN)
	{
		in_ffmpfile_file_close(&ffmpi->external_file_infos[streamtype_index]);
	}

	if(stream_number == INFFMPG_STREAM_INDEX_DISABLE)
	{
		if(ffmpi->codecctx_queues[streamtype_index].nb_packets)
			mpxplay_ffmpgdec_packetqueue_put(&ffmpi->codecctx_queues[streamtype_index], MPXPLAY_PACKETTYPE_AVCODECCTX, MPXPLAY_PACKETLISTFLAG_CLEARBUF, NULL, 0, NULL, NULL);
		if(ffmpi->packet_queues[streamtype_index].nb_packets)  // TODO: ???
			mpxplay_ffmpgdec_packetqueue_put(&ffmpi->packet_queues[streamtype_index], MPXPLAY_PACKETTYPE_AVPACKET, MPXPLAY_PACKETLISTFLAG_CLEARBUF, NULL, 0, NULL, NULL); //
		goto err_out_stinit;
	}

	if((stream_number >= INFFMPG_STREAM_INDEX_EXTERNALFILE_BEGIN) || (streamtype_index == MPXPLAY_STREAMTYPEINDEX_SEEKPREVIEW))
	{
		stream_count = in_ffmpfile_file_open_by_streamindex(ffmpi, streamtype_index, stream_number, &selected_av_st, &codec_av_new);
		if((stream_count > 0) && selected_av_st)
			selected_ff_streamindex = selected_av_st->index;
	}
	else
	{
		stream_count = in_ffmpstrm_streams_count(ffmpi, streamtype_index, &stream_number, &selected_ff_streamindex, &codec_av_new);
		if((stream_count > 0) && ffmpi->fctx->streams)
			selected_av_st = ffmpi->fctx->streams[selected_ff_streamindex];
	}

	if(stream_count <= 0)
	{
		retval = stream_count;
		goto err_out_stinit;
	}

	retval = in_ffmpstrm_stream_codec_open(ffmpi, streamtype_index, selected_av_st, &codec_av_new, &codec_ctx_new);
	if(retval < 0)
	{
		mpxplay_debugf(MPXPLAY_DEBUG_ERROR, "in_ffmpstrm_stream_codec_open FAILED ret:%d", retval);
		goto err_out_stinit;
	}

	ffmpi->selected_stream_number[streamtype_index] = stream_number;
	ffmpi->selected_ff_stream_index[streamtype_index] = selected_ff_streamindex;
	ffmpi->selected_avstream[streamtype_index] = selected_av_st;
	ffmpi->avcodecs[streamtype_index] = codec_av_new;
	ffmpi->codec_ctx[streamtype_index] = codec_ctx_new;
	ffmpi->codecctx_seq_counters[streamtype_index]++;

	packet_flags = 0;
	if( (streamtype_index == MPXPLAY_STREAMTYPEINDEX_VIDEO) || (streamtype_index >= MPXPLAY_STREAMTYPEINDEX_SEEKPREVIEW)
#ifndef INFFMP_USE_DEMUXER_AUINIT
	 || ((streamtype_index == MPXPLAY_STREAMTYPEINDEX_AUDIO) && funcbit_test(ffmpi->flags, INFFMPG_FLAG_IS_LIVESTREAM))
#endif
	){
		funcbit_enable(packet_flags, MPXPLAY_PACKETLISTFLAG_INIT);
	}
	if(ffmpi->stream_flags[streamtype_index] & INFFMPG_STREAMFLAG_CLEARBUF_DEMUX) // TODO: check CLEARBUF
		funcbit_enable(packet_flags, MPXPLAY_PACKETLISTFLAG_CLEARBUF);

	mpxplay_ffmpgdec_packetqueue_put(&ffmpi->codecctx_queues[streamtype_index], MPXPLAY_PACKETTYPE_AVCODECCTX, packet_flags, codec_ctx_new, ffmpi->codecctx_seq_counters[streamtype_index], selected_av_st, NULL);

	if(!funcbit_test(ffmpi->flags, INFFMPG_FLAG_LOADHEADONLY)){
		mpxplay_debugf(MPXPLAY_DEBUG_STRMCHG, "in_ffmpstrm_initialize_one_stream %8.8X sti:%d ctx:%8.8X", (mpxp_ptrsize_t)ffmpi, streamtype_index, (mpxp_ptrsize_t)codec_ctx_new);
	}

	//av_dump_format(ffmpi->fctx, ffmpi->selected_ff_stream_index[streamtype_index], NULL, 0);
	return stream_count;

err_out_stinit:
	ffmpi->selected_stream_number[streamtype_index] = INFFMPG_INVALID_STREAM_INDEX;
	ffmpi->selected_ff_stream_index[streamtype_index] = INFFMPG_INVALID_STREAM_INDEX;
	ffmpi->selected_avstream[streamtype_index] = NULL;
	ffmpi->avcodecs[streamtype_index] = NULL;
	ffmpi->codec_ctx[streamtype_index] = NULL;
	mpxplay_ffmpgdec_packetqueue_put(&ffmpi->codecctx_queues[streamtype_index], MPXPLAY_PACKETTYPE_AVCODECCTX, MPXPLAY_PACKETLISTFLAG_CLEARBUF, NULL, 0, NULL, NULL);
	mpxplay_ffmpgdec_packetqueue_put(&ffmpi->packet_queues[streamtype_index], MPXPLAY_PACKETTYPE_AVPACKET, MPXPLAY_PACKETLISTFLAG_CLEARBUF, NULL, 0, NULL, NULL);
	if(streamtype_index != MPXPLAY_STREAMTYPEINDEX_SUBTITLE)
	{
		mpxplay_debugf(MPXPLAY_DEBUG_ERROR, "in_ffmpstrm_initialize_one_stream FAILED %8.8X sti:%d ret:%d", (mpxp_ptrsize_t)ffmpi, streamtype_index, retval);
	}
	return retval;
}

// initialize primary audio, video and subtitle streams
mpxp_bool_t in_ffmpstrm_primary_streams_initialize(struct ffmpg_demuxer_data_s *ffmpi, mpxp_bool_t initial)
{
	mpxp_bool_t is_video_stream_change = FALSE;
	int i;
	for(i = 0; i <= ffmpi->streamtype_limit; i++)
	{
		if(!in_ffmpgdec_content_has_programs(ffmpi))
		{
			if(ffmpi->select_new_stream_number[i] == INFFMPG_STREAM_INDEX_LOST)
				ffmpi->select_new_stream_number[i] = 0; // TODO: use like in_ffmpstrm_primary_streams_select
		}
		if(ffmpi->select_new_stream_number[i] > INFFMPG_INVALID_STREAM_INDEX)
		{
			int retval, old_stream_num = ffmpi->selected_stream_number[i], old_ff_stridx = ffmpi->selected_ff_stream_index[i];
			if(i == MPXPLAY_STREAMTYPEINDEX_AUDIO)
			{
				mpxplay_debugf(MPXPLAY_DEBUG_STRMCHG, "in_ffmpstrm_primary_streams_initialize sti:%d -> sn:%d", i, ffmpi->select_new_stream_number[i]);
			}
			retval = in_ffmpstrm_initialize_one_stream(ffmpi, i, ffmpi->select_new_stream_number[i]);
			if(retval > 0)
			{
				if(!initial)
					funcbit_enable(ffmpi->stream_flags[i], INFFMPG_STREAMFLAG_CLEARBUF_DEMUX);
#ifdef INFFMP_USE_VIDEO_SYNC
//				else if(i == MPXPLAY_STREAMTYPEINDEX_AUDIO)
//					funcbit_enable(ffmpi->stream_flags[i], INFFMPG_STREAMFLAG_AVSYNC);  // FIXME: initial sync doesn't work
#endif
			}
			if((retval != INFFMPG_ERROR_STREAM_NOTPARSED) || (i >= MPXPLAY_STREAMTYPEINDEX_SUBTITLE))
			{
				ffmpi->select_new_stream_number[i] = INFFMPG_INVALID_STREAM_INDEX;
			}
			if(i <= MPXPLAY_STREAMTYPEINDEX_VIDEO)
			{
				ffmpi->avsync_timestamp = INFFMPG_INVALID_PTS;
				ffmpi->avsync_videotimestamp = INFFMPG_INVALID_PTS;
				if((i == MPXPLAY_STREAMTYPEINDEX_VIDEO) && ((old_stream_num != ffmpi->selected_stream_number[i]) || (old_ff_stridx != ffmpi->selected_ff_stream_index[i])))
				{
					is_video_stream_change = TRUE; // a video stream has changed
				}
			}
		}
	}
	in_ffmpstrm_all_programs_videostreams_update(ffmpi);
	return is_video_stream_change;
}

//--------------------------------------------------------------------------------------------------------------------------------------------------------
// videowall functions for demuxer side

// initialize video stream of all programs (for videowall) (note: high memory usage allocation)
mpxp_bool_t in_ffmpstrm_all_programs_videostreams_initialize(struct ffmpg_demuxer_data_s *ffmpi)
{
	mpxp_bool_t success = FALSE;
	int i, sti;

	if(!in_ffmpgdec_content_has_programs(ffmpi))
		return success;

	for(i = 0, sti = MPXPLAY_STREAMTYPEINDEX_VIDEOWALL; i < ffmpi->fctx->nb_programs; i++)
	{
		AVProgram *prg = ffmpi->fctx->programs[i];
		AVDictionaryEntry *t;

		if(!prg || (prg->id <= 0))
			continue;

		t = av_dict_get((const AVDictionary *)prg->metadata, "service_name", NULL, 0);
		if(!t || !t->value) // TODO: check, program elements without service names are ignored!
			continue;

		ffmpi->selected_avstream[sti] = in_ffmpstrm_find_videostream_and_decoder(ffmpi->fctx, sti, prg->id, &ffmpi->avcodecs[sti]);

		if(in_ffmpstrm_stream_codec_open(ffmpi, sti, ffmpi->selected_avstream[sti], &ffmpi->avcodecs[sti], &ffmpi->codec_ctx[sti]) >= 0)
		{
			ffmpi->codecctx_seq_counters[sti]++;
			if(!mpxplay_ffmpgdec_packetqueue_check(&ffmpi->codecctx_queues[sti]))
				mpxplay_ffmpgdec_packetqueue_init(&ffmpi->codecctx_queues[sti], sti);
			if(!mpxplay_ffmpgdec_packetqueue_check(&ffmpi->packet_queues[sti]))
				mpxplay_ffmpgdec_packetqueue_init(&ffmpi->packet_queues[sti], sti);
			ffmpi->selected_ff_stream_index[sti] = ffmpi->selected_avstream[sti]->index;
			if(ffmpi->selected_ff_stream_index[sti] == ffmpi->selected_ff_stream_index[MPXPLAY_STREAMTYPEINDEX_VIDEO])
				ffmpi->primary_video_stream_index = sti;
			mpxplay_ffmpgdec_packetqueue_put(&ffmpi->codecctx_queues[sti], MPXPLAY_PACKETTYPE_AVCODECCTX, MPXPLAY_PACKETLISTFLAG_INIT, ffmpi->codec_ctx[sti], ffmpi->codecctx_seq_counters[sti], ffmpi->selected_avstream[sti], NULL);
			sti++;
		}
		else
		{
			ffmpi->selected_ff_stream_index[sti] = INFFMPG_INVALID_STREAM_INDEX;
			ffmpi->selected_avstream[sti] = NULL;
		}
	}

	if(sti > MPXPLAY_STREAMTYPEINDEX_VIDEOWALL)
	{
		funcbit_enable(mpxplay_config_dvbepg_control_flags, MPXPLAY_CONFIG_DVBEPGCTRL_VIDEOWALL_WAS_ENABLED);
		success = TRUE;
	}

	for(; sti < MPXPLAY_STREAMTYPEINDEX_MAX; sti++)
	{
		ffmpi->selected_avstream[sti] = NULL;
		ffmpi->selected_ff_stream_index[sti] = INFFMPG_INVALID_STREAM_INDEX;
	}

	return success;
}

// update videowall settings (currently videowall_primary_streamindex only) after manual program change (from the video surface context menu)
static void in_ffmpstrm_all_programs_videostreams_update(struct ffmpg_demuxer_data_s *ffmpi)
{
	int sti;

	if(!ffmpi->videowall_mode_enabled)
		return;

	for(sti = MPXPLAY_STREAMTYPEINDEX_VIDEOWALL; sti < MPXPLAY_STREAMTYPEINDEX_MAX; sti++)
	{
		if(ffmpi->selected_ff_stream_index[sti] == ffmpi->selected_ff_stream_index[MPXPLAY_STREAMTYPEINDEX_VIDEO])
		{
			ffmpi->primary_video_stream_index = sti;
			break;
		}
	}
}

// get program id by streamtype_index (videowall index)
int in_ffmpstrm_all_programs_videostreams_program_id_get(struct ffmpg_demuxer_data_s *ffmpi, int streamtype_index)
{
	int program_id = -1;

	if(((streamtype_index < MPXPLAY_STREAMTYPEINDEX_VIDEOWALL) || (streamtype_index >= MPXPLAY_STREAMTYPEINDEX_MAX)) && (streamtype_index != MPXPLAY_STREAMTYPEINDEX_VIDEO))
		return program_id;

	if(ffmpi->selected_ff_stream_index[streamtype_index] >= INFFMPG_STREAM_INDEX_VALID)
	{
		AVProgram *program = av_find_program_from_stream(ffmpi->fctx, NULL, ffmpi->selected_ff_stream_index[streamtype_index]);
		if(program)
		{
			program_id = program->id;
		}
	}

	return program_id;
}

// manual program selection (by click on the video surface) by streamtype_index (videowall index)
mpxp_bool_t in_ffmpstrm_all_programs_videostreams_program_select(struct ffmpg_demuxer_data_s *ffmpi, int streamtype_index)
{
	int program_id = in_ffmpstrm_all_programs_videostreams_program_id_get(ffmpi, streamtype_index);
	mpxp_bool_t success = FALSE;

	if(program_id >= 0)
	{
		ffmpi->select_new_program_number = program_id;
		//funcbit_enable(ffmpi->flags, INFFMPG_FLAG_NONREWINDPRGCHG); // TODO:
		ffmpi->primary_video_stream_index = streamtype_index;
		success = TRUE;
	}

	return success;
}

void in_ffmpstrm_all_programs_videostreams_terminate(struct ffmpg_demuxer_data_s *ffmpi)
{
	int sti;

	ffmpi->primary_video_stream_index = MPXPLAY_STREAMTYPEINDEX_VIDEO;

	for(sti = MPXPLAY_STREAMTYPEINDEX_VIDEOWALL; sti < MPXPLAY_STREAMTYPEINDEX_MAX; sti++)
	{
		mpxplay_ffmpgdec_packetqueue_put(&ffmpi->codecctx_queues[sti], MPXPLAY_PACKETTYPE_AVCODECCTX, MPXPLAY_PACKETLISTFLAG_RESET|MPXPLAY_PACKETLISTFLAG_CLEARBUF, NULL, 0, NULL, NULL);
		mpxplay_ffmpgdec_packetqueue_put(&ffmpi->packet_queues[sti], MPXPLAY_PACKETTYPE_AVPACKET, MPXPLAY_PACKETLISTFLAG_CLEARBUF, NULL, 0, NULL, NULL);
		ffmpi->selected_ff_stream_index[sti] = INFFMPG_INVALID_STREAM_INDEX;
	}

	funcbit_disable(mpxplay_config_dvbepg_control_flags, MPXPLAY_CONFIG_DVBEPGCTRL_VIDEOWALL_WAS_ENABLED);
}

#endif //MPXPLAY_LINK_INFILE_FF_MPEG
