//**************************************************************************
//*                     This file is part of the                           *
//*                 Mpxplay/MMC - multimedia player.                       *
//*                  The source code of Mpxplay is                         *
//*        (C) copyright 1998-2019 by PDSoft (Attila Padar)                *
//*                http://mpxplay.sourceforge.net                          *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//function: language code handling
//based on https://www.loc.gov/standards/iso639-2/php/code_list.php

#include "ffmpgdec.h"

#ifdef MPXPLAY_LINK_INFILE_FF_MPEG

static const char *mpxplay_ffmplang_language_codes[] = {
 "all", // default selection
 "xxx", // custom selection
 "aar", // Afar
 "abk", // Abkhazian
 "ace", // Achinese
 "ach", // Acoli
 "ada", // Adangme
 "ady", // Adyghe
 "afa", // Afro-Asiatic (Other)
 "afh", // Afrihili
 "afr", // Afrikaans
 "ain", // Ainu
 "aka", // Akan
 "akk", // Akkadian
 "alb", // Albanian (B)  // "B" (bibliographic)
 "ale", // Aleut
 "alg", // Algonquian languages
 "alt", // Southern Altai
 "amh", // Amharic
 "ang", // English, Old (ca.450-1100)
 "apa", // Apache languages
 "ara", // Arabic
 "arc", // Aramaic
 "arg", // Aragonese
 "arm", // Armenian (B)
 "arn", // Araucanian
 "arp", // Arapaho
 "art", // Artificial (Other)
 "arw", // Arawak
 "asm", // Assamese
 "ast", // Asturian
 "ath", // Athapascan languages
 "aus", // Australian languages
 "ava", // Avaric
 "ave", // Avestan
 "awa", // Awadhi
 "aym", // Aymara
 "aze", // Azerbaijani
 "bad", // Banda
 "bai", // Bamileke languages
 "bak", // Bashkir
 "bal", // Baluchi
 "bam", // Bambara
 "ban", // Balinese
 "baq", // Basque (B)
 "bas", // Basa
 "bat", // Baltic (Other)
 "bej", // Beja
 "bel", // Belarusian
 "bem", // Bemba
 "ben", // Bengali
 "ber", // Berber (Other)
 "bho", // Bhojpuri
 "bih", // Bihari
 "bik", // Bikol
 "bin", // Bini
 "bis", // Bislama
 "bla", // Siksika
 "bnt", // Bantu (Other)
 "bos", // Bosnian
 "bra", // Braj
 "bre", // Breton
 "btk", // Batak (Indonesia)
 "bua", // Buriat
 "bug", // Buginese
 "bul", // Bulgarian
 "bur", // Burmese (B)
 "byn", // Blin
 "cad", // Caddo
 "cai", // Central American Indian (Other)
 "car", // Carib
 "cat", // Catalan
 "cau", // Caucasian (Other)
 "ceb", // Cebuano
 "cel", // Celtic (Other)
 "cha", // Chamorro
 "chb", // Chibcha
 "che", // Chechen
 "chg", // Chagatai
 "chi", // Chinese (simplified) (B)
 "chk", // Chuukese
 "chm", // Mari
 "chn", // Chinook jargon
 "cho", // Choctaw
 "chp", // Chipewyan
 "chr", // Cherokee
 "chu", // Church Slavic
 "chv", // Chuvash
 "chy", // Cheyenne
 "cmc", // Chamic languages
 "cop", // Coptic
 "cor", // Cornish
 "cos", // Corsican
 "cpe", // Creoles and pidgins, English based (Other)
 "cpf", // Creoles and pidgins, French-based (Other)
 "cpp", // Creoles and pidgins, Portuguese-based (Other)
 "cre", // Cree
 "crh", // Crimean Tatar
 "crp", // Creoles and pidgins (Other)
 "csb", // Kashubian
 "cus", // Cushitic (Other)' couchitiques, autres langues
 "cze", // Czech (B)
 "dak", // Dakota
 "dan", // Danish
 "dar", // Dargwa
 "day", // Dayak
 "del", // Delaware
 "den", // Slave (Athapascan)
 "dgr", // Dogrib
 "din", // Dinka
 "div", // Divehi
 "doi", // Dogri
 "dra", // Dravidian (Other)
 "dua", // Duala
 "dum", // Dutch, Middle (ca.1050-1350)
 "dut", // Dutch (B)
 "dyu", // Dyula
 "dzo", // Dzongkha
 "efi", // Efik
 "egy", // Egyptian (Ancient)
 "eka", // Ekajuk
 "elx", // Elamite
 "eng", // English
 "enm", // English, Middle (1100-1500)
 "epo", // Esperanto
 "est", // Estonian
 "ewe", // Ewe
 "ewo", // Ewondo
 "fan", // Fang
 "fao", // Faroese
 "fat", // Fanti
 "fij", // Fijian
 "fil", // Filipino
 "fin", // Finnish
 "fiu", // Finno-Ugrian (Other)
 "fon", // Fon
 "fre", // French (B)
 "frm", // French, Middle (ca.1400-1600)
 "fro", // French, Old (842-ca.1400)
 "fry", // Frisian
 "ful", // Fulah
 "fur", // Friulian
 "gaa", // Ga
 "gay", // Gayo
 "gba", // Gbaya
 "gem", // Germanic (Other)
 "geo", // Georgian
 "ger", // German (B)
 "gez", // Geez
 "gil", // Gilbertese
 "gla", // Gaelic
 "gle", // Irish
 "glg", // Galician
 "glv", // Manx
 "gmh", // German, Middle High (ca.1050-1500)
 "goh", // German, Old High (ca.750-1050)
 "gon", // Gondi
 "gor", // Gorontalo
 "got", // Gothic
 "grb", // Grebo
 "grc", // Greek, Ancient (to 1453)
 "gre", // Greek (B)
 "grn", // Guarani
 "guj", // Gujarati
 "gwi", // Gwich´in
 "hai", // Haida
 "hat", // Haitian
 "hau", // Hausa
 "haw", // Hawaiian
 "heb", // Hebrew
 "her", // Herero
 "hil", // Hiligaynon
 "him", // Himachali
 "hin", // Hindi
 "hit", // Hittite
 "hmn", // Hmong
 "hmo", // Hiri Motu
 "hrv", // Croatian
 "hun", // Hungarian
 "hup", // Hupa
 "iba", // Iban
 "ibo", // Igbo
 "ice", // Icelandic
 "ido", // Ido
 "iii", // Sichuan Yi
 "ijo", // Ijo
 "iku", // Inuktitut
 "ile", // Interlingue
 "ilo", // Iloko
 "ina", // Interlingua (International Auxiliary Language Asso
 "inc", // Indic (Other)
 "ind", // Indonesian
 "ine", // Indo-European (Other)
 "inh", // Ingush
 "ipk", // Inupiaq
 "ira", // Iranian (Other)
 "iro", // Iroquoian languages
 "ita", // Italian
 "jav", // Javanese
 "jpn", // Japanese
 "jpr", // Judeo-Persian
 "jrb", // Judeo-Arabic
 "kaa", // Kara-Kalpak
 "kab", // Kabyle
 "kac", // Kachin
 "kal", // Kalaallisut
 "kam", // Kamba
 "kan", // Kannada
 "kar", // Karen
 "kas", // Kashmiri
 "kau", // Kanuri
 "kaw", // Kawi
 "kaz", // Kazakh
 "kbd", // Kabardian
 "kha", // Khasi
 "khi", // Khoisan (Other)
 "khm", // Khmer
 "kho", // Khotanese
 "kik", // Kikuyu
 "kin", // Kinyarwanda
 "kir", // Kirghiz
 "kmb", // Kimbundu
 "kok", // Konkani
 "kom", // Komi
 "kon", // Kongo
 "kor", // Korean
 "kos", // Kosraean
 "kpe", // Kpelle
 "krc", // Karachay-Balkar
 "kro", // Kru
 "kru", // Kurukh
 "kua", // Kuanyama
 "kum", // Kumyk
 "kur", // Kurdish
 "kut", // Kutenai
 "lad", // Ladino
 "lah", // Lahnda
 "lam", // Lamba
 "lao", // Lao
 "lat", // Latin
 "lav", // Latvian
 "lez", // Lezghian
 "lim", // Limburgan
 "lin", // Lingala
 "lit", // Lithuanian
 "lol", // Mongo
 "loz", // Lozi
 "ltz", // Luxembourgish
 "lua", // Luba-Lulua
 "lub", // Luba-Katanga
 "lug", // Ganda
 "lui", // Luiseno
 "lun", // Lunda
 "luo", // Luo (Kenya and Tanzania)
 "lus", // lushai
 "mac", // Macedonian
 "mad", // Madurese
 "mag", // Magahi
 "mah", // Marshallese
 "mai", // Maithili
 "mak", // Makasar
 "mal", // Malayalam
 "man", // Mandingo
 "mao", // Maori
 "map", // Austronesian (Other)
 "mar", // Marathi
 "mas", // Masai
 "may", // Malay
 "mdf", // Moksha
 "mdr", // Mandar
 "men", // Mende
 "mga", // Irish, Middle (900-1200)
 "mic", // Mi'kmaq
 "min", // Minangkabau
 "mis", // Miscellaneous languages
 "mkh", // Mon-Khmer (Other)
 "mlg", // Malagasy
 "mlt", // Maltese
 "mnc", // Manchu
 "mni", // Manipuri
 "mno", // Manobo languages
 "moh", // Mohawk
 "mol", // Moldavian
 "mon", // Mongolian
 "mos", // Mossi
 "mwl", // Mirandese
 "mul", // Multiple languages
 "mun", // Munda languages
 "mus", // Creek
 "mwr", // Marwari
 "myn", // Mayan languages
 "myv", // Erzya
 "nah", // Nahuatl
 "nai", // North American Indian
 "nap", // Neapolitan
 "nau", // Nauru
 "nav", // Navajo
 "nbl", // Ndebele, South
 "nde", // Ndebele, North
 "ndo", // Ndonga
 "nds", // Low German
 "nep", // Nepali
 "new", // Nepal Bhasa
 "nia", // Nias
 "nic", // Niger-Kordofanian (Other)
 "niu", // Niuean
 "nno", // Norwegian Nynorsk
 "nob", // Norwegian Bokmal
 "nog", // Nogai
 "non", // Norse, Old
 "nor", // Norwegian
 "nso", // Northern Sotho
 "nub", // Nubian languages
 "nwc", // Classical Newari
 "nya", // Chichewa
 "nym", // Nyamwezi
 "nyn", // Nyankole
 "nyo", // Nyoro
 "nzi", // Nzima
 "oci", // Occitan
 "oji", // Ojibwa
 "ori", // Oriya
 "orm", // Oromo
 "osa", // Osage
 "oss", // Ossetian
 "ota", // Turkish, Ottoman (1500-1928)
 "oto", // Otomian languages
 "paa", // Papuan (Other)
 "pag", // Pangasinan
 "pal", // Pahlavi
 "pam", // Pampanga
 "pan", // Panjabi
 "pap", // Papiamento
 "pau", // Palauan
 "peo", // Persian, Old (ca.600-400 B.C.)
 "per", // Persian (B)
 "phi", // Philippine (Other)
 "phn", // Phoenician
 "pli", // Pali
 "pol", // Polish
 "pon", // Pohnpeian
 "por", // Portuguese
 "pra", // Prakrit languages
 "pro", // Provençal, Old (to 1500)
 "pus", // Pushto
 "que", // Quechua
 "raj", // Rajasthani
 "rap", // Rapanui
 "rar", // Rarotongan
 "roa", // Romance (Other)
 "roh", // Raeto-Romance
 "rom", // Romany
 "run", // Rundi
 "rup", // Aromanian
 "rus", // Russian
 "sad", // Sandawe
 "sag", // Sango
 "sah", // Yakut
 "sai", // South American Indian (Other)
 "sal", // Salishan languages
 "sam", // Samaritan Aramaic
 "san", // Sanskrit
 "sas", // Sasak
 "sat", // Santali
 "srp", // Serbian
 "scn", // Sicilian
 "sco", // Scots
 "sel", // Selkup
 "sem", // Semitic (Other)
 "sga", // Irish, Old (to 900)
 "sgn", // Sign Languages
 "shn", // Shan
 "sid", // Sidamo
 "sin", // Sinhalese
 "sio", // Siouan languages
 "sit", // Sino-Tibetan (Other)
 "sla", // Slavic (Other)
 "slo", // Slovak
 "slv", // Slovenian
 "sma", // Southern Sami
 "sme", // Northern Sami
 "smi", // Sami languages (Other)
 "smj", // Lule Sami
 "smn", // Inari Sami
 "smo", // Samoan
 "sms", // Skolt Sami
 "sna", // Shona
 "snd", // Sindhi
 "snk", // Soninke
 "sog", // Sogdian
 "som", // Somali
 "son", // Songhai
 "sot", // Sotho, Southern
 "spa", // Spanish
 "srd", // Sardinian
 "srr", // Serer
 "ssa", // Nilo-Saharan (Other)
 "ssw", // Swati
 "suk", // Sukuma
 "sun", // Sundanese
 "sus", // Susu
 "sux", // Sumerian
 "swa", // Swahili
 "swe", // Swedish
 "syr", // Syriac
 "tah", // Tahitian
 "tai", // Tai (Other)
 "tam", // Tamil
 "tat", // Tatar
 "tel", // Telugu
 "tem", // Timne
 "ter", // Tereno
 "tet", // Tetum
 "tgk", // Tajik
 "tgl", // Tagalog
 "tha", // Thai
 "tib", // Tibetan (B)
 "tig", // Tigre
 "tir", // Tigrinya
 "tiv", // Tiv
 "tkl", // Tokelau
 "tlh", // Klingon
 "tli", // Tlingit
 "tmh", // Tamashek
 "tog", // Tonga (Nyasa)
 "ton", // Tonga (Tonga Islands)
 "tpi", // Tok Pisin
 "tsi", // Tsimshian
 "tsn", // Tswana
 "tso", // Tsonga
 "tuk", // Turkmen
 "tum", // Tumbuka
 "tup", // Tupi languages
 "tur", // Turkish
 "tut", // Altaic (Other)
 "tvl", // Tuvalu
 "twi", // Twi
 "tyv", // Tuvinian
 "udm", // Udmurt
 "uga", // Ugaritic
 "uig", // Uighur
 "ukr", // Ukrainian
 "umb", // Umbundu
 "und", // Undetermined
 "urd", // Urdu
 "uzb", // Uzbek
 "vai", // Vai
 "ven", // Venda
 "vie", // Vietnamese
 "vol", // Volapük
 "vot", // Votic
 "wak", // Wakashan languages
 "wal", // Walamo
 "war", // Waray
 "was", // Washo
 "wel", // Welsh (B)
 "wen", // Sorbian languages
 "wln", // Walloon
 "wol", // Wolof
 "xal", // Kalmyk
 "xho", // Xhosa
 "yao", // Yao
 "yap", // Yapese
 "yid", // Yiddish
 "yor", // Yoruba
 "ypk", // Yupik languages
 "zap", // Zapotec
 "zen", // Zenaga
 "zha", // Zhuang
 "znd", // Zande
 "zul", // Zulu
 "zun", // Zuni
 "rum", // Romanian
 "pob", // Portuguese (BR)
 "mne", // Montenegrin
 "zht", // Chinese (traditional)
 "zhe", // Chinese bilingual
 "pom", // Portuguese (MZ)
 "ext", // Extremaduran
 NULL
};

static const char *mpxplay_ffmplang_language_names[] = {
 "All (select)",	// means deselected (none) language code
 "Custom (type bellow)",  // means fill "custom language" field
 "Afar",			// aar
 "Abkhazian",		// abk
 "Achinese",		// ace
 "Acoli",			// ach
 "Adangme",			// ada
 "Adyghe",			// ady
 "Afro-Asiatic",	// afa
 "Afrihili",		// afh
 "Afrikaans",		// afr
 "Ainu",			// ain
 "Akan",			// aka
 "Akkadian",		// akk
 "Albanian",		// alb
 "Aleut",			// ale
 "Algonquian",		// alg
 "Southern Altai",	// alt
 "Amharic",			// amh
 "English (old)",	// ang
 "Apache",			// apa
 "Arabic",			// ara
 "Aramaic",			// arc
 "Aragonese",		// arg
 "Armenian",		// arm
 "Araucanian",		// arn
 "Arapaho",			// arp
 "Artificial",		// art
 "Arawak",			// arw
 "Assamese",		// asm
 "Asturian",		// ast
 "Athapascan",		// ath
 "Australian",		// aus
 "Avaric",			// ava
 "Avestan",			// ave
 "Awadhi",			// awa
 "Aymara",			// aym
 "Azerbaijani",		// aze
 "Banda",			// bad
 "Bamileke",		// bai
 "Bashkir",			// bak
 "Baluchi",			// bal
 "Bambara",			// bam
 "Balinese",		// ban
 "Basque",			// baq
 "Basa",			// bas
 "Baltic",			// bat
 "Beja",			// bej
 "Belarusian",		// bel
 "Bemba",			// bem
 "Bengali",			// ben
 "Berber",			// ber
 "Bhojpuri",		// bho
 "Bihari",			// bih
 "Bikol",			// bik
 "Bini",			// bin
 "Bislama",			// bis
 "Siksika",			// bla
 "Bantu",			// bnt
 "Bosnian",			// bos
 "Braj",			// bra
 "Breton",			// bre
 "Batak (Indonesia)",// btk
 "Buriat",			// bua
 "Buginese",		// bug
 "Bulgarian",		// bul
 "Burmese",			// bur
 "Blin",			// byn
 "Caddo",			// cad
 "Central American Indian",	// cai
 "Carib",			// car
 "Catalan",			// cat
 "Caucasian",		// cau
 "Cebuano",			// ceb
 "Celtic",			// cel
 "Chamorro",		// cha
 "Chibcha",			// chb
 "Chechen",			// che
 "Chagatai",		// chg
 "Chinese (simplified)",	// chi
 "Chuukese",		// chk
 "Mari",			// chm
 "Chinook jargon",	// chn
 "Choctaw",			// cho
 "Chipewyan",		// chp
 "Cherokee",		// chr
 "Church Slavic",	// chu
 "Chuvash",			// chv
 "Cheyenne",		// chy
 "Chamic",			// cmc
 "Coptic",			// cop
 "Cornish",			// cor
 "Corsican",		// cos
 "Creoles and pidgins (english)",		// cpe
 "Creoles and pidgins (french)",		// cpf
 "Creoles and pidgins (portuguese)",	// cpp
 "Cree",			// cre
 "Crimean Tatar",	// crh
 "Creoles and pidgins",	// crp
 "Kashubian",		// csb
 "Cushitic",		// cus
 "Czech",			// cze
 "Dakota",			// dak
 "Danish",			// dan
 "Dargwa",			// dar
 "Dayak",			// day
 "Delaware",		// del
 "Slave (Athapascan)",	// den
 "Dogrib",			// dgr
 "Dinka",			// din
 "Divehi",			// div
 "Dogri",			// doi
 "Dravidian",		// dra
 "Duala",			// dua
 "Dutch (middle age)",	// dum
 "Dutch",			// dut
 "Dyula",			// dyu
 "Dzongkha",		// dzo
 "Efik",			// efi
 "Egyptian (ancient)",	// egy
 "Ekajuk",			// eka
 "Elamite",			// elx
 "English",			// eng
 "English (middle age)",// enm
 "Esperanto",		// epo
 "Estonian",		// est
 "Ewe",				// ewe
 "Ewondo",			// ewo
 "Fang",			// fan
 "Faroese",			// fao
 "Fanti",			// fat
 "Fijian",			// fij
 "Filipino",		// fil
 "Finnish",			// fin
 "Finno-Ugrian",	// fiu
 "Fon",				// fon
 "French",			// fre
 "French (middle)",	// frm
 "French (old)",	// fro
 "Frisian",			// fry
 "Fulah",			// ful
 "Friulian",		// fur
 "Ga",				// gaa
 "Gayo",			// gay
 "Gbaya",			// gba
 "Germanic",		// gem
 "Georgian",		// geo
 "German",			// ger
 "Geez",			// gez
 "Gilbertese",		// gil
 "Gaelic",			// gla
 "Irish",			// gle
 "Galician",		// glg
 "Manx",			// glv
 "German (middle)",	// gmh
 "German (old age)",// goh
 "Gondi",			// gon
 "Gorontalo",		// gor
 "Gothic",			// got
 "Grebo",			// grb
 "Greek (ancient)",	// grc
 "Greek",			// gre
 "Guarani",			// grn
 "Gujarati",		// guj
 "Gwich´in",		// gwi
 "Haida",			// hai
 "Haitian",			// hat
 "Hausa",			// hau
 "Hawaiian",		// haw
 "Hebrew",			// heb
 "Herero",			// her
 "Hiligaynon",		// hil
 "Himachali",		// him
 "Hindi",			// hin
 "Hittite",			// hit
 "Hmong",			// hmn
 "Hiri Motu",		// hmo
 "Croatian",		// hrv
 "Hungarian",		// hun
 "Hupa",			// hup
 "Iban",			// iba
 "Igbo",			// ibo
 "Icelandic",		// ice
 "Ido",				// ido
 "Sichuan Yi",		// iii
 "Ijo",				// ijo
 "Inuktitut",		// iku
 "Interlingue",		// ile
 "Iloko",			// ilo
 "Interlingua",		// ina
 "Indic (Other)",	// inc
 "Indonesian",		// ind
 "Indo-European",	// ine
 "Ingush",			// inh
 "Inupiaq",			// ipk
 "Iranian",			// ira
 "Iroquoian",		// iro
 "Italian",			// ita
 "Javanese",		// jav
 "Japanese",		// jpn
 "Judeo-Persian",	// jpr
 "Judeo-Arabic",	// jrb
 "Kara-Kalpak",		// kaa
 "Kabyle",			// kab
 "Kachin",			// kac
 "Kalaallisut",		// kal
 "Kamba",			// kam
 "Kannada",			// kan
 "Karen",			// kar
 "Kashmiri",		// kas
 "Kanuri",			// kau
 "Kawi",			// kaw
 "Kazakh",			// kaz
 "Kabardian",		// kbd
 "Khasi",			// kha
 "Khoisan",			// khi
 "Khmer",			// khm
 "Khotanese",		// kho
 "Kikuyu",			// kik
 "Kinyarwanda",		// kin
 "Kirghiz",			// kir
 "Kimbundu",		// kmb
 "Konkani",			// kok
 "Komi",			// kom
 "Kongo",			// kon
 "Korean",			// kor
 "Kosraean",		// kos
 "Kpelle",			// kpe
 "Karachay-Balkar",	// krc
 "Kru",				// kro
 "Kurukh",			// kru
 "Kuanyama",		// kua
 "Kumyk",			// kum
 "Kurdish",			// kur
 "Kutenai",			// kut
 "Ladino",			// lad
 "Lahnda",			// lah
 "Lamba",			// lam
 "Lao",				// lao
 "Latin",			// lat
 "Latvian",			// lav
 "Lezghian",		// lez
 "Limburgan",		// lim
 "Lingala",			// lin
 "Lithuanian",		// lit
 "Mongo",			// lol
 "Lozi",			// loz
 "Luxembourgish",	// ltz
 "Luba-Lulua",		// lua
 "Luba-Katanga",	// lub
 "Ganda",			// lug
 "Luiseno",			// lui
 "Lunda",			// lun
 "Luo (Kenya and Tanzania)",// luo
 "lushai",			// lus
 "Macedonian",		// mac
 "Madurese",		// mad
 "Magahi",			// mag
 "Marshallese",		// mah
 "Maithili",		// mai
 "Makasar",			// mak
 "Malayalam",		// mal
 "Mandingo",		// man
 "Maori",			// mao
 "Austronesian",	// map
 "Marathi",			// mar
 "Masai",			// mas
 "Malay",			// may
 "Moksha",			// mdf
 "Mandar",			// mdr
 "Mende",			// men
 "Irish (middle age)",	// mga
 "Mi'kmaq",			// mic
 "Minangkabau",		// min
 "Miscellaneous",	// mis
 "Mon-Khmer",		// mkh
 "Malagasy",		// mlg
 "Maltese",			// mlt
 "Manchu",			// mnc
 "Manipuri",		// mni
 "Manobo",			// mno
 "Mohawk",			// moh
 "Moldavian",		// mol
 "Mongolian",		// mon
 "Mossi",			// mos
 "Mirandese",		// mwl
 "Multiple languages",	// mul
 "Munda",			// mun
 "Creek",			// mus
 "Marwari",			// mwr
 "Mayan",			// myn
 "Erzya",			// myv
 "Nahuatl",			// nah
 "North American Indian",	// nai
 "Neapolitan",		// nap
 "Nauru",			// nau
 "Navajo",			// nav
 "Ndebele, South",	// nbl
 "Ndebele, North",	// nde
 "Ndonga",			// ndo
 "Low German",		// nds
 "Nepali",			// nep
 "Nepal Bhasa",		// new
 "Nias",			// nia
 "Niger-Kordofanian",	// nic
 "Niuean",			// niu
 "Norwegian Nynorsk",	// nno
 "Norwegian Bokmal",// nob
 "Nogai",			// nog
 "Norse, Old",		// non
 "Norwegian",		// nor
 "Northern Sotho",	// nso
 "Nubian languages",// nub
 "Classical Newari",// nwc
 "Chichewa",		// nya
 "Nyamwezi",		// nym
 "Nyankole",		// nyn
 "Nyoro",			// nyo
 "Nzima",			// nzi
 "Occitan",			// oci
 "Ojibwa",			// oji
 "Oriya",			// ori
 "Oromo",			// orm
 "Osage",			// osa
 "Ossetian",		// oss
 "Turkish, Ottoman",	// ota
 "Otomian languages",	// oto
 "Papuan",			// paa
 "Pangasinan",		// pag
 "Pahlavi",			// pal
 "Pampanga",		// pam
 "Panjabi",			// pan
 "Papiamento",		// pap
 "Palauan",			// pau
 "Persian, Old (B.C.)",// peo
 "Persian",			// per
 "Philippine",		// phi
 "Phoenician",		// phn
 "Pali",			// pli
 "Polish",			// pol
 "Pohnpeian",		// pon
 "Portuguese",		// por
 "Prakrit",			// pra
 "Provençal (old)",// pro
 "Pushto",			// pus
 "Quechua",			// que
 "Rajasthani",		// raj
 "Rapanui",			// rap
 "Rarotongan",		// rar
 "Romance (Other)",	// roa
 "Raeto-Romance",	// roh
 "Romany",			// rom
 "Rundi",			// run
 "Aromanian",		// rup
 "Russian",			// rus
 "Sandawe",			// sad
 "Sango",			// sag
 "Yakut",			// sah
 "South American Indian",	// sai
 "Salishan",		// sal
 "Samaritan Aramaic",	// sam
 "Sanskrit",		// san
 "Sasak",			// sas
 "Santali",			// sat
 "Serbian",			// srp
 "Sicilian",		// scn
 "Scots",			// sco
 "Selkup",			// sel
 "Semitic",			// sem
 "Irish (old)",		// sga
 "Sign Languages",	// sgn
 "Shan",			// shn
 "Sidamo",			// sid
 "Sinhalese",		// sin
 "Siouan languages",	// sio
 "Sino-Tibetan",	// sit
 "Slavic",			// sla
 "Slovak",			// slo
 "Slovenian",		// slv
 "Southern Sami",	// sma
 "Northern Sami",	// sme
 "Sami languages",	// smi
 "Lule Sami",		// smj
 "Inari Sami",		// smn
 "Samoan",			// smo
 "Skolt Sami",		// sms
 "Shona",			// sna
 "Sindhi",			// snd
 "Soninke",			// snk
 "Sogdian",			// sog
 "Somali",			// som
 "Songhai",			// son
 "Sotho, Southern",// sot
 "Spanish",			// spa
 "Sardinian",		// srd
 "Serer",			// srr
 "Nilo-Saharan",	// ssa
 "Swati",			// ssw
 "Sukuma",			// suk
 "Sundanese",		// sun
 "Susu",			// sus
 "Sumerian",		// sux
 "Swahili",			// swa
 "Swedish",			// swe
 "Syriac",			// syr
 "Tahitian",		// tah
 "Tai",				// tai
 "Tamil",			// tam
 "Tatar",			// tat
 "Telugu",			// tel
 "Timne",			// tem
 "Tereno",			// ter
 "Tetum",			// tet
 "Tajik",			// tgk
 "Tagalog",			// tgl
 "Thai",			// tha
 "Tibetan",			// tib
 "Tigre",			// tig
 "Tigrinya",		// tir
 "Tiv",				// tiv
 "Tokelau",			// tkl
 "Klingon",			// tlh
 "Tlingit",			// tli
 "Tamashek",		// tmh
 "Tonga (Nyasa)",	// tog
 "Tonga (Tonga Islands)",	// ton
 "Tok Pisin",		// tpi
 "Tsimshian",		// tsi
 "Tswana",			// tsn
 "Tsonga",			// tso
 "Turkmen",			// tuk
 "Tumbuka",			// tum
 "Tupi",			// tup
 "Turkish",			// tur
 "Altaic",			// tut
 "Tuvalu",			// tvl
 "Twi",				// twi
 "Tuvinian",		// tyv
 "Udmurt",			// udm
 "Ugaritic",		// uga
 "Uighur",			// uig
 "Ukrainian",		// ukr
 "Umbundu",			// umb
 "Undetermined",	// und
 "Urdu",			// urd
 "Uzbek",			// uzb
 "Vai",				// vai
 "Venda",			// ven
 "Vietnamese",		// vie
 "Volapük",		// vol
 "Votic",			// vot
 "Wakashan",		// wak
 "Walamo",			// wal
 "Waray",			// war
 "Washo",			// was
 "Welsh",			// wel
 "Sorbian",			// wen
 "Walloon",			// wln
 "Wolof",			// wol
 "Kalmyk",			// xal
 "Xhosa",			// xho
 "Yao",				// yao
 "Yapese",			// yap
 "Yiddish",			// yid
 "Yoruba",			// yor
 "Yupik",			// ypk
 "Zapotec",			// zap
 "Zenaga",			// zen
 "Zhuang",			// zha
 "Zande",			// znd
 "Zulu",			// zul
 "Zuni",			// zun
 "Romanian",		// rum
 "Portuguese (BR)",	// pob
 "Montenegrin",		// mne
 "Chinese (traditional)",	// zht
 "Chinese (bilingual)",	// zhe
 "Portuguese (MZ)",		// pom
 "Extremaduran",		// ext
 NULL
};

typedef struct ffmplang_langcode_conv_s {
	const char *alt_lang_code; // alternative language code ("T" or ISO639-1)
	const char *eq_lang_code;  // equivalent language code (ISO 639-2 / "B")
}ffmplang_langcode_conv_s;

// "T" (terminology) to "B" (bibliographic) conversions ("B" is the primary/used in the above fields)
static const struct ffmplang_langcode_conv_s mpxplay_ffmplang_iso639_TtoB_convs[] =
{
	{"sqi", "alb"},  // Albanian (T) -> Albanian (B)
	{"hye", "arm"},  // Armenian (T) -> Armenian (B)
	{"eus", "baq"},  // Basque (T) -> Basque (B)
	{"mya", "bur"},  // Burmese (T) -> Burmese (B)
	{"zho", "chi"},  // Chinese (simplified) (T) -> Chinese (simplified) (B)
	{"ces", "cze"},  // Czech (T) -> Czech (B)
	{"nld", "dut"},  // Dutch (T) -> Dutch (B)
	{"fra", "fre"},  // French (T) -> French (B)
	{"kat", "geo"},  // Georgian (T) -> Georgian (B)
	{"deu", "ger"},  // German (T) -> German (B)
	{"ell", "gre"},  // Greek (T) -> Greek (B)
	{"fas", "per"},  // Persian (T) -> Persian (B)
	{"bod", "tib"},  // Tibetan (T) -> Tibetan (B)
	{"cym", "wel"},  // Welsh (T) -> Welsh (B)
	{"isl", "ice"},  // Icelandic (T) -> Icelandic (B)
	{"mkd", "mac"},  // Macedonian (T) -> Macedonian (B)
	{"mri", "mao"},  // Maori (T) -> Maori (B)
	{"msa", "may"},  // Malay (T) -> Malay (B)
	{"ron", "rum"},  // Romanian (T) -> Romanian (B)
	{"slk", "slo"},  // Slovak (T) -> Slovak (B)
	{NULL, NULL}
};

// ISO 639-1 to ISO 639-2 ("B") conversions
static const struct ffmplang_langcode_conv_s mpxplay_ffmplang_iso639_1to2_convs[] =
{
	{"aa", "aar"},  // Afar
	{"ab", "abk"},  // Abkhazian
	{"af", "afr"},  // Afrikaans
	{"ak", "aka"},  // Akan
	{"sq", "alb"},  // Albanian
	{"am", "amh"},  // Amharic
	{"ar", "ara"},  // Arabic
	{"an", "arg"},  // Aragonese
	{"hy", "arm"},  // Armenian
	{"as", "asm"},  // Assamese
	//{"at", "ast"},  // Asturian   // it's not in the ISO standard
	{"av", "ava"},  // Avaric
	{"ae", "ave"},  // Avestan
	{"ay", "aym"},  // Aymara
	{"az", "aze"},  // Azerbaijani
	{"ba", "bak"},  // Bashkir
	{"bm", "bam"},  // Bambara
	{"eu", "baq"},  // Basque
	{"be", "bel"},  // Belarusian
	{"bn", "ben"},  // Bengali
	{"bh", "bih"},  // Bihari languages
	{"bi", "bis"},  // Bislama
	{"bs", "bos"},  // Bosnian
	{"br", "bre"},  // Breton
	{"bg", "bul"},  // Bulgarian
	{"my", "bur"},  // Burmese
	{"ca", "cat"},  // Catalan
	{"ch", "cha"},  // Chamorro
	{"ce", "che"},  // Chechen
	{"zh", "chi"},  // Chinese
	{"cu", "chu"},  // Church Slavic
	{"cv", "chv"},  // Chuvash
	{"kw", "cor"},  // Cornish
	{"co", "cos"},  // Corsican
	{"cr", "cre"},  // Cree
	{"cs", "cze"},  // Czech
	{"da", "dan"},  // Danish
	{"dv", "div"},  // Divehi
	{"nl", "dut"},  // Dutch
	{"dz", "dzo"},  // Dzongkha
	{"el", "gre"},  // Greek, Modern
	{"en", "eng"},  // English
	{"eo", "epo"},  // Esperanto
	{"et", "est"},  // Estonian
	{"ee", "ewe"},  // Ewe
	{"fo", "fao"},  // Faroese
	{"fj", "fij"},  // Fijian
	{"fi", "fin"},  // Finnish
	{"fr", "fre"},  // French
	{"fy", "fry"},  // Western Frisian
	{"ff", "ful"},  // Fulah
	{"ka", "geo"},  // Georgian
	{"de", "ger"},  // German
	{"gd", "gla"},  // Gaelic
	{"ga", "gle"},  // Irish
	{"gl", "glg"},  // Galician
	{"gv", "glv"},  // Manx
	{"gn", "grn"},  // Guarani
	{"gu", "guj"},  // Gujarati
	{"ht", "hat"},  // Haitian
	{"ha", "hau"},  // Hausa
	{"he", "heb"},  // Hebrew
	{"hz", "her"},  // Herero
	{"hi", "hin"},  // Hindi
	{"ho", "hmo"},  // Hiri Motu
	{"hr", "hrv"},  // Croatian
	{"hu", "hun"},  // Hungarian
	{"ig", "ibo"},  // Igbo
	{"is", "ice"},  // Icelandic
	{"io", "ido"},  // Ido
	{"ii", "iii"},  // Sichuan Yi
	{"iu", "iku"},  // Inuktitut
	{"ie", "ile"},  // Interlingue
	{"ia", "ina"},  // Interlingua
	{"id", "ind"},  // Indonesian
	{"ik", "ipk"},  // Inupiaq
	{"it", "ita"},  // Italian
	{"jv", "jav"},  // Javanese
	{"ja", "jpn"},  // Japanese
	{"kl", "kal"},  // Kalaallisut
	{"kn", "kan"},  // Kannada
	{"ks", "kas"},  // Kashmiri
	{"kr", "kau"},  // Kanuri
	{"kk", "kaz"},  // Kazakh
	{"km", "khm"},  // Khmer
	{"ki", "kik"},  // Kikuyu
	{"rw", "kin"},  // Kinyarwanda
	{"ky", "kir"},  // Kirghiz
	{"kv", "kom"},  // Komi
	{"kg", "kon"},  // Kongo
	{"ko", "kor"},  // Korean
	{"kj", "kua"},  // Kuanyama
	{"ku", "kur"},  // Kurdish
	{"lo", "lao"},  // Lao
	{"la", "lat"},  // Latin
	{"lv", "lav"},  // Latvian
	{"li", "lim"},  // Limburgan
	{"ln", "lin"},  // Lingala
	{"lt", "lit"},  // Lithuanian
	{"lb", "ltz"},  // Luxembourgish
	{"lg", "lug"},  // Ganda
	{"mk", "mac"},  // Macedonian
	{"mh", "mah"},  // Marshallese
	{"ml", "mal"},  // Malayalam
	{"mi", "mao"},  // Maori
	{"mr", "mar"},  // Marathi
	{"ms", "may"},  // Malay
	{"mg", "mlg"},  // Malagasy
	{"mt", "mlt"},  // Maltese
	{"ma", "mni"},  // Manipuri
	{"mo", "mol"},  // Moldavian
	{"mn", "mon"},  // Mongolian
	{"na", "nau"},  // Nauru
	{"nv", "nav"},  // Navajo
	{"ng", "ndo"},  // Ndonga
	{"ne", "nep"},  // Nepali
	{"nn", "nno"},  // Norwegian
	{"nb", "nob"},  // Norwegian
	{"no", "nor"},  // Norwegian
	{"ny", "nya"},  // Chichewa
	{"oc", "oci"},  // Occitan
	{"oj", "oji"},  // Ojibwa
	{"or", "ori"},  // Oriya
	{"om", "orm"},  // Oromo
	{"os", "oss"},  // Ossetian
	{"pa", "pan"},  // Panjabi
	{"fa", "per"},  // Persian
	{"pi", "pli"},  // Pali
	{"pl", "pol"},  // Polish
	{"pt", "por"},  // Portuguese
	{"ps", "pus"},  // Pushto
	{"qu", "que"},  // Quechua
	{"rm", "roh"},  // Raeto-Romance
	{"rn", "run"},  // Rundi
	{"ru", "rus"},  // Russian
	{"sg", "sag"},  // Sango
	{"sa", "san"},  // Sanskrit
	{"sr", "srp"},  // Serbian
	{"si", "sin"},  // Sinhalese
	{"sk", "slo"},  // Slovak
	{"sl", "slv"},  // Slovenian
	{"se", "sme"},  // Northern
	{"sm", "smo"},  // Samoan
	{"sn", "sna"},  // Shona
	{"sd", "snd"},  // Sindhi
	{"so", "som"},  // Somali
	{"st", "sot"},  // Sotho, Southern
	{"es", "spa"},  // Spanish
	{"sc", "srd"},  // Sardinian
	{"ss", "ssw"},  // Swati
	{"su", "sun"},  // Sundanese
	{"sw", "swa"},  // Swahili
	{"sv", "swe"},  // Swedish
	{"sy", "syr"},  // Syriac
	{"ty", "tah"},  // Tahitian
	{"ta", "tam"},  // Tamil
	{"tt", "tat"},  // Tatar
	{"te", "tel"},  // Telugu
	{"tg", "tgk"},  // Tajik
	{"tl", "tgl"},  // Tagalog
	{"th", "tha"},  // Thai
	{"bo", "tib"},  // Tibetan
	{"ti", "tir"},  // Tigrinya
	{"to", "ton"},  // Tonga
	{"tn", "tsn"},  // Tswana
	{"ts", "tso"},  // Tsonga
	{"tk", "tuk"},  // Turkmen
	{"tr", "tur"},  // Turkish
	{"tw", "twi"},  // Twi
	{"ug", "uig"},  // Uighur
	{"uk", "ukr"},  // Ukrainian
	{"ur", "urd"},  // Urdu
	{"uz", "uzb"},  // Uzbek
	{"ve", "ven"},  // Venda
	{"vi", "vie"},  // Vietnamese
	{"vo", "vol"},  // Volapk
	{"cy", "wel"},  // Welsh
	{"wa", "wln"},  // Walloon
	{"wo", "wol"},  // Wolof
	{"xh", "xho"},  // Xhosa
	{"yi", "yid"},  // Yiddish
	{"yo", "yor"},  // Yoruba
	{"za", "zha"},  // Zhuang
	{"zu", "zul"},  // Zulu
	{"ro", "rum"},  // Romanian
	{"pb", "pob"},  // Portuguese
	{"me", "mne"},  // Montenegrin
	{"zt", "zht"},  // Chinese
	{"ze", "zhe"},  // Chinese
	{"pm", "pom"},  // Portuguese
	{"ex", "ext"},  // Extremaduran
	{NULL, NULL}
};

static const char *ffmplang_search_in_TtoB_convs(const char *langcode_T_to_search)
{
	const struct ffmplang_langcode_conv_s *iso639_TtoB_ptr;
	const char *langcode_B = NULL;
	mpxp_uint32_t langcode_search_in_num;
	char strtmp[4];

	pds_strcpy(strtmp, (char *)langcode_T_to_search); // length of input string is already checked against overflow
	pds_str_lowercase(strtmp);
	langcode_search_in_num = PDS_GETB_LEU32(&strtmp[0]); // storing string and comparing in uint32 for faster search

	iso639_TtoB_ptr = &mpxplay_ffmplang_iso639_TtoB_convs[0];
	do{
		const char *langcode_t = iso639_TtoB_ptr->alt_lang_code;
		if(!langcode_t)
			break;
		if(PDS_GETB_LEU32(langcode_t) == langcode_search_in_num)
		{
			langcode_B = iso639_TtoB_ptr->eq_lang_code;
			break;
		}
		iso639_TtoB_ptr++;
	}while(1);

	return langcode_B;
}

// convert ISO639-1 (2 chars codes) to ISO639-2 (3 chars codes) (like "hu" to "hun")
const char *mpxplay_ffmplang_search_in_iso639_1to2_convs(const char *langcode_iso639_1_to_search)
{
	const struct ffmplang_langcode_conv_s *iso639_1to2_ptr;
	const char *langcode_B = NULL;
	mpxp_uint16_t langcode_search_in_num;
	char strtmp[4];

	if(pds_strlen((char *)langcode_iso639_1_to_search) != INFFMPG_LANGCODE_ISO6391_LEN)
		return langcode_B;

	pds_strcpy(strtmp, (char *)langcode_iso639_1_to_search); // length of input string is already checked against overflow
	pds_str_lowercase(strtmp);
	langcode_search_in_num = PDS_GETB_LEU16(&strtmp[0]); // storing string and comparing in uint16 for faster search

	iso639_1to2_ptr = &mpxplay_ffmplang_iso639_1to2_convs[0];
	do{
		const char *langcode_1 = iso639_1to2_ptr->alt_lang_code;
		if(!langcode_1)
			break;
		if(PDS_GETB_LEU16(langcode_1) == langcode_search_in_num)
		{
			langcode_B = iso639_1to2_ptr->eq_lang_code;
			break;
		}
		iso639_1to2_ptr++;
	}while(1);

	return langcode_B;
}

// get language name by language code ("hun" to "Hungarian")
const char *mpxplay_ffmplang_getlangname_by_langcode(const char *lang_code_to_search)
{
	const char **lang_codes_ptr, **lang_names_ptr, *lang_cv_code, *lang_name = NULL;
	mpxp_uint32_t langcode_search_in_num;
	char strtmp[4];

	if(!lang_code_to_search || !lang_code_to_search[0])
		return lang_name;

	lang_cv_code = mpxplay_ffmplang_search_in_iso639_1to2_convs(lang_code_to_search);
	if(lang_cv_code)                             // ISO639-1 code has found, converted to an ISO639-2 code
		lang_code_to_search = lang_cv_code;
	else
	{
		if(pds_strlen((char *)lang_code_to_search) != INFFMPG_LANGCODE_ISO6392_LEN) // unnecessary to search for a string with a non ISO639-2 length
			return lang_name;
		lang_cv_code = ffmplang_search_in_TtoB_convs(lang_code_to_search);
		if(lang_cv_code)
			lang_code_to_search = lang_cv_code;
	}

	pds_strcpy(strtmp, (char *)lang_code_to_search);     // length of input string is already checked against overflow
	pds_str_lowercase(strtmp);                           // our language codes are stored in lower case format
	langcode_search_in_num = PDS_GETB_LEU32(&strtmp[0]); // storing string and comparing in uint32 for faster search

	lang_codes_ptr = &mpxplay_ffmplang_language_codes[0];
	lang_names_ptr = &mpxplay_ffmplang_language_names[0];
	do{
		const char *langcode = *lang_codes_ptr;
		if(!langcode)
			break;
		if(PDS_GETB_LEU32(langcode) == langcode_search_in_num)
		{
			lang_name = *lang_names_ptr;
			break;
		}
		lang_codes_ptr++; lang_names_ptr++;
	}while(1);

	return lang_name;
}

// returns begin pointers of language_codes and language_names and the number of elements
int mpxplay_ffmplang_get_language_fields(const char ***lang_codes_ptr, const char ***lang_names_ptr)
{
	if(!lang_codes_ptr || !lang_names_ptr)
		return 0;

	*lang_codes_ptr = &mpxplay_ffmplang_language_codes[0];
	*lang_names_ptr = &mpxplay_ffmplang_language_names[0];

	return (sizeof(mpxplay_ffmplang_language_codes) / sizeof(const char *) - 1); // -1 is the NULL terminator
}

#endif // MPXPLAY_LINK_INFILE_FF_MPEG
