/*
 * a packet driver shim for picoTCP
 * Copyright (C) 2015 Mateusz Viste
 *
 * http://picotcp4dos.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include <dos.h>
#include <string.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h> /* malloc(), free() */
#include "picotcp.h"

#include "pktdrv.h" /* include self for control */


/* uncomment the line below for debug outputs */
/* #define DEBUG printf */

/* BUFSIZE is the size of our single-packet buffer for incoming packets */
#define INBUFFSIZE 1520

/* if DEBUG() not defined, use a dummy declaration */
#ifndef DEBUG
  #define DEBUG(...)
#endif


/* a few global variables and packet buffer */
struct picodos_t {
  short inbufferlen;         /* length of the frame in buffer, 0 means "free" */
  unsigned short pkthandle;  /* handler returned by the packet driver */
  unsigned short pktint;     /* software interrupt of the packet driver */
  unsigned char inbuffer[INBUFFSIZE];
} *buff = NULL;


/* releases the handle previously registered with the packet driver */
static int pktdrv_releasetype(void) {
  union REGS inregs, outregs;
  struct SREGS segregs;

  inregs.h.ah = 0x3;
  inregs.x.bx = buff->pkthandle;
  int86x(buff->pktint, &inregs, &outregs, &segregs);

  if (outregs.x.cflag != 0) {
    return(-1);
  } else {
    return(0);
  }
}


/* this function is called two times by the packet driver. One time for
 * telling that a packet is incoming, and how big it is, so the application
 * can prepare a buffer for it and hand it back to the packet driver. the
 * second call is just let know that the frame has been copied into the
 * buffer.
 * even though this is properly declared as an interrupt routine, the epilog
 * have to be hand-tuned (this has been lurked from mTCP). */
static void far interrupt pktdrv_recv(union INTPACK r) {
  /* DEBUG("pktdrv_recv() called with AX=%u / CX=%u\n", r.w.ax, r.w.cx); */
  if (r.w.ax == 0) { /* first call: the packet driver needs a buffer */
    /* DEBUG("pktdrv requests %d bytes for incoming frame\n", r.w.cx); */
    if ((r.w.cx > INBUFFSIZE) || (buff->inbufferlen != 0)) { /* no room or packet too big */
      DEBUG("no room or packet too big\n");
      r.w.es = 0;
      r.w.di = 0;
    } else { /* we can handle it. give out the first buffer on the free stack */
      /* DEBUG("buffer assigned\n"); */
      buff->inbufferlen = r.w.cx;
      buff->inbufferlen = 0 - buff->inbufferlen; /* switch value to negative until we receive actual data */
      r.w.es = FP_SEG(buff->inbuffer);
      r.w.di = FP_OFF(buff->inbuffer);
    }
  } else {
    /* DEBUG("pktdrv_recv() second call\n"); */
    /* this is the second pktdrv call - I am supposed to find a new packet
     * inside inbuffer. I switch inbufferlen back to a positive value so the
     * TCP/IP stack can see there's something available there. */
    buff->inbufferlen = 0 - buff->inbufferlen;
    /* DEBUG("a frame of %d bytes is available in buffer now\n", inbufferlen); */
  }

  /* custom epilog code (copied from mTCP). Rationale: "Some packet drivers
   * can handle the normal compiler generated epilog, but the Xircom PE3-10BT
   * drivers definitely can not." */
  _asm {
    pop ax
    pop ax
    pop es
    pop ds
    pop di
    pop si
    pop bp
    pop bx
    pop bx
    pop dx
    pop cx
    pop ax
    retf
  };
}


/* registers a packet driver handle to use on subsequent calls */
static int pktdrv_accesstype(void) {
  union REGS inregs, outregs;
  struct SREGS segregs;

  inregs.h.ah = 0x2;
  inregs.h.al = 0x1;
  inregs.x.bx = 0xFFFFu;
  inregs.h.dl = 0;
  segregs.ds = FP_SEG(NULL);
  inregs.x.si = FP_OFF(NULL);
  inregs.x.cx = 0;
  segregs.es = FP_SEG(pktdrv_recv);
  inregs.x.di = FP_OFF(pktdrv_recv);

  int86x(buff->pktint, &inregs, &outregs, &segregs);

  if (outregs.x.cflag) return(-1);
  buff->pkthandle = outregs.x.ax;
  return(0);
}


/************ PUBLIC FUNCTIONS ************/


/* get my own MAC addr. target MUST point to a space of at least 6 chars */
void pktdrv_getaddr(unsigned char *dst) {
  union REGS inregs, outregs;
  struct SREGS segregs;

  inregs.h.ah = 0x6;
  inregs.x.bx = buff->pkthandle;
  segregs.es = FP_SEG(dst);
  inregs.x.di = FP_OFF(dst);
  inregs.x.cx = 6;

  int86x(buff->pktint, &inregs, &outregs, &segregs);
}


int pktdrv_init(unsigned short pktintparam) {
  uint16_t far *intvect = (uint16_t far *)MK_FP(0, pktintparam * 4);
  uint16_t pktdrvfuncoffs = *intvect;
  uint16_t pktdrvfuncseg = *(intvect+1);
  char far *pktdrvfunc = (char far *)MK_FP(pktdrvfuncseg, pktdrvfuncoffs);

  pktdrvfunc += 3; /* skip three bytes of executable code */
  if (memcmp("PKT DRVR", pktdrvfunc, 8) != 0) return(-1);

  DEBUG("packet driver found at int 0x%02X\n", pktintparam);

  /* if buffer already allocated, quit */
  if (buff != NULL) return(-1);

  buff = calloc(sizeof(struct picodos_t), 1);
  if (buff == NULL) return(-1); /* out of memory */

  buff->pktint = pktintparam;
  return(pktdrv_accesstype());
}


/* clean up and quit */
void pktdrv_quit(void) {
  if (buff == NULL) return; /* not inited, abort */
  pktdrv_releasetype();
  free(buff);
  buff = NULL;
}


/* */
int pktdrv_send(struct pico_device *device, void *buffptr, int bufferlen) {
  union REGS inregs, outregs;
  struct SREGS segregs;
  unsigned char *buffer = (unsigned char *)buffptr;

  inregs.h.ah = 0x4;
  inregs.x.cx = bufferlen;

  inregs.x.si = FP_OFF(buffer);
  segregs.ds = FP_SEG(buffer);

  DEBUG("frame %d bytes sent by picoTCP: %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X..\n", bufferlen, buffer[0], buffer[1], buffer[2], buffer[3], buffer[4], buffer[5], buffer[6], buffer[7], buffer[8], buffer[9], buffer[10], buffer[11], buffer[12], buffer[13]);
  int86x(buff->pktint, &inregs, &outregs, &segregs);

  /* this should return either the length of the frame (on success), or 0 ("retry later") */
  return(bufferlen);
}


/* this is called by picoTCP periodically. if a frame is available, it should
 * be delivered to picoTCP via a pico_stack_recv() call. this function can
 * process more than one frame at once, but no more than loop_score. */
int pktdrv_poll(struct pico_device *device, int loop_score) {
  long res;
  _disable();
  if ((buff->inbufferlen > 0) && (loop_score > 0)) {
    loop_score -= 1;
    DEBUG("frame %d bytes recv by picoTCP: %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X..\n", buff->inbufferlen, buff->inbuffer[0], buff->inbuffer[1], buff->inbuffer[2], buff->inbuffer[3], buff->inbuffer[4], buff->inbuffer[5], buff->inbuffer[6], buff->inbuffer[7], buff->inbuffer[8], buff->inbuffer[9], buff->inbuffer[10], buff->inbuffer[11], buff->inbuffer[12], buff->inbuffer[13]);
    res = pico_stack_recv(device, buff->inbuffer, buff->inbufferlen);
    if (res != 0) {
      DEBUG("pico_stack_recv() returned a non-zero value: %ld\n", res);
    }
    buff->inbufferlen = 0;
  }
  _enable();
  return(loop_score);
}
