//    MARAUDER
//
//    genes.cpp: genetic evolution object routines
//
//    By Shawn Hargreaves, 1995.
//
//    C++ source code for djgpp, using the 
//    Allegro game programming library.


#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <stdio.h>
#include <limits.h>
#include <allegro.h>
#include "marauder.h"
#include "sprite.h"
#include "bullet.h"
#include "genes.h"


extern DATAFILE *dat;
extern BITMAP *s2;

extern multidir *alien_pics[];
extern int alien_col[];

int _line;


void ERROR(char *s)
{
    char buf[80];
    sprintf(buf, "Error at line %d:", _line);
    alert(buf, s, NULL, "OK", NULL, 13, 0);
}



int in_set(char c, char *set)
{
    while (*set) {
	if (*set == c)
	    return TRUE;
	set++;
    }
    return FALSE;
}



int read_int(char *buf, int len, int *amount_read, char *str, int *val, int low, int high)
{
    char b[80];
    int c;

    while ((*amount_read < len) && (in_set(buf[*amount_read], " \t\r\n"))) {
	if (buf[*amount_read] == '\r')
	    _line++;
	(*amount_read)++;
    }

    for (c=0; str[c]; c++) {
	if ((*amount_read >= len) || (str[c] != buf[*amount_read])) {
	    sprintf(b, "Expecting %s statement", str);
	    ERROR(b);
	    return FALSE;
	}
	(*amount_read)++;
    }

    c=0;
    while ((*amount_read < len) && 
	   (((buf[*amount_read] >= '0') && (buf[*amount_read] <= '9')) ||
	    (buf[*amount_read] == '-'))) {
	b[c] = buf[*amount_read];
	(*amount_read)++;
	c++;
    }

    b[c] = 0;
    *val = atoi(b); 

    if ((*val < low) || (*val > high) || (b[0] == 0)) {
	sprintf(b, "Expecting integer %d-%d", low, high);
	ERROR(b);
	return FALSE;
    }

    return TRUE;
}



int get_value(char *buf, int len, int *amount_read, char **table, int *val, int *val_flag)
{
    while ((*amount_read < len) && (in_set(buf[*amount_read], " \t")))
	(*amount_read)++;

    if (*amount_read >= len)
	return FALSE;

    if (((buf[*amount_read] >= '0') && (buf[*amount_read] <= '9')) ||
	(buf[*amount_read] == '-')) {
	*val_flag = TRUE;
	return read_int(buf, len, amount_read, "", val, INT_MIN, INT_MAX);
    }

    char b[40];
    int pos = 0;
    int c, c2;

    do {
	b[pos++] = buf[(*amount_read)++];
	b[pos] = 0;

	for (c=0; table[c]; c++) {
	    c2=0; 
	    while ((b[c2]==table[c][c2]) && (b[c2]))
		c2++;

	    if ((b[c2]==0) && (table[c][c2]==0)) {
		*val = c;
		*val_flag = FALSE;
		return TRUE; 
	    }
	}
    } while ((*amount_read < len) && (pos < 30));

    ERROR("Illegal Value");
    return FALSE;
}



int get_operator(char *buf, int len, int *amount_read, int *val)
{
    while ((*amount_read < len) && (in_set(buf[*amount_read], " \t")))
	(*amount_read)++;

    switch (buf[*amount_read]) {
	case '=':
	    *val = EQUAL;
	    break;
	case '!':
	    *val = NOT_EQUAL;
	    break;
	case '>':
	    *val = GREATER;
	    break;
	case '<':
	    *val = LESS;
	    break;
	case '}':
	    *val = GREAT_OR_EQUAL;
	    break;
	case '{':
	    *val = LESS_OR_EQUAL;
	    break;
	case '@':
	    *val = APPROXIMATE;
	    break;
	default:
	    ERROR("Illegal Operator");
	    return FALSE;
    }

    (*amount_read)++;
    return TRUE;
}



gene::gene()
{
    rule1.op = rule2.op = UNUSED;
    memset(&act, 0, sizeof(act));
    towards = -1;
    set_v1 = -2;
    set_v2 = -2;
}



int gene::read_action(char *buf, int len, int *amount_read)
{
    static char *table[] = {
	"Treasure",
	"Planet",
	"Ship",
	"Player",
	"Laser",
	"Missile",
	"Left",         // 6
	"Right",        // 7
	"Forwards",     // 8
	"Fire",         // 9
	"V1",           // 10
	"V2",           // 11
	NULL
    };

    int flag;
    int val;

    if (!get_value(buf, len, amount_read, table, &val, &flag))
	return FALSE;

    if (flag) {
	ERROR("Illegal action");
	return FALSE;
    }

    switch (val) {
	case 6:
	    if (!read_int(buf, len, amount_read, "=", &act.left, INT_MIN, INT_MAX))
		return FALSE;
	    break;
	case 7:
	    if (!read_int(buf, len, amount_read, "=", &act.right, INT_MIN, INT_MAX))
		return FALSE;
	    break;
	case 8:
	    if (!read_int(buf, len, amount_read, "=", &act.forwards, INT_MIN, INT_MAX))
		return FALSE;
	    break;
	case 9:
	    if (!read_int(buf, len, amount_read, "=", &act.fire, INT_MIN, INT_MAX))
		return FALSE;
	    break;
	case 10:
	    if (!read_int(buf, len, amount_read, "=", &set_v1, -1, INT_MAX))
		return FALSE;
	    break;
	case 11:
	    if (!read_int(buf, len, amount_read, "=", &set_v2, -1, INT_MAX))
		return FALSE;
	    break;
	default:
	    towards = val;
	    if (!read_int(buf, len, amount_read, "+", &to_angle, 0, 7))
		return FALSE;
	    if (!read_int(buf, len, amount_read, "=", &to_priority, INT_MIN, INT_MAX))
		return FALSE;
	    break;
    }

    return TRUE; 
}



int gene::read(char *buf, int len, int *amount_read)
{
    static char *table[] = {
	"TreasureDir",
	"PlanetDir",
	"ShipDir",
	"PlayerDir",
	"LaserDir",
	"MissileDir",
	"TreasureDist",
	"PlanetDist",
	"ShipDist",
	"PlayerDist",
	"LaserDist",
	"MissileDist",
	"BuyPlanet",
	"SellPlanet",
	"Money",
	"Cargo",
	"Damage",
	"V1",
	"V2",
	NULL
    };

    while ((*amount_read < len) && (in_set(buf[*amount_read], " \t\r\n"))) {
	if (buf[*amount_read] == '\r')
	    _line++;
	(*amount_read)++;
    }

    if (!get_value(buf, len, amount_read, table, &rule1.value1, &rule1.is_value1))
	return FALSE;
    if (!get_operator(buf, len, amount_read, &rule1.op))
	return FALSE;
    if (!get_value(buf, len, amount_read, table, &rule1.value2, &rule1.is_value2))
	return FALSE;

    while ((*amount_read < len) && (in_set(buf[*amount_read], " \t")))
	(*amount_read)++;

    if ((*amount_read < len) && (buf[*amount_read] == '&')) {
	(*amount_read)++;
	if (!get_value(buf, len, amount_read, table, &rule2.value1, &rule2.is_value1))
	    return FALSE;
	if (!get_operator(buf, len, amount_read, &rule2.op))
	    return FALSE;
	if (!get_value(buf, len, amount_read, table, &rule2.value2, &rule2.is_value2))
	    return FALSE;
    }

    while ((*amount_read < len) && (!in_set(buf[*amount_read], "\r\n"))) {
	while ((*amount_read < len) && (in_set(buf[*amount_read], " \t")))
	    (*amount_read)++;
	if (!read_action(buf, len, amount_read))
	    return FALSE;
    }

    return TRUE; 
}



int get_info_val(int val, info *inf)
{
    switch (val) {
	case TREASURE_DIR:  return inf->treasure.dir;
	case PLANET_DIR:    return inf->planet.dir;
	case SHIP_DIR:      return inf->ship.dir;
	case PLAYER_DIR:    return inf->player.dir;
	case LASER_DIR:     return inf->laser.dir;
	case MISSILE_DIR:   return inf->missile.dir;
	case TREASURE_DIST: return inf->treasure.dist >> FIX_TO_PIX_SHIFT;
	case PLANET_DIST:   return inf->planet.dist >> FIX_TO_PIX_SHIFT;
	case SHIP_DIST:     return inf->ship.dist >> FIX_TO_PIX_SHIFT;
	case PLAYER_DIST:   return inf->player.dist >> FIX_TO_PIX_SHIFT;
	case LASER_DIST:    return inf->laser.dist >> FIX_TO_PIX_SHIFT;
	case MISSILE_DIST:  return inf->missile.dist >> FIX_TO_PIX_SHIFT;
	case BUY_PLANET:    return inf->buy_planet;
	case SELL_PLANET:   return inf->sell_planet;
	case MONEY:         return inf->money;
	case CARGO:         return inf->cargo;
	case DAMAGE:        return inf->damage;
	case V1:            return inf->v1;
	case V2:            return inf->v2;
	default: return 0;
    }
}



int min(int x, int y)
{
    if (x < y)
	return x;
    else
	return y;
}



int max(int x, int y)
{
    if (x > y)
	return x;
    else
	return y;
}



int evaluate(int v1, int v2, int op)
{
    switch (op) {
	case EQUAL:             return (v1 == v2);
	case NOT_EQUAL:         return (v1 != v2);
	case GREATER:           return (v1 > v2);
	case LESS:              return (v1 < v2);
	case GREAT_OR_EQUAL:    return (v1 >= v2);
	case LESS_OR_EQUAL:     return (v1 <= v2);
	case APPROXIMATE:       return ((myabs(v1-v2) < 2) ||
					((min(v1,v2)==0) && (max(v1,v2)==7)));
    }
    return 0;
}



int evaluate(rule *r, info *inf)
{
    int val1, val2;

    if (!r->op)
	return TRUE;

    val1 = (r->is_value1) ? r->value1 : get_info_val(r->value1, inf);
    val2 = (r->is_value2) ? r->value2 : get_info_val(r->value2, inf);
    return evaluate(val1, val2, r->op);
}



void gene::evaluate(struct info *inf, action *a)
{
    if (!::evaluate(&rule1, inf))
	return;

    if (!::evaluate(&rule2, inf))
	return;

    a->left += act.left;
    a->right += act.right;
    a->forwards += act.forwards;
    a->fire += act.fire;

    if (towards != -1) {
	int angle = get_info_val(towards, inf);
	angle += to_angle;
	angle &= 7;

	if (angle == 0)
	    a->forwards += to_priority;
	else if (angle < 4)
	    a->right += to_priority;
	else
	    a->left += to_priority;
    }

    if (set_v1 != -2)
	if (set_v1 == -1)
	    inf->v1--;
	else
	    inf->v1 = set_v1;

    if (set_v2 != -2)
	if (set_v2 == -1)
	    inf->v2--;
	else
	    inf->v2 = set_v2;
}



geneinstance::geneinstance()
{
    speed = 0;                  // range 0-2
    turn_speed = 0;             // range 0-2
    shields = 0;                // range 0 upwards
    weapon = WEAPON_LASER;      // laser, proton gun or matter cannon
    rules = NULL;
}



geneinstance::geneinstance(geneinstance *other)
{
    speed = other->speed;
    turn_speed = other->turn_speed;
    weapon = other->weapon;
    shields = other->shields;
    rule_count = other->rule_count;
    if (rule_count > 0) {
	rules = new gene[rule_count];
	for (int c=0; c<rule_count; c++)
	    rules[c] = other->rules[c];
    }
    else
	rules = NULL;
}



geneinstance::~geneinstance()
{
    if (rules)
	delete rules;
}



int geneinstance::read(char *buf, int len, int *amount_read)
{
    if (rules) {
	delete rules;
	rules = NULL;
    }

    *amount_read = 0;

    if (!read_int(buf, len, amount_read, "Speed=", &speed, 0, 2))
	return FALSE;

    if (!read_int(buf, len, amount_read, "TurnSpeed=", &turn_speed, 0, 2))
	return FALSE;

    if (!read_int(buf, len, amount_read, "Weapon=", &weapon, 0, 2))
	return FALSE;

    if (!read_int(buf, len, amount_read, "Shields=", &shields, 0, INT_MAX))
	return FALSE;

    if (!read_int(buf, len, amount_read, "RuleCount=", &rule_count, 0, 99))
	return FALSE;

    rules = new gene[rule_count];
    for (int c=0; c<rule_count; c++)
	if (!rules[c].read(buf, len, amount_read))
	    return FALSE;

    return TRUE;
}



void geneinstance::evaluate(struct info *inf, action *act)
{
    act->left = act->right = act->forwards = act->fire = 0;
    for (int c=0; c<rule_count; c++)
	rules[c].evaluate(inf, act);
}



genepool::genepool()
{
    filename[0] = 0;
    gene_pos = 0;
    for (int c=0; c<ALIEN_COUNT; c++)
	g[c] = NULL;
}



genepool::~genepool()
{
    for (int c=0; c<ALIEN_COUNT; c++)
	if (g[c])
	    delete g[c];
}



int genepool::read(char *buf, int len)
{
    _line = 1;
    int pos = 0;
    int c = 0;

    while ((pos < len) && (in_set(buf[pos], " \t\r\n"))) {
	if (buf[pos] == '\r')
	    _line++;
	pos++;
    }

    while (pos < len) {
	if (c >= ALIEN_COUNT) {
	    ERROR("Too many gene definitions");
	    return FALSE;
	}

	int skip;
	g[c] = new geneinstance; 
	if (!g[c]->read(buf+pos, len-pos, &skip))
	    return FALSE;
	pos += skip;
	c++;

	while ((pos < len) && (in_set(buf[pos], " \t\r\n"))) {
	    if (buf[pos] == '\r')
		_line++;
	    pos++;
	}
    }

    pos=0;
    while (c < ALIEN_COUNT)
	g[c++] = new geneinstance(g[pos++]);

    return TRUE;
}



geneinstance *genepool::get_genes(multidir **spr, int *col)
{
    gene_pos++;
    if (gene_pos >= ALIEN_COUNT)
	gene_pos = 0;

    *spr = alien_pics[gene_pos];
    *col = alien_col[gene_pos];

    return g[gene_pos];
}



int my_button_proc(int msg, DIALOG *d, int c);
int load_list_proc(int msg, DIALOG *d, int c);
char *list_getter(int index, int *list_size);


static DIALOG load_dlg[] = {
    { d_text_proc, 20, 8, 102, 32, 160, -1, 0, 0, 0, 0, "Preset level:" },
    { d_text_proc, 184, 16, 64, 42, 160, -1, 0, 0, 0, 0, "Load AI from:" },
    { my_button_proc, 28, 40,  112, 32, 96, 0, '1', D_EXIT, 0, 0, "Novice" },
    { my_button_proc, 28, 80,  112, 32, 56, 0, '2', D_EXIT | D_GOTFOCUS, 0, 0, "Pilot" },
    { my_button_proc, 28, 120, 112, 32, 160, 0, '3', D_EXIT, 0, 0, "Master" },
    { my_button_proc, 28, 160, 112, 32, 24, 0, '4', D_EXIT, 0, 0, "Guru" },
    { load_list_proc, 192, 76, 96, 99, 32, 0, 0, D_EXIT, 0, 0, list_getter },
    { NULL, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, NULL }
};



int my_button_proc(int msg, DIALOG *d, int c)
{
    if (msg == MSG_DRAW) {
	text_mode(-1);
	textout_centre(screen, (FONT *)dat[BIG_FONT].dat, (char *)d->dp, 
		       d->x+d->w/2, d->y+8, d->fg);

	int c = (d->flags & D_GOTFOCUS) ? d->fg : getpixel(screen, d->x-1, d->y-1);
	rect(screen, d->x, d->y, d->x+d->w, d->y+d->h, c);

	return D_O_K; 
    }

    return d_button_proc(msg, d, c);
}



#define FLIST_SIZE      128

static char *fileptr;

typedef struct FLIST
{
   int size;
   char f[FLIST_SIZE][14];
} FLIST;

FLIST *flist = NULL;



void flist_putter(char *str, int attrib, int unused)
{
    attrib = attrib;

    int c, c2;
    char *s = get_filename(str);
    *(get_extension(s)-1) = 0;
    strupr(s);

    if ((flist->size < FLIST_SIZE) && (strcmp(s,".")!=0)) {
	for (c=0; c<flist->size; c++) {
	    if (strcmp(s,flist->f[c]) < 0)
		break;
	}

	for (c2=flist->size; c2 > c; c2--)
	    strcpy(flist->f[c2], flist->f[c2-1]);

	strcpy(flist->f[c], s);
	flist->size++;
    }
}



char *list_getter(int index, int *list_size)
{
    if (index < 0) {
	if (list_size)
	    *list_size = flist->size;
	return NULL;
    }
    return flist->f[index];
}



int load_list_proc(int msg, DIALOG *d, int c)
{
    if (msg==MSG_START)
	return D_O_K;

    if (msg==MSG_END) {
	if ((d->d1 >= 0) && (d->d1 < flist->size)) {
	    strcpy(fileptr, flist->f[d->d1]);
	    strcat(fileptr, ".MAR");
	}
	else
	    *fileptr = 0;
    }

    return d_list_proc(msg, d, c);
}



int genepool::load_genes()
{
    fileptr = filename;

    if (!flist)
	flist = (FLIST *)malloc(sizeof(FLIST));
    if (!flist) {
	alert(NULL,"Out of memory",NULL,"OK",NULL,13,0);
	return FALSE; 
    }
    flist->size = 0;
    for_each_file("*.MAR", FA_RDONLY | FA_ARCH, flist_putter, 0);

    blit((BITMAP *)dat[BACKGROUND_BMP].dat, screen, 0, 0, 0, 0, 640, 480);
    centre_dialog(load_dlg);
    int nowhere;
    dialog_message(load_dlg, MSG_DRAW, 0, &nowhere);

    do {
    } while (key[KEY_ENTER] || key[KEY_SPACE]);
    clear_keybuf();

    int result = do_dialog(load_dlg, find_dialog_focus(load_dlg));

    show_mouse(NULL);

    if (flist)
	free(flist);
    flist = NULL;

    switch (result) {
	case 2: 
	    filename[0] = 0; 
	    return read((char *)dat[PRESET1].dat, dat[PRESET1].size);
	case 3: 
	    filename[0] = 0; 
	    return read((char *)dat[PRESET2].dat, dat[PRESET2].size);
	case 4: 
	    filename[0] = 0; 
	    return read((char *)dat[PRESET3].dat, dat[PRESET3].size);
	case 5: 
	    filename[0] = 0; 
	    return read((char *)dat[PRESET4].dat, dat[PRESET4].size);
	case -1:
	    return FALSE;
    }

    int size = file_size(filename);

    PACKFILE *f = pack_fopen(filename, F_READ);
    if (!f) {
	alert("Error opening", filename, NULL, "Oh dear", NULL, 13, 0);
	return FALSE;
    }

    char *buf = (char *)malloc(size);
    if (pack_fread(buf, size, f) != size) {
	pack_fclose(f);
	free(buf);
	alert("Error reading", filename, NULL, "Oh dear", NULL, 13, 0);
	return FALSE;
    } 

    pack_fclose(f);
    int ret = read(buf, size);
    free(f);
    return ret;
}
