;*******************************************;
; WASM Video Module, Mode Routines          ;
; By Eric Tauck                             ;
;                                           ;
; Defines:                                  ;
;                                           ;
;   VidTyp  return the adapter type         ;
;   ModSet  switch to a mode                ;
;   VidOpn  initialize the video routines   ;
;   VidClo  uninitialize the video routines ;
;                                           ;
; Requires:                                 ;
;                                           ;
;   VIDEO1.ASM                              ;
;*******************************************;

        call    VidTyp          ;get video type
        mov     _vid_type, al   ;save it
        jmp     _video2_end

_vid_type       DB      ?       ;video type

;--- mode data

_vid_oldm       DB      0       ;mode before opening
_vid_oldg       DB      0       ;hardware mode (i.e. saved graphics mode)

;--- adapter types

MDA             EQU     1       ;monochrome display adapter
CGAMON          EQU     2       ;color graphics adapter (monochrome)
CGACOL          EQU     3       ;color graphics adapter
EGAMON          EQU     4       ;enhanced graphics adapter, monochrome
EGACOL          EQU     5       ;enhanced graphics adapter, color
VGAMON          EQU     6       ;video graphics array, monochrome
VGACOL          EQU     7       ;video graphics array, color
MCGAMON         EQU     8       ;multi-color graphics adapter, monochrome
MCGACOL         EQU     9       ;multi-color graphics adapter, color

;========================================
; Determine the adapter type from the
; current video mode. Used locally if the
; adapter type cannot be otherwise
; determined.
;
; Out: AL= adapter type; CY= set if error.

VidTypM PROC  NEAR
        mov     ah, 15          ;get video information
        int     10H             ;execute
        cmp     al, 7           ;check maximum valid mode
        ja      _vdtym1

;--- return mode from table

        mov     bl, al                  ;load BL
        sub     bh, bh                  ;BX has index
        mov     al, [_vid_modetrans+bx] ;return adapter type
        clc
        ret

;--- error

_vdtym1 sub     al, al  ;return zero
        stc
        ret

;--- video mode to adapter translation table, 0 to 7

_vid_modetrans  DB      CGAMON, CGACOL, CGAMON, CGACOL
                DB      CGACOL, CGAMON, CGACOL, MDA
        ENDP

;========================================
; Determine the current adapter type.
;
; Out: AL= adapter type; CY= set if
;      error.

VidTyp  PROC   NEAR

;--- try reading display combination code

        mov     ax, 1A00H       ;get display combination code function
        int     10H             ;execute
        cmp     al, 1AH         ;check if function supported
        jne     _vdtyp1         ;jump if not

        cmp     bl, 12          ;check range
        ja      _vdtyp4         ;jump if error
        sub     bh, bh
        mov     al, [_vid_BIOStrans+bx] ;load adapter type
        or      al, al                  ;check if invalid type
        je      _vdtyp4
        cmp     al, CGACOL      ;check if CGA
        jne     _vdtyp3
        push    ax
        call    VidTypM         ;check mode
        pop     dx
        cmp     al, CGAMON      ;see if in mono mode
        je      _vdtyp3
        mov     al, dl          ;return original mode
        jmps    _vdtyp3

;--- combination code failed, try EGA query

_vdtyp1 mov     ah, 12H         ;EGA alternate function
        mov     bl, 10H         ;return information subfunction
        int     10H             ;execute
        cmp     bl, 10H         ;check if supported
        je      _vdtyp2         ;jump if not

        xor     al, 1           ;set ZF if mono
        mov     al, EGAMON      ;load mono type
        jz      _vdtyp3
        mov     al, EGACOL      ;load color type
        jmps    _vdtyp3

;--- EGA query failed, go by mode

_vdtyp2 call    VidTypM         ;determine adapter by mode
        jc      _vdtyp4

;--- finished

_vdtyp3 clc
        ret

;--- error

_vdtyp4 sub     al, al          ;return zero
        stc
        ret

;--- BIOS adapter type to adapter translation table, 0 to 12

_vid_BIOStrans  DB      0, MDA, CGACOL, 0, EGACOL, EGAMON, 0
                DB      VGAMON, VGACOL, 0, 0, MCGAMON, MCGACOL

        ENDP

;========================================
; Set the current mode.
;
; In: AL= mode
;
; Out: CY= set if error.

ModSet  PROC    NEAR
        push    ax
        call    ModGet          ;get mode
        mov     dl, al
        pop     ax
        cmp     dl, al          ;check if same
        je      _mdset6         ;exit if so

;--- find data for setting mode

        push    ax
        call    VidTyp          ;get the adapter type
        pop     bx
        jc      _mdset2         ;jump if error
        dec     ax
        xchg    ax, bx          ;swap mode and adapter
        sub     bh, bh
        shl     bx
        shl     bx                      ;adapter * 4, offset into index table
        mov     cx, [_vid_adpttabs+bx+2] ;load number of valid modes
        mov     bx, [_vid_adpttabs+bx]  ;load address of mode data

_mdset1 cmp     al, [bx]        ;check if mode matches
        je      _mdset3         ;jump if so
        add     bx, 4           ;next entry
        loop    _mdset1         ;loop back

;--- error

_mdset2 stc
        ret

;--- found mode data, set scan lines

_mdset3 push    ax              ;save mode

        mov     al, [bx+2]      ;load scan lines
        cmp     al, _vid_SLNONE ;check if set scan lines
        je      _mdset4         ;skip if not

        push    bx
        mov     ah, 12H         ;alternate select function
        mov     bl, 30H         ;set text scan lines subfunction
        int     10H             ;execute
        pop     bx

;--- set hardware mode

_mdset4 push    bx
        mov     ah, 0           ;set mode function
        mov     al, [bx+1]      ;load mode
        int     10H             ;execute
        pop     bx

;--- set font

        mov     al, [bx+3]      ;load font
        cmp     al, _vid_FNNONE ;check if set font
        je      _mdset5         ;skip if not

        mov     ah, 11H         ;character generator function
        sub     bl, bl          ;block zero
        int     10H             ;execute

;--- finished

_mdset5 call    ModGet          ;get mode
        pop     dx
        jc      _mdset2         ;exit if error
        cmp     dl, al          ;check if same
        jne     _mdset2         ;exit if error

_mdset6 call    VidInit         ;initialize data
        ret

;--- index table to mode switching data

_vid_adpttabs LABEL      WORD
        DW      OFFSET _vid_MDA, 1
        DW      OFFSET _vid_CGAMON, 2
        DW      OFFSET _vid_CGACOL, 2
        DW      OFFSET _vid_EGAMON, 2
        DW      OFFSET _vid_EGACOL, 4
        DW      OFFSET _vid_VGAMON, 5
        DW      OFFSET _vid_VGACOL, 14
        DW      OFFSET _vid_MCGAMON, 1
        DW      OFFSET _vid_MCGACOL, 1

;--- scan line constants

_vid_SL200      EQU     000H    ;200
_vid_SL350      EQU     001H    ;350
_vid_SL400      EQU     002H    ;400
_vid_SLNONE     EQU     0FFH    ;not defined

;--- font pointer constants

_vid_FN8X8      EQU     012H    ;8x8
_vid_FN8X14     EQU     011H    ;8x14
_vid_FN8X16     EQU     014H    ;8x16
_vid_FNNONE     EQU     0FFH    ;not defined

;--- MDA modes

_vid_MDA LABEL  BYTE
        DB      TEXT8025, 07H, _vid_SLNONE, _vid_FNNONE

;--- modes

_vid_CGAMON LABEL       BYTE
        DB      TEXT4025, 00H, _vid_SLNONE, _vid_FNNONE
        DB      TEXT8025, 02H, _vid_SLNONE, _vid_FNNONE

;--- CGACOL modes

_vid_CGACOL LABEL       BYTE
        DB      TEXT4025, 01H, _vid_SLNONE, _vid_FNNONE
        DB      TEXT8025, 03H, _vid_SLNONE, _vid_FNNONE

;--- EGAMON modes

_vid_EGAMON LABEL       BYTE
        DB      TEXT8025, 07H, _vid_SLNONE, _vid_FNNONE
        DB      TEXT8043, 07H, _vid_SLNONE, _vid_FN8X8

;--- EGACOL modes

_vid_EGACOL LABEL       BYTE
        DB      TEXT4025, 01H, _vid_SLNONE, _vid_FNNONE
        DB      TEXT4043, 01H, _vid_SLNONE, _vid_FN8X8
        DB      TEXT8025, 03H, _vid_SLNONE, _vid_FNNONE
        DB      TEXT8043, 03H, _vid_SLNONE, _vid_FN8X8

;--- VGAMON modes

_vid_VGAMON LABEL       BYTE
        DB      TEXT8021, 7, _vid_SL350, _vid_FN8X16
        DB      TEXT8025, 7, _vid_SL400, _vid_FNNONE
        DB      TEXT8028, 7, _vid_SL400, _vid_FN8X14
        DB      TEXT8043, 7, _vid_SL350, _vid_FN8X8
        DB      TEXT8050, 7, _vid_SL400, _vid_FN8X8

;--- VGACOL modes

_vid_VGACOL LABEL       BYTE
        DB      TEXT4012, 1, _vid_SL200, _vid_FN8X16
        DB      TEXT4014, 1, _vid_SL200, _vid_FN8X14
        DB      TEXT4021, 1, _vid_SL350, _vid_FN8X16
        DB      TEXT4025, 1, _vid_SL400, _vid_FNNONE
        DB      TEXT4028, 1, _vid_SL400, _vid_FN8X14
        DB      TEXT4043, 1, _vid_SL350, _vid_FN8X8
        DB      TEXT4050, 1, _vid_SL400, _vid_FN8X8
        DB      TEXT8012, 3, _vid_SL200, _vid_FN8X16
        DB      TEXT8014, 3, _vid_SL200, _vid_FN8X14
        DB      TEXT8021, 3, _vid_SL350, _vid_FN8X16
        DB      TEXT8025, 3, _vid_SL400, _vid_FNNONE
        DB      TEXT8028, 3, _vid_SL400, _vid_FN8X14
        DB      TEXT8043, 3, _vid_SL350, _vid_FN8X8
        DB      TEXT8050, 3, _vid_SL400, _vid_FN8X8

;--- MCGAMON modes

_vid_MCGAMON LABEL      BYTE
        DB      TEXT8025, 7, _vid_SLNONE, _vid_FNNONE

;--- MCGACOL modes

_vid_MCGACOL LABEL      BYTE
        DB      TEXT8025, 03H, _vid_SLNONE, _vid_FNNONE

        ENDP

;========================================
; Initialize video data. Preserve the
; current mode and switch to new mode.
;
; In: AL= text mode.
;
; Out: CY= set if error.

VidOpn  PROC   NEAR
        push    ax

;--- save the text mode

        call    ModGet          ;get mode
        mov     _vid_oldm, al   ;save it
        mov     ax, 1200H       ;enable palette setting
        jc      _vdopn1         ;jump if invalid text mode
        inc     al              ;disable palette setting

_vdopn1 cmp     _vid_type, VGAMON       ;check video type
        jb      _vdopn2                 ;skip if not VGA
        mov     bl, 31H                 ;subfunction
        int     10H                     ;execute

;--- save the hardware mode (in case invalid text mode above)

_vdopn2 mov     ah, 0FH         ;video information function
        int     10H
        mov     _vid_oldg, al   ;save hardware mode

;--- switch to the new mode

        pop     ax              ;restore mode
        call    ModSet          ;switch to mode
        jc      _vdopn4

        cmp     _vid_type, VGAMON       ;check video type
        jb      _vdopn3                 ;skip if not VGA

        mov     ax, 1201H       ;disable palette setting
        mov     bl, 31H         ;subfunction
        int     10H             ;execute
_vdopn3 clc
        ret

;--- error, close video

_vdopn4 call    VidClo          ;reset video
        stc
        ret
        ENDP

;========================================
; Close video. Restore old mode.

VidClo  PROC NEAR
        mov     al, _vid_oldm   ;symbolic mode
        or      al, al          ;check if valid
        jnz     _vdclo2

;--- restore unrecognized mode

        cmp     _vid_type, VGAMON       ;check video type
        jb      _vdclo1                 ;skip if not VGA

        mov     ax, 1200H       ;enable palette loading
        mov     bl, 31H         ;subfunction
        int     10H             ;execute

_vdclo1 mov     al, _vid_oldg   ;use saved hardware mode
        sub     ah, ah
        int     10H
        ret

;--- restore text mode

_vdclo2 call    ModSet          ;set mode

        cmp     _vid_type, VGAMON       ;check video type
        jb      _vdclo3                 ;skip if not VGA

        mov     ax, 1200H       ;enable palette loading
        mov     bl, 31H         ;subfunction
        int     10H             ;execute

_vdclo3 ret
        ENDP

_video2_end
