{***************************************************************************
 * VESA Generalized Timing Formula(GTF) calculator for VESA VBE 3.0 Driver *
 ***************************************************************************

 Last updated: 27.2.2011 [pascal]
 Source origin:
     Copyright (C) 2000-2011 SciTech Software
     Modified output for VGA13/VESADRV library by Martin Rehak
     Translated into pascal by Laaca

 Contact: rayer@seznam.cz, rehakm2@feld.cvut.cz
          laaca@seznam.cz
***************************************************************************}




{Define the structures for holding the horizontal and vertical
 CRTC parameters for a mode.

 Note: The sync timings are defined in both VGA compatible timings
 (sync start and sync end positions) and also in GTF compatible
 modes with the front porch, sync width and back porch defined.}


type

GTF_hCRTC = record
    hTotal:longint;
    hDisp:longint;
    hSyncStart:longint;
    hSyncEnd:longint;
    hFrontPorch:longint;
    hSyncWidth:longint;
    hBackPorch:longint;
end;

GTF_vCRTC = record
    vTotal:longint;                    {vertical total}
    vDisp:longint;                     {vertical displayed}
    vSyncStart:longint;                {vertical sync start}
    vSyncEnd:longint;                  {vertical sync end}
    vFrontPorch:longint;               {vertical front porch}
    vSyncWidth:longint;                {vertical sync width}
    vBackPorch:longint;                {vertical back porch}
end;


{Define the main structure for holding generated GTF timings}
GTF_timings = packed record
    h:gtf_hcrtc;                       {horizontalni CRTC parametry}
    v:gtf_vcrtc;                       {vertikalni CRTC parametry}
    hSyncPol:boolean;                  {horizontalni sync. polarita}
    vSyncPol:boolean;                  {vertikalni sync. polarita}
    interlace:boolean;                 {interlace ano/ne}
    vFreq:double;                      {vertikalni frekvence (Hz)}
    hFreq:double;                      {horizontalni frekvence (KHz)}
    dotClock:double;                   {Pixel clock (MHz)}
end;


{Define the structure for holding standard GTF formula constants}
GTF_constants = record
  margin:double;                    {jake % obrazovky zabira ramecek}
  cellGran:double;                  {granularita bunky textoveho znaku}
  minPorch:double;                  {Minimum front porch in lines/chars}
  vSyncRqd:double;                  {Width of V sync in lines}
  hSync:double;                     {Width of H sync as percent of total}
  minVSyncBP:double;                {Minimum vertical sync + back porch (us)}
  m:double;                         {Blanking formula gradient}
  c:double;                         {Blanking formula offset}
  k:double;                         {Blanking formula scaling factor}
  j:double;                         {Blanking formula scaling factor weight}
end;

const
GTF_lockVF  = 1;      {lock to vertical frequency}
GTF_lockHF = 2;       {lock to horizontal frequency}
GTF_lockPF  = 3;      {Lock to pixel clock frequency}
CFGFN = 'CRTC.CFG';


{------------------------- Global Variables --------------------------------}
gc:GTF_constants = (margin:1.8; cellgran:8; minPorch:1; vSyncRQD: 3;
                    hSync: 8; minVSyncBP:550; m:600; c:40; k:128; j:20);


Function Round(v:real):longint;
var l:longint;
begin
l:=Trunc(v+0.5);
if l<0 then Round:=l-1 else Round:=l;
end;


Procedure GetInternalConstants(var c:GTF_constants);
{ Vypocita celociselne zaokrouhlenou sadu GTF konstant. Tyto konstanty se lisi
  od skutecnych GTF konstant, ktere mohou byt nastaveny pro monitor.}
begin
c.margin:=GC.margin;
c.cellgran:=round(gc.cellGran);
c.minPorch:=round(gc.minPorch);
c.vSyncRQD:=round(gc.vSyncRQD);
c.hsync:=GC.hsync;
c.minVSyncBP:=GC.minvsyncBP;
if GC.k = 0 then c.k:=0.001 else c.k:=GC.k;
c.m:= (c.k/256) * GC.m;
c.c:= (GC.c - GC.j) * (c.k/256) + GC.j;
c.j:=GC.j;
end;


Function boby(b:boolean):byte;
begin
if b then boby:=1 else boby:=0;
end;


Procedure GTF_calcTimings(hPixels,vLines:double;freq:double;typ:byte;
               wantmargins,wantinterlace:boolean;var t:GTF_timings);
{Parameters:  hPixels  - X resolution
              vLines   - Y resolution
              freq     - Frequency (Hz, KHz or MHz depending on type)
              type     - 1 - vertical, 2 - horizontal, 3 - dot clock
              margins  - True if margins should be generated
              interlace - True if interlaced timings to be generated
              t        - Place to store the resulting timings

Description:  Calculates a set of GTF timing parameters given a specified
              resolution and vertical frequency. The horizontal frequency
              and dot clock will be automatically generated by this
              routines.

              For interlaced modes the CRTC parameters are calculated for
              a single field, so will be half what would be used in
              a non-interlaced mode.}

var
vFieldRate, hPeriod:double;
topMarginLines, botMarginLines:double;
leftMarginPixels, rightMarginPixels:double;
hPeriodEst, vSyncBP, vBackPorch:double;
vTotalLines, vFieldRateEst:double;
hTotalPixels, hTotalActivePixels, hBlankPixels:double;
idealDutyCycle,hSyncWidth,hSyncBP,hBackPorch:double;
idealHPeriod:double;
dotclock,vFreq,hFreq:double;
c:GTF_constants;
interlace:boolean;

begin
idealDutyCycle:=0;
vTotalLines:=0;
hPeriodEst:=0;
vSyncBP:=0;
vBackPorch:=0;
hPeriod:=0;


GetInternalConstants(c);
{Get rounded GTF constants used for internal calculations}

vFreq:=freq;
hFreq:=freq;
dotClock:=freq;
{Move input parameters into appropriate variables}

hPixels:=round(hPixels / c.cellGran) * c.cellGran;
{Round pixels to character cell granularity}


vFieldRate:=vFreq;
interlace:=false;
if (wantInterlace) then dotClock:=dotClock*2;
{For interlaced mode halve the vertical parameters, and double the required
 field refresh rate.}


if (wantMargins) then {Determine the lines for margins}
   begin
   topMarginLines:=round(c.margin / 100 * vLines);
   botMarginLines:=round(c.margin / 100 * vLines);
   end
   else begin
   topMarginLines:=0;
   botMarginLines:=0;
   end;

  if typ<>GTF_lockPF then
     BEGIN
     if typ=GTF_lockVF then
        begin
        hPeriodEst:=((1/vFieldRate)-(c.minVSyncBP/1000000))/(vLines+(2*topMarginLines)+c.minPorch+boby(interlace))*1000000;
        {Estimate the horizontal period}
        vSyncBP:=round(c.minVSyncBP / hPeriodEst);
        {Find the number of lines in vSync + back porch}
        end

     else if typ=GTF_lockHF then
        vSyncBP:=round((c.minVSyncBP * hFreq) / 1000);
        {Find the number of lines in vSync + back porch}


     vBackPorch:=vSyncBP - c.vSyncRqd;
     {Find the number of lines in the V back porch alone}


     vTotalLines:=vLines + topMarginLines + botMarginLines + vSyncBP + boby(interlace) + c.minPorch;
     {Find the total number of lines in the vertical period}

     if typ=GTF_lockVF then
        begin
        vFieldRateEst:=1000000 / (hPeriodEst * vTotalLines);
        {Estimate the vertical frequency}

        hPeriod:=(hPeriodEst * vFieldRateEst) / vFieldRate;
        {Find the actual horizontal period}

        vFieldRate:=1000000 / (hPeriod * vTotalLines);
        {Find the actual vertical field frequency}
        end
        else if typ=GTF_lockHF then vFieldRate:=(hFreq / vTotalLines) * 1000;
                {Find the actual vertical field frequency}
     END; {typ<>GTF_lockPF}

  if wantMargins then {Find the number of pixels in the left and right margins}
     begin
     leftMarginPixels:=round(hPixels * c.margin) / (100 * c.cellGran);
     rightMarginPixels:=round(hPixels * c.margin) / (100 * c.cellGran);
     end
     else begin
     leftMarginPixels:=0;
     rightMarginPixels:=0;
     end;

  hTotalActivePixels:=hPixels + leftMarginPixels + rightMarginPixels;
  {Find the total number of active pixels in image + margins}

  if typ=GTF_lockVF then idealDutyCycle:=c.c - ((c.m * hPeriod) / 1000)
     {Find the ideal blanking duty cycle}

  else if typ=GTF_lockHF then idealDutyCycle:=c.c - (c.m / hFreq)
     {Find the ideal blanking duty cycle}

  else if typ=GTF_lockPF then
     begin
     idealHPeriod:=(((c.c-100)+(sqrt(((100-c.c)*(100-c.c))+(0.4*c.m*(hTotalActivePixels+
                   rightMarginPixels+leftMarginPixels)/dotClock))))/(2*c.m))*1000;
     {Find ideal horizontal period from blanking duty cycle formula}
     idealDutyCycle:=c.c - ((c.m * idealHPeriod) / 1000);
     {Find the ideal blanking duty cycle}
     end;

  hBlankPixels:=round((hTotalActivePixels * idealDutyCycle) / ((100 - idealDutyCycle) * c.cellGran)) * c.cellGran;
  {Find the number of pixels in blanking time}
  hTotalPixels:=hTotalActivePixels + hBlankPixels;
  {Find the total number of pixels}
  hBackPorch:=round((hBlankPixels / 2) / c.cellGran) * c.cellGran;
  {Find the horizontal back porch}
  hSyncWidth:=round(((c.hSync/100) * hTotalPixels) / c.cellGran) * c.cellGran;
  {Find the horizontal sync width}
  hSyncBP:=hBackPorch + hSyncWidth;
  {Find the horizontal sync + back porch}

  if typ=GTF_lockPF then
     begin
     hFreq:=(dotClock / hTotalPixels) * 1000;
     {Find the horizontal frequency}
     vSyncBP:=round((c.minVSyncBP * hFreq) / 1000);
     {Find the number of lines in vSync + back porch}
     vBackPorch:=vSyncBP - c.vSyncRqd;
     {Find the number of lines in the V back porch alone}
     vTotalLines:=vLines + topMarginLines + botMarginLines + vSyncBP
        + boby(interlace) + c.minPorch;
     {Find the total number of lines in the vertical period}
     vFieldRate:=(hFreq / vTotalLines) * 1000;
     {Find the actual vertical field frequency}
     end
     else begin
     if typ=GTF_lockVF then hFreq:=1000 / hPeriod else
      {Find the horizontal frequency}
     if typ=GTF_lockHF then hPeriod:=1000 / hFreq;
      {Find the horizontal frequency}

     dotClock:=hTotalPixels / hPeriod;
     {Find the pixel clock frequency}
     end;


  t.vFreq:=vFieldRate;
  t.hFreq:=hFreq;
  t.dotClock:=dotClock;
  {Return the computed frequencies}

  t.h.hTotal:=Trunc(hTotalPixels);
  t.h.hDisp:=Trunc(hTotalActivePixels);
  t.h.hSyncStart:=t.h.hTotal - Trunc(hSyncBP);
  t.h.hSyncEnd:=t.h.hTotal - Trunc(hBackPorch);
  t.h.hFrontPorch:= t.h.hSyncStart - t.h.hDisp;
  t.h.hSyncWidth:=Trunc(hSyncWidth);
  t.h.hBackPorch:=Trunc(hBackPorch);
  {Determine the horizontal timing parameters}

  t.v.vTotal:=Trunc(vTotalLines);
  t.v.vDisp:=Trunc(vLines);
  t.v.vSyncStart:=t.v.vTotal - Trunc(vSyncBP);
  t.v.vSyncEnd:=t.v.vTotal - Trunc(vBackPorch);
  t.v.vFrontPorch:=t.v.vSyncStart - t.v.vDisp;
  t.v.vSyncWidth:=Trunc(c.vSyncRqd);
  t.v.vBackPorch:=Trunc(vBackPorch);
  {Determine the vertical timing parameters}

  if wantInterlace then
     begin {Halve the timings for interlaced modes}
     t.v.vTotal:=t.v.vTotal div 2;
     t.v.vDisp:=t.v.vDisp div 2;
     t.v.vSyncStart:=t.v.vSyncStart div 2;
     t.v.vSyncEnd:=t.v.vSyncEnd div 2;
     t.v.vFrontPorch:=t.v.vFrontPorch div 2;
     t.v.vSyncWidth:=t.v.vSyncWidth div 2;
     t.v.vBackPorch:=t.v.vBackPorch div 2;
     t.dotClock:=t.dotClock / 2;
     end;

  {Mark as GTF timing using the sync polarities}
  t.interlace:=wantInterlace;
  t.hSyncPol:=false;
  t.vSyncPol:=true;
end;


Procedure GTF_getConstants(var c:GTF_constants);
begin
c:=gc;
end;

Procedure GTF_setConstants(c:GTF_constants);
begin
GC:=c;
end;


Function MyVal(s:string):real;
var i:real;
    j:integer;
begin
Val(s,i,j);
MyVal:=i;
end;

Function Bolog(b:boolean):string;
begin
if B then Bolog:='''+''' else Bolog:='''-''';
end;

{============================================================================}
var f:text;
    xpixels,ypixels,freq:real;
    interlace:boolean;
    t:gtf_timings;
    ps4:string;
    k:char;


BEGIN
{hlavni program}
writeln;
Writeln('GTFCALC 1.3 (C) 2000-2011 by SciTech Software & Martin Rehak; rayer@seznam.cz');
Writeln('Translated into pascal by Laaca in 5/2011');
writeln;

if not (ParamCount in [4,5]) then
   begin
   writeln('Usage: GTFCALC <xPixels> <yPixels> <freq> [[Hz] [KHz] [MHz]] [I]');
   writeln('where <xPixels> is the horizontal resolution of the mode, <yPixels> is the');
   writeln('vertical resolution of the mode. The <freq> value will be the frequency to');
   writeln('drive the calculations, and will be either the vertical frequency (in Hz)');
   writeln('the horizontal frequency (in KHz) or the dot clock (in MHz). To generate');
   writeln('timings for an interlaced mode, add "I" to the end of the command line.');
   writeln('For example to generate timings for 640x480 at 60Hz vertical:');
   writeln('    GTFCALC 640 480 60 Hz');
   writeln('For example to generate timings for 640x480 at 31.5KHz horizontal:');
   writeln('    GTFCALC 640 480 31.5 KHz');
   writeln('For example to generate timings for 640x480 with a 25.175Mhz dot clock:');
   writeln('    GTFCALC 640 480 25.175 MHz');
   writeln('GTFCALC will print the summary of calculated results and dump the CRTC');
   writeln('values to CRTC.CFG config file in the format used by Martin''s VGA13/VESA driver');
   writeln('and SciTech Display Doctor.');
   Halt;
   end;

{Get values from command line}
xPixels:=MyVal(ParamStr(1));
yPixels:=MyVal(ParamStr(2));
freq:=MyVal(ParamStr(3));
interlace:=(ParamCount=5) and ((ParamStr(5)='I') or (ParamStr(5)='i'));


{Compute the CRTC timings}
ps4:=Paramstr(4);
if UpCase(ps4[1])='H' then
   GTF_calcTimings(xPixels,yPixels,freq,GTF_lockVF,false,interlace,t) else

if UpCase(ps4[1])='K' then
   GTF_calcTimings(xPixels,yPixels,freq,GTF_lockHF,false,interlace,t) else

if UpCase(ps4[1])='M' then
   GTF_calcTimings(xPixels,yPixels,freq,GTF_lockPF,false,interlace,t) else
   begin
   writeln('Unknown command line!');
   Halt;
   end;


{Dump summary info to standard output}
writeln('CRTC values for ',xPixels:3:0,'x',yPixels:3:0,' @ ',freq:2:0,ParamStr(4));
writeln('  hTotal = ',t.h.hTotal,#9#9#9,'vTotal = ',t.v.vTotal);
writeln('  hDisp = ',t.h.hDisp,#9#9#9,'vDisp = ',t.v.vDisp);
writeln('  hSyncStart = ',t.h.hSyncStart,#9#9'vSyncStart = ',t.v.vSyncStart);
writeln('  hSyncEnd = ',t.h.hSyncEnd,#9#9'vSyncEnd = ',t.v.vSyncEnd);
writeln('  hFrontPorch = ',t.h.hFrontPorch,#9#9'vFrontPorch = ',t.v.vFrontPorch);
writeln('  hSyncWidth = ',t.h.hSyncWidth,#9#9'vSyncWidth = ',t.v.vSyncWidth);
writeln('  hBackPorch = ',t.h.hBackPorch,#9#9'vBackPorch = ',t.v.vBackPorch);
if t.interlace then writeln('   Interlaced : Yes') else writeln('   Interlaced : No');
writeln('  H sync pol = ',t.hSyncPol);
writeln('  V sync pol = ',t.vSyncPol);
writeln('  Vert freq = ',t.vFreq:2:2,'Hz');
writeln('  Horiz freq = ',t.hFreq:2:2,'KHz');
writeln('  Dot Clock = ',t.dotClock:2:2,'MHz');

{$I-}
Assign(f,CFGFN);
Reset(f);
if IOresult<>2 then
   begin
   Close(f);
   write('Warning: file ',CFGFN,' already exist, overwrite it? [y/n] ');
   read(k);
   if upcase(k)='N' then halt;
   end;

Rewrite(f);
if IOresult=0 then
   begin
   writeln(f,'[',xPixels:3:0,' ',yPixels:3:0,']');
   write(f,t.h.hTotal,' ',t.h.hDisp,' ',t.h.hSyncStart,' ',t.h.hSyncEnd,' ',bolog(t.hSyncPol),' ');
   if t.interlace then writeln(f,'I') else writeln(f,'NI');
   writeln(f,t.v.vTotal,' ',t.v.vDisp,' ',t.v.vSyncStart,' ',t.v.vSyncEnd,' ',bolog(t.vSyncPol));
   writeln(f,t.dotClock:2:2);
   writeln(f,'# end of GTFCALC configuration data'#13#10);
   writeln(f,'# set calculated CRTC data for current videomode, for calc. run GTFCALC program [0/1]');
   writeln(f,'set_crtc = 1');      {default accept CRTC values'}
   writeln(f,'# bits per pixel [8/15/16/24/32]');
   writeln(f,'bpp = 32');          {default bpp=32}
   writeln(f,'# use Linear Frame Buffer mode if available (on VESA VBE 2.0+ adapters) [0/1], set 0 if run under Windows NT');
   writeln(f,'use_lfb = 1');       {use LFB}
   writeln(f,'# use MTRR Write-Combining memory mode for LFB/BS window on P6+/K7+ systems [0/1]');
   writeln(f,'use_mtrr_wc = 0');   {allow MTRR WC set (works only on P6+)}
   writeln(f,'use vertical synchronization [0/1]');
   writeln(f,'use_vsync = 1');     {allow vsync}
   Close(f);
   writeln('File ',CFGFN,' was successfully created.');
   end
   else begin
   writeln('ERROR: cannot create file ',CFGFN);
   Halt;
   end;
end.
