/*
    This file is part of the CLib sub-project of the FreeDOS project
    Copyright (C) 1997 by the author see below

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/* $RCSfile: STRARGTO.C $
   $Locker:  $	$Name:  $	$State: Exp $

	char_t *strargtoken(char_t **line, char_t *delimiter,
		char_t *squote, char_t *miquote, char_t *moquote)

	Tokenizer that handles single- and multi-character quotes and
	leaves the string unchanged.

	delimiters contains the token delimiters, if NULL, only '\0' is
	used.

	squote contains the single character quotes. Any character immediately
	following a s.c. quote is skipped, unless it's '\0'. The quote
	remains within the argument.

	miquotes contains the left multi character quote and moquotes the
	right one. These quotes are removed from the arguments, if the
	argument immediately starts and finishs with the same quote, e.g.:
	if miquotes = moquotes = "\""
		"..."		will remove '"', but
		arg="..."	will not, nor will
		"..."arg	, nor will
		arg"..."arg	.
	If strlen(moquote) < strlen(miquote), all quotes of miquote with the
	index > strlen(moquote) will quote the rest of the line completely.
	If moquote == NULL, moquote == "" is assumed.

	Input:
		line != NULL, *line != NULL
		delimiter, squote, miquote, moquote may be NULL

	Return:
		== NULL: *line points to delimiters only (or '\0')
		!= NULL: end of argument, *line updated to the start of the arg

	Note:
		<none>

	Conforms to:
		<none>

	See also:
		strargbrk, strtok

	Target compilers:
		Any C compiler

	Origin:
		1997/10/10 Steffen Kaiser (ska)

	Revised by:
		<none>

	File Revision:    Revision 1.3  1998/02/08 04:55:59  ska
*/

#include <_clib.h>			/* standard include, must be the first! */
#include <_string.h>

#ifdef RCS_Version
static char const rcsid[] = 
	"$Id: STRARGTO.C 1.3 1998/02/08 04:55:59 ska Exp $";
#endif

_CLibFunc char_t ___PTR *
strargtoken(char_t ___PTR * ___PTR * line
	,char ___PTR *delimiter
	,char ___PTR *squote
	,char ___PTR *miquote
	,char ___PTR *moquote)
{	char_t ___PTR *p;		/* temp pointer into line */
	char_t ___PTR *q;		/* temp pointer into line */
	int outquote;				/* multi character "out" quote */
	int ch;						/* current character */
	int numquotes;				/* number of processed quotes */
	int outquotes;				/* number of right quote chars */
	char_t ___PTR *lquote, ___PTR *rquote;	/* left & right quote */

	assert(line != NULL);
	assert(*line != NULL);

	if(delimiter)
		*line += strspn(*line, delimiter);

	p = *line;
	outquote = NUL;
	outquotes = moquote? strlen(moquote) - 1: -1;
	lquote = rquote = NULL;
	numquotes = 0;

	while((ch = *p) != NUL && (outquote || !stpchr(delimiter, ch))) {
		if(stpchr(squote, ch)) {
			if(!p[1]) break;		/* '\0' cannot be quoted */
			++p;					/* skip next character */
		}
		else if(!outquote) {
			if((q = stpchr(miquote, ch)) != NULL) {
				++numquotes;
				lquote = p;
				if((outquote = q - miquote) > outquotes) {
					/* quote the rest of the line */
					if(p == *line)
						/* remove left quote */
						++*line;
					p = strchr(p, NUL);		/* end of line */
					break;
				}
				outquote = moquote[outquote];	/* right quote character */
			}
		}
		else if(ch == outquote) {
			rquote = p;
			outquote = NUL;
		}
		++p;
	}

	if(numquotes == 1 && *line == lquote && p - 1 == rquote) {
		/* argument completely surrounded by the quote */
		++*line;
		--p;
	}

	return (!numquotes && p == *line)? (char_t ___PTR*)NULL	/* no argument found */
	 : p;			/* argument found, probably a quoted "" */
}
