/*
    This file is part of the CLib sub-project of the FreeDOS project
    Copyright (C) 1997 by the author see below

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/* $RCSfile: _HPSPLIT.C $
   $Locker: ska $	$Name:  $	$State: Exp $

	void _heapsplit(_heapNode *node, size_t size)

	Splits the node in that way the node becomes size bytes large.
	If possible, the new (unused) node is joined with an immediately
	following free node.

	Input:
		node != NULL, node must exist
		size <= node->length

	Return:
		<none>

	Note:
		Local heap only

	Conforms to:
		<none>

	See also:
		free, malloc, calloc, _fmalloc

	Target compilers:
		Any C compiler

	Origin:
		1997/10/10 Steffen Kaiser (ska)

	Revised by:
		<none>

	File Revision:    Revision 1.4  1998/02/11 07:40:34  ska
*/

#include <_clib.h>			/* standard include, must be the first! */
#include <_alloc.h>
#include <string.h>

#ifdef RCS_Version
static char const rcsid[] = 
	"$Id: _HPSPLIT.C 1.4 1998/02/11 07:40:34 ska Exp ska $";
#endif

_CLibFunc void
_heapsplit(_pHeapNode node
       ,size_t size)
{	REG size_t length;		/* of current node incl. overhead */
	REG size_t size2;		/* left over bytes of block */
	_pHeapNode nNode;		/* next node (old) */
	_pHeapNode tNode;		/* next node if block must be splitted */

	assert(node != NULL);
	assert(_heapfind(nodeData(node), NULL) != NULL);
	assert(nodeLength(node) >= size);

	if((size2 = nodeLength(node) - size) == 0)		/* nothing to do */
		return;

	nNode = nxtNode(node);	/* start of next node */
	tNode = (_pHeapNode)((_pHeapData)node + sizeof(_heapNode) + size);
#ifdef DEBUG_HEAP		/* zap all the free area */
	if(dbgHeapFreeCh)
		memset(tNode, dbgHeapFreeCh, size2);
#endif

	switch(nNode->type) {
	case HEAP_END:		/* simple: just cut the new block */
		tNode->type = HEAP_END;
#if _IN_ALLOC == 2
		_frgcollect(tNode);		/* resize fragment */
#endif 
		break;

	case HEAP_UNUSED:	/* join this node with the free bytes */
		setNxtNode(tNode, nxtNode(nNode));
		tNode->type = HEAP_UNUSED;
#ifdef DEBUG_HEAP		/* zap the next node's control structure */
		if(dbgHeapFreeCh)
			memset(nNode, dbgHeapFreeCh, sizeof(*nNode));
#endif
		break;

	default:			/* we must create a new block if possible */
		if(size2 < sizeof(_heapNode))
			/* The remaining space is too small to put the block overhead
				in --> no new block can be created -> the block
				is left the size as it has now. */
			/* No break because of trailing commands */
			return;

		/* to split this node is possible */
		setNodeLength(tNode, size2 - sizeof(_heapNode));
		tNode->type = HEAP_UNUSED;
		break;
	}
	setNodeLength(node, size);			/* this node was splitted */
}
