/*
    This file is part of the CLib sub-project of the FreeDOS project
    Copyright (C) 1997 by the author see below

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/* $RCSfile: SETVBUF.C $
   $Locker:  $	$Name:  $	$State: Exp $

	int setvbuf(FILE *fp, char *buf, int mode, size_t size)
	int setlinebuf(FILE *fp)

	Change the buffer mode of the stream.

	setlinebuf() is a macro and associates the stream with a dynamically
	allocated buffer and puts the stream into line-buffering.

	setvbuf() is controlled by these parameters:
		mode == _IOFBF:		put stream into full-buffering
		mode == _IOLBF:		put stream into line-buffering
		mode == _IONBF:		put stream into unbuffered mode
	
	If buf == NULL, the buffer is dynamically allocated. If there is
	not enough memory to allocate the buffer _before_ de-associating
	the old one, the current buffer mode is not changed.

	If size <= 0, the buffer size is assumed to be BUFSIZ (at
	compile-time of the C Library).

	Input:
		fp != NULL, a valid FILE pointer

	Return:
		0: on success
		else: on failure

	Note:
		setvbuf() uses setbuffer() to (de-)associate the buffer to
		the stream.

	Conforms to:
		ANSI

	See also:
		setbuffer

	Target compilers:
		Any C compiler

	Origin:
		1997/11/03 Robert de Bath (see CONTRIB\STDIO2.ZIP)

	Revised by:
		1997/11/23 Steffen Kaiser (ska)

	File Revision:    Revision 1.2  1998/01/29 07:10:01  ska
*/

#include <_clib.h>			/* standard include, must be the first! */
#include "stdio.h"
#include <alloc.h>			/* malloc() */

#ifdef RCS_Version
static char const rcsid[] = 
	"$Id: SETVBUF.C 1.2 1998/01/29 07:10:01 ska Exp $";
#endif

_CLibFunc int
setvbuf(FILE *fp, char *buf, int mode, size_t size)
{
	assert(fp != NULL);

	switch(mode) {
	case _IONBF:
		setbuffer(fp, NULL, 0);
		return 0;
	case _IOFBF: case _IOLBF:
		if(size <= 0) size = BUFSIZ;
		if(!buf) {
			if((buf = malloc(size)) == NULL)
				break;
			mode |= __MODE_FREEBUF;
		}
		setbuffer(fp, buf, size);
#if _IOFBF		/* Possibly someone likes to change _IOFBF == 0 */
		fp->mode &= ~__MODE_BUF;
#endif
		fp->mode |= mode;
		return 0;
	}
	return EOF;
}
