/*
    This file is part of the CLib sub-project of the FreeDOS project
    Copyright (C) 1997 by the author see below

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/* $RCSfile: FSEEK.C $
   $Locker:  $	$Name:  $	$State: Exp $

	int fseek(FILE *fp, off_t offset, int whence)

	Seek the file position of the stream.

	whence controls how to interprete offset:
		SEEK_SET: from beginning of the stream
		SEEK_CUR: from the current position
		SEEK_END: from the end of the stream

	Note: There are un-seekable streams, e.g. input streams connected
	to a TTY. To perform a seek on such streams results in undefined
	behaviour.

	fseek() may succeed even if the new position points to an invalid
	place into the stream.

	To seek files opened in text mode can cause unpredictable results.
	The only conform action is to seek to place with whence == SEEK_SET
	and an offset returned by ftell().

	fseek() does _not_ necessarily synchronize the system's file
	position with the stream position, thus, the caller cannot assume
	to savely intermix stream and handle based functions by calling
	this function! Use fflush() instead.

	Input:
		fp != NULL; a valid FILE pointer

	Return:
		0: on success
		else: on failure

	Note:
		In read mode and if no character was ungetc()'ed into the
		stream buffer and if offset is so small that the new position
		remains within the currently loaded buffer, the buffer is
		not flushed.

	Conforms to:
		ANSI

	See also:
		fflush, lseek, rewind

	Target compilers:
		Any C compiler

	Origin:
		1997/11/03 Robert de Bath (see CONTRIB\STDIO2.ZIP)

	Revised by:
		1997/11/23 Steffen Kaiser (ska)

	File Revision:    Revision 1.2  1998/01/29 07:10:01  ska
*/

#include <_clib.h>			/* standard include, must be the first! */
#include "stdio.h"
#include <io.h>				/* lseek() */

#ifdef RCS_Version
static char const rcsid[] = 
	"$Id: FSEEK.C 1.2 1998/01/29 07:10:01 ska Exp $";
#endif

_CLibFunc int
fseek(FILE *fp, off_t offset, int ref)
{
#if 1
   /* if __MODE_READING and no ungetc ever done can just move pointer */
   /* This needs testing! */
   off_t pos;

	assert(fp != NULL);

	if((fp->mode & (__MODE_READING | __MODE_UNGOT)) == __MODE_READING) {
		switch(ref) {
		case SEEK_SET:		/* transform into SEEK_CUR */
			if((pos = tell(fp->fd)) == _SEEK_FAIL)
				return EOF;			/* no tell, no seek */
			/* new offset := destination - normalized current offset */
			offset -= pos - (fp->bufread - fp->bufpos);
			ref = SEEK_CUR;

			/* fall through */
		case SEEK_CUR:		/* offset contains difference */
			pos = offset < 0? -offset: offset;
			if(pos < (fp->bufread - fp->bufstart)) {
				/* if offset is quite large, the pointer arethmetics would
					fail badly */
				char *p = fp->bufpos + (int)offset;		/* new position */
				if(fp->bufstart <= p && p <= fp->bufread) {
					/* OK, the pointer remains within the read buffer */
					fp->bufpos = p;
					return 0;
				}
			}
			/* fall through */
		default:
			break;
		}
	}
#else
	assert(fp != NULL);
#endif

   /* Use fflush to sync the pointers */

   if (fflush(fp) || lseek(fp->fd, offset, ref) == _SEEK_FAIL)
      return EOF;
   return 0;
}
