/*
    This file is part of the CLib sub-project of the FreeDOS project
    Copyright (C) 1997 by the author see below

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/* $RCSfile: FGET.C $
   $Locker:  $	$Name:  $	$State: Exp $

	size_t fget(char *buf, size_t buflen, FILE *fp)

	Read up to buflen characters from stream fp into buffer buf.

	This function transfers the data in _binary_ mode, no character
	translation is performed!

	Input:
		buf != NULL
		fp != NULL; a valid FILE pointer

	Return:
		0: on EOF or failure
		else: number of read characters

	Note:
		If the buffer is or becomes empty and the remaining request is
		larger than one bufferful (stream buffer), the read is performed
		as one chunk directly into the buffer bypassing the stream
		buffer. As this function is often called to read large chunks of
		data, this method can be a big win.
		It is the core function for fread() and fgetc().

	Conforms to:
		<none>

	See also:
		fread, fgetc

	Target compilers:
		Any C compiler

	Origin:
		1997/11/03 Robert de Bath (see CONTRIB\STDIO2.ZIP)

	Revised by:
		1997/11/23 Steffen Kaiser (ska)

	File Revision:    Revision 1.2  1998/01/29 07:10:01  ska
*/

#include <_clib.h>			/* standard include, must be the first! */
#include "stdio.h"
#include <io.h>
#include <string.h>

#ifdef RCS_Version
static char const rcsid[] = 
	"$Id: FGET.C 1.2 1998/01/29 07:10:01 ska Exp $";
#endif

_CLibFunc size_t
fget(char *buf, size_t bytes, FILE *fp)
{
	int   v;
	size_t len, got = 0;
	Inline_call

	assert(buf != NULL);
	assert(fp != NULL);

	v = fp->mode;

	/* Want to do this to bring the file pointer up to date */
	if (v & __MODE_WRITING)
		fflush(fp);

	/* Can't read or there's been an EOF or error then return zero */
	if ((v & (__MODE_READ | __MODE_EOF | __MODE_ERR)) != __MODE_READ)
		return 0;

	len = fp->bufread - fp->bufpos;
	if (len >= bytes) {		/* Enough buffered */
		memcpy(buf, fp->bufpos, (unsigned) bytes);
		fp->bufpos += bytes;
		return bytes;
	}

	if (len > 0) {		/* Some buffered */
		memcpy(buf, fp->bufpos, len);
		bytes -= got = len;
	}

	/* empty the buffer and, because this read does bypass the buffer,
		we must ensure that fast seek is disabled or, at least, a read()
		is performed when one tries to read a byte. If we fill the
		buffer, the pointers are updated correctly.		- 1997/11/15 ska*/
	fp->bufpos = fp->bufread = fp->bufwrite = fp->bufstart;

	if(bytes < fp->bufend - fp->bufstart) {
		/* We should read with as large chunks as possible.  Therefore
			we completely fill the buffer and extract only the requested
			part from it */
		len = read(fp->fd, fp->bufstart, fp->bufend - fp->bufstart);

		if(len && len != _IO_err) {		/* OK, at least one byte read */
			fp->bufread += len;

					/* read suceeded? More bytes read than requested? */
			if(len > bytes)	
				len = bytes;

			memcpy(buf + got, fp->bufpos, len);
			fp->bufpos += len;
			fp->mode |= __MODE_READING;
			fp->mode &= ~__MODE_UNGOT;
		}
	}
	else
	   /* Need more; do it with a direct read */
	   len = read(fp->fd, buf + got, bytes);

   /* Possibly for now _or_ later */
   switch(len) {
   case _IO_err:
		fp->mode |= __MODE_ERR;
		len = 0;
		break;
	case 0:
		fp->mode |= __MODE_EOF;
		break;
    }

   return got + len;
}
