/*
** $Id: mpf.c, initiated August 21, 2020 $
** GNU Multiple Precision Floating-Point Reliable Library (MPFR) binding
** See Copyright Notice in agena.h
**
** NOTE: MinGW cannot compile this file if its name would be `mpfr.c` as this seems to collide with the mpfr.h header file.
**
*/

#define AGENA_LIBVERSION	"mpf 1.1.0 for Agena as of August 28, 2023\n"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <limits.h>
#include <signal.h>

#include <mpfr.h>
#include <gmp.h>

#define mpflib_c
#define LUA_LIB

#include "agena.h"

#include "agnxlib.h"
#include "agenalib.h"
#include "agncmpt.h"


#if !(defined(LUA_DOS) || defined(__OS2__) || defined(LUA_ANSI))
#define AGENA_MPFLIBNAME "mpf"
LUALIB_API int (luaopen_mpf) (lua_State *L);
#endif


static int MPFR_PRECISION = 128;
static int MPFR_ROUNDING = MPFR_RNDN;  /* round to nearest, with ties to even */

gmp_randstate_t randomstate;

#define MPFR_NULL                 ((mpfr_ptr)0)

#define Mpfr_t                    mpfr_t
#define Mpfr_init                 mpfr_init
#define Mpfr_set_d(mx,d)          mpfr_set_d(mx, d, MPFR_ROUNDING)
#define Mpfr_set_str(mx,s)        mpfr_set_str(mx, s, 10, MPFR_ROUNDING)
#define Mpfr_init_set_d(mx,x)     mpfr_init_set_d(mx, x, MPFR_ROUNDING)
#define Mpfr_init_set_str(mx,x)   mpfr_init_set_str(mx, x, 10, MPFR_ROUNDING)
#define Mpfr_get_d(mx)            mpfr_get_d(mx, MPFR_ROUNDING)
#define Mpfr_clear(mx)            mpfr_clear(mx)
#define Mpfr_clears               mpfr_clears
#define Mpfr_cmp                  mpfr_cmp
#define Mpfr_set_default_prec     mpfr_set_default_prec(MPFR_PRECISION)

#define Mpfr_add(mr, mx, my)      mpfr_add(mr, mx, my, MPFR_ROUNDING)
#define Mpfr_sub(mr, mx, my)      mpfr_sub(mr, mx, my, MPFR_ROUNDING)
#define Mpfr_mul(mr, mx, my)      mpfr_mul(mr, mx, my, MPFR_ROUNDING)
#define Mpfr_div(mr, mx, my)      mpfr_div(mr, mx, my, MPFR_ROUNDING)
#define Mpfr_pow(mr, mx, my)      mpfr_pow(mr, mx, my, MPFR_ROUNDING)
#define Mpfr_copysign(mr, mx, my) mpfr_copysign(mr, mx, my, MPFR_ROUNDING)
#define Mpfr_fma(mr, mx, my, mz)  mpfr_fma(mr, mx, my, mz, MPFR_ROUNDING)
#define Mpfr_fms(mr, mx, my, mz)  mpfr_fms(mr, mx, my, mz, MPFR_ROUNDING)

#define Mpfr_neg(mr, mx)          mpfr_neg(mr, mx, MPFR_ROUNDING)
#define Mpfr_abs(mr, mx)          mpfr_abs(mr, mx, MPFR_ROUNDING)

#define Mpfr_sqrt(mr, mx)         mpfr_sqrt(mr, mx, MPFR_ROUNDING)
#define Mpfr_rec_sqrt(mr,mx)      mpfr_rec_sqrt(mr, mx, MPFR_ROUNDING)
#define Mpfr_cbrt(mr, mx)         mpfr_cbrt(mr, mx, MPFR_ROUNDING)
#define Mpfr_log(mr, mx)          mpfr_log(mr, mx, MPFR_ROUNDING)
#define Mpfr_log2(mr, mx)         mpfr_log2(mr, mx, MPFR_ROUNDING)
#define Mpfr_log10(mr, mx)        mpfr_log10(mr, mx, MPFR_ROUNDING)
#define Mpfr_exp(mr, mx)          mpfr_exp(mr, mx, MPFR_ROUNDING)
#define Mpfr_exp2(mr, mx)         mpfr_exp2(mr, mx, MPFR_ROUNDING)
#define Mpfr_exp10(mr, mx)        mpfr_exp10(mr, mx, MPFR_ROUNDING)

#define Mpfr_sin(mr, mx)          mpfr_sin(mr, mx, MPFR_ROUNDING)
#define Mpfr_cos(mr, mx)          mpfr_cos(mr, mx, MPFR_ROUNDING)
#define Mpfr_tan(mr, mx)          mpfr_tan(mr, mx, MPFR_ROUNDING)
#define Mpfr_sec(mr, mx)          mpfr_sec(mr, mx, MPFR_ROUNDING)
#define Mpfr_csc(mr, mx)          mpfr_csc(mr, mx, MPFR_ROUNDING)
#define Mpfr_cot(mr, mx)          mpfr_cot(mr, mx, MPFR_ROUNDING)
#define Mpfr_asin(mr, mx)         mpfr_asin(mr, mx, MPFR_ROUNDING)
#define Mpfr_acos(mr, mx)         mpfr_acos(mr, mx, MPFR_ROUNDING)
#define Mpfr_atan(mr, mx)         mpfr_atan(mr, mx, MPFR_ROUNDING)
#define Mpfr_atan2(mr, mx, my)    mpfr_atan2(mr, mx, my, MPFR_ROUNDING)
#define Mpfr_dim(mr, mx, my)      mpfr_dim(mr, mx, my, MPFR_ROUNDING)
#define Mpfr_hypot(mr, mx, my)    mpfr_hypot(mr, mx, my, MPFR_ROUNDING)
#define Mpfr_modf(mr, mx, my)     mpfr_modf(mr, mx, my, MPFR_ROUNDING)
#define Mpfr_fmod(mr, mx, my)     mpfr_fmod(mr, mx, my, MPFR_ROUNDING)
#define Mpfr_sinh(mr, mx)         mpfr_sinh(mr, mx, MPFR_ROUNDING)
#define Mpfr_cosh(mr, mx)         mpfr_cosh(mr, mx, MPFR_ROUNDING)
#define Mpfr_tanh(mr, mx)         mpfr_tanh(mr, mx, MPFR_ROUNDING)
#define Mpfr_sech(mr, mx)         mpfr_sech(mr, mx, MPFR_ROUNDING)
#define Mpfr_csch(mr, mx)         mpfr_csch(mr, mx, MPFR_ROUNDING)
#define Mpfr_coth(mr, mx)         mpfr_coth(mr, mx, MPFR_ROUNDING)
#define Mpfr_acosh(mr, mx)        mpfr_acosh (mr, mx, MPFR_ROUNDING)
#define Mpfr_asinh(mr, mx)        mpfr_asinh (mr, mx, MPFR_ROUNDING)
#define Mpfr_atanh(mr, mx)        mpfr_atanh (mr, mx, MPFR_ROUNDING)

#define Mpfr_ceil(mr, mx)         mpfr_ceil(mr, mx)
#define Mpfr_trunc(mr, mx)        mpfr_trunc(mr, mx)
#define Mpfr_floor(mr, mx)        mpfr_floor(mr, mx)
#define Mpfr_round(mr, mx)        mpfr_round(mr, mx)

#define Mpfr_min(mr,mx,my)        mpfr_min(mr, mx, my, MPFR_ROUNDING)
#define Mpfr_max(mr,mx,my)        mpfr_max(mr, mx, my, MPFR_ROUNDING)
#define Mpfr_reldiff(mr,mx,my)    mpfr_reldiff(mr, mx, my, MPFR_ROUNDING)

#define Mpfr_eint(mr,mx)          mpfr_eint(mr,mx,MPFR_ROUNDING)
#define Mpfr_li2(mr,mx)           mpfr_li2(mr,mx,MPFR_ROUNDING)
#define Mpfr_gamma(mr,mx)         mpfr_gamma(mr,mx,MPFR_ROUNDING)
#define Mpfr_lngamma(mr,mx)       mpfr_lngamma(mr,mx,MPFR_ROUNDING)
#define Mpfr_digamma(mr,mx)       mpfr_digamma(mr,mx,MPFR_ROUNDING)
#ifndef DEBIAN
#define Mpfr_beta(mr,mx,my)       mpfr_beta(mr,mx,my,MPFR_ROUNDING)
#endif
#define Mpfr_zeta(mr,mx)          mpfr_zeta(mr,mx,MPFR_ROUNDING)
#define Mpfr_erf(mr,mx)           mpfr_erf(mr,mx,MPFR_ROUNDING)
#define Mpfr_erfc(mr,mx)          mpfr_erfc(mr,mx,MPFR_ROUNDING)
#define Mpfr_j0(mr,mx)            mpfr_j0(mr,mx,MPFR_ROUNDING)
#define Mpfr_j1(mr,mx)            mpfr_j1(mr,mx,MPFR_ROUNDING)
#define Mpfr_jn(mr,mx,n)          mpfr_jn(mr,n,mx,MPFR_ROUNDING)
#define Mpfr_y0(mr,mx)            mpfr_y0(mr,mx,MPFR_ROUNDING)
#define Mpfr_y1(mr,mx)            mpfr_y1(mr,mx,MPFR_ROUNDING)
#define Mpfr_yn(mr,mx,n)          mpfr_yn(mr,n,mx,MPFR_ROUNDING)
#define Mpfr_agm(mr,mx,my)        mpfr_agm(mr,mx,my,MPFR_ROUNDING)
#define Mpfr_ai(mr,mx)            mpfr_ai(mr,mx,MPFR_ROUNDING)

#define Mpfr_const_log2(mr)       mpfr_const_log2(mr, MPFR_ROUNDING)
#define Mpfr_const_pi(mr)         mpfr_const_pi(mr, MPFR_ROUNDING)
#define Mpfr_const_euler(mr)      mpfr_const_euler(mr, MPFR_ROUNDING)
#define Mpfr_const_catalan(mr)    mpfr_const_catalan(mr, MPFR_ROUNDING)

#define Mpfr_set_nan(mr)          mpfr_set_nan(mr)
#define Mpfr_set_inf(mr,s)        mpfr_set_inf(mr,s)
#define Mpfr_set_zero(mr,s)       mpfr_set_zero(mr,s)
#define Mpfr_swap(mx, my)         mpfr_swap(mx, my)
#define Mpfr_get_str              mpfr_get_str

typedef struct Mpfr {
  Mpfr_t val;
} Mpfr;


#define creatempf(x) { \
  x = (Mpfr *)lua_newuserdata(L, sizeof(Mpfr)); \
  lua_setmetatabletoobject(L, -1, "mpf", 1); \
  Mpfr_init(x->val); \
}

/* #define checkmpf(L, n)            (Mpfr *)luaL_checkudata(L, n, "mpf") */

Mpfr INLINE *checkmpf (lua_State *L, int n) {
  if (agn_isnumber(L, n)) {
    Mpfr *mx;
    creatempf(mx);
    Mpfr_set_d(mx->val, agn_tonumber(L, n));
    lua_replace(L, n);  /* move (and pop) */
  }
  return (Mpfr *)luaL_checkudata(L, n, "mpf");
}


/* Create a constant: log2, pi, euler, catalan, inf, nan */
#define template_create_constant(procname) { \
  Mpfr *mr; \
  creatempf(mr); \
  procname(mr->val); \
}


/* Creates an MPFR floating-point object (mpfr_t MPFR userdata object) from a number, or a string str representing a number.
   See also: `mp.setstring`. */
static int Mpf_new (lua_State *L) {
  Mpfr *mr;
  if (agn_isnumber(L, 1)) {
    lua_Number x = agn_tonumber(L, 1);
    creatempf(mr);
    if (isnan(x)) {  /* 3.3.7 fix */
      Mpfr_set_nan(mr->val);
    } else if (isinf(x)) {
      Mpfr_set_inf(mr->val, tools_sign(x));
    } else {
      Mpfr_set_d(mr->val, x);
    }
  } else if (agn_isstring(L, 1)) {  /* mpfr_init_set_* crashes in Windows with mpfr-3.1.5 */
    const char *str = agn_tostring(L, 1);
    creatempf(mr);  /* first scan args, then push userdata */
    if (Mpfr_set_str(mr->val, str) == -1)
      luaL_error(L, "Error in " LUA_QS ": allocation failed, probably due to a non-numeric string.", "mp.new");
  } else
    luaL_error(L, "Error in " LUA_QS ": expected a number or string, got %s.", "mp.new", luaL_typename(L, 1));
  return 1;  /* leave userdata on the top of the stack */
}


/* Clones an MPFR value and returns it. The rounding mode of the MPFR value returned will be the current one, not necessarily
   the one with which the value to be duplicated has been created. */
static int Mpf_clone (lua_State *L) {
  Mpfr *mr, *mx;
  mx = checkmpf(L, 1);
  creatempf(mr);
  mpfr_set_prec(mr->val, mpfr_get_prec(mx->val));  /* copy precision */
  mpfr_set(mr->val, mx->val, MPFR_ROUNDING);
  return 1;
}


/*****************************************************************************************************

	Macro templates

******************************************************************************************************/

/* take two mpf num arguments a, b, apply procname to them and push the result onto the stack in new mpf num r. */
#define template_2args_1ret(procname) { \
  Mpfr *mx, *my, *mr; \
  mx = checkmpf(L, 1); \
  my = checkmpf(L, 2); \
  creatempf(mr); \
  procname(mr->val, mx->val, my->val); \
}

/* take three mpf num arguments a, b, c, apply procname to them and push the result onto the stack in new mpf num r. */
#define template_3args_1ret(procname) { \
  Mpfr *mx, *my, *mz, *mr; \
  mx = checkmpf(L, 1); \
  my = checkmpf(L, 2); \
  mz = checkmpf(L, 3); \
  creatempf(mr); \
  procname(mr->val, mx->val, my->val, mz->val); \
}

/* take three mpf num arguments r, a, b, apply procname to them; the result is implicitly stored to mpf num r. */
#define template_3args_composite(procname) { \
  Mpfr *mx, *my, *mr; \
  mr = checkmpf(L, 1); \
  mx = checkmpf(L, 2); \
  my = checkmpf(L, 3); \
  lua_settop(L, 1); \
  procname(mr->val, mx->val, my->val); \
}

/* take two mpf num arguments a, b, apply procname to them and push the result - a Lua/Agena number - onto the stack  */
#define template_2args_1num(procname) { \
  Mpfr *mx, *my; \
  mx = checkmpf(L, 1); \
  my = checkmpf(L, 2); \
  lua_pushnumber(L, procname(mx->val, my->val)); \
}

/* take a mpf num, apply procname to it and push the result onto the stack as a new mpf num r. */
#define template_1arg_1ret(procname) { \
  Mpfr *mx, *mr; \
  mx = checkmpf(L, 1); \
  creatempf(mr); \
  procname(mr->val, mx->val); \
}


static int Mpf_Pi (lua_State *L) {
  template_create_constant(Mpfr_const_pi);
  return 1;
}

static int Mpf_Log2 (lua_State *L) {
  template_create_constant(Mpfr_const_log2);
  return 1;
}

static int Mpf_Euler (lua_State *L) {
  template_create_constant(Mpfr_const_euler);
  return 1;
}

static int Mpf_Catalan (lua_State *L) {
  template_create_constant(Mpfr_const_catalan);
  return 1;
}

static int Mpf_Nan (lua_State *L) {
  template_create_constant(Mpfr_set_nan);
  return 1;
}


/* Depending on the sign of its argument, an integer: if it is non-negative, returns +infinity as an MPFR object, and
   -infinity as an MPFR object, otherwise. 2.9.10 */

#define template_InfZero(procname) { \
  Mpfr *mr; \
  int sgn; \
  sgn = (int)luaL_checkint32_t(L, 1); \
  creatempf(mr); \
  procname(mr->val, sgn); \
}

static int Mpf_Inf (lua_State *L) {
  template_InfZero(Mpfr_set_inf);
  return 1;
}


/* Depending on the sign of its argument, an integer: if it is non-negative, returns 0 as an MPFR object, and -0 as an
   MPFR object, otherwise. 2.9.10 */
static int Mpf_Zero (lua_State *L) {
  template_InfZero(Mpfr_set_zero);
  return 1;
}



/*****************************************************************************************************

	Basic Arithmetic

******************************************************************************************************/

/* mpf.add(a, b): Adds two mpf num a, b, and returns a new mpf num. Used by __add metamethod, i.e.:
   mpf.add(a, b) = a + b. */
static int Mpf_add (lua_State *L) {  /* a + b */
  template_2args_1ret(Mpfr_add);
  return 1;
}


/* mpf.subtract(a, b): Subtracts two mpf num a, b and returns a new mpf num. Used by __sub metamethod, i.e.:
   mpf.subtract(a, b) = a - b. */
static int Mpf_subtract (lua_State *L) {  /* a - b  */
  template_2args_1ret(Mpfr_sub);
  return 1;
}


/* mp.multiply(a, b): Multiplies two mpints a, b and returns a new mpint. Used by __mul metamethod, i.e.:
   mp.multiply(a, b) = a * b. */
static int Mpf_multiply (lua_State *L) {  /* a * b */
  template_2args_1ret(Mpfr_mul);
  return 1;
}


/* mp.divide(a, b): Divides two mpints a, b and returns a new mpint. Used by __div metamethod, i.e.:
   mp.divide(a, b) = a / b. */

#include <mpfr-impl.h>  /* must be included here, not in the header */
static int Mpf_divide (lua_State *L) {  /* extended 3.4.2 explicit check for zero denominator */
  Mpfr *mx, *my, *mr;
  mx = checkmpf(L, 1);
  my = checkmpf(L, 2);
  creatempf(mr);
  if (MPFR_IS_ZERO(my->val)) {
    MPFR_SET_NAN(mr->val);  /* same as template_create_constant(Mpfr_set_nan) */
    /* cannot use MPFR_RET_NAN as it cannot be linked */
  } else {
    Mpfr_div(mr->val, mx->val, my->val);
  }
  return 1;
}


/* __pow metamethod + mpf.pow(): Exponentiation a^b: raises mpf num a to the power of mpf num b and returns
   a new mpf num. Used by __pow metamethod. */
static int Mpf_pow (lua_State *L) {
  template_2args_1ret(Mpfr_pow);
  return 1;
}


/* fast multiply-add */
static int Mpf_fma (lua_State *L) {
  template_3args_1ret(Mpfr_fma);
  return 1;
}


/* fast multiply-subtract */
static int Mpf_fms (lua_State *L) {
  template_3args_1ret(Mpfr_fms);
  return 1;
}


/* dim: returns a-b if a > b, 0 if a <= b, or `undefined` of a or b is `undefined`. */
static int Mpf_dim (lua_State *L) {
  template_2args_1ret(Mpfr_dim);
  return 1;
}


/*****************************************************************************************************

	Miscellaneous

******************************************************************************************************/

/* mpf.tonumber(a): Returns the numeric value in mpf num a as a number. */
static int Mpf_tonumber (lua_State *L) {
  Mpfr *a = checkmpf(L, 1);
  lua_pushnumber(L, Mpfr_get_d(a->val));
  return 1;
}


/* Gets or sets the overall precision, in bits. If a number in the range 2 .. 2,147,483,647 is being passed, the function
   sets the precision for all values subsequently allocated. If no argument is given, the current setting is returned.
   The default precision at invocation of the package is 128. */
static int Mpf_precision (lua_State *L) {
  if (lua_gettop(L) != 0) {
    int precision = agn_checkinteger(L, 1);
    if (precision < MPFR_PREC_MIN || precision > MPFR_PREC_MAX)
      luaL_error(L, "Error in " LUA_QS ": precision out of range %d .. %d.", "mpf.precision", MPFR_PREC_MIN, MPFR_PREC_MAX);
    MPFR_PRECISION = precision;
    mpfr_set_default_prec(precision);
  }
  lua_pushinteger(L, MPFR_PRECISION);
  return 1;
}


/* Gets or sets the current rounding mode. If a string rmode is passed, the function sets the rounding mode for all values
   subsequently allocated. Valid settings for rmode are:
     rndn = MPFR_RNDN=0,  round to nearest, with ties to even
     rndz = MPFR_RNDZ,    round toward zero
     rndu = MPFR_RNDU,    round toward +Inf
     rndd = MPFR_RNDD,    round toward -Inf
   If no argument is given, the current rounding mode is returned. The default rounding mode at invocation of the package
   is 'rndn', i.e. rounding to nearest. */
static int Mpf_rounding (lua_State *L) {
  static const char *const opts[] = {"rndn", "rndz", "rndu", "rndd", NULL};
  static const char *const desc[] = {
    "round to nearest, with ties to even",
    "round toward zero",
    "round toward +infinity",
    "round toward -infinity",
    NULL
  };
  if (lua_gettop(L) != 0) {
    int roundingmode = agnL_checkoption(L, 1, "rndn", opts, 0);
    MPFR_ROUNDING = roundingmode;
  }
  lua_pushstring(L, opts[MPFR_ROUNDING]);
  lua_pushstring(L, desc[MPFR_ROUNDING]);
  return 2;
}


/* mpf.swap(a,b): swaps the values of a and b efficiently. Returns nothing. 2.9.10 */
static int Mpf_swap (lua_State *L) {
  Mpfr *mx, *my;
  mx = checkmpf(L, 1);
  my = checkmpf(L, 2);
  Mpfr_swap(mx->val, my->val);
  return 0;
}


/*****************************************************************************************************

	Metamethods

******************************************************************************************************/

static int mt_tostring (lua_State *L) {  /*  */
  Mpfr *a;
  a = checkmpf(L, 1);  /* push userdata on stack */
  lua_settop(L, 1);    /* don't let the stack be polluted by futile additional arguments */
  if (agn_getutype(L, 1)) {
    int deciloca;
    char *r;
    mpfr_exp_t mpfrDeciloca;
    char *buf = Mpfr_get_str(NULL, &mpfrDeciloca, 10, 0, a->val, MPFR_ROUNDING);
    if (!buf)
      luaL_error(L, "Error in " LUA_QS ": memory allocation failed.", "mpf.__tostring");
    /* see: https://machinecognitis.github.io/Math.Mpfr.Native/html/2408892b-5aea-259c-00be-b1a44fa53c1f.htm */
    if (tools_streq(buf, "@NaN@")) {
      mpfr_free_str(buf);
      lua_pushstring(L, "undefined");
      return 1;
    } else if (tools_streq(buf, "-@Inf@")) {
      mpfr_free_str(buf);
      lua_pushstring(L, "-infinity");
      return 1;
    } else if (tools_streq(buf, "@Inf@")) {
      mpfr_free_str(buf);
      lua_pushstring(L, "infinity");
      return 1;
    }
    deciloca = mpfrDeciloca;
    if (deciloca < 0) {
      char *r0;
      r0 = str_insert(buf, ".", 1 + (buf[0] == '-'));
      r = str_concat(r0, "e", tools_itoa(--deciloca, 10), NULL);
      if (!r) {
        luaL_error(L, "Error in " LUA_QS ": memory allocation failed.", "__tostring metamethod");
      }
      xfree(r0);
    } else {
      if (deciloca == 0 && (buf[0] == '0' || (buf[0] == '-' && buf[1] == '0'))) deciloca = 1;
      r = str_insert(buf, ".", deciloca + (buf[0] == '-'));
    }
    lua_pushfstring(L, "(%s)", r);
    lua_concat(L, 2);
    mpfr_free_str(buf);
    xfree(r);
  } else
    luaL_error(L, "Error in " LUA_QS ": invalid mpf object.", "mpf.__tostring");
  return 1;
}


#define template_cmp(cond) { \
  Mpfr *a, *b; \
  a = checkmpf(L, 1); \
  b = checkmpf(L, 2); \
  lua_pushboolean(L, (cond)); \
  return 1; \
}

static int mt_isequal (lua_State *L) {
  template_cmp(0 == Mpfr_cmp(a->val, b->val));
}

static int mt_islessthan (lua_State *L) {
  template_cmp(0 > Mpfr_cmp(a->val, b->val));
}

static int mt_islessorequal (lua_State *L) {
  template_cmp(0 >= Mpfr_cmp(a->val, b->val));
}


static int mpf_cmpd (lua_State *L) {  /* 3.4.2 */
  Mpfr *mx = checkmpf(L, 1);
  lua_pushinteger(L, mpfr_cmp_d(mx->val, agn_checknumber(L, 2)));
  return 1;
}


/* mp.neg(a): Returns -a, with a a mpf num, as a new mpf num. Used by __unm metamethod, i.e. mp.neg(a) <=> -a. */
static int mt_neg (lua_State *L) {
  template_1arg_1ret(Mpfr_neg);
  return 1;
}


/* __abs methamethod and mpf.argument: determimes the absolute value */
static int mt_abs (lua_State *L) {
  template_1arg_1ret(Mpfr_abs);
  return 1;
}


/* __sign methamethod: determimes sign and returns a number: -1, 0 or +1 */
static int mt_sign (lua_State *L) {
  Mpfr *a = checkmpf(L, 1);
  lua_pushinteger(L, mpfr_sgn(a->val));
  return 1;
}


/* __square metamethod, 3.3.7 */
static int mt_square (lua_State *L) {
  Mpfr *mx, *mr;
  mx = checkmpf(L, 1);
  creatempf(mr);
  mpfr_sqr(mr->val, mx->val, MPFR_ROUNDING);
  return 1;
}


/* __cube metamethod, 3.3.7 */
static int mt_cube (lua_State *L) {
  Mpfr *mx, *mr;
  mpfr_t my;
  mx = checkmpf(L, 1);
  creatempf(mr);
  mpfr_init2(my, MPFR_PREC(mx->val));
  Mpfr_set_d(my, 3.0);
  Mpfr_pow(mr->val, mx->val, my);
  mpfr_clear(my);
  return 1;
}


/* __recip metamethod, 3.3.7 */
static int mt_recip (lua_State *L) {
  Mpfr *mx, *mr;
  mx = checkmpf(L, 1);
  creatempf(mr);
  if (MPFR_IS_ZERO(mx->val)) {
    MPFR_SET_NAN(mr->val);  /* same as template_create_constant(Mpfr_set_nan) */
    /* cannot use MPFR_RET_NAN as it cannot be linked */
  } else {
    mpfr_ui_div(mr->val, 1, mx->val, MPFR_ROUNDING);
  }
  return 1;
}


static int mt_mpfgc (lua_State *L) {  /* __gc method */
  lua_lock(L);
  if (luaL_isudata(L, 1, "mpf")) {  /* avoid panics/crashes during restart, 3.4.6 */
    Mpfr *a = lua_touserdata(L, 1);
    lua_setmetatabletoobject(L, 1, NULL, 1);
    Mpfr_clear(a->val);
  }
  lua_unlock(L);
  return 0;
}


static void mpfcleanup (void) {  /* cleanup is not called when pressing CTRL+C */
  mpfr_free_cache();
  mpfr_clear_underflow();
  mpfr_clear_overflow();
  mpfr_clear_flags();
#ifndef DEBIAN
  mpfr_mp_memory_cleanup();
#endif
  gmp_randclear(randomstate);
}


static void mpfsigcleanup (int sig) {  /* for CTRL+C */
  mpfr_free_cache();
  mpfr_clear_underflow();
  mpfr_clear_overflow();
  mpfr_clear_flags();
#ifndef DEBIAN
  mpfr_mp_memory_cleanup();
#endif
  gmp_randclear(randomstate);
}


/*****************************************************************************************************

	Transcendental Functions

******************************************************************************************************/

/* __sqrt methamethod: determimes square root */
static int mt_sqrt (lua_State *L) {
  template_1arg_1ret(Mpfr_sqrt);
  return 1;
}


/* determimes inverse square root, 2.21.10 */
static int Mpf_recsqrt (lua_State *L) {
  template_1arg_1ret(Mpfr_rec_sqrt);
  return 1;
}


/* __cbrt: determimes cubic root */
static int Mpf_cbrt (lua_State *L) {
  template_1arg_1ret(Mpfr_cbrt);
  return 1;
}


#if !(defined(DEBIAN))
static int Mpf_root (lua_State *L) {  /* 3.4.2 */
  Mpfr *mx, *mr;
  uint32_t k;
  mx = checkmpf(L, 1);
  k = agn_checkuint32_t(L, 2);
  creatempf(mr);
  if (k == 0) {
    MPFR_SET_NAN(mr->val);
  } else {
    mpfr_rootn_ui(mr->val, mx->val, k, MPFR_PRECISION);
  }
  return 1;
}
#endif


/* __ln methamethod: determimes natural logarithm */
static int mt_ln (lua_State *L) {
  template_1arg_1ret(Mpfr_log);
  return 1;
}


/* determimes logarithm to base 2 */
static int Mpf_log2 (lua_State *L) {
  template_1arg_1ret(Mpfr_log2);
  return 1;
}


/* determimes logarithm to base 10 */
static int Mpf_log10 (lua_State *L) {
  template_1arg_1ret(Mpfr_log10);
  return 1;
}


/* __exp methamethod: determines exponential function to the base E = 2.71828... */
static int mt_exp (lua_State *L) {
  template_1arg_1ret(Mpfr_exp);
  return 1;
}


/* determimes exponential function to the base 2 */
static int Mpf_exp2 (lua_State *L) {
  template_1arg_1ret(Mpfr_exp2);
  return 1;
}


/* determimes exponential function to the base 10 */
static int Mpf_exp10 (lua_State *L) {
  template_1arg_1ret(Mpfr_exp10);
  return 1;
}


/* __sin methamethod: determimes sine */
static int mt_sin (lua_State *L) {
  template_1arg_1ret(Mpfr_sin);
  return 1;
}


/* __sin methamethod: determimes cosine */
static int mt_cos (lua_State *L) {
  template_1arg_1ret(Mpfr_cos);
  return 1;
}


/* __tan methamethod: determimes tangent */
static int mt_tan (lua_State *L) {
  template_1arg_1ret(Mpfr_tan);
  return 1;
}


/* __arcsin methamethod: determimes arcsine */
static int mt_arcsin (lua_State *L) {
  template_1arg_1ret(Mpfr_asin);
  return 1;
}


/* __arccos methamethod: determimes arccosine */
static int mt_arccos (lua_State *L) {
  template_1arg_1ret(Mpfr_acos);
  return 1;
}


/* __arctan methamethod: determimes arctangent */
static int mt_arctan (lua_State *L) {
  template_1arg_1ret(Mpfr_atan);
  return 1;
}


/* mpf.sec determimes secant */
static int Mpf_sec (lua_State *L) {
  template_1arg_1ret(Mpfr_sec);
  return 1;
}


/* mpf.csc determimes cosecant */
static int Mpf_csc (lua_State *L) {
  template_1arg_1ret(Mpfr_csc);
  return 1;
}


/* mpf.cot determimes cotangent */
static int Mpf_cot (lua_State *L) {
  template_1arg_1ret(Mpfr_cot);
  return 1;
}


/* mpf.arctan2(): arc-tangent2 of y and x  */
static int Mpf_arctan2 (lua_State *L) {
  template_2args_1ret(Mpfr_atan2);
  return 1;
}


/* mpf.hypot(): hypotenuse */
static int Mpf_hypot (lua_State *L) {
  template_2args_1ret(Mpfr_hypot);
  return 1;
}


static int Mpf_hypot4 (lua_State *L) {  /* 3.4.2 */
  Mpfr *mx, *my, *mr;
  mx = checkmpf(L, 1);
  my = checkmpf(L, 2);
  creatempf(mr);
  if (mpfr_cmp(mx->val, my->val) < 0) {
    MPFR_SET_NAN(mr->val);
  } else {
    mpfr_t mxsq, mysq;
    mpfr_prec_t prec = MPFR_PREC(mx->val) + 2;
    mpfr_init2(mxsq, prec);
    mpfr_init2(mysq, prec);
    mpfr_sqr(mxsq, mx->val, MPFR_ROUNDING);
    mpfr_sqr(mysq, my->val, MPFR_ROUNDING);
    mpfr_sub(mxsq, mxsq, mysq, MPFR_ROUNDING);
    mpfr_sqrt(mr->val, mxsq, MPFR_ROUNDING);
    mpfr_clear(mxsq);
    mpfr_clear(mysq);
  }
  return 1;
}


static int Mpf_pytha (lua_State *L) {  /* 3.4.2 */
  Mpfr *mx, *my, *mr;
  mx = checkmpf(L, 1);
  my = checkmpf(L, 2);
  creatempf(mr);
  mpfr_t mxsq, mysq;
  mpfr_prec_t prec = MPFR_PREC(mx->val) + 2;
  mpfr_init2(mxsq, prec);
  mpfr_init2(mysq, prec);
  mpfr_sqr(mxsq, mx->val, MPFR_ROUNDING);
  mpfr_sqr(mysq, my->val, MPFR_ROUNDING);
  mpfr_add(mr->val, mxsq, mysq, MPFR_ROUNDING);
  mpfr_clear(mxsq);
  mpfr_clear(mysq);
  return 1;
}


static int Mpf_pytha4 (lua_State *L) {  /* 3.4.2 */
  Mpfr *mx, *my, *mr;
  mx = checkmpf(L, 1);
  my = checkmpf(L, 2);
  creatempf(mr);
  mpfr_t mxsq, mysq;
  mpfr_prec_t prec = MPFR_PREC(mx->val) + 2;
  mpfr_init2(mxsq, prec);
  mpfr_init2(mysq, prec);
  mpfr_sqr(mxsq, mx->val, MPFR_ROUNDING);
  mpfr_sqr(mysq, my->val, MPFR_ROUNDING);
  mpfr_sub(mr->val, mxsq, mysq, MPFR_ROUNDING);
  mpfr_clear(mxsq);
  mpfr_clear(mysq);
  return 1;
}


static int Mpf_relerror (lua_State *L) {  /* 3.4.2 */
  template_2args_1ret(Mpfr_reldiff);
  return 1;
}


/* __sinh methamethod: determimes hyperbolic sine */
static int mt_sinh (lua_State *L) {
  template_1arg_1ret(Mpfr_sinh);
  return 1;
}


/* __cosh methamethod: determimes hyperbolic cosine */
static int mt_cosh (lua_State *L) {
  template_1arg_1ret(Mpfr_cosh);
  return 1;
}


/* __tanh methamethod: determimes hyperbolic tangent */
static int mt_tanh (lua_State *L) {
  template_1arg_1ret(Mpfr_tanh);
  return 1;
}


/* sech: determimes hyperbolic secant, 3.3.7 */
static int mt_sech (lua_State *L) {
  template_1arg_1ret(Mpfr_sech);
  return 1;
}


/* csch: determimes hyperbolic cosecant, 3.3.7 */
static int mt_csch (lua_State *L) {
  template_1arg_1ret(Mpfr_csch);
  return 1;
}


/* csch: determimes hyperbolic cotangent, 3.3.7 */
static int mt_coth (lua_State *L) {
  template_1arg_1ret(Mpfr_coth);
  return 1;
}


/* arccosh: determimes inverse hyperbolic cosine, 3.3.7 */
static int mt_arccosh (lua_State *L) {
  template_1arg_1ret(Mpfr_acosh);
  return 1;
}


/* arcsinh: determimes inverse hyperbolic sine, 3.3.7 */
static int mt_arcsinh (lua_State *L) {
  template_1arg_1ret(Mpfr_asinh);
  return 1;
}


static int mt_arccsch (lua_State *L) {  /* 3.4.2 */
  Mpfr *mx, *mr;
  mx = checkmpf(L, 1);
  creatempf(mr);
  if (MPFR_IS_ZERO(mx->val)) {
    MPFR_SET_NAN(mr->val);  /* same as template_create_constant(Mpfr_set_nan) */
    /* cannot use MPFR_RET_NAN as it cannot be linked */
  } else {
    Mpfr_set_d(mr->val, 1.0);
    Mpfr_div(mr->val, mr->val, mx->val);
    Mpfr_asinh(mr->val, mr->val);
  }
  return 1;
}


static int mt_arcsech (lua_State *L) {  /* 3.4.2 */
  Mpfr *mx, *mr;
  mx = checkmpf(L, 1);
  creatempf(mr);
  if (MPFR_IS_NEG(mx->val) || MPFR_IS_SINGULAR(mx->val)) {
    if (MPFR_IS_ZERO(mx->val) || MPFR_IS_NEG(mx->val) || MPFR_IS_NAN(mx->val)) {
      MPFR_SET_NAN(mr->val);  /* same as template_create_constant(Mpfr_set_nan) */
    } else {  /* inf */
      MPFR_SET_INF(mr->val);
    }
  } else {
    mpfr_d_div(mr->val, 1.0, mx->val, MPFR_ROUNDING);
    Mpfr_acosh(mr->val, mr->val);
  }
  return 1;
}


static int mt_arccoth (lua_State *L) {  /* 3.4.2 */
  Mpfr *mx, *mr;
  int isundef;
  mx = checkmpf(L, 1);
  isundef = mpfr_cmp_d(mx->val, 1.0) < 1;  /* x <= 1 ? */
  creatempf(mr);
  if (isundef || MPFR_IS_SINGULAR(mx->val)) {
    if (isundef || MPFR_IS_NAN(mx->val)) {
      MPFR_SET_NAN(mr->val);  /* same as template_create_constant(Mpfr_set_nan) */
    } else if (MPFR_IS_ZERO(mx->val)) {
      MPFR_SET_ZERO(mr->val);
    } else {  /* inf */
      MPFR_SET_INF(mr->val);
    }
  } else {
    mpfr_prec_t prec = MPFR_PREC(mx->val) + 2;
    mpfr_t mxm1, mxp1;
    mpfr_init2(mxm1, prec);
    mpfr_init2(mxp1, prec);
    /* 0.5*(ln(x + 1) - ln(x - 1)) */
    mpfr_add_d(mxp1, mx->val, 1.0, MPFR_ROUNDING);
    mpfr_log(mxp1, mxp1, MPFR_ROUNDING);
    mpfr_sub_d(mxm1, mx->val, 1.0, MPFR_ROUNDING);
    mpfr_log(mxm1, mxm1, MPFR_ROUNDING);
    mpfr_sub(mr->val, mxp1, mxm1, MPFR_ROUNDING);
    mpfr_div_d(mr->val, mr->val, 2.0, MPFR_ROUNDING);
    mpfr_clear(mxm1);
    mpfr_clear(mxp1);
  }
  return 1;
}


/* arcsinh: determimes inverse hyperbolic tangent, 3.3.7 */
static int mt_arctanh (lua_State *L) {
  template_1arg_1ret(Mpfr_atanh);
  return 1;
}


/* exponential integral */
static int Mpf_eint (lua_State *L) {
  template_1arg_1ret(Mpfr_eint);
  return 1;
}


/* real part of the dilogarithm of its argument */
static int Mpf_li2 (lua_State *L) {
  template_1arg_1ret(Mpfr_li2);
  return 1;
}


/* Gamma function */
static int Mpf_gamma (lua_State *L) {
  template_1arg_1ret(Mpfr_gamma);
  return 1;
}


/* logarithm of the Gamma function */
static int Mpf_lngamma (lua_State *L) {
  template_1arg_1ret(Mpfr_lngamma);
  return 1;
}


/* Digamma (Psi) function */
static int Mpf_digamma (lua_State *L) {
  template_1arg_1ret(Mpfr_digamma);
  return 1;
}


/* Riemann Zeta function */
static int Mpf_zeta (lua_State *L) {
  template_1arg_1ret(Mpfr_zeta);
  return 1;
}


/* error function */
static int Mpf_erf (lua_State *L) {
  template_1arg_1ret(Mpfr_erf);
  return 1;
}


/* complementary error function */
static int Mpf_erfc (lua_State *L) {
  template_1arg_1ret(Mpfr_erfc);
  return 1;
}


/* first kind Bessel function of order 0 */
static int Mpf_j0 (lua_State *L) {
  template_1arg_1ret(Mpfr_j0);
  return 1;
}


/* first kind Bessel function of order 1 */
static int Mpf_j1 (lua_State *L) {
  template_1arg_1ret(Mpfr_j1);
  return 1;
}


/* second kind Bessel function of order 0 */
static int Mpf_y0 (lua_State *L) {
  template_1arg_1ret(Mpfr_y0);
  return 1;
}


/* second kind Bessel function of order 1 */
static int Mpf_y1 (lua_State *L) {
  template_1arg_1ret(Mpfr_y1);
  return 1;
}


/* Airy function */
static int Mpf_ai (lua_State *L) {
  template_1arg_1ret(Mpfr_ai);
  return 1;
}


/* Beta function */
#if !(defined(DEBIAN))
static int Mpf_beta (lua_State *L) {
  template_2args_1ret(Mpfr_beta);
  return 1;
}
#endif


/* arithmetic-geometric mean function */
static int Mpf_agm (lua_State *L) {
  template_2args_1ret(Mpfr_agm);
  return 1;
}


/* take two mpf num arguments a, b, apply procname to them and push the result - a Lua/Agena number - onto the stack  */
#define template_Bessel(procname) { \
  Mpfr *mr, *mx; \
  long int n; \
  mx = checkmpf(L, 1); \
  n = (long int)luaL_checkint32_t(L, 2); \
  creatempf(mr); \
  procname(mr->val, mx->val, n); \
}


/* Bessel function of the first kind with order n */
static int Mpf_jn (lua_State *L) {
  template_Bessel(Mpfr_jn);
  return 1;
}


/* Bessel function of the second kind with order n */
static int Mpf_yn (lua_State *L) {
  template_Bessel(Mpfr_yn);
  return 1;
}


/*****************************************************************************************************

	Miscellaneous Numeric Functions

******************************************************************************************************/

/* determimes if mpf num is `undefined` */
static int Mpf_isundefined (lua_State *L) {
  Mpfr *a = checkmpf(L, 1);
  lua_pushboolean(L, mpfr_nan_p(a->val) != 0);
  return 1;
}


/* determimes if mpf num is `infinity` */
static int Mpf_isinfinite (lua_State *L) {
  Mpfr *a = checkmpf(L, 1);
  lua_pushboolean(L, mpfr_inf_p(a->val) != 0);
  return 1;
}


/* determimes if mpf num is finite, i.e. neither `undefined` or `infinity` */
static int Mpf_isfinite (lua_State *L) {
  Mpfr *a = checkmpf(L, 1);
  lua_pushboolean(L, mpfr_number_p(a->val) != 0);
  return 1;
}


/* __zero metamethod + mpf.iszero: determimes if mpf num is zero */
static int mt_iszero (lua_State *L) {
  Mpfr *a = checkmpf(L, 1);
  lua_pushboolean(L, mpfr_zero_p(a->val) != 0);
  return 1;
}


/* __nonzero metamethod + mpf.isnonzero: determimes if mpf num is non-zero */
static int mt_isnonzero (lua_State *L) {
  Mpfr *a = checkmpf(L, 1);
  lua_pushboolean(L, mpfr_zero_p(a->val) == 0);
  return 1;
}


/* mpf.modf() */
static int Mpf_modf (lua_State *L) {
  template_2args_1ret(Mpfr_modf);
  return 1;
}


/* mpf.fmod() */
static int Mpf_fmod (lua_State *L) {
  template_2args_1ret(Mpfr_fmod);
  return 1;
}


/* Like `math.nextafter`, but for MPFR values. Note that this function does _not_ change the argument you are passing. */
static int Mpf_nexttoward (lua_State *L) {
  mpfr_t r;
  Mpfr *mr, *mx, *my;
  mx = checkmpf(L, 1);
  my = checkmpf(L, 2);
  mpfr_init(r);
  mpfr_set_prec(r, mpfr_get_prec(mx->val));  /* copy precision */
  mpfr_set(r, mx->val, MPFR_ROUNDING);
  mpfr_nexttoward(r, my->val);
  creatempf(mr);
  mpfr_set_prec(mr->val, mpfr_get_prec(mx->val));  /* copy precision */
  mpfr_set(mr->val, r, MPFR_ROUNDING);
  mpfr_clear(r);
  return 1;
}


/* Returns a uniformly distributed random float on the interval [0, 1]. 2.21.10 */
static int Mpf_random (lua_State *L) {
  Mpfr *mr;
  creatempf(mr);
  mpfr_urandom(mr->val, randomstate, MPFR_ROUNDING);
  return 1;
}


/* Resets the random nuumber generator. 2.21.10 */
static int Mpf_randinit (lua_State *L) {
  gmp_randinit_default(randomstate);
  return 0;
}


/*****************************************************************************************************

	Rounding Functions

******************************************************************************************************/

/* mpf.ceil rounds up to to the next higher or equal integer */
static int Mpf_ceil (lua_State *L) {
  template_1arg_1ret(Mpfr_ceil);
  return 1;
}


/* mpf.trunc rounds to the next integer toward zero */
static int Mpf_trunc (lua_State *L) {
  template_1arg_1ret(Mpfr_trunc);
  return 1;
}


/* mpf.floor rounds to the next lower or equal integer */
static int Mpf_floor (lua_State *L) {
  template_1arg_1ret(Mpfr_floor);
  return 1;
}


/* mpf.round rounds to the nearest integer, rounding halfway cases away from zero */
static int Mpf_round (lua_State *L) {
  template_1arg_1ret(Mpfr_round);
  return 1;
}


/* minimum of a and b, 2.21.10 */
static int Mpf_min (lua_State *L) {
  template_2args_1ret(Mpfr_min);
  return 1;
}


/* maximumof a and b, 2.21.10 */
static int Mpf_max (lua_State *L) {
  template_2args_1ret(Mpfr_max);
  return 1;
}


/* mpf.copysign(): like math.copysign. 2.21.10 */
static int Mpf_copysign (lua_State *L) {
  template_2args_1ret(Mpfr_copysign);
  return 1;
}


/* mpf.signbit(): like math.signbit, i.e. checks the sign bit and returns `true` (value is negative) or `false`; 2.21.10 */
static int Mpf_signbit (lua_State *L) {
  Mpfr *mx;
  mx = checkmpf(L, 1);
  lua_pushboolean(L, mpfr_signbit(mx->val));
  return 1;
}


/*****************************************************************************************************/

static const struct luaL_Reg mt_mpflib [] = {
  {"__tostring",  mt_tostring},  /* for output at the console, e.g. print(n) */
  {"__unm",       mt_neg},
  {"__add",       Mpf_add},
  {"__sub",       Mpf_subtract},
  {"__mul",       Mpf_multiply},
  {"__div",       Mpf_divide},
  {"__recip",     mt_recip},
  {"__abs",       mt_abs},
  {"__sign",      mt_sign},
  {"__pow",       Mpf_pow},
  {"__sqrt",      mt_sqrt},
  {"__square",    mt_square},
  {"__cube",      mt_cube},
  {"__exp",       mt_exp},
  {"__ln",        mt_ln},
  {"__sin",       mt_sin},
  {"__cos",       mt_cos},
  {"__tan",       mt_tan},
  {"__sinh",      mt_sinh},
  {"__cosh",      mt_cosh},
  {"__tanh",      mt_tanh},
  {"__arcsin",    mt_arcsin},
  {"__arccos",    mt_arccos},
  {"__arctan",    mt_arctan},
  {"__nonzero",   mt_isnonzero},
  {"__zero",      mt_iszero},
  {"__eq",        mt_isequal},
  {"__lt",        mt_islessthan},
  {"__le",        mt_islessorequal},
  {"__gc",        mt_mpfgc},
  {NULL, NULL}
};


static const luaL_Reg mpflib[] = {
  {"add",         Mpf_add},              /* added on August 21, 2020 */
  {"agm",         Mpf_agm},              /* added on August 25, 2020 */
  {"ai",          Mpf_ai},               /* added on August 25, 2020 */
  {"arccosh",     mt_arccosh},           /* added on August 23, 2023 */
  {"arccoth",     mt_arccoth},           /* added on August 28, 2023 */
  {"arccsch",     mt_arccsch},           /* added on August 28, 2023 */
  {"arcsech",     mt_arcsech},           /* added on August 28, 2023 */
  {"arcsinh",     mt_arcsinh},           /* added on August 23, 2023 */
  {"arctanh",     mt_arctanh},           /* added on August 23, 2023 */
  {"arctan2",     Mpf_arctan2},          /* added on August 21, 2020 */
#if !(defined(DEBIAN))
  {"beta",        Mpf_beta},             /* added on August 25, 2020 */
#endif
  {"cbrt",        Mpf_cbrt},             /* added on August 21, 2020 */
  {"ceil",        Mpf_ceil},             /* added on August 21, 2020 */
  {"clone",       Mpf_clone},            /* added on August 30, 2020 */
  {"cmpd",        mpf_cmpd},             /* added on August 28, 2023 */
  {"copysign",    Mpf_copysign},         /* added on August 30, 2020 */
  {"cot",         Mpf_cot},              /* added on August 21, 2020 */
  {"coth",        mt_coth},              /* added on August 23, 2023 */
  {"csc",         Mpf_csc},              /* added on August 21, 2020 */
  {"csch",        mt_csch},              /* added on August 23, 2023 */
  {"digamma",     Mpf_digamma},          /* added on August 25, 2020 */
  {"dim",         Mpf_dim},              /* added on August 21, 2020 */
  {"divide",      Mpf_divide},           /* added on August 21, 2020 */
  {"eint",        Mpf_eint},             /* added on August 25, 2020 */
  {"erf",         Mpf_erf},              /* added on August 25, 2020 */
  {"erfc",        Mpf_erfc},             /* added on August 25, 2020 */
  {"exp10",       Mpf_exp10},            /* added on August 21, 2020 */
  {"exp2",        Mpf_exp2},             /* added on August 21, 2020 */
  {"floor",       Mpf_floor},            /* added on August 21, 2020 */
  {"fma",         Mpf_fma},              /* added on August 21, 2020 */
  {"fmod",        Mpf_fmod},             /* added on August 21, 2020 */
  {"fms",         Mpf_fms},              /* added on August 21, 2020 */
  {"gamma",       Mpf_gamma},            /* added on August 25, 2020 */
  {"hypot",       Mpf_hypot},            /* added on August 21, 2020 */
  {"hypot4",      Mpf_hypot4},           /* added on August 28, 2023 */
  {"Inf",         Mpf_Inf},              /* added on August 27, 2020 */
  {"isfinite",    Mpf_isfinite},         /* added on August 21, 2020 */
  {"isinfinite",  Mpf_isinfinite},       /* added on August 21, 2020 */
  {"isnonzero",   mt_isnonzero},         /* added on August 21, 2020 */
  {"isundefined", Mpf_isundefined},      /* added on August 21, 2020 */
  {"iszero",      mt_iszero},            /* added on August 21, 2020 */
  {"j0",          Mpf_j0},               /* added on August 25, 2020 */
  {"j1",          Mpf_j1},               /* added on August 25, 2020 */
  {"jn",          Mpf_jn},               /* added on August 25, 2020 */
  {"li2",         Mpf_li2},              /* added on August 25, 2020 */
  {"lgamma",      Mpf_lngamma},          /* added on August 25, 2020 */
  {"log10",       Mpf_log10},            /* added on August 21, 2020 */
  {"log2",        Mpf_log2},             /* added on August 21, 2020 */
  {"max",         Mpf_max},              /* added on August 27, 2020 */
  {"min",         Mpf_min},              /* added on August 27, 2020 */
  {"modf",        Mpf_modf},             /* added on August 21, 2020 */
  {"multiply",    Mpf_multiply},         /* added on August 21, 2020 */
  {"Nan",         Mpf_Nan},              /* added on August 27, 2020 */
  {"new",         Mpf_new},              /* added on August 21, 2020 */
  {"nexttoward",  Mpf_nexttoward},       /* added on August 30, 2020 */
  {"pow",         Mpf_pow},              /* added on August 21, 2020 */
  {"precision",   Mpf_precision},        /* added on August 21, 2020 */
  {"pytha",       Mpf_pytha},            /* added on August 28, 2023 */
  {"pytha4",      Mpf_pytha4},           /* added on August 28, 2023 */
  {"randinit",    Mpf_randinit},         /* added on August 30, 2020 */
  {"random",      Mpf_random},           /* added on August 30, 2020 */
  {"recsqrt",     Mpf_recsqrt},          /* added on August 27, 2020 */
  {"relerror",    Mpf_relerror},         /* added on August 28, 2023 */
  {"invsqrt",     Mpf_recsqrt},          /* added on August 27, 2020 */
#if !(defined(DEBIAN))
  {"root",        Mpf_root},             /* added on August 28, 2023 */
#endif
  {"round",       Mpf_round},            /* added on August 21, 2020 */
  {"rounding",    Mpf_rounding},         /* added on August 25, 2020 */
  {"sec",         Mpf_sec},              /* added on August 21, 2020 */
  {"sech",        mt_sech},              /* added on August 23, 2023 */
  {"signbit",     Mpf_signbit},          /* added on August 30, 2020 */
  {"subtract",    Mpf_subtract},         /* added on August 21, 2020 */
  {"swap",        Mpf_swap},             /* added on August 27, 2020 */
  {"tonumber",    Mpf_tonumber},         /* added on August 21, 2020 */
  {"tostring",    mt_tostring},          /* added on August 21, 2020 */
  {"trunc",       Mpf_trunc},            /* added on August 21, 2020 */
  {"y0",          Mpf_y0},               /* added on August 25, 2020 */
  {"y1",          Mpf_y1},               /* added on August 25, 2020 */
  {"yn",          Mpf_yn},               /* added on August 25, 2020 */
  {"zeta",        Mpf_zeta},             /* added on August 25, 2020 */
  {"Zero",        Mpf_Zero},             /* added on August 27, 2020 */
  {NULL, NULL}
};


static int nopened = 0;

/*
** Open mpf library
*/

LUALIB_API int luaopen_mpf (lua_State *L) {
  /* #if defined(_WIN32) && !defined(WINLEGACY)
  struct WinVer winversion;
  if (getWindowsVersion(&winversion) < MS_WINS2008) {
    luaL_error(L, "Error in " LUA_QS " package: need at least Windows 7 or 2008 Server or later.\nYou may download and install the Agena Legacy version instead.", "mpf");
  }
  #endif */
  luaL_checkstack(L, 3, "not enough stack space");  /* 3.18.4 fix */
  nopened++;
  Mpfr_set_default_prec;
  luaL_newmetatable(L, "mpf");
  luaL_register(L, NULL, mt_mpflib);  /* associate __gc method */
  luaL_register(L, AGENA_MPFLIBNAME, mpflib);
  lua_rawsetstringstring(L, -1, "initstring", AGENA_LIBVERSION);
  /* constants */
  Mpf_Log2(L);
  lua_setfield(L, -2, "Ln2");
  Mpf_Pi(L);
  lua_setfield(L, -2, "Pi");
  Mpf_Euler(L);
  lua_setfield(L, -2, "Euler");
  Mpf_Catalan(L);
  lua_setfield(L, -2, "Catalan");
  /* clean-up at exit */
  if (nopened == 1) {
    gmp_randinit_default(randomstate);  /* 3.4.2 fix, initialise only once or space will not be cleaned up */
    atexit(mpfcleanup);
    signal(SIGTERM, mpfsigcleanup);  /* for CTRL+c */
  }
  return 1;
}

