/*
** $Id: lreglib.c,v 1.38 2005/10/23 17:38:15 roberto Exp $
** Library for Register Manipulation
** See Copyright Notice in agena.h
*/

#include <stddef.h>
#include <string.h>

#define lreglib_c
#define LUA_LIB

#include "agena.h"

#include "agnxlib.h"
#include "agenalib.h"
#include "llex.h"
#include "ltablib.h"


static int reg_settop (lua_State *L) {  /* 2.3.0 RC 3 */
  luaL_argcheck(L, lua_isreg(L, 1), 1, "register expected");
  lua_pushvalue(L, 2);
  /* agnReg_settop checks the 2nd argument automatically and would return false in case of a wrong type */
  lua_pushboolean(L, agn_regsettop(L, 1));
  return 1;
}


static int reg_resize (lua_State *L) {  /* 4.6.4 (2.3.0 RC 3) */
  int nil;
  luaL_argcheck(L, lua_isreg(L, 1), 1, "register expected");
  nil = agnL_optboolean(L, 3, 1);
  lua_pushboolean(L, agn_regresize(L, 1, agn_checkinteger(L, 2), nil));
  return 1;
}


/* creates a register with values a, a+step ..., b-step, b; based on calc.fseq
   uses a modified version of Kahan's summation algorithm to avoid roundoff errors
   0.30.0, December 30, 2009; merged with calc.fseq 1.3.3, February 01, 2011

   calc.fseq comments: creates a sequence with values f(a), ..., f(b); June 30, 2008;
   20 % faster than an Agena implementation; extended August 20, 2009, 0.27.0
   extended 0.30.0, December 29, 2009 with Kahan summation algorithm to avoid roundoff errors;
   extended 1.8.8, November 07, 2012, to also return non-numeric sequences; extended 2.4.1,
   January 15, 2015, to process the number of elements to be returned.

   For the idea on the Matlab version please see:
   http://de.mathworks.com/help/matlab/ref/linspace.html
   http://www.che.utah.edu/~sutherland/wiki/index.php/Matlab_Arrays#Linspace_and_Logspace */

static int reg_new (lua_State *L) {
  /* use `volatile` so that the compiler does not render the Kahan code effectless */
  /* volatile lua_Number idx, step, c, y, t; formerly nreg, 2.28.1; created October 15, 2014 */
  int isfunc, isint, isdefault;
  size_t counter, i, nargs, total, offset;
  lua_Number a, b, eps;
  volatile lua_Number s, c, cs, ccs, cc, t, idx, step, x;
  luaL_aux_nstructure(L, "registers.new", &nargs, &offset, &a, &b, &step, &eps, &total, &isfunc, &isdefault, &isint);
  luaL_checkstack(L, 1 + isdefault, "not enough stack space");  /* 3.15.4 fix */
  agn_createreg(L, total);
  counter = 0;
  cs = ccs = 0;
  s = idx = a;
  if (isdefault) {  /* create a sequence of n slots and fill it with one and the same default of any type */
    agn_pairgeti(L, 2, 2);
  }
  /* total > counter: prevents that the last element is inserted even if a roundoff error occurred. */
  if (isfunc) {  /* function passed ? */
    int slots = 2 + (nargs >= 4 + offset)*(nargs - 4 - (int)offset + 1);
    while (idx <= b || tools_approx(idx, b, eps)) {
      luaL_checkstack(L, slots, "not enough stack space");  /* 3.5.5/3.15.4 fix */
      lua_pushvalue(L, offset);  /* push function */
      lua_pushnumber(L, (fabs(idx) < AGN_HEPSILON) ? 0 : idx);  /* quite dirty hack to avoid roundoff errors with 0 */
      for (i=4 + offset; i <= nargs; i++) lua_pushvalue(L, i);
      lua_call(L, slots - 1, 1);
      agn_regseti(L, -2, ++counter);
      if (isint) {
        idx += step;
      } else {
        x = step;  /* Kahan-Babuska */
        t = s + x;
        c = (fabs(s) >= fabs(x)) ? (s - t) + x : (x - t) + s;
        s = t;
        t = cs + c;
        cc = (fabs(cs) >= fabs(c)) ? (cs - t) + c : (c - t) + cs;
        cs = t;
        ccs += cc;
        idx = s + cs + ccs;
      }
    }
  } else {
    while (idx <= b || tools_approx(idx, b, AGN_HEPSILON)) {
      if (isdefault) {  /* fill with default value  */
        lua_pushvalue(L, -1);
        agn_regseti(L, -3, idx);
      } else
        agn_regsetinumber(L, -1, ++counter, (fabs(idx) < AGN_HEPSILON) ? 0 : idx);
      if (isint) {  /* 2.12.2 */
        idx += step;
      } else {
        x = step;  /* Kahan-Babuska */
        t = s + x;
        c = (fabs(s) >= fabs(x)) ? (s - t) + x : (x - t) + s;
        s = t;
        t = cs + c;
        cc = (fabs(cs) >= fabs(c)) ? (cs - t) + c : (c - t) + cs;
        cs = t;
        ccs += cc;
        idx = s + cs + ccs;
      }
    }
  }
  if (isdefault) agn_poptop(L);
  return 1;
}


static int reg_numunion (lua_State *L) {  /* 3.10.0 */
  luaL_argcheck(L, lua_type(L, 1) == LUA_TREG, 1, "argument is not a register");
  luaL_argcheck(L, lua_type(L, 2) == LUA_TREG, 2, "argument is not a register");
  lua_pushinteger(L, agn_numunion(L, 1, 2));
  return 1;
}


static int reg_numintersect (lua_State *L) {  /* 3.10.0 */
  luaL_argcheck(L, lua_type(L, 1) == LUA_TREG, 1, "argument is not a register");
  luaL_argcheck(L, lua_type(L, 2) == LUA_TREG, 2, "argument is not a register");
  lua_pushinteger(L, agn_numintersect(L, 1, 2));
  return 1;
}


static int reg_numminus (lua_State *L) {  /* 3.10.0 */
  luaL_argcheck(L, lua_type(L, 1) == LUA_TREG, 1, "argument is not a register");
  luaL_argcheck(L, lua_type(L, 2) == LUA_TREG, 2, "argument is not a register");
  lua_pushinteger(L, agn_numminus(L, 1, 2));
  return 1;
}


static int reg_isall (lua_State *L) {  /* 3.10.2 */
  agn_regisall(L, 1, "registers.isall");
  return 1;
}


/* }====================================================== */

static const luaL_Reg reg_funcs[] = {
  {"isall", reg_isall},                /* added February 04, 2024 */
  {"new", reg_new},                    /* added May 30, 2022 */
  {"numunion", reg_numunion},          /* added January 26, 2024 */
  {"numintersect", reg_numintersect},  /* added January 26, 2024 */
  {"numminus", reg_numminus},          /* added January 26, 2024 */
  {"resize", reg_resize},              /* added December 16, 2024 */
  {"settop", reg_settop},              /* added October 14, 2014 */
  {NULL, NULL}
};


LUALIB_API int luaopen_registers (lua_State *L) {
  luaL_register(L, AGENA_REGLIBNAME, reg_funcs);
  return 1;
}

