/*
** $Id: combinat.c, v 1.0.1 by Alexander Walz - initiated June 19, 2007
** Combinatorics library
** See Copyright Notice in agena.h
*/

#include <stdlib.h>
#include <string.h>
#include <math.h>

/* the following package ini declarations must be included after `#include <` and before `include #` ! */

#define combinat_c
#define LUA_LIB

#include "agena.h"
#include "agnxlib.h"
#include "agenalib.h"
#include "agncmpt.h"  /* for TRUNC */
#include "agnhlps.h"  /* for tools_intpow, isnan, PI2, quicksort, k-th smallest */
#include "lstate.h"   /* for agn_getdblepsilon */


/* Returns the n-th Bell number, i.e. counts the possible partitions of a set. 2.22.1,
   See: https://www.geeksforgeeks.org/bell-numbers-number-of-ways-to-partition-a-set/ */
static int combinat_bell (lua_State *L) {
  int i, j, n;
  n = agn_checknonnegint(L, 1);
  if (n > 218) {  /* avoid crashes with n > 500, 3.5.6 */
    lua_pushnumber(L, HUGE_VAL);
  } else {
    lua_Number bell[n + 1][n + 1];
    bell[0][0] = 1;
    for (i=1; i <= n; i++) {
      /* explicitly fill for j = 0 */
      bell[i][0] = bell[i - 1][i - 1];
      /* fill for remaining values of j */
      for (j=1; j <= i; j++)
        bell[i][j] = bell[i - 1][j - 1] + bell[i][j - 1];
    }
    lua_pushnumber(L, bell[n][0]);
  }
  return 1;
}


static int combinat_catalan (lua_State *L) {  /* 3.7.0 */
  int n;
  n = agn_checknonnegint(L, 1);
  lua_pushnumber(L, tools_binomial(2*n, n)/(n + 1));
  return 1;
}


/* Taken from the Fortran 90 source file:
   https://people.sc.fsu.edu/~jburkardt/f77_src/special_functions/special_functions.f, subroutine eulerb

  cc EULERB computes the Euler number En.
  c
  c  Licensing:
  c
  c    This routine is copyrighted by Shanjie Zhang and Jianming Jin.  However,
  c    they give permission to incorporate this routine into a user program
  c    provided that the copyright is acknowledged.
  c
  c  Modified:
  c
  c    09 July 2012
  c
  c  Author:
  c
  c    Shanjie Zhang, Jianming Jin
  c
  c  Reference:
  c
  c    Shanjie Zhang, Jianming Jin,
  c    Computation of Special Functions,
  c    Wiley, 1996,
  c    ISBN: 0-471-11963-6,
  c    LC: QA351.C45.
*/

#define TWOoPISQUARED  0.40528473456935108576
#define TWOoPICUBED    0.25801227546559591346

#define TWOoPISQUAREDld  0.405284734569351085775517852840
#define TWOoPICUBEDld    0.258012275465595913475376421509

/* Table taken from: http://www.mymathlib.com/functions/euler_numbers.html,
   Copyright  2004 RLH. All rights reserved. */
static long double euler_numbers[] = {
  +1.00000000000000000000000000000000000L,
  -1.00000000000000000000000000000000000L,
  +5.00000000000000000000000000000000000L,
  -6.10000000000000000000000000000000000e+1L,
  +1.38500000000000000000000000000000000e+3L,
  -5.05210000000000000000000000000000000e+4L,
  +2.70276500000000000000000000000000000e+6L,
  -1.99360981000000000000000000000000000e+8L,
  +1.93915121450000000000000000000000000e+10L,
  -2.40487967544100000000000000000000000e+12L,
  +3.70371188237525000000000000000000000e+14L,
  -6.93488743931379010000000000000000000e+16L,
  +1.55145341635570869050000000000000000e+19L,
  -4.08707250929312389236100000000000000e+21L,
  +1.25225964140362986546828500000000000e+24L,
  -4.41543893249023104553682821000000000e+26L,
  +1.77519391579539289436664789665000000e+29L,
  -8.07232992358878980621682474532810000e+31L,
  +4.12220603395177021223470796712590450e+34L,
  -2.34895805270431082520178285761989477e+37L,
  +1.48511507181149800178771567814058267e+40L,
  -1.03646227335196121193979573047451860e+43L,
  +7.94757942259759270360804051008807062e+45L,
  -6.66753751668554497743502847477374820e+48L,
  +6.09627864556854215869168574287684315e+51L,
  -6.05328524818862189631438378511164909e+54L,
  +6.50616248668460884771587063408082298e+57L,
  -7.54665993900873909806143256588973674e+60L,
  +9.42032189642024120420228623769058323e+63L,
  -1.26220192518062187199034092372874893e+67L,
  +1.81089114965792304965458077416521587e+70L,
  -2.77571017020715805973669809083715274e+73L,
  +4.53581033300178891747468878715677624e+76L,
  -7.88628420666178941810072074223999042e+79L,
  +1.45618443801396315007150470094942327e+83L,
  -2.85051783223697718732198729556739340e+86L,
  +5.90574720777544365455135032296439571e+89L,
  -1.29297366418786417049760323593869875e+93L,
  +2.98692818328457695093074365221714061e+96L,
  -7.27060171401686414380328065169928185e+99L,
  +1.86229157584126970444824923030431260e+103L,
  -5.01310494081097966129086936788810094e+106L,
  +1.41652557597856259916722069410021670e+110L,
  -4.19664316404024471322573414069418892e+113L,
  +1.30215959052404639812585869133081868e+117L,
  -4.22724068613990906470558992921459310e+120L,
  +1.43432127919765834061336826405785659e+124L,
  -5.08179907245804251645597576430907360e+127L,
  +1.87833293645293026402007579184179893e+131L,
  -7.23653438103385777657187661736782293e+134L,
  +2.90352834666109749705460383476443588e+138L,
  -1.21229373789292182105392954978560988e+142L,
  +5.26306424961699070600224073584236661e+145L,
  -2.37407307193676634703461698760652652e+149L,
  +1.11189009424828230249702335881757893e+153L,
  -5.40307865979529320561911549426347699e+156L,
  +2.72234108557222702137153414458909549e+160L,
  -1.42130105480096698118085204572231882e+164L,
  +7.68426182064690265317095628366647794e+167L,
  -4.29962192543974964281889033648632755e+171L,
  +2.48839157478298716316902455408489408e+175L,
  -1.48875820890620408401048810913362396e+179L,
  +9.20261411885209418840864126560312709e+182L,
  -5.87424445729243560747806550051798443e+186L,
  +3.87013355417592724899726125339465800e+190L,
  -2.63038464627282201918918005755736145e+194L,
  +1.84342186190681643216739318103276967e+198L,
  -1.33150076083199759777989619061195919e+202L,
  +9.90773407946409970275719941594148144e+205L,
  -7.59161615376086554230567716763177264e+209L,
  +5.98738690421595478060934030092899051e+213L,
  -4.85853153680527007166022567445774339e+217L,
  +4.05474737750791455464680535308584710e+221L,
  -3.47892371339090601415585327133292340e+225L,
  +3.06749738825108489449144357479461161e+229L,
  -2.77857404780457414987248665136951661e+233L,
  +2.58465603902711815098815082730837912e+237L,
  -2.46817048046364050455631133967404223e+241L,
  +2.41875397603671333264713788326666700e+245L,
  -2.43169264709107277171036789982532904e+249L,
  +2.50718300057371449601915222347628344e+253L,
  -2.65025200052581375350895159803901660e+257L,
  +2.87130197316667968492991621100369935e+261L,
  -3.18736021623541104699251674698644208e+265L,
  +3.62424164505845624987618515668413679e+269L,
  -4.22000551313026080825687414912160887e+273L,
  +5.03034557853150041609481420707106604e+277L,
  -6.13696178494213385049453688204944205e+281L,
  +7.66062813846337323811799348691311731e+285L,
  -9.78178011283967454892036825005468034e+289L,
  +1.27733166367198064207287773215186928e+294L,
  -1.70535141854472052178024263787253627e+298L,
  +2.32725003482003005917234767874590751e+302L,
  -3.24554745838924695277710327883293385e+306L
};

#define EULER_NUMBERS_SIZE 186   /* = 2*sizeof(euler_numbers)/sizeof(long double) - 2; */

static int combinat_euler (lua_State *L) {  /* 2.20.2, 2.34.10 switch to long doubles, moved from calc package 3.7.0 */
  long double r1, r2, s, isgn;
  int k, m, n;
  n = agn_checknonnegint(L, 1);
  switch (n) {
    case 0:
      lua_pushnumber(L, 1);
      break;
    case 2:
      lua_pushnumber(L, -1.0);
      break;
    default: {
      lua_Number r;
      if (n & 1) {  /* euler(odd integer) = 0 */
        r = 0;
      } else if (n <= EULER_NUMBERS_SIZE) {  /* 2.41.0 extension */
        r = euler_numbers[n >> 1];
      } else {
        lua_Number eps = agnL_optpositive(L, 2, agn_getdblepsilon(L));
        r1 = -4.0*TWOoPICUBEDld;
        r2 = 0.0;
        for (m=4; m <= n; m += 2) {
          r1 = - r1*(m - 1)*m*TWOoPISQUAREDld;  /* 0.405.. =  TWOoPI* TWOoPI */
          r2 = 1.0;
          isgn = 1.0;
          for (k=3; k <= 1000; k += 2) {
            isgn = -isgn;
            s = tools_powil(1.0/k, m + 1);  /* tuned 2.29.5 */
            r2 = r2 + isgn*s;
            if (s < eps) break;
          }
        }
        r = r1*r2;  /* <=> bn(m) = r1 * r2 */
      }
      lua_pushnumber(L, r);
    }
  }
  return 1;
}


/* Taken from the Fortran 90 source file:
   https://people.sc.fsu.edu/~jburkardt/f77_src/special_functions/special_functions.f, subroutine bernob

  cc BERNOB computes the Bernoulli number Bn.
  c
  c  Licensing:
  c
  c    This routine is copyrighted by Shanjie Zhang and Jianming Jin.  However,
  c    they give permission to incorporate this routine into a user program
  c    provided that the copyright is acknowledged.
  c
  c  Modified:
  c
  c    11 July 2012
  c
  c  Author:
  c
  c    Shanjie Zhang, Jianming Jin
  c
  c  Reference:
  c
  c    Shanjie Zhang, Jianming Jin,
  c    Computation of Special Functions,
  c    Wiley, 1996,
  c    ISBN: 0-471-11963-6,
  c    LC: QA351.C45.
*/

/* Computes the n'th Bernoulli number Bn. Moved from calc package 3.7.0 */

#define ONESIXTHld      (0.166666666666666666666666666667L)
#define PI2SQUAREDld    (39.4784176043574344753379639995L)   /* = (2*Pi)^2 */
#define ONEoPISQUAREDld (0.101321183642337771443879463210L)/* = 1/Pi^2 */

static int combinat_bernoulli (lua_State *L) {  /* 2.20.2, 2.34.10 switched to long doubles */
  long double r1, r2, s;
  int k, m, n;
  n = agn_checknonnegint(L, 1);
  switch (n) {
    case 0:
      lua_pushnumber(L, 1);
      break;
    case 1:
      lua_pushnumber(L, -0.5);
      break;
    case 2:
      lua_pushnumber(L, ONESIXTHld);
      break;
    default: {
      lua_Number eps = agnL_optpositive(L, 2, agn_getdblepsilon(L));
      if (n & 1) {  /* bernoulli(odd integer) = 0 */
        lua_pushnumber(L, 0);
        return 1;
      }
      r1 = ONEoPISQUAREDld;
      r2 = 0.0;
      for (m=4; m <= n; m += 2) {
        r1 = -r1*(m - 1)*m/PI2SQUAREDld;  /* PI2SQUARED = PI2 * PI2 */
        r2 = 1.0;
        s  = 10000.0;  /* 3.5.6 change */
        for (k=2; s >= eps && k <= 10000; k++) {
          s = tools_powil(1.0/k, m);  /* tuned 2.29.5 */
          r2 += s;
        }
      }
      lua_pushnumber(L, r1*r2);  /* <=> bn(m) = r1 * r2 */
    }
  }
  return 1;
}


/* Computes Stirling number of the first kind for n, k. 2.13.0 */
static int combinat_stirling1 (lua_State *L) {
  int i, j, n, k, maxj, *a;
  n = agn_checkinteger(L, 1);
  k = agn_checkinteger(L, 2);
  if (tools_isnegint(n) || tools_isnegint(k)) {  /* for both 1st and 2nd kind */
    lua_pushnumber(L, 0);
    return 1;
  }
  if (n < k || n == k) {
    lua_pushinteger(L, (n < k) ? 0 : 1);
    return 1;
  }
  maxj = n - k;
  a = agn_malloc(L, (maxj + 1) * sizeof(int), "combinat.stirling1", NULL);  /* 4.11.5 fix */
  /* Source: https://stackoverflow.com/questions/5133050/dynamic-programming-approach-to-calculating-stirlings-number */
  for (i=0; i <= maxj; i++) a[i] = 0;
  a[0] = 1;
  for (i=1; i <= k; i++) {
    for (j=1; j <= maxj; j++)
      a[j] -= (i + j - 1)*a[j - 1];
  }
  lua_pushinteger(L, a[maxj]);
  xfree(a);
  return 1;
}


/* Computes Stirling number of the second kind for n, k. 2.13.0 */
static int combinat_stirling2 (lua_State *L) {
  int i, j, n, k, maxj, *a;
  n = agn_checkinteger(L, 1);
  k = agn_checkinteger(L, 2);
  if (tools_isnegint(n) || tools_isnegint(k)) {  /* for both 1st and 2nd kind */
    lua_pushnumber(L, 0);
    return 1;
  }
  if (n < k || n == k) {
    lua_pushinteger(L, (n < k) ? 0 : 1);
    return 1;
  }
  maxj = n - k;
  a = agn_malloc(L, (maxj + 1) * sizeof(int), "combinat.stirling2", NULL);  /* 4.11.5 fix */
  /* Source: https://stackoverflow.com/questions/5133050/dynamic-programming-approach-to-calculating-stirlings-number */
  for (i=0; i <= maxj; i++) a[i] = 1;
  for (i=2; i <= k; i++) {
    for (j=1; j <= maxj; j++)
      a[j] += i*a[j - 1];
  }
  lua_pushinteger(L, a[maxj]);
  xfree(a);
  return 1;
}


/* See: https://stackoverflow.com/questions/35149865/computes-the-number-of-ways-to-partition-n-into-the-sum-of-positive-integers-c */
static int intpartition (int n, int k) {
  if (k == 0) return 0;
  if (n == 0) return 1;
  if (n < 0)  return 0;
  return intpartition(n, k - 1) + intpartition(n - k, k);
}

/* Computes the number of partitions of n, the partition numbers, taken r at a time. By default, r = n. Behaves like Maple V's
   combinat.numbperm. See: https://oeis.org/A000041; 2.22.1
   The approximation tools_numbpartapx is slower than recursion. */
static int combinat_numbpart (lua_State *L) {
  double n, r;
  int approx;
  n = agn_checknonnegint(L, 1);
  r = agnL_optnonnegint(L, 2, n);
  approx = agnL_optboolean(L, 3, 0);
  if (n == 0) n = 1;
  if (r == 0) n = 0;
  lua_pushnumber(L, intpartition(n, r));
  if (approx)  /* UNDOC */
    lua_pushnumber(L, tools_numbpartapx(n));
  return 1 + approx;
}


/* Counts the combinations of the elements of n taken k at a time and returns a number. 1.11.2 u1, C port 3.5.3.

   stats.numbcomb := proc(x :: {number, set}, y :: number) is  # 2.12.1 modification
      return if x :: set then binomial(size x, y) else binomial(x, y) fi
   end; */

static int combinat_numbcomb (lua_State *L) {
  int type = lua_type(L, 1);
  luaL_typecheck(L, type == LUA_TNUMBER || type == LUA_TSET, 1, "number of set expected", type);
  lua_pushnumber(L, tools_binomial(lua_isset(L, 1) ? agn_ssize(L, 1) : agn_tonumber(L, 1), agn_checknumber(L, 2)));
  return 1;
}


/* Number of permutations; 14.03.2012, extended 18.04.2013

   stats.numbperm := proc(x, y :: number) is  # C port 3.5.3
   if x :: set then
      return stats.numbperm(size x, y)
   elif x :- number then
      argerror(x, 'stats.numbperm', 'expected an integer or a set')
   fi;
   if float(x) or float(y) or x < 0 or y < 0 then  # faster than calling not(isnonnegint)
      return undefined
   elif x < y then
      return 0
   else
      return exp(lngamma(x + 1) - lngamma(x - y + 1))
   fi
end; */

static int combinat_numbperm (lua_State *L) {
  lua_Number x, y;
  int type = lua_type(L, 1);
  luaL_typecheck(L, type == LUA_TNUMBER || type == LUA_TSET, 1, "number of set expected", type);
  x = (lua_isset(L, 1)) ? agn_ssize(L, 1) : agn_tonumber(L, 1);
  y = agn_checknumber(L, 2);
  if (tools_isfrac(x) || tools_isfrac(y) || x < 0 || y < 0)
    lua_pushundefined(L);
  else if (x < y)
    lua_pushnumber(L, 0);
  else
    lua_pushnumber(L, sun_exp(sun_lgamma(x + 1) - sun_lgamma(x - y + 1)));
  return 1;
}


static const luaL_Reg combinatlib[] = {
  {"bell",      combinat_bell},             /* from stats, added on October 22, 2020 */
  {"bernoulli", combinat_bernoulli},        /* from calc, added on May 17, 2020 */
  {"catalan",   combinat_catalan},          /* added on December 06, 2023 */
  {"euler",     combinat_euler},            /* from calc, added on May 17, 2020 */
  {"numbcomb",  combinat_numbcomb},         /* from stats, added on December 05, 2023 */
  {"numbpart",  combinat_numbpart},         /* from stats, added on October 22, 2020 */
  {"numbperm",  combinat_numbperm},         /* from stats, added on December 05, 2023 */
  {"stirling1", combinat_stirling1},        /* from math.stirnum, added on August 13, 2018 */
  {"stirling2", combinat_stirling2},        /* from math.stirnum, added on August 13, 2018 */
  {NULL, NULL}
};


/*
** Open combinat library
*/
LUALIB_API int luaopen_combinat (lua_State *L) {
  luaL_register(L, AGENA_COMBINATLIBNAME, combinatlib);
  return 1;
}

/* ====================================================================== */

