; Project name:  Master Boot Loader (mbldr)
; File name:     mbldr.asm
; See also:      mbldr.h, mbldr.lst (autogenerated)
; Author:        Arnold Shade
; Creation date: 24 April 2006
; License type:  BSD
; URL:           http://mbldr.sourceforge.net/
; Description:   Assembly source of a master boot
; loader program intended to be a part of MBR.
; The output binary file should be exactly 512
; bytes long. Compile this assembly source with
; NASM in bin format

; Old origin of the MBR and a destination for BR
OldMBROrigin  EQU  7C00h
; New origin of the MBR
NewMBROrigin     EQU  0600h

; Assume the code is already moved to new origin
              ORG  NewMBROrigin

; ***********************************************************
; Here goes some code stored in MBR
; ***********************************************************

; Initialization: load segement registers, prepare stack and move program from
; old origin to the new one
; DL here contains the number of hard drive being used (80h most probably)
              CLI                    ; Disable interrupts
              XOR  CX,CX             ; Stack grows downwards from
              MOV  SS,CX             ; SS:SP = 0:7C00h
              MOV  SP,OldMBROrigin
              MOV  SI,SP             ; SI = 7C00h
              PUSH CX                ; ES = 0000h
              POP  ES
              PUSH CX                ; DS = 0000h
              POP  DS
              CLD                    ; Clear direction flag for REP command
              MOV  DI,NewMBROrigin
              INC  CH                ; CX=256
              REP  MOVSW             ; Copy to 0:0600h
              JMP  (BeginMain + NewMBROrigin - OldMBROrigin)

; Display boot menu (SI will point to OffsetsTable after the execution of
; this code)
BeginMain:    MOV  SI,BootMenuText   ; It is an offset known to
                                     ; external installation program
NextChar:     LODSB                  ; Load [SI] to AL, then increment SI
              TEST AL,AL             ; Check end of string
              JZ   SetupTimer        ; Finish output if 0 has been found
              MOV  AH,0Eh            ; Teletype mode
                                     ; (BX is ignored in text mode)
              INT  10h               ; Write symbol in teletype mode
              JMP  NextChar          ; Next iteration

; Install handler for interrupt 1Ch (user timer) or 08h (system timer)
SetupTimer:   MOV  EBX,TimerHandler  ; We install handler to routine in the
                                     ; new origin because it could be changed
                                     ; (to prevent from saving modified routine)
                                     ; Higher word contains 0, what is equal to
                                     ; the code segment value (CS)
              XCHG EBX,[1Ch*4]       ; Get old/set new offset and segment of
                                     ; the vector
                                     ; (patched from installation: 1Ch/08h)
              STI                    ; Enable interrupts
              MOV  DH,'9'            ; Progress bar will start from '9' char

; Wait for time out interval to elapse, or for the user selection
CheckKeyboard:
              MOV  AH,1              ; Get keyboard status
              INT  16h
              JZ   NotPressed        ; Is key pressed?
              MOV  AH,0              ; If yes, then read it
              INT  16h
              CMP  AL,0Dh            ; Is Enter pressed?
              JE   BootDefaultOS     ; Enter activates booting default OS
              CMP  AL,1Bh            ; Esc code interrupts timer
                                     ; (patched from installation: Esc/Space)
              JNE  TestKey           ; Go to checking of OS number choice key
              MOV  BYTE [TimerHandler],90h
                                     ; Write NOP opcode in the beginning of
                                     ; interrupt routine to disable timed boot
TestKey:      MOV  AL,AH             ; Move scan-code to AL instead of ASCII
                                     ; code (could be filled with NOP opcodes
                                     ; from installation program)
              SUB  AL,'1'            ; convert ASCII/scan code to OS number
                                     ; (patched from installation)
              CMP  AL,9              ; Check maximum available OS number to boot
                                     ; (patched from installation)
              JB   KeyPressed
NotPressed:   MOV  AX,18             ; How often to output a progress bar symbols
                                     ; (patched from installation)
                                     ; ThisValue*10*55ms=DefaultBootTimeout
              CMP  CX,AX             ; Is not it a time to output a symbol
                                     ; of progress bar?
              JB   CheckKeyboard
              SUB  CX,AX
              MOV  AH,0Eh            ; Teletype mode for int 10h
              MOV  AL,DH             ; Symbol to output for int 10h
;              MOV  AL,'*'            ; Symbol to output for int 10h
              INT  10h               ; Output symbol in teletype mode
                                     ; Substituted with NOP opcodes to disable
                                     ; progress-bar (patched from installation)
              DEC  DH                ; Switch to next symbol
              CMP  DH,'0'            ; Check if time is out
              JAE  CheckKeyboard
BootDefaultOS:
              MOV  AL,0              ; If timeout has occured, set default OS
                                     ; (patched from installation)

; Store currently chosen OS number to be used as a default one at next boot
KeyPressed:   MOV  [OldMBROrigin - NewMBROrigin + BootDefaultOS + 1],AL
                                     ; (could be filled with NOP opcodes from
                                     ; installation)
; Put old vector of timer interrupt back
              CLI
              MOV  [1Ch*4],EBX       ; Segment and offset of old interrupt
                                     ; (patched from installation: 1Ch/08h)
              STI

; Get sectors offset of chosen operating system
              CBW                    ; convert byte to word, ax = os number
              SHL  AX,2              ; convert number to offset in a table
              ADD  SI,AX             ; SI points to proper element in offsets
                                     ; table
              MOV  EBP,[SI]          ; load offset in sectors of desired OS
              INC  EBP               ; Check "skip boot" menu item
              JNZ  BootAttempt
              INT  18h               ; Tell BIOS, boot attempt has been failed
                                     ; User has chosen not to boot

; Set desired partition to active and hide remaining partitions
; we modify partition in the old origin since it would be saved to MBR
BootAttempt:  DEC  EBP               ; Restore EBP after temporary increment
              MOV  SI,OldMBROrigin - NewMBROrigin + PartitionTable
              MOV  CX,4              ; number of records in partition table
ProcessPartitionRecord:
              MOV  AL,0              ; inactive flag
              CMP  EBP,[SI+8]        ; check relative sectors value
              JNE  SetActiveFlag
              MOV  AL,80h            ; hard disk 0 (active flag)
SetActiveFlag:
              MOV  [SI],AL           ; set partition (in)active flag
                                     ; (patched with NOPs from installation to
                                     ; avoid unnecessary marking partitions as
                                     ; active or inactive depending on what to
                                     ; boot)
              MOV  AL,[SI+4]         ; load partition system id byte
              OR   AL,10h            ; Set HIDDEN bit (to simplify comparison
                                     ; operations, so this allows to ignore
                                     ; whether the partition is hidden or not)
              CMP  AL,11h            ; Visible/hidden DOS 12-bit FAT
              JE   ModifyPartitionVisibility
              CMP  AL,14h            ; Visible/hidden DOS 3.0+ 16-bit FAT (up to 32M)
              JE   ModifyPartitionVisibility
              CMP  AL,16h            ; Visible/hidden DOS 3.31+ 16-bit FAT (over 32M)
              JE   ModifyPartitionVisibility
              CMP  AL,17h            ; Visible/hidden Windows NT NTFS or OS/2 HPFS (IFS)
              JE   ModifyPartitionVisibility
              CMP  AL,1Bh            ; Visible/hidden WIN95_OSR2/Win98 FAT32
              JE   ModifyPartitionVisibility
              CMP  AL,1Ch            ; Visible/hidden WIN95_OSR2/Win98 FAT32, LBA-mapped
              JE   ModifyPartitionVisibility
              CMP  AL,1Eh            ; Visible/hidden WIN95/98: DOS 16-bit FAT, LBA-mapped
              JNE  SkipVisibility
ModifyPartitionVisibility:
              CMP  EBP,[SI+8]        ; check relative sectors value
              JNE  WriteVisibilityFlag
                                     ; make other partitions invisible (patched
                                     ; from installation: "WriteVisibilityFlag"
                                     ; to "SkipVisibility" to avoid hiding other
                                     ; primary partitions)
              AND  AL,0EFh           ; make bootable partition visible
WriteVisibilityFlag:
              MOV  [SI+4],AL         ; save updated system id byte
SkipVisibility:
              ADD  SI,10h            ; next record in partition table
              LOOP ProcessPartitionRecord

; Write modified partition table to MBR (we assume DL still contains the
; number of drive since we have not overwritten it till this place)
              MOV  SI,DevAdrPkt
              MOV  AX,4300h          ; Extended write, no verify
              INT  13h               ; we do not check for errors in order
                                     ; to save space in MBR

; Load boot sector of OS (we use extended int13h)
              TEST EBP,EBP           ; Take care about next drive if requested
                                     ; if EBP=0
              JNE  ReadBR
              INC  DL                ; Next disk please
ReadBR:       MOV  [SI+8],EBP        ; save offset (relative sectors)
              MOV  AH,42h            ; Extended read
              INT  13h
              JNC  CheckSignature
UseMBR:       MOV  DL,80h            ; Try to use default boot disk
              XOR  EBP,EBP           ; Use MBR in case of error
              JMP  ReadBR            ; Try again

; Check for signature in boot record
CheckSignature:
              CMP  WORD [OldMBROrigin+510],0AA55h
              JNE  UseMBR            ; Load MBR again instead of invalid BR
              JMP  OldMBROrigin      ; Execute OS boot loader (DL is a disk
                                     ; number what could be important for the
                                     ; new BR/MBR being loaded)

; System/user timer interrupt routine is called every 55msec (approx 1/18sec)
; We modify the value in new origin to prevent saving non-zero value back to MBR
; The first opcode here could be patched from installation (with 90h - NOP to
; disable timed boot (at installation time it is patched in actual master boot
; record which is saved to HDD, but at run-time the same is done with the
; Esc/Space keypress in the copy of MBR to prevent saving the modified code
; back to MBR)
TimerHandler: INC  CX                ; This opcode is patched from installation
                                     ; to prevent loading of default OS by
                                     ; timeout. CX initially is equal to 0
              PUSH EBX               ; Push CS and IP of old interrupt vector
                                     ; The first byte of this PUSH command can
                                     ; be patched from installation program
                                     ; (replaced with IRET opcode) for user
                                     ; interrupt 1Ch
              RETF                   ; Go to original handler of the interrupt

; ***********************************************************
; Here goes some data stored in MBR
; ***********************************************************

; Device address packet which is used in extended read operation
; (which should be performed in order to work around a problem
; with booting an operating system located higher than 1024 cyl.)
DevAdrPkt:    DB    10h       ; Size of this structure in bytes, shall be 10h
              DB    0         ; Reserved field, should be set to 0
              DB    1         ; Number of blocks to transfer, we need 1 sector
              DB    0         ; Reserved field, should be set to 0
              DW    OldMBROrigin
                              ; Address of transfer buffer (offset)
              DW    0         ; Address of transfer buffer (segment)
              DD    0         ; This double word and the next one (DQ) mean
              DD    0         ; starting logical block address in form of
                              ; (Cyl*MaxHead+SelHead)*MaxSector+SelSector-1
                              ; initially it (DQ 0) points to MBR

; Text for user menu (patched from installation, real data should reflect
; user's boot menu text). The default entry is marked with "*" sign (">"
; could also be used)
BootMenuText: DB    0Ah       ; This LF is used to avoid boot menus visually
                              ; flowing together into one solid block of text
                              ; f.ex. when choosing a corrupted menu entry
              TIMES 136 DB 20h
                              ; It is a maximum number of characters for
                              ; user's boot menu text when 9 partitions are
                              ; used. For example 2 partitions will result
                              ; in 138+(9-2)*4=166 availble characters
              DB    0

; Table representing offsets of bootable partitions. (patched from
; installation, real offsets should be written by the installation program)
OffsetsTable: TIMES 9 DD 0

; DiskSignature (or so-called NT Drive Serial Number) for Microsoft Windows
; These bytes should not be overwritten since it breaks boot process under
; Windows Vista
DiskSignature:
              DD    0
              TIMES 2 DB 0

; Padding to 512 bytes (all this space could be used by installation routine
; to place text for boot menu and offsets table). This is also used to check
; how many bytes are left unused in MBR (for text of boot menu and offsets
; table)
              TIMES 446-($-$$) DB 0

; Partition table (patched from installation, should be overwritten with
; actual data at installation time)
PartitionTable:
              TIMES 16 DB 00h
              TIMES 16 DB 00h
              TIMES 16 DB 00h
              TIMES 16 DB 00h

; BIOS boot signature
              DW    0AA55h

