/*
    Template for local message declaration file for msgComp
    Copyright (C) 1995 Steffen Kaiser

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/


#include <stdio.h>
#include <file.h>
#include <freedos.h>
#include "yerror.h"
#include "cdp.h"

#ifndef lint
static char rcsid[] = 
	"$Id: cdp.c 1.7 1996/03/12 23:56:28 ska Exp ska $";
#endif

#define REQ_MODE 1
#define REQ_QUIT 2
#define REQ_READ 0
#define REQ_WRITE 1
#define REQ_DONE 8
#define REQ_BUSY 9
#define REQ_ERROR 15
#define INF_4CHANNELS	(1 << 7)
#define INF_NOAUDIO		(1 << 6)
#define INF_NOCOPY		(1 << 5)
#define INF_PREEMPH		(1 << 4)

#define ERR_NR 0xff
#define ERR_IS 0x8000
#define ERR_IOCTL 0x4000	/* an error with this bit set indicates an DOS
	error. With this bit cleared, this is a driver error. */

#define ACCESSATTEMPTS 15

#define STAT_OPEN 1
#define STAT_UNLOCKED 2
#define STAT_RAWREAD 4
#define STAT_WRITABLE 8
#define STAT_AUDIOPLAY 16
#define STAT_INTERLEAVING 32
#define STAT_PREFETCH 128
#define STAT_CHANNELCNTRL 256
#define STAT_REDBOOK 512
#define STAT_XA 1024
#define STAT_NODISK 2048
#define STAT_RW 4096

struct REQ {
	unsigned char req_len;
	unsigned char req_unit;
	unsigned char req_cmd;
	unsigned      req_status;
	unsigned char req_dummy[8];
	unsigned char req_addr;
	unsigned char req_start[4];
	unsigned char req_secs[4];
};
struct REQ_IO {			/* template for driver request block for IOCTL I/O fcts */
	unsigned char ri_len;
	unsigned char ri_unit;
	unsigned char ri_cmd;
	unsigned      ri_status;
	unsigned char ri_dummy[8];
	unsigned char ri_media;
	unsigned      ri_addrLo;
	unsigned      ri_addrHi;
	unsigned      ri_tranLen;
	unsigned      ri_secNum;
	unsigned char ri_dmy2[4];
};
struct GEN_IOCTL {
	byte GI_fct;
};
struct RedBook {
	unsigned char rb_frame, rb_second, rb_minute, rb_dummy;
};
struct CD_INFO {
	unsigned char cd_firstTrack, cd_lastTrack;
	struct RedBook cd_leadOut;
	unsigned cd_drvOff, cd_drvSeg;
};

unsigned autoUnlock = 0;
unsigned drvUnit = 0;
char dname[9] = "MSCD001";
char *cmds[] = {
		"play",
		"status",
		"stop",
		"tracks",
		"eject",
		"lock",
		"unlock",
		"close",
		"toc",
		"notplaying",
		"isplaying",
		"dooropened",
		"doorclosed",
		NULL };
char *drvName = &dname;
char *statName = NULL;
extern skprintf(char *poi, unsigned len, char *poi1);

register int fd_printf(/*const char **/msg)	/* need two args */
{
	asm {
		shl ax, 1	; amount of bytes used by all arguments
		lea bx, 2[bp]	; address below last arg
		add bx, ax	; address of msg
	}
	stdout;
	asm {
		push ax	; 1st arg: file pointer
		xor ax, ax
		push ax	; 2nd arg: 0 => indicate FILE pointer
		push Word Ptr [bx]	; formt string
		dec bx
		dec bx		; address of 1st printf()''s arg
		push bx		; addrss of 1st arg for printf()
		call _skprintf
		add sp, 8
	}
}

callDriver(/*unsigned **/addr, /*struct REQ **/req)
asm {
		mov bx, 6[bp]				; poi to driver address
		les bx, [bx]				; ES:BX := driver address
		mov Word Ptr cs:seg1, es	; initialize CALL FAR
		mov Word Ptr cs:seg2, es
		mov ax, es:[bx+6]			; load strategy address
		mov Word Ptr cs:strp, ax
		mov ax, es:[bx+8]			; load interrupt address
		mov Word Ptr cs:intp, ax
		jmp short q1				; make sure patches in code seg are fetched
q1:		mov ax, ds
		mov es, ax
		mov bx, 4[bp]				; ES:BX := driver request structure
		db 9ah						; call far
strp	dw 0
seg1	dw 0
		db 9ah						; call far
intp	dw 0
seg2	dw 0
}

ioctlCD_(fd_10, mode_8, buf_6, len_4)
asm {
	mov cx, 4[bp]
	mov dx, 6[bp]
	mov ax, 8[bp]
	mov bx, 10[bp]
	add ax, 4402h
	int 21h
	jnc ioctlCD1
	or ah, 0c0h
ioctlCD1:
}

ioctlCD(struct CD_INFO *cd, int mode, struct GEN_IOCTL *buf, int len)
{	struct REQ_IO reqIoctl;

	reqIoctl.ri_len = sizeof(reqIoctl);
	reqIoctl.ri_unit = drvUnit;
	reqIoctl.ri_cmd = (mode & REQ_MODE) == REQ_READ? 0x03: 0x0c;		/* IOCTL */
	reqIoctl.ri_addrLo = buf;
	reqIoctl.ri_addrHi = get_ds();
	reqIoctl.ri_tranLen = len;
	callDriver(&(cd->cd_drvOff), reqIoctl);
	return reqIoctl.ri_status;
}

errCD(int nr1)
{	int nr;

	nr = nr1 & ERR_NR;
	if(nr1 & ERR_IOCTL) {
fprintf(stderr, "IOCTL error %d\n", nr);
/*
		if(nr < 0x30) switch(nr) {
			case 0x00: fputs("no error\n", stderr); break;
			case 0x01: fputs("function number invalid\n", stderr); break;
			case 0x02: fputs("file not found\n", stderr); break;
			case 0x03: fputs("path not found\n", stderr); break;
			case 0x04: fputs("too many open files (no handles available)\n", stderr); break;
			case 0x05: fputs("access denied\n", stderr); break;
			case 0x06: fputs("invalid handle\n", stderr); break;
			case 0x07: fputs("memory control block destroyed\n", stderr); break;
			case 0x08: fputs("insufficient memory\n", stderr); break;
			case 0x09: fputs("memory block address invalid\n", stderr); break;
			case 0x0A: fputs("environment invalid (usually >32K in length)\n", stderr); break;
			case 0x0B: fputs("format invalid\n", stderr); break;
			case 0x0C: fputs("access code invalid\n", stderr); break;
			case 0x0D: fputs("data invalid\n", stderr); break;
			case 0x0E: fputs("reserved\n", stderr); break;
			case 0x0F: fputs("invalid drive\n", stderr); break;
			case 0x10: fputs("attempted to remove current directory\n", stderr); break;
			case 0x11: fputs("not same device\n", stderr); break;
			case 0x12: fputs("no more files\n", stderr); break;
			case 0x13: fputs("disk write-protected\n", stderr); break;
			case 0x14: fputs("unknown unit\n", stderr); break;
			case 0x15: fputs("drive not ready\n", stderr); break;
			case 0x16: fputs("unknown command\n", stderr); break;
			case 0x17: fputs("data error (CRC)\n", stderr); break;
			case 0x18: fputs("bad request structure length\n", stderr); break;
			case 0x19: fputs("seek error\n", stderr); break;
			case 0x1A: fputs("unknown media type (non-DOS disk)\n", stderr); break;
			case 0x1B: fputs("sector not found\n", stderr); break;
			case 0x1C: fputs("printer out of paper\n", stderr); break;
			case 0x1D: fputs("write fault\n", stderr); break;
			case 0x1E: fputs("read fault\n", stderr); break;
			case 0x1F: fputs("general failure\n", stderr); break;
			case 0x20: fputs("sharing violation\n", stderr); break;
			case 0x21: fputs("lock violation\n", stderr); break;
			case 0x22: fputs("disk change invalid (ES:DI -> media ID structure)(see #0961)\n", stderr); break;
			case 0x23: fputs("FCB unavailable\n", stderr); break;
			case 0x24: fputs("sharing buffer overflow\n", stderr); break;
			case 0x25: fputs("(DOS 4.0+) code page mismatch\n", stderr); break;
			case 0x26: fputs("(DOS 4.0+) cannot complete file operation (out of input)\n", stderr); break;
			case 0x27: fputs("(DOS 4.0+) insufficient disk space\n", stderr); break;
		}
		else if(nr < 0x60) switch(nr) {
			case 0x32: fputs("network request not supported\n", stderr); break;
			case 0x33: fputs("remote computer not listening\n", stderr); break;
			case 0x34: fputs("duplicate name on network\n", stderr); break;
			case 0x35: fputs("network name not found\n", stderr); break;
			case 0x36: fputs("network busy\n", stderr); break;
			case 0x37: fputs("network device no longer exists\n", stderr); break;
			case 0x38: fputs("network BIOS command limit exceeded\n", stderr); break;
			case 0x39: fputs("network adapter hardware error\n", stderr); break;
			case 0x3A: fputs("incorrect response from network\n", stderr); break;
			case 0x3B: fputs("unexpected network error\n", stderr); break;
			case 0x3C: fputs("incompatible remote adapter\n", stderr); break;
			case 0x3D: fputs("print queue full\n", stderr); break;
			case 0x3E: fputs("queue not full\n", stderr); break;
			case 0x3F: fputs("not enough space to print file\n", stderr); break;
			case 0x40: fputs("network name was deleted\n", stderr); break;
			case 0x41: fputs("network: Access denied\n", stderr); break;
			case 0x42: fputs("network device type incorrect\n", stderr); break;
			case 0x43: fputs("network name not found\n", stderr); break;
			case 0x44: fputs("network name limit exceeded\n", stderr); break;
			case 0x45: fputs("network BIOS session limit exceeded\n", stderr); break;
			case 0x46: fputs("temporarily paused\n", stderr); break;
			case 0x47: fputs("network request not accepted\n", stderr); break;
			case 0x48: fputs("network print/disk redirection paused\n", stderr); break;
			case 0x49: fputs("network software not installed ;; (LANtastic) invalid network version\n", stderr); break;
			case 0x4A: fputs("unexpected adapter close;; (LANtastic) account expired\n", stderr); break;
			case 0x4B: fputs("(LANtastic) password expired\n", stderr); break;
			case 0x4C: fputs("(LANtastic) login attempt invalid at this time\n", stderr); break;
			case 0x4D: fputs("(LANtastic v3+) disk limit exceeded on network node\n", stderr); break;
			case 0x4E: fputs("(LANtastic v3+) not logged in to network node\n", stderr); break;
			case 0x4F: fputs("reserved\n", stderr); break;
			case 0x50: fputs("file exists\n", stderr); break;
			case 0x51: fputs("reserved\n", stderr); break;
			case 0x52: fputs("cannot make directory\n", stderr); break;
			case 0x53: fputs("fail on INT 24h\n", stderr); break;
			case 0x54: fputs("(DOS 3.3+) too many redirections\n", stderr); break;
			case 0x55: fputs("(DOS 3.3+) duplicate redirection\n", stderr); break;
			case 0x56: fputs("(DOS 3.3+) invalid password\n", stderr); break;
			case 0x57: fputs("(DOS 3.3+) invalid parameter\n", stderr); break;
			case 0x58: fputs("(DOS 3.3+) network write fault\n", stderr); break;
			case 0x59: fputs("(DOS 4.0+) function not supported on network\n", stderr); break;
			case 0x5A: fputs("(DOS 4.0+) required system component not installed\n", stderr); break;
		}
		else switch(nr) {
			case 0x64: fputs("(MSCDEX) unknown error\n", stderr); break;
			case 0x65: fputs("(MSCDEX) not ready\n", stderr); break;
			case 0x66: fputs("(MSCDEX) EMS memory no longer valid\n", stderr); break;
			case 0x67: fputs("(MSCDEX) not High Sierra or ISO-9660 format\n", stderr); break;
			case 0x68: fputs("(MSCDEX) door open\n", stderr); break;
			default: fputs("Unknown Error\n", stderr); break;
		}*/
	}
	else {
		fprintf(stderr, "Driver error #%u: ", nr);
		switch(nr) {
			case 0x0 : fputs("write-protect violation\n", stderr); break;
			case 0x1 : fputs("unknown unit\n", stderr); break;
			case 0x2 : fputs("drive not ready\n", stderr); break;
			case 0x3 : fputs("unknown command\n", stderr); break;
			case 0x4 : fputs("CRC error\n", stderr); break;
			case 0x5 : fputs("bad drive request structure length\n", stderr); break;
			case 0x6 : fputs("seek error\n", stderr); break;
			case 0x7 : fputs("unknown media\n", stderr); break;
			case 0x8 : fputs("sector not found\n", stderr); break;
			case 0x9 : fputs("printer out of paper\n", stderr); break;
			case 0xA : fputs("write fault\n", stderr); break;
			case 0xB : fputs("read fault\n", stderr); break;
			case 0xC : fputs("general failure\n", stderr); break;
			case 0xE : fputs("(CD-ROM) media unavailable\n", stderr); break;
			case 0xF : fputs("invalid disk change\n", stderr); break;
			default: fprintf(stderr, "Unknown error\n", nr); break;
		}
	}

	if(nr1 >= 0)
		exit(nr);
}

int EioctlCD(struct CD_INFO *cd, int mode, char *buf, unsigned len)
{	int e;

	e = ioctlCD(cd, mode, buf, len);
	if((e & ERR_IS) && (mode & REQ_QUIT))
		errCD(e & 0x7fff);
	return e;
}

void reset(struct CD_INFO *cd)
{	
	ioctlCD(cd, REQ_WRITE, "\2", 1);	/* reset drive */
}

unsigned statCD(struct CD_INFO *cd)
{	struct {
		unsigned char r_fct;
		unsigned      r_info1, r_info2;
	} ioctlBlk;

	ioctlBlk.r_fct = 0x06;		/* get driver status */
	EioctlCD(cd, REQ_READ | REQ_QUIT, ioctlBlk, sizeof(ioctlBlk));
	return ioctlBlk.r_info1;
}

int isPlaying(struct CD_INFO *cd)
{	char tranArea[11];
	int e;

	tranArea[0] = 4;		/* request audio channel info */
	e = ioctlCD(cd, REQ_READ, tranArea, sizeof(tranArea));
	return (e & ERR_IS)? 0: e & (1 << REQ_BUSY);
}

void drvOpen(struct CD_INFO *cd)
{	int fd, e;
	struct {
		unsigned char d_fct;
		unsigned d_drvOff, d_drvSeg;
	} *ioctlBlk;

	if((fd = open(drvName, F_READ)) <= 0)
		error(E_drvOpen, drvName);
	ioctlBlk = (char*)&(cd->cd_drvOff) - 1;
	ioctlBlk->d_fct = 0x00;		/* request driver address */
	if((e = ioctlCD_(fd, REQ_READ, ioctlBlk, 5)) & ERR_IS)
		errCD(e & 0x7fff);
	close(fd);
}

void closeTray(struct CD_INFO *cd)
{	struct {
		unsigned char q_fct;
		unsigned char q_lowest, q_highest;
		struct RedBook rb;
	} ioctlBlk;
	struct REQ req;
	int e, f;

	if(statCD(cd) & 1) {
		ioctlBlk.q_fct = 0x05;	/* close tray */
		ioctlCD(cd, REQ_WRITE, ioctlBlk, sizeof(ioctlBlk));
	}
	req.req_len = sizeof(req);
	req.req_unit = drvUnit;
	req.req_cmd = 0x85;				/* Stop audio play */
	callDriver(&(cd->cd_drvOff), req);	/* ignore if call failed */

	ioctlBlk.q_fct = 0xa;			/* request audio disk info */
	for(f = ACCESSATTEMPTS; --f; delay(200))	/* allow time-out */
		if(((e = ioctlCD(cd, REQ_READ, ioctlBlk, sizeof(ioctlBlk))) & (ERR_IS | ERR_NR)) != 0x8002)
				/* 0x8002 is "Drive not ready" and is set, while the
					tray is moving. */
			break;
#ifndef NDEBUG
		else printf("closeTray: req audio nfo: %u\n", e &0x7fff);
	printf("closeTray: req audio nfo: %u times\n", ACCESSATTEMPTS - f);
#endif
	if(statCD(cd) & STAT_NODISK) {	/* Failed, because no Disk in drive? */
		/* no disk in drive */
		memset(&cd->cd_firstTrack, 0, sizeof(ioctlBlk) - 1);
		return;
	}
	if(e & ERR_IS)
		EioctlCD(cd, REQ_READ | REQ_QUIT, ioctlBlk, sizeof(ioctlBlk));

	memcpy(&cd->cd_firstTrack, &ioctlBlk.q_lowest, sizeof(ioctlBlk) - 1);
}

#define drvClose(a)
/*void drvClose(struct CD_INFO *cd)
{	
}*/

void chkDisk(struct CD_INFO *cd)
{	unsigned stat;

	statCD(cd);		/* allow the drive to react */
	stat = statCD(cd);
	if(stat & STAT_OPEN)	error(E_clTray);
	if(stat & STAT_NODISK)	error(E_noDisk);
}

RB2HSG(char *hsg, struct RedBook *rb)
/* (HSG sector = minute * 4500 + second * 75 + frame - 150) */
/* Red Book info: frame/second/minute/dummy */
{	char t[4];
	char m[4];

	longset(hsg, rb->rb_minute);
	longset(t, 4500);
	longmul(hsg, t);
	longset(m, rb->rb_second);
	longset(t, 75);
	longmul(m, t);
	longadd(hsg, m);
	longset(t, rb->rb_frame);
	longadd(hsg, t);
	longset(t, 150);
	longsub(hsg, t);
}

void diffRB(dst, src1, src2)
struct RedBook *dst, *src1, *src2;	/* dst := src1 - src2 */
{	int i;

	if((i = (int)src1->rb_frame - (int)src2->rb_frame) < 0)
		dst->rb_frame = 75 + i, i = 1;
	else dst->rb_frame = i, i = 0;
	if((i = (int)src1->rb_second - (int)src2->rb_second - i) < 0)
		dst->rb_second = 60 + i, i = 1;
	else dst->rb_second = i, i = 0;
	dst->rb_minute = (int)src1->rb_minute - (int)src2->rb_minute - i;
}

char *ml(MSGID msg)
{	return msgLock(msg); }

void cd_status(void)
/* Display the status of the CD driver */
{	struct CD_INFO cd;
	unsigned stat;
	char *reply;

	drvOpen(cd);
	stat = statCD(cd);
	fd_printf(M_devStat, stat);
#define ww(nr,msg) fd_printf(msg, reply = ml((stat & (1 << (nr)))? M_yes : M_no)); msgUnlockString(reply)
	stat ^= STAT_UNLOCKED | STAT_NODISK;		/* Text has alternate meaning */
	ww(11, M_empty);
	ww(0, M_open);
	ww(1, M_locked);
	fd_printf(M_isPlay, reply = msgLock(isPlaying(cd)? M_yes: M_no)); msgUnlockString(reply);
	ww(2, M_raw);
	ww(3, M_write);
	ww(4, M_audio);
	ww(5, M_interl);
	ww(7, M_pref);
	ww(8, M_aCntrl);
	ww(9, M_redBook);
	ww(10, M_XAdecode);
	ww(12, M_rw);
#undef ww
	drvClose(cd);
}

void cd_play(char *tracks)
{	struct CD_INFO cd;
	unsigned char first, last;
	struct {
		unsigned char w_fct;
		unsigned char w_track;
		struct RedBook w_start;
		unsigned char w_flags;
	} ioctlBlk;
	struct REQ req;

	if(tracks) {
		first = last = 0;
		while(isdigit(*tracks))
			first = first * 10 + *tracks++ - '0';
		if(*tracks == '-') {
			if(isdigit(*++tracks))
				while(isdigit(*tracks))
					last = last * 10 + *tracks++ - '0';
			else
				last = ~0;
		}
		else last = first;
		if(*tracks)
			error(E_playArg);
	}
	else first = 0, last = ~0;

	drvOpen(cd);
	closeTray(cd);
	chkDisk(cd);

	if(first < cd.cd_firstTrack)
		first = cd.cd_firstTrack;
	if(last > cd.cd_lastTrack)
		last = cd.cd_lastTrack;

	while(first <= last) {
		ioctlBlk.w_fct = 0xb;			/* request audio track info */
		ioctlBlk.w_track = first;
		if(ioctlCD(cd, REQ_READ, ioctlBlk, sizeof(ioctlBlk)) >= 0
		 && !(ioctlBlk.w_flags & INF_NOAUDIO)) break;
		++first;
	}
	while(first <= last) {
		ioctlBlk.w_fct = 0xb;			/* request audio track info */
		ioctlBlk.w_track = last;
		if(ioctlCD(cd, REQ_READ, ioctlBlk, sizeof(ioctlBlk)) >= 0
		 && !(ioctlBlk.w_flags & INF_NOAUDIO)) break;
		--last;
	}
	if(first > last)
		error(E_noAudioTracks);
	informative(M_playing, first, last);

	/* calculate #secs of end address */
	ioctlBlk.w_fct = 0xb;			/* request audio track info */
	ioctlBlk.w_track = last + 1;
	if(last == cd.cd_lastTrack ||
	 ioctlCD(cd, REQ_READ, ioctlBlk, sizeof(ioctlBlk)) < 0) /* use end of audio space */
		RB2HSG(req.req_secs, cd.cd_leadOut);
	else RB2HSG(req.req_secs, ioctlBlk.w_start);

	/* calculate start sector */
	ioctlBlk.w_fct = 0xb;			/* request audio track info */
	ioctlBlk.w_track = first;
	EioctlCD(cd, REQ_READ | REQ_QUIT, ioctlBlk, sizeof(ioctlBlk));
	RB2HSG(req.req_start, ioctlBlk.w_start);

	/* calculate #secs to play */
	longsub(req.req_secs, req.req_start);

	req.req_len = sizeof(req);
	req.req_unit = drvUnit;
	req.req_cmd = 0x84;				/* Play audio */
	req.req_status = 0;
	req.req_addr = 0;				/* sectors in HSG format */
	callDriver(&cd.cd_drvOff, req);

	drvClose(cd);
}

void cd_stop(void)
{	struct CD_INFO cd;

	drvOpen(cd);
	closeTray(cd);
	drvClose(cd);
}

void cd_tok(void)
{	struct CD_INFO cd;
	char *r1, *r2, *r3;
	union {
		struct {
			unsigned char w_fct;
			unsigned char w_track;
			struct RedBook w_start;
			unsigned char w_flags;
		} io_w;
		struct {
			unsigned char w_fct;
			unsigned char w_adr;
			unsigned char w_upc[7];
			unsigned char w_dmy;
			unsigned char w_frm;
			unsigned char w_dmyu[8];
		} io_b;
	} ioctlBlk, ioBlk;
	int track;
	struct RedBook diff;
	
	drvOpen(cd);
	closeTray(cd);
	chkDisk(cd);

/*
---function 0Eh---
 01h    BYTE    CONTROL and ADR byte
 02h  7 BYTEs   UPC/EAN code (13 BCD digits,low-order nybble of last byte is 0)
 09h    BYTE    zero
 0Ah    BYTE    "AFRAME" */
	ioctlBlk.w_fct = 0xb;			/* request audio track info */
	ioctlBlk.w_track = 1;
	EioctlCD(cd, REQ_READ | REQ_QUIT, ioctlBlk, sizeof(ioctlBlk));
	diffRB(diff, cd.cd_leadOut, ioctlBlk.w_start);

	message(stdout, M_tocHead,
		cd.cd_firstTrack, cd.cd_lastTrack,
		diff.rb_minute, diff.rb_second, diff.rb_frame);

 	ioBlk.w_fct = 0x0e;			/* get UPC code */
 	ioBlk.w_adr = 3;
 	message(stdout, (ioctlCD(cd, REQ_READ, ioBlk, sizeof(ioBlk)) & ERR_IS)
 	 ? M_noUPC: M_UPC,	ioBlk.w_upc[0], ioBlk.w_upc[1],
						ioBlk.w_upc[2], ioBlk.w_upc[3],
						ioBlk.w_upc[4], ioBlk.w_upc[5],
						ioBlk.w_upc[6] >> 4);
//printf("adr: %x   frm: %x\n", ioBlk.w_adr, ioBlk.w_frm);

	message(stdout, M_table);
 	track = 1;
	do {
		ioctlBlk.w_fct = 0xb;			/* request audio track info */
		ioctlBlk.w_track = track++;
		if(ioctlCD(cd, REQ_READ, ioctlBlk, sizeof(ioctlBlk)) & ERR_IS)
			continue;
		ioBlk.w_fct = 0xb;			/* request audio track info */
		ioBlk.w_track = track;
		if((ioctlBlk.w_track = track) > cd.cd_lastTrack
		 || ioctlCD(cd, REQ_READ, ioBlk, sizeof(ioBlk)) & ERR_IS)
			diffRB(diff, cd.cd_leadOut, ioctlBlk.w_start);
		else
			diffRB(diff, ioBlk.w_start, ioctlBlk.w_start);
		printf("  %03u  %02u %02u %02u  %02u %02u %02u  %-6s %-3s  %s  %s\n",
			track - 1,
			ioctlBlk.w_start.rb_minute,
			ioctlBlk.w_start.rb_second,
			ioctlBlk.w_start.rb_frame,
			diff.rb_minute,
			diff.rb_second,
			diff.rb_frame,
#define ww(w,ja,nein) (ioctlBlk.w_flags & (w)? ja: nein)
			r1 = msgLock(ww(INF_NOAUDIO, M__cdrom, M__audio)),
			r2 = msgLock(ww(INF_NOCOPY, M_no, M_yes)),
			ww(INF_4CHANNELS, "4", "2"),
			r3 = msgLock(ww(INF_PREEMPH, M_yes, M_no))
#undef ww
		);
		msgUnlockString(r1);
		msgUnlockString(r2);
		msgUnlockString(r3);
	} while(track <= cd.cd_lastTrack);

	drvClose(cd);
}

void cd_eject(void)
{	struct CD_INFO cd;

	drvOpen(cd);
	EioctlCD(cd, REQ_WRITE | REQ_QUIT, "\0", 2);	/* eject disk */
	if(!(statCD(cd) & STAT_OPEN))
		warning(W_noOpenDoor);
	drvClose(cd);
}

int cd_locked(void)
{	struct CD_INFO cd;
	int mode;

	drvOpen(cd);
	mode = !(statCD(cd) & STAT_UNLOCKED);
	drvClose(cd);

	return mode;
}

void cd_lock(int mode)
{	struct CD_INFO cd;
	struct {
		unsigned char t_fct;
		unsigned char t_mode;
	} ioctlBlk;

	ioctlBlk.t_mode = mode &= 1;
	drvOpen(cd);
	if(!!(statCD(cd) & STAT_UNLOCKED) ^ mode)
		goto end;				/* already requested mode */
	ioctlBlk.t_fct = 0x01;	/* Lock/unlock door */
	EioctlCD(cd, REQ_WRITE | REQ_QUIT, ioctlBlk, sizeof(ioctlBlk));
	if(!!(statCD(cd) & STAT_UNLOCKED) ^ mode)
		goto end;				/* requested mode set */
	warning(W_noLockCntrl);
end:
	drvClose(cd);
}

void cd_noplay(int mode)
{	struct CD_INFO cd;

	drvOpen(cd);
	mode ^= !!isPlaying(cd);
	drvClose(cd);
	if(mode)					/* status does NOT match */
		exit(100);				/* indicate, state don't match */
}

void cd_opened(int mode)
{	struct CD_INFO cd;

	drvOpen(cd);
	mode ^= !!(statCD(cd) & STAT_OPEN);
	drvClose(cd);
	if(mode)
		exit(101);
}


int idCmd(char *cmd)
{	
	int idx;
#define CMD_PLAY 0
#define CMD_STATUS 1
#define CMD_STOP 2
#define CMD_TOK 3
#define CMD_EJECT 4
#define CMD_LOCK 5
#define CMD_UNLOCK 6
#define CMD_CLOSE 7
#define CMD_TOK2 8
#define CMD_NOPLAY 9
#define CMD_ISPLAY 10
#define CMD_ISOPEN 11
#define CMD_ISCLOSED 12

	for(idx = 0; cmds[idx]; ++idx)
		if(strbeg(cmds[idx], cmd))
			return idx;
	return -1;
}

void spawnCmd(int nr, char *p)
{	switch(nr) {
#define argc ARGC
#define argv ARGV
		default:
			if(!isdigit(*p))
				error(E_cmd, p);
			--optind;		/* emulate "play" keyword */
		case CMD_PLAY:
			if(optind + 1 < argc && isdigit(argv[optind + 1][0]))
				cd_play(argv[++optind]);	/* skip the "play" keyword */
			else cd_play(NULL);
			break;
		case CMD_STATUS:
			cd_status();
			break;
		case CMD_CLOSE:
		case CMD_STOP:
			cd_stop();
			break;
		case CMD_TOK:
		case CMD_TOK2:
			cd_tok();
			break;
		case CMD_EJECT:
			if(autoUnlock) cd_lock(0);			/* unlock door */
			else if(cd_locked())
				error(E_locked);
			cd_eject();
			break;
		case CMD_UNLOCK:
			cd_lock(0);
			break;
		case CMD_LOCK:
			cd_lock(1);
			break;
		case CMD_NOPLAY:	/* is audio NOT playing? */
			cd_noplay(0);
			break;
		case CMD_ISPLAY:
			cd_noplay(1);	/* is audio playing? */
			break;
		case CMD_ISOPEN:
			cd_opened(1);	/* is door opened? */
			break;
		case CMD_ISCLOSED:	/* is door closed? */
			cd_opened(0);
			break;
	}
#undef argc
#undef argv
}
struct {
	char *cmd;
	int nr;
} mapCmd[3] = {
	int "", CMD_PLAY,		/* the 1st entry must default to "" */
	"EJECT", CMD_EJECT,
	"CLOSE", CMD_STOP };

int defCmd(void)
{	int c;

	for(c = sizeof(mapCmd)/sizeof(mapCmd[0])
	 ; !strbeg(appName(), mapCmd[--c].cmd););
	return mapCmd[c].nr;
}

void processOpt(int c, char *argv, int stat) {
	switch(c) {
		case 'A': 	if(optarg) autoUnlock = *optarg == '+';
					else autoUnlock = !autoUnlock;
					break;
		case 'D':	if(statName) free(statName);
					if(!stat && optarg) {
						if(!(optarg = statName = strdup(optarg)))
							fatal(E_noMem);
					}
					else statName = NULL;

					if((drvName = optarg) == NULL)
						drvName = dname;
					break;
		case 'U':	if(optarg) {
						if(strlen(optarg) != strspn(optarg, "0123456789"))
							error(E_numOpt, argv[optind]);
						drvUnit = atoi(optarg);
					}
					else drvUnit = 0;
					break;
		case 'V':	message(stdout, E_version, PRGNAME, SKAUS + 7);
					exit(msgErrorNumber(E_version));
		default:	hlpScreen();
	}
}

void chkEnvOpt(char *var, char *opt, char *argopt)
{	char buf[256], *pb[3];
	int c;

	if(getenv(var, buf)) {
		pb[0] = pb[2] = 0;
		if(pb[1] = strtok(buf, " \t\n\r")) do {
			while((c = getopt1(2, pb, opt, argopt)) != EOF) 
				processOpt(c, pb, 0);
			optind = 1;
			optchar = 0;
		} while(pb[1] = strtok(NULL, " \t\n\r"));
	}
}


main(int argc, char **argv)
{	int c, cmdCnt;
	char *p;

	msgInit();

	chkEnvOpt(GSKAUS, "?AHV", "DU");

	cmdCnt = 0;
	do {
	/* read options */
		while((c = getopt1(argc, argv, "?AHV", "DU")) != EOF) 
			processOpt(c, argv, 1);

		if(!(p = argv[optind])) break;
		if(!*p) continue;

	/* read command */
		++cmdCnt;
		strlwr(p);
		spawnCmd(idCmd(p), p);
		++optind;			/* skip actual keyword */
	} while(1);

	if(!cmdCnt)
		spawnCmd(defCmd(), "");
}

void hlpScreen(void)
{	message(stdout, E_hlpScreen, appName(), GSKAUS, cmds[defCmd()]);
	exit(msgErrorNumber(E_hlpScreen));

	CSKAUS;
}



/*
--------d-214403-----------------------------
INT 21 - CD-ROM device driver - IOCTL OUTPUT
        AX = 4403h
        BX = file handle referencing character device for CD-ROM driver
        CX = number of bytes to write
        DS:DX -> control block (see #0818)
Return: CF clear if successful
            AX = number of bytes actually written
        CF set on error
            AX = error code (01h,05h,06h,0Dh) (see #0960 at AH=59h)
SeeAlso: AX=4402h"CD-ROM",INT 2F/AX=0802h

Format of CR-ROM control block:
Offset  Size    Description     (Table 0818)
 00h    BYTE    function code
                00h eject disk
                01h lock/unlock door
                02h reset drive
                03h control audio channel
                04h write device control string
                05h close tray
---functions 00h,02h,05h---
 no further fields
---function 01h---
 01h    BYTE    lock function
                00h unlock door
                01h lock door
---function 03h---
 01h    BYTE    input channel (0-3) for output channel 0
 02h    BYTE    volume for output channel 0
 03h    BYTE    input channel (0-3) for output channel 1
 04h    BYTE    volume for output channel 1
 05h    BYTE    input channel (0-3) for output channel 2
 06h    BYTE    volume for output channel 2
 07h    BYTE    input channel (0-3) for output channel 3
 08h    BYTE    volume for output channel 3
Note:   output channels 0 and 1 are left and right, 2 and 3 are left prime and
          right prime; a volume of 00h is off
---function 04h---
 01h  N BYTEs   bytes to send directly to the CD-ROM drive without
                  interpretation
--------D-2F0802-----------------------------
INT 2F U - DRIVER.SYS support - EXECUTE DEVICE DRIVER REQUEST
        AX = 0802h
        ES:BX -> device driver request header (see #1845)
Return: request header updated as per requested operation
Notes:  supported by DR-DOS 5.0
        DOS 3.2 executes this function on any AL value from 02h through F7h;
          DOS 4.0+ executes this function on AL=02h and AL=04h-F7h
        the command codes (see #1843) and structures described below apply
          to all drivers which support the appropriate commands; this call is
          just one of a number of ways in which a device driver request may
          be invoked
        this call reportedly leaves one word on the stack under MS-DOS 6.2,
          so the stack pointer should be saved and restored around the call
          to this function
SeeAlso: AX=0800h,AX=0801h,AX=0803h,INT 21/AH=52h,INT 21/AH=99h,INT 21/AH=9Ah

(Table 1843)
Values for device driver command code:
 00h    INIT
 01h    MEDIA CHECK (block devices)
 02h    BUILD BPB (block devices)
 03h    IOCTL INPUT
 04h    INPUT
 05h    NONDESTRUCTIVE INPUT, NO WAIT (character devices)
 06h    INPUT STATUS (character devices)
 07h    INPUT FLUSH (character devices)
 08h    OUTPUT
 09h    OUTPUT WITH VERIFY
 0Ah    OUTPUT STATUS (character devices)
 0Bh    OUTPUT FLUSH (character devices)
 0Ch    IOCTL OUTPUT
 0Dh    (DOS 3.0+) DEVICE OPEN
 0Eh    (DOS 3.0+) DEVICE CLOSE
 0Fh    (DOS 3.0+) REMOVABLE MEDIA (block devices)
 10h    (DOS 3.0+) OUTPUT UNTIL BUSY (character devices)
 11h    (European MS-DOS 4.0) STOP OUTPUT (console screen drivers only)
 12h    (European MS-DOS 4.0) RESTART OUTPUT (console screen drivers only)
 13h    (DOS 3.2+) GENERIC IOCTL
 14h    unused
 15h    (European MS-DOS 4.0) RESET UNCERTAIN MEDIA FLAG
 16h    unused
 17h    (DOS 3.2+) GET LOGICAL DEVICE
 18h    (DOS 3.2+) SET LOGICAL DEVICE
 19h    (DOS 5.0+) CHECK GENERIC IOCTL SUPPORT
 80h    (CD-ROM) READ LONG
 81h    (CD-ROM) reserved
 82h    (CD-ROM) READ LONG PREFETCH
 83h    (CD-ROM) SEEK
 84h    (CD-ROM) PLAY AUDIO
 85h    (CD-ROM) STOP AUDIO
 86h    (CD-ROM) WRITE LONG
 87h    (CD-ROM) WRITE LONG VERIFY
 88h    (CD-ROM) RESUME AUDIO

Bitfields for device request status:
Bit(s)  Description     (Table 1844)
 15     error
 14-11  reserved
 10     ??? set by DOS kernel on entry to some driver calls
 9      busy
 8      done (may be clear on return under European MS-DOS 4.0)
 7-0    error code if bit 15 set (see #1846)

Format of device driver request header:
Offset  Size    Description     (Table 1845)
 00h    BYTE    length of request header
 01h    BYTE    subunit within device driver
 02h    BYTE    command code (see #1843)
 03h    WORD    status (filled in by device driver) (see #1844)
---DOS---
 05h  4 BYTEs   reserved (unused in DOS 2.x and 3.x)
 09h    DWORD   (European MS-DOS 4.0 only) pointer to next request header in
                          device's request queue
                (other versions) reserved (unused in DOS 2.x and 3.x)
---STARLITE architecture---
 05h    DWORD   pointer to next request header
 09h  4 BYTEs   reserved
---command code 00h---
 0Dh    BYTE    (ret) number of units
 0Eh    DWORD   (call) pointer to DOS device helper function (see #1847)
                          (European MS-DOS 4.0 only)
                (call) pointer past end of memory available to driver (DOS 5+)
                (ret) address of first free byte following driver
 12h    DWORD   (call) pointer to commandline arguments
                (ret) pointer to BPB array (block drivers) or
                          0000h:0000h (character drivers)
 16h    BYTE    (DOS 3.0+) drive number for first unit of block driver (0=A)
   ---European MS-DOS 4.0---
 17h    DWORD   pointer to function to save registers on stack
   ---DOS 5+ ---
 17h    WORD    (ret) error-message flag
                0001h MS-DOS should display error msg on init failure
---command code 01h---
 0Dh    BYTE    media descriptor
 0Eh    BYTE    (ret) media status
                00h don't know
                01h media has not changed
                FFh media has been changed
 0Fh    DWORD   (ret, DOS 3.0+) pointer to previous volume ID if the
                  OPEN/CLOSE/RM bit in device header is set and disk changed
---command code 02h---
 0Dh    BYTE    media descriptor
 0Eh    DWORD   transfer address
                -> scratch sector if NON-IBM FORMAT bit in device header set
                -> first FAT sector otherwise
 12h    DWORD   pointer to BPB (set by driver) (see #0950 at INT 21/AH=53h)
---command codes 03h,0Ch--- (see also INT 21/AX=4402h,INT 21/AX=4403h)
 0Dh    BYTE    media descriptor (block devices only)
 0Eh    DWORD   transfer address
 12h    WORD    (call) number of bytes to read/write
                (ret) actual number of bytes read or written
---command codes 04h,08h,09h (except Compaq DOS 3.31, DR-DOS 6)---
 0Dh    BYTE    media descriptor (block devices only)
 0Eh    DWORD   transfer address
 12h    WORD    byte count (character devices) or sector count (block devices)
 14h    WORD    starting sector number (block devices only)
 16h    DWORD   (DOS 3.0+) pointer to volume ID if error 0Fh returned
 1Ah    DWORD   (DOS 4.0+) 32-bit starting sector number (block devices with
                  device attribute word bit 1 set only) if starting sector
                  number above is FFFFh (see INT 21/AH=52h)
---command codes 04h,08h,09h (Compaq DOS 3.31, DR-DOS 6)---
 0Dh    BYTE    media descriptor (block devices only)
 0Eh    DWORD   transfer address
 12h    WORD    byte count (character devices) or sector count (block devices)
 14h    DWORD   32-bit starting sector number (block devices only)
        Note:   to reliably determine which variant of the request block for
                  functions 04h,08h,09h has been passed to the driver, check
                  the length field as well as the word at offset 14h.  If the
                  length is 1Eh and 14h=FFFFh, use the DWORD at 1Ah as the
                  starting sector number; if the length is 18h, use the DWORD
                  at 14h; otherwise, use the WORD at 14h.
---command code 05h---
 0Dh    BYTE    byte read from device if BUSY bit clear on return
---command codes 06h,07h,0Ah,0Bh,0Dh,0Eh,0Fh---
 no further fields
---command code 10h---
 0Dh    BYTE    unused
 0Eh    DWORD   transfer address
 12h    WORD    (call) number of bytes to write
                (ret) actual number of bytes written
---command codes 11h,12h---
 0Dh    BYTE    reserved
---command code 15h---
 no further fields
---command codes 13h,19h---
 0Dh    BYTE    category code
                00h unknown
                01h COMn:
                03h CON
                05h LPTn:
                07h mouse (European MS-DOS 4.0)
                08h disk
                9Eh (STARLITE) Media Access Control driver
 0Eh    BYTE    function code
                00h (STARLITE) MAC Bind request
 0Fh    WORD    copy of DS at time of IOCTL call (apparently unused in DOS 3.3)
                SI contents (European MS-DOS 4.0)
 11h    WORD    offset of device driver header
                DI contents (European MS-DOS 4.0)
 13h    DWORD   pointer to parameter block from INT 21/AX=440Ch or AX=440Dh
---command codes 80h,82h---
 0Dh    BYTE    addressing mode
                00h HSG (default)
                01h Phillips/Sony Red Book
 0Eh    DWORD   transfer address (ignored for command 82h)
 12h    WORD    number of sectors to read
                (if 0 for command 82h, request is an advisory seek)
 14h    DWORD   starting sector number
                logical sector number in HSG mode
                frame/second/minute/unused in Red Book mode
                (HSG sector = minute * 4500 + second * 75 + frame - 150)
 18h    BYTE    data read mode
                00h cooked (2048 bytes per frame)
                01h raw (2352 bytes per frame, including EDC/ECC)
 19h    BYTE    interleave size (number of sectors stored consecutively)
 1Ah    BYTE    interleave skip factor
                (number of sectors between consecutive portions)
---command code 83h---
 0Dh    BYTE    addressing mode (see above)
 0Eh    DWORD   transfer address (ignored)
 12h    WORD    number of sectors to read (ignored)
 14h    DWORD   starting sector number (see also above)
---command code 84h---
 0Dh    BYTE    addressing mode (see above)
 0Eh    DWORD   starting sector number (see also above)
 12h    DWORD   number of sectors to play
---command codes 85h,88h---
 no further fields
---command codes 86h,87h---
 0Dh    BYTE    addressing mode (see above)
 0Eh    DWORD   transfer address (ignored in write mode 0)
 12h    WORD    number of sectors to write
 14h    DWORD   starting sector number (see also above)
 18h    BYTE    write mode
                00h mode 0 (write all zeros)
                01h mode 1 (default) (2048 bytes per sector)
                02h mode 2 form 1 (2048 bytes per sector)
                03h mode 2 form 2 (2336 bytes per sector)
 19h    BYTE    interleave size (number of sectors stored consecutively)
 1Ah    BYTE    interleave skip factor
                (number of sectors between consecutive portions)

(Table 1846)
Values for device driver error code:
 00h    write-protect violation
 01h    unknown unit
 02h    drive not ready
 03h    unknown command
 04h    CRC error
 05h    bad drive request structure length
 06h    seek error
 07h    unknown media
 08h    sector not found
 09h    printer out of paper
 0Ah    write fault
 0Bh    read fault
 0Ch    general failure
 0Dh    reserved
 0Eh    (CD-ROM) media unavailable
 0Fh    invalid disk change

--------d-214402-----------------------------
INT 21 - CD-ROM device driver - IOCTL INPUT
        AX = 4402h
        BX = file handle referencing character device for CD-ROM driver
        CX = number of bytes to read
        DS:DX -> control block (see #0745)
Return: CF clear if successful
            AX = number of bytes actually read
        CF set on error
            AX = error code (01h,05h,06h,0Dh) (see #0960 at AH=59h)
Note:   the data returned depends on the first byte of the control block; the
          remainder of the control block is filled by the driver
SeeAlso: AX=4403h"CD-ROM",INT 2F/AX=0802h

(Table 0744)
Values for CD-ROM data being requested:
 00h    device driver header address
 01h    drive head location
 02h    reserved
 03h    error statistics
 04h    audio channel info
 05h    raw drive bytes (uninterpreted and device-specific)
 06h    device status
 07h    sector size
 08h    volume size
 09h    media change status
 0Ah    audio disk info
 0Bh    audio track info
 0Ch    audio Q-Channel info
 0Dh    audio sub-channel info
 0Eh    UPC code
 0Fh    audio status info

Format of CD-ROM control block:
Offset  Size    Description     (Table 0745)
 00h    BYTE    data being requested (see #0744)
---function 00h---
 01h    DWORD   device driver header address (see also AH=52h)
---function 01h---
 01h    BYTE    addressing mode
                00h HSG
                01h Red Book
 02h    DWORD   current location of drive's head
                logical sector number in HSG mode
                frame/second/minute/unused in Red Book mode
                (HSG sector = minute * 4500 + second * 75 + frame - 150)
---function 03h---
 01h  N BYTEs   undefined as of 5 Aug 88 specification
---function 04h---
 01h    BYTE    input channel (0-3) for output channel 0
 02h    BYTE    volume for output channel 0
 03h    BYTE    input channel (0-3) for output channel 1
 04h    BYTE    volume for output channel 1
 05h    BYTE    input channel (0-3) for output channel 2
 06h    BYTE    volume for output channel 2
 07h    BYTE    input channel (0-3) for output channel 3
 08h    BYTE    volume for output channel 3
Notes:  output channels 0 and 1 are left and right, 2 and 3 are left prime and
          right prime; a volume of 00h is off
        the default setting is for each input channel to be assigned to the
          same-numbered output channel at full (FFh) volume
---function 05h---
 01h    BYTE    number of bytes read
 02h 128 BYTEs  buffer for drive bytes
---function 06h---
 01h    DWORD   device parameters (see #0746)
---function 07h---
 01h    BYTE    read mode
                00h cooked
                01h raw
 02h    WORD    sector size in bytes
---function 08h---
 01h    DWORD   volume size in sectors
---function 09h---
 01h    BYTE    media change status
                00h don't know
                01h media unchanged
                FFh media has been changed
---function 0Ah---
 01h    BYTE    lowest audio track number
 02h    BYTE    highest audio track number
 03h    DWORD   start address of lead-out track (Red Book format)
--function 0Bh---
 01h    BYTE    track number (set by caller)
 02h    DWORD   starting point of track (Red Book format)
 06h    BYTE    track control info
                bits 15,14,12: track type (notice: bits not contiguous!)
                        000 two audio channels, no pre-emphasis
                        001 two audio channels with pre-emphasis
                        010 data track
                        100 four audio channels, no pre-emphasis
                        101 four audio channels with pre-emphasis
                        other reserved
                bit 13: digital copy permitted
---function 0Ch---
 01h    BYTE    CONTROL and ADR byte (as received from drive)
 02h    BYTE    track number
 03h    BYTE    point or index
 04h    BYTE    minute  \
 05h    BYTE    second   > running time within track
 06h    BYTE    frame   /
 07h    BYTE    zero
 08h    BYTE    "AMIN" or "PMIN"     \
 09h    BYTE    "ASEC" or "PSEC"      > running time on disk
 0Ah    BYTE    "AFRAME" or "PFRAME" /
---function 0Dh---
 01h    DWORD   starting frame address (Red Book format)
 05h    DWORD   transfer address
 09h    DWORD   number of sectors to read
Note:   copies 96 bytes of sub-channel info per sector into buffer
---function 0Eh---
 01h    BYTE    CONTROL and ADR byte
 02h  7 BYTEs   UPC/EAN code (13 BCD digits,low-order nybble of last byte is 0)
 09h    BYTE    zero
 0Ah    BYTE    "AFRAME"
---function 0Fh---
 ??? documentation not yet available
 01h    WORD    pause status (0000h not paused, 0001h paused)
 03h    DWORD   audio play start address
 07h    DWORD   ??? audio play length or end address

Bitfields for CD-ROM device parameters:
Bit(s)  Description     (Table 0746)
 0      door open
 1      door unlocked
 2      supports raw reading in addition to cooked
 3      writable
 4      can play audio/video tracks
 5      supports interleaving
 6      reserved
 7      supports prefetch requests
 8      supports audio channel control
 9      supports Red Book addressing in addition to HSG
 10     audio is playing
 11     no disk in drive
 12     supports R-W subchannels
*/
