/* SPDX-License-Identifier: GPL-3.0-or-later */
/*
 * Text output for SDL2
 * Copyright (c) 2020-2023 Andreas K. Foerster <akf@akfoerster.de>
 *
 * This file is part of AKFText.
 *
 * AKFText is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * AKFText is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "akftext.h"
#include <stdbool.h>

#define UNDERLINED  0x1
#define UNDERLINED2 0x2
#define STRIKE      0x4

#define BACKSPACE 8

// based on https://www.cl.cam.ac.uk/~mgk25/ucs-fonts.html
extern const unsigned char Text_Font[];
extern const unsigned int Text_Font_len;

static SDL_Renderer *renderer;
static SDL_Texture *font;
static SDL_Rect char_area, area;
static int flags;
static bool wrap;

static const char16_t Extra1[] = {
  0x0102, 0x0103, 0x0104, 0x0105, 0x0106, 0x0107, 0x0108, 0x0109,
  0x010A, 0x010B, 0x010C, 0x010D, 0x010E, 0x010F, 0x0110, 0x0111,
  0x0118, 0x0119, 0x011A, 0x011B, 0x011C, 0x011D, 0x011E, 0x011F,
  0x0120, 0x0121, 0x0124, 0x0125, 0x0126, 0x0127, 0x0130, 0x0131,
  0x0132, 0x0133, 0x0134, 0x0135, 0x0139, 0x013A, 0x013D, 0x013E,
  0x0141, 0x0142, 0x0143, 0x0144, 0x0147, 0x0148, 0x0150, 0x0151,
  0x0152, 0x0153, 0x0154, 0x0155, 0x0158, 0x0159, 0x015A, 0x015B,
  0x015C, 0x015D, 0x015E, 0x015F, 0x0160, 0x0161, 0x0162, 0x0163,
  0x0164, 0x0165, 0x016C, 0x016D, 0x016E, 0x016F, 0x0170, 0x0171,
  0x0178, 0x0179, 0x017A, 0x017B, 0x017C, 0x017D, 0x017E, 0x017F,
  0x0192, 0x02C6, 0x02C7, 0x02D8, 0x02D9, 0x02DB, 0x02DC, 0x02DD,
  0x03C0, 0x0490, 0x0491
};

static const char16_t Extra2[] = {
  0x1E9E, 0x2013, 0x2014, 0x2018, 0x2019, 0x201A, 0x201C, 0x201D,
  0x201E, 0x2020, 0x2021, 0x2022, 0x2024, 0x2025, 0x2026, 0x2030,
  0x2039, 0x203A, 0x20AC, 0x2116, 0x2122, 0x2190, 0x2191, 0x2192,
  0x2193, 0x2194, 0x2195, 0x2196, 0x2197, 0x2198, 0x2199, 0x221a,
  0x221E, 0x2248, 0x2260, 0x2261, 0x2264, 0x2265, 0x2320, 0x2321,
  0x23BA, 0x23BB, 0x23BC, 0x23BD, 0x2400, 0x2409, 0x240A, 0x240B,
  0x240C, 0x240D, 0x241B, 0x2423, 0x2424, 0x2500, 0x2502, 0x250C,
  0x2510, 0x2514, 0x2518, 0x251C, 0x2524, 0x252C, 0x2534, 0x253C,
  0x2588, 0x2591, 0x2592, 0x2593, 0x25A0, 0x25C6, 0x2610, 0x2611,
  0x2612, 0x263A, 0x263B, 0x2639, 0x2640, 0x2642, 0x2660, 0x2663,
  0x2665, 0x2666, 0x26A0, 0x2713
};


static SDL_INLINE void glyph (SDL_Rect *, int, int);
static SDL_INLINE void glyphlist (SDL_Rect *, char16_t, size_t,
				  const char16_t *, size_t);
static SDL_INLINE bool combining (char32_t);
static SDL_INLINE bool nospace (char32_t);
static SDL_INLINE bool whitespace (char32_t);
static SDL_INLINE int Get_column (void);
static bool control (char32_t);
static char32_t Alias (char32_t);
static char32_t combined_letters (char32_t);
static SDL_INLINE void glyph (SDL_Rect *, int, int);
static SDL_INLINE void glyphlist (SDL_Rect *, char16_t, size_t,
				  const char16_t *, size_t);


extern int
Text_Init (SDL_Renderer * r)
{
  SDL_Surface *s;

  SDL_SetHint (SDL_HINT_RENDER_SCALE_QUALITY, "nearest");
  SDL_memset (&area, 0, sizeof (area));

  char_area.x = char_area.y = 0;
  char_area.w = Text_CharacterWidth;
  char_area.h = Text_CharacterHeight;
  flags = 0;
  wrap = false;

  if (font)
    {
      if (r == renderer)
	return 0;		// already initialized
      else			// new renderer
	{
	  SDL_DestroyTexture (font);
	  font = NULL;
	}
    }

  s = SDL_LoadBMP_RW (SDL_RWFromConstMem (Text_Font, Text_Font_len), 1);
  if (s)
    {
      SDL_SetColorKey (s, SDL_TRUE, 0);
      font = SDL_CreateTextureFromSurface (r, s);
      SDL_FreeSurface (s);
    }

  if (!font)
    return -1;

  renderer = r;

  return 0;
}


extern void
Text_Done (void)
{
  if (font)
    SDL_DestroyTexture (font);
  SDL_memset (&area, 0, sizeof (area));
  SDL_memset (&char_area, 0, sizeof (char_area));

  font = NULL;
  renderer = NULL;
  SDL_SetError ("d0e84b1a6e385ffe1bbf6f9d9d31d87625de6b5aa7a68fdb377e2701");
}


extern void
Text_Wrap (bool x)
{
  wrap = x;
}


extern void
Text_Color (Uint8 red, Uint8 green, Uint8 blue)
{
  if (font)
    SDL_SetTextureColorMod (font, red, green, blue);
}


extern void
Text_Alpha (Uint8 a)
{
  if (font)
    SDL_SetTextureAlphaMod (font, a);
}


extern void
Text_Underline (bool x)
{
  if (x)
    flags |= UNDERLINED;
  else
    flags &= ~UNDERLINED;
}


extern void
Text_Strike (bool x)
{
  if (x)
    flags |= STRIKE;
  else
    flags &= ~STRIKE;
}


extern void
Text_Home (void)
{
  char_area.x = area.x;
  char_area.y = area.y;
}


extern void
Text_Position (int x, int y)
{
  char_area.x = area.x + x;
  char_area.y = area.y + y;
}


extern int
Text_Get_Y (void)
{
  return (char_area.y - area.y);
}


extern int
Text_Get_X (void)
{
  return (char_area.x - area.x);
}


extern void
Text_Size (int width, int height)
{
  if (width <= 0)
    width = Text_CharacterWidth;

  if (height <= 0)
    height = Text_CharacterHeight;

  char_area.w = width;
  char_area.h = height;
}


extern void
Text_Area (const SDL_Rect * r)
{
  if (r)
    SDL_memcpy (&area, r, sizeof (area));
  else
    SDL_memset (&area, 0, sizeof (area));

  char_area.x = area.x;
  char_area.y = area.y;
}


extern SDL_Rect *
Text_Get_Area (void)
{
  return &area;
}


static SDL_INLINE bool
outside (void)
{
  if (!SDL_RectEmpty (&area))
    {
      if ((char_area.x < area.x)
	  || (char_area.y < area.y)
	  || (char_area.x + char_area.w > area.x + area.w)
	  || (char_area.y + char_area.h > area.y + area.h))
	return true;
    }

  return false;
}


static SDL_INLINE bool
behind_line (void)
{
  return (char_area.x + char_area.w > area.x + area.w);
}


// control and zero width characters, not combining characters
static SDL_INLINE bool
nospace (char32_t c)
{
  // TODO: incomplete
  return (c < 0x20 || (0x7F <= c && c <= 0x9F) || c == 0xAD
	  || (0x200B <= c && c <= 0x200F) || c == 0x2060
	  || (0xFFF9 <= c && c <= 0xFFFB)
	  || c == 0xFEFF || (0xE0000 <= c && c <= 0xE007F));
}


// only supported range
static SDL_INLINE bool
combining (char32_t c)
{
  return (0x0300 <= c && c <= 0x036F);
}


static SDL_INLINE bool
whitespace (char32_t c)
{
  return (c == 0x20 || c == 0xA0 || (0x2000 <= c && c <= 0x200A)
	  || c == 0x202F || c == 0x205F || c == 0x1680 || c == 0x3000);
}


// unreliable if Text_Position is used or Text_Size within a line
static SDL_INLINE int
Get_column (void)
{
  return ((char_area.x - area.x) / char_area.w);
}


static bool
control (char32_t c)
{
  switch (c)
    {
    case '\n':			// LF
    case '\v':			// VT
    case '\f':			// FF
    case 0x85:			// NEL
    case 0x2028:		// LINE SEPARATOR
    case 0x2029:		// PARAGRAPH SEPARATOR
      char_area.y += char_area.h;
      char_area.x = area.x;
      break;

    case 0x84:			// IND
      char_area.y += char_area.h;
      break;

    case 0x8D:			// RI
      char_area.y -= char_area.h;
      break;

    case '\r':			// CR
      char_area.x = area.x;
      break;

    case '\t':			// HT
      char_area.x += (8 - (Get_column () % 8)) * char_area.w;
      break;

    case '\b':			// BS
      if (char_area.x - char_area.w >= area.x)
	char_area.x -= char_area.w;
      break;

    case 0x0E:			// SO
      flags |= UNDERLINED;
      break;

    case 0x0F:			// SI
      flags &= ~UNDERLINED;
      break;
    }

  // 0x1A (substitute) is handled by Alias()

  return nospace (c);
}


// no side effects
static char32_t
Alias (char32_t c)
{
  char32_t result;

  // Unicode is huge!

  switch (c)
    {
    case 0x1A:			// substitute
      result = 0xFFFD;		// replacement character
      break;

    case 0x2010:		// hyphen
    case 0x2011:		// non-breaking hyphen
    case 0x2012:		// figure dash
    case 0x2043:		// hyphen bullet
    case 0x2212:		// minus sign
      result = '-';
      break;

    case 0x2015:		// horizontal bar
      result = 0x2014;		// em-dash
      break;

    case 0x2236:		// ratio
    case 0xA789:		// modifier letter colon
      result = ':';
      break;

    case 0x2044:		// fraction slash
    case 0x2215:		// division slash
    case 0x27CB:		// mathematical rising diagonal
      result = '/';
      break;

    case 0x2216:		// set minus
      result = '\\';
      break;

    case 0x2217:		// asterisk operator
      result = '*';
      break;

    case 0x0138:		// kra
      result = 0x043A;
      break;

    case 0x01C0:		// latin letter dental click
    case 0x2223:		// divides
    case 0x2758:		// light vertical bar
      result = '|';
      break;

    case 0x2027:		// hyphenation point
    case 0x22C5:		// dot operator
    case 0x2E31:		// word separator middle dot
    case 0xA78F:		// latin letter sinological dot
      result = 0xB7;		// middle dot
      break;

    case 0x2219:		// bullet operator
      result = 0x2022;		// bullet
      break;

    case 0x2715:		// multiplication X
    case 0x2716:
    case 0x2717:
    case 0x2718:
    case 0x1F5D9:
    case 0x1F5F4:
    case 0x1F5F6:
      result = 0xD7;		// multiplication sign
      break;

    case 0x2714:		// heavy check mark
    case 0x1F5F8:		// light checkmark
      result = 0x2713;		// check mark
      break;

    case 0x1F5F5:		// ballot box with script x
    case 0x1F5F7:		// ballot box with bold script x
      result = 0x2612;
      break;

    case 0x1F5F9:		// ballot box with bold check
      result = 0x2611;
      break;

    case 0x27E8:		// mathematical left angle bracket
      result = '<';
      break;

    case 0x27E9:		// mathematical right double angle bracket
      result = '>';
      break;

    case 0x27EA:		// mathematical left double angle bracket
      result = 0xAB;
      break;

    case 0x27EB:		// mathematical right double angle bracket
      result = 0xBB;
      break;

    case 0x0218:		// S with comma below
      result = 0x015E;		// S with cedilla
      break;

    case 0x0219:		// s with comma below
      result = 0x015F;		// s with cedilla
      break;

    case 0x021A:		// T with comma below
      result = 0x0162;		// T with cedilla
      break;

    case 0x021B:		// t with comma below
      result = 0x0163;		// t with cedilla
      break;

    case 0x0400:		// E with grave (cyrillic)
      result = 0xC8;
      break;

      // combining characters
    case 0x0300:		// grave
      result = 0x60;
      break;

    case 0x0301:		// acute
      result = 0xB4;
      break;

    case 0x0302:		// circumflex
      result = 0x02C6;
      break;

    case 0x0303:		// tilde
      result = 0x02DC;
      break;

    case 0x0304:		// macron
      result = 0xAF;
      break;

    case 0x0305:		// overline
      result = 0x23BA;
      break;

    case 0x0306:		// breve
      result = 0x02D8;
      break;

    case 0x0307:		// dot above
      result = 0x02D9;
      break;

    case 0x0308:		// double dot above
      result = 0xA8;
      break;

/*
    case 0x030A:		// ring above
      result = 0xB0;
      break;
*/

    case 0x030B:		// double acute
      result = 0x02DD;
      break;

    case 0x030C:		// caron
      result = 0x02C7;
      break;

    case 0x0327:		// cedille
      result = 0xB8;
      break;

    case 0x0328:		// ogonek
      result = 0x02DB;
      break;

      // map different borders to single line
    case 0x2501:
    case 0x2504:
    case 0x2505:
    case 0x2508:
    case 0x2509:
    case 0x254C:
    case 0x254D:
    case 0x2550:
    case 0x257C:
    case 0x257E:
      result = 0x2500;
      break;

    case 0x2503:
    case 0x2506:
    case 0x2507:
    case 0x250A:
    case 0x250B:
    case 0x254E:
    case 0x254F:
    case 0x2551:
    case 0x257D:
    case 0x257F:
      result = 0x2502;
      break;

    case 0x250D:
    case 0x250E:
    case 0x250F:
    case 0x2552:
    case 0x2553:
    case 0x2554:
    case 0x256D:
      result = 0x250C;
      break;

    case 0x2511:
    case 0x2512:
    case 0x2513:
    case 0x2555:
    case 0x2556:
    case 0x2557:
    case 0x256E:
      result = 0x2510;
      break;

    case 0x2515:
    case 0x2516:
    case 0x2517:
    case 0x2558:
    case 0x2559:
    case 0x255A:
    case 0x2570:
      result = 0x2514;
      break;

    case 0x2519:
    case 0x251A:
    case 0x251B:
    case 0x255B:
    case 0x255C:
    case 0x255D:
    case 0x256F:
      result = 0x2518;
      break;

    case 0x251D:
    case 0x251E:
    case 0x251F:
    case 0x2520:
    case 0x2521:
    case 0x2522:
    case 0x2523:
    case 0x255E:
    case 0x255F:
    case 0x2560:
      result = 0x251C;
      break;

    case 0x2525:
    case 0x2526:
    case 0x2527:
    case 0x2528:
    case 0x2529:
    case 0x252A:
    case 0x252B:
    case 0x2561:
    case 0x2562:
    case 0x2563:
      result = 0x2524;
      break;

    case 0x252C:
    case 0x252D:
    case 0x252E:
    case 0x252F:
    case 0x2530:
    case 0x2531:
    case 0x2532:
    case 0x2533:
    case 0x2564:
    case 0x2565:
    case 0x2566:
      result = 0x252C;
      break;

    case 0x2535:
    case 0x2536:
    case 0x2537:
    case 0x2538:
    case 0x2539:
    case 0x253A:
    case 0x253B:
    case 0x2567:
    case 0x2568:
    case 0x2569:
      result = 0x2534;
      break;

    case 0x253D:
    case 0x253E:
    case 0x253F:
    case 0x2540:
    case 0x2541:
    case 0x2542:
    case 0x2543:
    case 0x2544:
    case 0x2545:
    case 0x2546:
    case 0x2547:
    case 0x2548:
    case 0x2549:
    case 0x254A:
    case 0x254B:
    case 0x256A:
    case 0x256B:
    case 0x256C:
      result = 0x253C;
      break;


    default:
      result = c;
      break;
    }

  return result;
}


static char32_t
combined_letters (char32_t c)
{
  char32_t base_letter, combination;

  // These combinations are not ideal, but better than nothing
  // Capital letters are most problematic

  switch (c)
    {
    case 0x0100:		// A with macron
      base_letter = 'A';
      combination = 0x00AF;	// macron
      break;

    case 0x0101:		// a with macron
      base_letter = 'a';
      combination = 0x00AF;	// macron
      break;

    case 0x0112:		// E with macron
      base_letter = 'E';
      combination = 0x00AF;	// macron
      break;

    case 0x0113:		// e with macron
      base_letter = 'e';
      combination = 0x00AF;	// macron
      break;

    case 0x0114:		// E with breve
      base_letter = 'E';
      combination = 0x02D8;	// breve
      break;

    case 0x0115:		// e with breve
      base_letter = 'e';
      combination = 0x02D8;	// breve
      break;

    case 0x0116:		// E with dot
      base_letter = 'E';
      combination = 0x02D9;	// dot
      break;

    case 0x0117:		// e with dot
      base_letter = 'e';
      combination = 0x02D9;	// dot
      break;

    case 0x0122:		// G with cedille
      base_letter = 'G';
      combination = 0x00B8;	// cedille
      break;

    case 0x0123:		// g with cedille
      base_letter = 'g';
      combination = 0x00B4;	// acute
      break;

    case 0x0128:		// I with tilde FIXME
      base_letter = 'I';
      combination = 0x02DC;
      break;

    case 0x0129:		// i with tilde
      base_letter = 0x0131;
      combination = 0x02DC;
      break;

    case 0x012A:		// I with macron FIXME
      base_letter = 'I';
      combination = 0x00AF;	// macron
      break;

    case 0x012B:		// i with macron
      base_letter = 0x0131;
      combination = 0x00AF;	// macron
      break;

    case 0x012C:		// I with breve
      base_letter = 'I';
      combination = 0x02D8;	// breve
      break;

    case 0x012D:		// i with breve
      base_letter = 0x0131;
      combination = 0x02D8;	// breve
      break;

    case 0x012E:		// I with ogonek
      base_letter = 'I';
      combination = 0x02DB;	// ogonek
      break;

    case 0x012F:		// i with ogonek
      base_letter = 'i';
      combination = 0x02DB;	// ogonek
      break;

    case 0x0136:		// K with cedille
      base_letter = 'K';
      combination = 0x00B8;	// cedille
      break;

    case 0x0137:		// k with cedille
      base_letter = 'k';
      combination = 0x00B8;	// cedille
      break;

    case 0x013B:		// L with cedille
      base_letter = 'L';
      combination = 0x00B8;	// cedille
      break;

    case 0x013C:		// l with cedille
      base_letter = 'l';
      combination = 0x00B8;	// cedille
      break;

    case 0x013F:		// L with middle dot
      base_letter = 'L';
      combination = 0xB7;	// middle dot
      break;

    case 0x0140:		// l with middle dot
      base_letter = 'l';
      combination = 0xB7;	// middle dot
      break;

    case 0x0145:		// N with cedille
      base_letter = 'N';
      combination = 0x00B8;	// cedille
      break;

    case 0x0146:		// n with cedille
      base_letter = 'n';
      combination = 0x00B8;	// cedille
      break;

    case 0x014C:		// O with macron
      base_letter = 'O';
      combination = 0x00AF;	// macron
      break;

    case 0x014D:		// o with macron
      base_letter = 'o';
      combination = 0x00AF;	// macron
      break;

    case 0x014E:		// O with breve
      base_letter = 'O';
      combination = 0x02D8;	// breve
      break;

    case 0x014F:		// o with breve
      base_letter = 'o';
      combination = 0x02D8;	// breve
      break;

    case 0x0156:		// R with cedille
      base_letter = 'R';
      combination = 0xB8;	// cedille
      break;

    case 0x0157:		// r with cedille
      base_letter = 'r';
      combination = 0x00B8;	// cedille
      break;

    case 0x0168:		// U with tilde FIXME
      base_letter = 'U';
      combination = 0x02DC;
      break;

    case 0x0169:		// u with tilde
      base_letter = 'u';
      combination = 0x02DC;
      break;

    case 0x016A:		// U with macron
      base_letter = 'U';
      combination = 0x00AF;	// macron
      break;

    case 0x016B:		// u with macron
      base_letter = 'u';
      combination = 0x00AF;	// macron
      break;

    case 0x0172:		// U with ogonek
      base_letter = 'U';
      combination = 0x02DB;	// ogonek
      break;

    case 0x0173:		// u with ogonek
      base_letter = 'u';
      combination = 0x02DB;	// ogonek
      break;

    case 0x0174:		// W with circumflex
      base_letter = 'W';
      combination = 0x02C6;	// circumflex
      break;

    case 0x0175:		// w with circumflex
      base_letter = 'w';
      combination = 0x02C6;	// circumflex
      break;

    case 0x0176:		// Y with circumflex
      base_letter = 'Y';
      combination = 0x02C6;	// circumflex
      break;

    case 0x0177:		// y with circumflex
      base_letter = 'y';
      combination = 0x02C6;	// circumflex
      break;

    default:
      base_letter = c;
      combination = 0;
      break;
    }

  if (combination)
    {
      Text_Character (combination);
      control (BACKSPACE);
    }

  return base_letter;
}


// selects glyph from map
static SDL_INLINE void
glyph (SDL_Rect * r, int row, int column)
{
  r->x = column * Text_CharacterWidth;
  r->y = row * Text_CharacterHeight;
}


// selects glyph from map via list (only for char16_t)
static SDL_INLINE void
glyphlist (SDL_Rect * r, char16_t c, size_t row,
	   const char16_t * list, size_t length)
{
  for (size_t i = 0; i < length; ++i)
    if (c == list[i])
      {
	glyph (r, row, i);
	break;
      }
}


extern void
Text_Character (char32_t c)
{
  SDL_Rect r;

  if (!font)
    return;

  // combining character
  if (combining (c))
    control (BACKSPACE);

  c = Alias (c);

  // note: control characters may change the position
  if (control (c))
    return;

  // wrap?
  if (wrap && behind_line ())
    control ('\n');

  if (outside ())
    return;

  c = combined_letters (c);

  r.w = Text_CharacterWidth;
  r.h = Text_CharacterHeight;

  if (flags & (UNDERLINED | UNDERLINED2))
    {
      glyph (&r, 0, '_' - 0x21);
      SDL_RenderCopy (renderer, font, &r, &char_area);

      if (flags & UNDERLINED2)
	{
	  glyph (&r, 4, 43);
	  SDL_RenderCopy (renderer, font, &r, &char_area);
	}
    }

  if (flags & STRIKE)
    {
      glyph (&r, 4, 53);	// U+2500
      SDL_RenderCopy (renderer, font, &r, &char_area);
    }

  // replacement character
  // 0xFFFD is not supported directly but implicit
  glyph (&r, 0, 0x7F - 0x21);

  if (whitespace (c))
    goto next_position;
  else if (0x21 <= c && c <= 0x7E)	// ASCII
    glyph (&r, 0, c - 0x21);
  else if (0xA1 <= c && c <= 0xFF)	// Latin-1
    glyph (&r, 1, c - 0xA1);
  else if (0x0401 <= c && c <= 0x045F)	// Cyrillic
    glyph (&r, 2, c - 0x0401);
  else if (0x0102 <= c && c <= 0x0491)	// Extra1
    glyphlist (&r, c, 3, Extra1, SDL_arraysize (Extra1));
  else if (0x1E9E <= c && c <= 0x2713)	// Extra2
    glyphlist (&r, c, 4, Extra2, SDL_arraysize (Extra2));

  SDL_RenderCopy (renderer, font, &r, &char_area);

next_position:
  char_area.x += char_area.w;
}
