/*
 * ported to Javascript/DOjs in 2020 by Andre Seidelt <superilu@yahoo.com>
 * Original code from https://github.com/DhrBaksteen/ArduinoOPL2
 */

/*
 * ________ __________.____    ________      _____            .___.__         .____    ._____.
 * \_____  \\______   \    |   \_____  \    /  _  \  __ __  __| _/|__| ____   |    |   |__\_ |__
 *  /   |   \|     ___/    |    /  ____/   /  /_\  \|  |  \/ __ | |  |/  _ \  |    |   |  || __ \
 * /    |    \    |   |    |___/       \  /    |    \  |  / /_/ | |  (  <_> ) |    |___|  || \_\ \
 * \_______  /____|   |_______ \_______ \ \____|__  /____/\____ | |__|\____/  |_______ \__||___  /
 *         \/                 \/       \/ _____   \/           \/                     \/       \/
 *                                      _/ ____\___________
 *                                      \   __\/  _ \_  __ \
 *                                       |  | (  <_> )  | \/
 *                                       |__|  \____/|__|
 *               _____            .___    .__                  ____    __________.__
 *              /  _  \_______  __| _/_ __|__| ____   ____    /  _ \   \______   \__|
 *             /  /_\  \_  __ \/ __ |  |  \  |/    \ /  _ \   >  _ </\  |     ___/  |
 *            /    |    \  | \/ /_/ |  |  /  |   |  (  <_> ) /  <_\ \/  |    |   |  |
 *            \____|__  /__|  \____ |____/|__|___|  /\____/  \_____\ \  |____|   |__|
 *                    \/           \/             \/                \/
 *
 * YM3812 OPL2 Audio Library for Arduino, Raspberry Pi and Orange Pi v1.5.0
 * Code by Maarten Janssen (maarten@cheerful.nl) 2016-12-18
 *
 * Look for example code on how to use this library in the examples folder.
 *
 * Connect the OPL2 Audio Board as follows. To learn how to connect your favorite development platform visit the wiki at
 * https://github.com/DhrBaksteen/ArduinoOPL2/wiki/Connecting.
 *    OPL2 Board | Arduino | Raspberry Pi
 *   ------------+---------+--------------
 *      Reset    |    8    |      18
 *      A0       |    9    |      16
 *      Latch    |   10    |      12
 *      Data     |   11    |      19
 *      Shift    |   13    |      23
 *
 *
 * IMPORTANT: Make sure you set the correct BOARD_TYPE in OPL2.h. Default is set to Arduino.
 *
 *
 * Last updated 2019-11-10
 * Most recent version of the library can be found at my GitHub: https://github.com/DhrBaksteen/ArduinoOPL2
 * Details about the YM3812 and OPL chips can be found at http://www.shikadi.net/moddingwiki/OPL_chip
 *
 * This library is open source and provided as is under the MIT software license, a copy of which is provided as part of
 * the project's repository. This library makes use of Gordon Henderson's Wiring Pi.
 */


/**
 * Instantiate the OPL2-Object.
 */
exports.OPL2 = function OPL2() {
	if (MIDI_AVAILABLE == true) {
		throw new Error("OPL2 does only work when Allegro FM sound is disabled (-f on command line or in DOjS.INI)");
	}

	this.oplRegisters = [];
	for (var i = 0; i < 256; i++) {
		this.oplRegisters.push(0);
	}

	this.fIntervals = [
		0.048, 0.095, 0.190, 0.379, 0.759, 1.517, 3.034, 6.069
	];
	this.noteFNumbers = [
		0x156, 0x16B, 0x181, 0x198, 0x1B0, 0x1CA,
		0x1E5, 0x202, 0x220, 0x241, 0x263, 0x287
	];
	this.blockFrequencies = [
		48.503, 97.006, 194.013, 388.026,
		776.053, 1552.107, 3104.215, 6208.431
	];
	this.registerOffsets = [
		[0x00, 0x01, 0x02, 0x08, 0x09, 0x0A, 0x10, 0x11, 0x12],   /*  initializers for operator 1 */
		[0x03, 0x04, 0x05, 0x0B, 0x0C, 0x0D, 0x13, 0x14, 0x15]     /*  initializers for operator 2 */
	];
	this.drumRegisterOffsets = [
		[0x10, 0xFF, 0x12, 0xFF, 0x11],
		[0x13, 0x14, 0xFF, 0x15, 0xFF]
	];
	this.drumOffsets = [
		0x10, 0x13, 0x14, 0x12, 0x15, 0x11
	];
	this.drumChannels = [
		6, 7, 8, 8, 7
	];
	this.drumBits = [
		0x10, 0x08, 0x04, 0x02, 0x01
	];
	this.instrumentBaseRegs = [
		0x20, 0x40, 0x60, 0x80, 0xE0, 0xC0
	];

	this.OPL2_NUM_CHANNELS = 9;

	// Operator definitions.
	this.OPERATOR1 = 0;
	this.OPERATOR2 = 1;
	this.MODULATOR = 0;
	this.CARRIER = 1;

	// Synthesis type definitions.
	this.FREQ_MODULATION = false;
	this.ADDITIVE_SYNTH = true;

	// Drum sounds.
	this.DRUM_BASS = 0x10;
	this.DRUM_SNARE = 0x08;
	this.DRUM_TOM = 0x04;
	this.DRUM_CYMBAL = 0x02;
	this.DRUM_HI_HAT = 0x01;

	// Instrument type definitions.
	this.INSTRUMENT_TYPE_MELODIC = 0;
	this.INSTRUMENT_TYPE_BASS = 6;
	this.INSTRUMENT_TYPE_SNARE = 7;
	this.INSTRUMENT_TYPE_TOM = 8;
	this.INSTRUMENT_TYPE_CYMBAL = 9;
	this.INSTRUMENT_TYPE_HI_HAT = 10;

	this.Reset();
};


/**
 * detect the presence of a FM card.
 */
exports.OPL2.prototype.Detect = function () {
	var A, B;

	this.Write(1, 0);
	this.Write(4, 0x60);
	this.Write(4, 0x80);
	A = InPortByte(0x388);
	this.Write(2, 0xFF);
	this.Write(4, 0x21);
	Sleep(80);
	B = InPortByte(0x388);
	this.Write(4, 0x60);
	this.Write(4, 0x80);
	if ((A & 0xE0) == 0 && (B & 0xE0) == 0xC0) {
		return true;
	} else {
		return false;
	}
}


/**
 * Hard reset the OPL2 chip. This should be done before sending any register data to the chip.
 */
exports.OPL2.prototype.Reset = function () {
	for (var i = 0; i < 256; i++) {
		this.oplRegisters[i] = 0x00;
		this.Write(i, 0x00);
	}
};

/**
 * Send the given byte of data to the given register of the OPL2 chip.
 */
exports.OPL2.prototype.Write = function (reg, val) {
	// Select register
	OutPortByte(0x388, reg);
	// Wait for card to accept value
	for (var i = 1; i < 25; i++) {
		InPortByte(0x388);
	}
	// Send value
	OutPortByte(0x389, val);
	// Wait for card to accept value
	for (var i = 1; i < 100; i++) {
		InPortByte(0x388);
	}
};

/**
 * Get the current value of the given register.
 */
exports.OPL2.prototype.GetRegister = function (reg) {
	return this.oplRegisters[reg];
};


/**
 * Sets the given register to the given value.
 */
exports.OPL2.prototype.SetRegister = function (reg, value) {
	this.oplRegisters[reg] = value;
	this.Write(reg, value);
	return reg;
};


/**
 * Calculate register offet based on channel and operator.
 */
exports.OPL2.prototype.GetRegisterOffset = function (channel, operatorNum) {
	channel = Math.max(0, min(channel, 8));
	operatorNum = Math.max(0, min(operatorNum, 1));
	return this.registerOffsets[operatorNum][channel];
};


/**
 * Get the F-number for the given frequency for a given channel. When the F-number is calculated the current frequency
 * block of the channel is taken into account.
 */
exports.OPL2.prototype.GetFrequencyFNumber = function (channel, frequency) {
	var fInterval = this.GetFrequencyStep(channel);
	return Math.floor(Math.max(0, Math.min(frequency / fInterval, 1023)));
};


/**
 * Get the F-Number for the given note. In this case the block is assumed to be the octave.
 */
exports.OPL2.prototype.GetNoteFNumber = function (note) {
	return this.noteFNumbers[Math.max(0, Math.min(note, 11))];
};

/**
 * Get the frequency step per F-number for the current block on the given channel.
 */
exports.OPL2.prototype.GetFrequencyStep = function (channel) {
	return this.fIntervals[this.GetBlock(channel)];
};


/**
 * Get the optimal frequency block for the given frequency.
 */
exports.OPL2.prototype.GetFrequencyBlock = function (frequency) {
	for (var i = 0; i < 8; i++) {
		if (frequency < this.blockFrequencies[i]) {
			return i;
		}
	}
	return 7;
};


/**
 * Create and return a new empty instrument.
 */
exports.OPL2.prototype.CreateInstrument = function () {
	var instrument = {
		feedback: 0,
		isAdditiveSynth: false,
		type: this.INSTRUMENT_TYPE_MELODIC,
		operators: [{}, {}]
	};

	for (var op = this.OPERATOR1; op <= this.OPERATOR2; op++) {
		instrument.operators[op].hasTremolo = false;
		instrument.operators[op].hasVibrato = false;
		instrument.operators[op].hasSustain = false;
		instrument.operators[op].hasEnvelopeScaling = false;
		instrument.operators[op].frequencyMultiplier = 0;
		instrument.operators[op].keyScaleLevel = 0;
		instrument.operators[op].outputLevel = 0;
		instrument.operators[op].attack = 0;
		instrument.operators[op].decay = 0;
		instrument.operators[op].sustain = 0;
		instrument.operators[op].release = 0;
		instrument.operators[op].waveForm = 0;
	}

	return instrument;
};


/**
 * Create an instrument and load it with instrument parameters from the given instrument data pointer.
 */
exports.OPL2.prototype.LoadInstrument = function (instrumentData) {
	var instrument = this.CreateInstrument();

	var data = [];
	for (var i = 0; i < 12; i++) {
		data.push(instrumentData[i]);
	}

	for (var op = this.OPERATOR1; op <= this.OPERATOR2; op++) {
		instrument.operators[op].hasTremolo = data[op * 6 + 1] & 0x80 ? true : false;
		instrument.operators[op].hasVibrato = data[op * 6 + 1] & 0x40 ? true : false;
		instrument.operators[op].hasSustain = data[op * 6 + 1] & 0x20 ? true : false;
		instrument.operators[op].hasEnvelopeScaling = data[op * 6 + 1] & 0x10 ? true : false;
		instrument.operators[op].frequencyMultiplier = (data[op * 6 + 1] & 0x0F);
		instrument.operators[op].keyScaleLevel = (data[op * 6 + 2] & 0xC0) >> 6;
		instrument.operators[op].outputLevel = data[op * 6 + 2] & 0x3F;
		instrument.operators[op].attack = (data[op * 6 + 3] & 0xF0) >> 4;
		instrument.operators[op].decay = data[op * 6 + 3] & 0x0F;
		instrument.operators[op].sustain = (data[op * 6 + 4] & 0xF0) >> 4;
		instrument.operators[op].release = data[op * 6 + 4] & 0x0F;
		instrument.operators[op].waveForm = data[op * 6 + 5] & 0x03;
	}

	instrument.feedback = (data[6] & 0x0E) >> 1;
	instrument.isAdditiveSynth = data[6] & 0x01 ? true : false;
	instrument.type = data[0];

	return instrument;
};


/**
 * Create a new instrument from the given OPL2 channel.
 */
exports.OPL2.prototype.GetInstrument = function (channel) {
	var instrument = {
		feedback: this.GetFeedback(channel),
		isAdditiveSynth: this.GetSynthMode(channel),
		type: this.INSTRUMENT_TYPE_MELODIC,
		operators: [{}, {}]
	};

	for (var op = this.OPERATOR1; op <= this.OPERATOR2; op++) {
		instrument.operators[op].hasTremolo = this.GetTremolo(channel, op);
		instrument.operators[op].hasVibrato = this.GetVibrato(channel, op);
		instrument.operators[op].hasSustain = this.GetMaintainSustain(channel, op);
		instrument.operators[op].hasEnvelopeScaling = this.GetEnvelopeScaling(channel, op);
		instrument.operators[op].frequencyMultiplier = this.GetMultiplier(channel, op);
		instrument.operators[op].keyScaleLevel = this.GetScalingLevel(channel, op);
		instrument.operators[op].outputLevel = this.GetVolume(channel, op);
		instrument.operators[op].attack = this.GetAttack(channel, op);
		instrument.operators[op].decay = this.GetDecay(channel, op);
		instrument.operators[op].sustain = this.GetSustain(channel, op);
		instrument.operators[op].release = this.GetRelease(channel, op);
		instrument.operators[op].waveForm = this.GetWaveForm(channel, op);
	}

	return instrument;
};


/**
 * Create a new drum instrument that can be used in percussion mode. The drumType is one of INSTRUMENT_TYPE_* other than
 * MELODIC and this determines the channel operator(s) to be loaded into the instrument object.
 */
exports.OPL2.prototype.GetDrumInstrument = function (drumType) {
	var channel = this.drumChannels[drumType - this.INSTRUMENT_TYPE_BASS];
	var instrument = this.CreateInstrument();

	for (var op = this.OPERATOR1; op <= this.OPERATOR2; op++) {
		if (drumRegisterOffsets[op][drumType - this.INSTRUMENT_TYPE_BASS] != 0xFF) {
			instrument.operators[op].hasTremolo = this.GetTremolo(channel, op);
			instrument.operators[op].hasVibrato = this.GetVibrato(channel, op);
			instrument.operators[op].hasSustain = this.GetMaintainSustain(channel, op);
			instrument.operators[op].hasEnvelopeScaling = this.GetEnvelopeScaling(channel, op);
			instrument.operators[op].frequencyMultiplier = this.GetMultiplier(channel, op);
			instrument.operators[op].keyScaleLevel = this.GetScalingLevel(channel, op);
			instrument.operators[op].outputLevel = this.GetVolume(channel, op);
			instrument.operators[op].attack = this.GetAttack(channel, op);
			instrument.operators[op].decay = this.GetDecay(channel, op);
			instrument.operators[op].sustain = this.GetSustain(channel, op);
			instrument.operators[op].release = this.GetRelease(channel, op);
			instrument.operators[op].waveForm = this.GetWaveForm(channel, op);
		}
	}

	instrument.feedback = 0x00;
	instrument.isAdditiveSynth = false;
	instrument.type = drumType;

	return instrument;
};


/**
 * Set the given instrument to a channel. An optional volume may be provided to assign to proper output levels for the
 * operators.
 */
exports.OPL2.prototype.SetInstrument = function (channel, instrument, volume) {
	channel = Math.max(0, Math.min(channel, 8));
	volume = Math.max(0.0, Math.min(volume, 1.0));

	this.SetWaveFormSelect(true);
	for (var op = this.OPERATOR1; op <= this.OPERATOR2; op++) {
		var outputLevel = 63 - Math.floor(((63.0 - instrument.operators[op].outputLevel) * volume));
		var registerOffset = this.registerOffsets[op][channel];

		this.Write(0x20 + registerOffset,
			(instrument.operators[op].hasTremolo ? 0x80 : 0x00) +
			(instrument.operators[op].hasVibrato ? 0x40 : 0x00) +
			(instrument.operators[op].hasSustain ? 0x20 : 0x00) +
			(instrument.operators[op].hasEnvelopeScaling ? 0x10 : 0x00) +
			(instrument.operators[op].frequencyMultiplier & 0x0F));
		this.Write(0x40 + registerOffset,
			((instrument.operators[op].keyScaleLevel & 0x03) << 6) +
			(outputLevel & 0x3F));
		this.Write(0x60 + registerOffset,
			((instrument.operators[op].attack & 0x0F) << 4) +
			(instrument.operators[op].decay & 0x0F));
		this.Write(0x80 + registerOffset,
			((instrument.operators[op].sustain & 0x0F) << 4) +
			(instrument.operators[op].release & 0x0F));
		this.Write(0xE0 + registerOffset,
			(instrument.operators[op].waveForm & 0x03));
	}

	this.Write(0xC0 + channel,
		((instrument.feedback & 0x07) << 1) +
		(instrument.isAdditiveSynth ? 0x01 : 0x00));
};


/**
 * Set the given drum instrument that can be used in percussion mode. Depending on the type of drum the instrument
 * parameters will be loaded into the appropriate channel operator(s). An optional volume may be provided to set the
 * proper output levels for the operator(s).
 */
exports.OPL2.prototype.SetDrumInstrument = function (instrument, volume) {
	volume = Math.max(0.0, Math.min(volume, 1.0));

	this.SetWaveFormSelect(true);
	for (var op = this.OPERATOR1; op <= this.OPERATOR2; op++) {
		var outputLevel = 63 - Math.floor(((63.0 - instrument.operators[op].outputLevel) * volume));
		var registerOffset = drumRegisterOffsets[op][instrument.type - this.INSTRUMENT_TYPE_BASS];

		if (registerOffset != 0xFF) {
			this.Write(0x20 + registerOffset,
				(instrument.operators[0].hasTremolo ? 0x80 : 0x00) +
				(instrument.operators[0].hasVibrato ? 0x40 : 0x00) +
				(instrument.operators[0].hasSustain ? 0x20 : 0x00) +
				(instrument.operators[0].hasEnvelopeScaling ? 0x10 : 0x00) +
				(instrument.operators[0].frequencyMultiplier & 0x0F));
			this.Write(0x40 + registerOffset,
				((instrument.operators[0].keyScaleLevel & 0x03) << 6) +
				(outputLevel & 0x3F));
			this.Write(0x60 + registerOffset,
				((instrument.operators[0].attack & 0x0F) << 4) +
				(instrument.operators[0].decay & 0x0F));
			this.Write(0x80 + registerOffset,
				((instrument.operators[0].sustain & 0x0F) << 4) +
				(instrument.operators[0].release & 0x0F));
			this.Write(0xE0 + registerOffset,
				(instrument.operators[0].waveForm & 0x03));
		}
	}

	this.Write(0xC0 + this.drumChannels[instrument.type - this.INSTRUMENT_TYPE_BASS], 0x00);
};


/**
 * Play a note of a certain octave on the given channel.
 */
exports.OPL2.prototype.PlayNote = function (channel, octave, note) {
	this.SetKeyOn(channel, false);
	this.SetBlock(channel, Math.max(0, Math.min(octave, 7)));
	this.SetFNumber(channel, this.noteFNumbers[Math.max(0, Math.min(note, 11))]);
	this.SetKeyOn(channel, true);
};


/**
 * Is wave form selection currently enabled.
 */
exports.OPL2.prototype.GetWaveFormSelect = function () {
	return this.oplRegisters[0x01] & 0x20;
};


/**
 * Enable wave form selection for each operator.
 */
exports.OPL2.prototype.SetWaveFormSelect = function (enable) {
	if (enable) {
		return this.SetRegister(0x01, this.oplRegisters[0x01] | 0x20);
	} else {
		return this.SetRegister(0x01, this.oplRegisters[0x01] & 0xDF);
	}
};


/**
 * Is amplitude modulation enabled for the given operator?
 */
exports.OPL2.prototype.GetTremolo = function (channel, operatorNum) {
	return this.oplRegisters[0x20 + this.GetRegisterOffset(channel, operatorNum)] & 0x80;
};


/**
 * Apply amplitude modulation when set to true. Modulation depth is controlled globaly by the AM-depth flag in the 0xBD
 * register.
 */
exports.OPL2.prototype.SetTremolo = function (channel, operatorNum, enable) {
	var reg = 0x20 + this.GetRegisterOffset(channel, operatorNum);
	if (enable) {
		return this.SetRegister(reg, this.oplRegisters[reg] | 0x80);
	} else {
		return this.SetRegister(reg, this.oplRegisters[reg] & 0x7F);
	}
};


/**
 * Is vibrator enabled for the given channel?
 */
exports.OPL2.prototype.GetVibrato = function (channel, operatorNum) {
	return this.oplRegisters[0x20 + this.GetRegisterOffset(channel, operatorNum)] & 0x40;
};


/**
 * Apply vibrato when set to true. Vibrato depth is controlled globally by the VIB-depth flag in the 0xBD register.
 */
exports.OPL2.prototype.SetVibrato = function (channel, operatorNum, enable) {
	var reg = 0x20 + this.GetRegisterOffset(channel, operatorNum);
	if (enable) {
		return this.SetRegister(reg, this.oplRegisters[reg] | 0x40);
	} else {
		return this.SetRegister(reg, this.oplRegisters[reg] & 0xBF);
	}
};


/**
 * Is sustain being maintained for the given channel?
 */
exports.OPL2.prototype.GetMaintainSustain = function (channel, operatorNum) {
	return this.oplRegisters[0x20 + this.GetRegisterOffset(channel, operatorNum)] & 0x20;
};


/**
 * When set to true the sustain level of the voice is maintained until released. When false the sound begins to decay
 * immediately after hitting the sustain phase.
 */
exports.OPL2.prototype.SetMaintainSustain = function (channel, operatorNum, enable) {
	var reg = 0x20 + this.GetRegisterOffset(channel, operatorNum);
	if (enable) {
		return this.SetRegister(reg, oplRegisters[reg] | 0x20);
	} else {
		return this.SetRegister(reg, oplRegisters[reg] & 0xDF);
	}
};


/**
 * Is envelope scaling being applied to the given channel?
 */
exports.OPL2.prototype.GetEnvelopeScaling = function (channel, operatorNum) {
	return this.oplRegisters[0x20 + this.GetRegisterOffset(channel, operatorNum)] & 0x10;
};


/**
 * Enable or disable envelope scaling. When set to true higher notes will be shorter than lower ones.
 */
exports.OPL2.prototype.SetEnvelopeScaling = function (channel, operatorNum, enable) {
	var reg = 0x20 + this.GetRegisterOffset(channel, operatorNum);
	if (enable) {
		return this.SetRegister(reg, oplRegisters[reg] | 0x10);
	} else {
		return this.SetRegister(reg, oplRegisters[reg] & 0xEF);
	}
};


/**
 * Get the frequency multiplier for the given channel.
 */
exports.OPL2.prototype.GetMultiplier = function (channel, operatorNum) {
	return this.oplRegisters[0x20 + this.GetRegisterOffset(channel, operatorNum)] & 0x0F;
};


/**
 * Set frequency multiplier for the given channel. Note that a multiplier of 0 will apply a 0.5 multiplication.
 */
exports.OPL2.prototype.SetMultiplier = function (channel, operatorNum, multiplier) {
	var reg = 0x20 + this.GetRegisterOffset(channel, operatorNum);
	return this.SetRegister(reg, (this.oplRegisters[reg] & 0xF0) | (multiplier & 0x0F));
};


/**
 * Get the scaling level for the given channel.
 */
exports.OPL2.prototype.GetScalingLevel = function (channel, operatorNum) {
	return (this.oplRegisters[0x40 + this.GetRegisterOffset(channel, operatorNum)] & 0xC0) >> 6;
};


/**
 * Decrease output levels as frequency increases.
 * 00 - No change
 * 01 - 1.5 dB/oct
 * 10 - 3.0 dB/oct
 * 11 - 6.0 dB/oct
 */
exports.OPL2.prototype.SetScalingLevel = function (channel, operatorNum, scaling) {
	var reg = 0x40 + this.GetRegisterOffset(channel, operatorNum);
	return this.SetRegister(reg, (this.oplRegisters[reg] & 0x3F) | ((scaling & 0x03) << 6));
};


/**
 * Get the volume of the given channel. 0x00 is laudest, 0x3F is softest.
 */
exports.OPL2.prototype.GetVolume = function (channel, operatorNum) {
	return this.oplRegisters[0x40 + this.GetRegisterOffset(channel, operatorNum)] & 0x3F;
};


/**
 * Set the volume of the channel.
 * Note that the scale is inverted! 0x00 for loudest, 0x3F for softest.
 */
exports.OPL2.prototype.SetVolume = function (channel, operatorNum, volume) {
	var reg = 0x40 + this.GetRegisterOffset(channel, operatorNum);
	return this.SetRegister(reg, (this.oplRegisters[reg] & 0xC0) | (volume & 0x3F));
};


/**
 * Get the attack rate of the given channel.
 */
exports.OPL2.prototype.GetAttack = function (channel, operatorNum) {
	return (this.oplRegisters[0x60 + this.GetRegisterOffset(channel, operatorNum)] & 0xF0) >> 4;
};


/**
 * Attack rate. 0x00 is slowest, 0x0F is fastest.
 */
exports.OPL2.prototype.SetAttack = function (channel, operatorNum, attack) {
	var reg = 0x60 + this.GetRegisterOffset(channel, operatorNum);
	return this.SetRegister(reg, (this.oplRegisters[reg] & 0x0F) | ((attack & 0x0F) << 4));
};


/**
 * Get the decay rate of the given channel.
 */
exports.OPL2.prototype.GetDecay = function (channel, operatorNum) {
	return this.oplRegisters[0x60 + this.GetRegisterOffset(channel, operatorNum)] & 0x0F;
};


/**
 * Decay rate. 0x00 is slowest, 0x0F is fastest.
 */
exports.OPL2.prototype.SetDecay = function (channel, operatorNum, decay) {
	var reg = 0x60 + this.GetRegisterOffset(channel, operatorNum);
	return this.SetRegister(reg, (this.oplRegisters[reg] & 0xF0) | (decay & 0x0F));
};


/**
 * Get the sustain level of the given channel. 0x00 is laudest, 0x0F is softest.
 */
exports.OPL2.prototype.GetSustain = function (channel, operatorNum) {
	return (this.oplRegisters[0x80 + this.GetRegisterOffset(channel, operatorNum)] & 0xF0) >> 4;
};


/**
 * Sustain level. 0x00 is laudest, 0x0F is softest.
 */
exports.OPL2.prototype.SetSustain = function (channel, operatorNum, sustain) {
	var reg = 0x80 + this.GetRegisterOffset(channel, operatorNum);
	return this.SetRegister(reg, (this.oplRegisters[reg] & 0x0F) | ((sustain & 0x0F) << 4));
};


/**
 * Get the release rate of the given channel.
 */
exports.OPL2.prototype.GetRelease = function (channel, operatorNum) {
	return this.oplRegisters[0x80 + this.GetRegisterOffset(channel, operatorNum)] & 0x0F;
};


/**
 * Release rate. 0x00 is flowest, 0x0F is fastest.
 */
exports.OPL2.prototype.SetRelease = function (channel, operatorNum, release) {
	var reg = 0x80 + this.GetRegisterOffset(channel, operatorNum);
	return this.SetRegister(reg, (this.oplRegisters[reg] & 0xF0) | (release & 0x0F));
};


/**
 * Get the frequenct F-number of the given channel.
 */
exports.OPL2.prototype.GetFNumber = function (channel) {
	var offset = Math.max(0x00, Math.min(channel, 0x08));
	return ((this.oplRegisters[0xB0 + offset] & 0x03) << 8) + this.oplRegisters[0xA0 + offset];
};


/**
 * Set frequency F-number [0, 1023] for the given channel.
 */
exports.OPL2.prototype.SetFNumber = function (channel, fNumber) {
	var reg = 0xA0 + Math.max(0x00, Math.min(channel, 0x08));
	this.SetRegister(reg, fNumber & 0x00FF);
	this.SetRegister(reg + 0x10, (this.oplRegisters[reg + 0x10] & 0xFC) | ((fNumber & 0x0300) >> 8));
	return reg;
};


/**
 * Get the frequency for the given channel.
 */
exports.OPL2.prototype.GetFrequency = function (channel) {
	var fInterval = this.GetFrequencyStep(channel);
	return this.GetFNumber(channel) * fInterval;
};


/**
 * Set the frequenct of the given channel and if needed switch to a different block.
 */
exports.OPL2.prototype.SetFrequency = function (channel, frequency) {
	var block = this.GetFrequencyBlock(frequency);
	if (this.GetBlock(channel) != block) {
		this.SetBlock(channel, block);
	}
	var fNumber = this.GetFrequencyFNumber(channel, frequency);
	return this.SetFNumber(channel, fNumber);
};


/**
 * Get the frequency block of the given channel.
 */
exports.OPL2.prototype.GetBlock = function (channel) {
	var offset = Math.max(0x00, Math.min(channel, 0x08));
	return (this.oplRegisters[0xB0 + offset] & 0x1C) >> 2;
};


/**
 * Set frequency block. 0x00 is lowest, 0x07 is highest. This determines the frequency interval between notes.
 * 0 - 0.048 Hz, Range: 0.047 Hz ->   48.503 Hz
 * 1 - 0.095 Hz, Range: 0.094 Hz ->   97.006 Hz
 * 2 - 0.190 Hz, Range: 0.189 Hz ->  194.013 Hz
 * 3 - 0.379 Hz, Range: 0.379 Hz ->  388.026 Hz
 * 4 - 0.759 Hz, Range: 0.758 Hz ->  776.053 Hz
 * 5 - 1.517 Hz, Range: 1.517 Hz -> 1552.107 Hz
 * 6 - 3.034 Hz, Range: 3.034 Hz -> 3104.215 Hz
 * 7 - 6.069 Hz, Range: 6.068 Hz -> 6208.431 Hz
 */
exports.OPL2.prototype.SetBlock = function (channel, block) {
	var reg = 0xB0 + Math.max(0x00, Math.min(channel, 0x08));
	return this.SetRegister(reg, (this.oplRegisters[reg] & 0xE3) | ((block & 0x07) << 2));
};


/**
 * Is the voice of the given channel currently enabled?
 */
exports.OPL2.prototype.GetKeyOn = function (channel) {
	var offset = Math.max(0x00, Math.min(channel, 0x08));
	return this.oplRegisters[0xB0 + offset] & 0x20;
};


/**
 * Enable voice on channel.
 */
exports.OPL2.prototype.SetKeyOn = function (channel, keyOn) {
	var reg = 0xB0 + Math.max(0x00, Math.min(channel, 0x08));
	if (keyOn) {
		return this.SetRegister(reg, this.oplRegisters[reg] | 0x20);
	} else {
		return this.SetRegister(reg, this.oplRegisters[reg] & 0xDF);
	}
};


/**
 * Get the feedback strength of the given channel.
 */
exports.OPL2.prototype.GetFeedback = function (channel) {
	var offset = Math.max(0x00, Math.min(channel, 0x08));
	return (this.oplRegisters[0xC0 + offset] & 0xE0) >> 1;
};


/**
 * Set feedback strength. 0x00 is no feedback, 0x07 is strongest.
 */
exports.OPL2.prototype.SetFeedback = function (channel, feedback) {
	var reg = 0xC0 + Math.max(0x00, Math.min(channel, 0x08));
	return this.SetRegister(reg, (this.oplRegisters[reg] & 0x01) | ((feedback & 0x07) << 1));
};


/**
 * Is the decay algorythm enabled for the given channel?
 */
exports.OPL2.prototype.GetSynthMode = function (channel) {
	var offset = Math.max(0x00, Math.min(channel, 0x08));
	return this.oplRegisters[0xC0 + offset] & 0x01;
};


/**
 * Set decay algorithm. When false operator 1 modulates operator 2 (operator 2 is the only one to produce sounde). If
 * set to true both operator 1 and operator 2 will produce sound.
 */
exports.OPL2.prototype.SetSynthMode = function (channel, isAdditive) {
	var reg = 0xC0 + Math.max(0x00, Math.min(channel, 0x08));
	if (isAdditive) {
		return this.SetRegister(reg, this.oplRegisters[reg] | 0x01);
	} else {
		return this.SetRegister(reg, this.oplRegisters[reg] & 0xFE);
	}
};


/**
 * Is deeper amplitude modulation enabled?
 */
exports.OPL2.prototype.GetDeepTremolo = function () {
	return this.oplRegisters[0xBD] & 0x80;
};


/**
 * Set deeper aplitude modulation depth. When false modulation depth is 1.0 dB, when true modulation depth is 4.8 dB.
 */
exports.OPL2.prototype.SetDeepTremolo = function (enable) {
	if (enable) {
		return this.SetRegister(0xBD, this.oplRegisters[0xBD] | 0x80);
	} else {
		return this.SetRegister(0xBD, this.oplRegisters[0xBD] & 0x7F);
	}
};


/**
 * Is deeper vibrato depth enabled?
 */
exports.OPL2.prototype.GetDeepVibrato = function () {
	return this.oplRegisters[0xBD] & 0x40;
};


/**
 * Set deeper vibrato depth. When false vibrato depth is 7/100 semi tone, when true vibrato depth is 14/100.
 */
exports.OPL2.prototype.SetDeepVibrato = function (enable) {
	if (enable) {
		return this.SetRegister(0xBD, this.oplRegisters[0xBD] | 0x40);
	} else {
		return this.SetRegister(0xBD, this.oplRegisters[0xBD] & 0xBF);
	}
};


/**
 * Is percussion mode currently enabled?
 */
exports.OPL2.prototype.GetPercussion = function () {
	return this.oplRegisters[0xBD] & 0x20;
};


/**
 * Enable or disable percussion mode. When set to false there are 9 melodic voices, when true there are 6 melodic
 * voices and channels 6 through 8 are used for drum sounds. KeyOn for these channels must be off.
 */
exports.OPL2.prototype.SetPercussion = function (enable) {
	if (enable) {
		return this.SetRegister(0xBD, this.oplRegisters[0xBD] | 0x20);
	} else {
		return this.SetRegister(0xBD, this.oplRegisters[0xBD] & 0xDF);
	}
};


/**
 * Return which drum sounds are enabled.
 */
exports.OPL2.prototype.GetDrums = function () {
	return this.oplRegisters[0xBD] & 0x1F;
};


/**
 * Enable or disable various drum sounds.
 * Note that keyOn for channels 6, 7 and 8 must be false in order to use rhythms.
 */
exports.OPL2.prototype.SetDrums = function (bass, snare, tom, cymbal, hihat) {
	var drums = 0;
	drums += bass ? DRUM_BASS : 0x00;
	drums += snare ? DRUM_SNARE : 0x00;
	drums += tom ? DRUM_TOM : 0x00;
	drums += cymbal ? DRUM_CYMBAL : 0x00;
	drums += hihat ? DRUM_HI_HAT : 0x00;
	this.SetRegister(0xBD, oplRegisters[0xBD] & ~drums);
	return this.SetRegister(0xBD, oplRegisters[0xBD] | drums);
};


/**
 * Get the wave form currently set for the given channel.
 */
exports.OPL2.prototype.GetWaveForm = function (channel, operatorNum) {
	return this.oplRegisters[0xE0 + this.GetRegisterOffset(channel, operatorNum)] & 0x03;
};


/**
 * Select the wave form to use.
 */
exports.OPL2.prototype.SetWaveForm = function (channel, operatorNum, waveForm) {
	var reg = 0xE0 + this.GetRegisterOffset(channel, operatorNum);
	return this.SetRegister(reg, (this.oplRegisters[reg] & 0xFC) | (waveForm & 0x03));
};

// Note to frequency mapping.
exports.NOTE = {};
exports.NOTE.C = 0;
exports.NOTE.CS = 1;
exports.NOTE.D = 2;
exports.NOTE.DS = 3;
exports.NOTE.E = 4;
exports.NOTE.F = 5;
exports.NOTE.FS = 6;
exports.NOTE.G = 7;
exports.NOTE.GS = 8;
exports.NOTE.A = 9;
exports.NOTE.AS = 10;
exports.NOTE.B = 11;



/**
 * This file contains instrument defenitions from the Adlib standard instrument library for use with the OPL2 Audio
 * Board library.
 *
 * Instrument definition is based on Adlib instrument bank format.
 *  0 - Rhythm mode drum channel
 *		Drum channels are predefined by the YM3812 and cannot be redefined. Regular instruments have their channel set
 *		to 0x00 and can be assigned to any channel by the setInstrument function. Rhythm mode instruments can only be
 *		used when rhythm mode is active (see OPL2.setPercussion).
 *
 *  1 - Channel c, operator 1, register 0x20
 *		Tremolo(1) | Vibrato(1) | Sustain(1) | KSR(1) | Frequency multiplier (4)
 *
 *  2 - Channel c, operator 1, register 0x40
 *		Key scale level(2) | Output level(6)
 *
 *  3 - Channel c, operator 1, register 0x60
 *		Attack(4) | Decay(4)
 *
 *  4 - Channel c, operator 1, register 0x80
 *		Sustain(4) | Release(4)
 *
 *  5 - Channel c, operator 1, register 0xE0
 *		Undefined(5) | Waveform(3)
 *
 *  6 - Channel c, register 0xC0
 *		Undefined(4) | Modulation feedback factor(3) | Synth type(1)
 *
 *  7 - Channel c, operator 2, register 0x20
 *  8 - Channel c, operator 2, register 0x40
 *  9 - Channel c, operator 2, register 0x60
 * 10 - Channel c, operator 2, register 0x80
 * 11 - Channel c, operator 2, register 0xE0
 */
exports.INSTRUMENT = {};
// exports.INSTRUMENT.ACCORDN = [0x00, 0x24, 0x4F, 0xF2, 0x0B, 0x00, 0x0E, 0x31, 0x00, 0x52, 0x0B, 0x00];
// exports.INSTRUMENT.BAGPIPE1 = [0x00, 0x31, 0x43, 0x6E, 0x17, 0x01, 0x02, 0x22, 0x05, 0x8B, 0x0C, 0x02];
// exports.INSTRUMENT.BAGPIPE2 = [0x00, 0x30, 0x00, 0xFF, 0xA0, 0x03, 0x00, 0xA3, 0x00, 0x65, 0x0B, 0x02];
// exports.INSTRUMENT.BANJO1 = [0x00, 0x31, 0x87, 0xA1, 0x11, 0x00, 0x08, 0x16, 0x80, 0x7D, 0x43, 0x00];
// exports.INSTRUMENT.BASS1 = [0x00, 0x01, 0x15, 0x25, 0x2F, 0x00, 0x0A, 0x21, 0x80, 0x65, 0x6C, 0x00];
// exports.INSTRUMENT.BASS2 = [0x00, 0x01, 0x1D, 0xF2, 0xEF, 0x00, 0x0A, 0x01, 0x00, 0xF5, 0x78, 0x00];
// exports.INSTRUMENT.BASSHARP = [0x00, 0xC0, 0x6D, 0xF9, 0x01, 0x01, 0x0E, 0x41, 0x00, 0xF2, 0x73, 0x00];
// exports.INSTRUMENT.BASSOON1 = [0x00, 0x30, 0xC8, 0xD5, 0x19, 0x00, 0x0C, 0x71, 0x80, 0x61, 0x1B, 0x00];
// exports.INSTRUMENT.BASSTRLG = [0x00, 0xC1, 0x4F, 0xB1, 0x53, 0x03, 0x06, 0xE0, 0x00, 0x12, 0x74, 0x03];
// exports.INSTRUMENT.BDRUM1 = [0x06, 0x00, 0x0B, 0xA8, 0x4C, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00];
// exports.INSTRUMENT.BELLONG = [0x00, 0x64, 0xDB, 0xFF, 0x01, 0x00, 0x04, 0x3E, 0xC0, 0xF3, 0x62, 0x00];
// exports.INSTRUMENT.BELLS = [0x00, 0x07, 0x4F, 0xF2, 0x60, 0x00, 0x08, 0x12, 0x00, 0xF2, 0x72, 0x00];
// exports.INSTRUMENT.BELSHORT = [0x00, 0x64, 0xDB, 0xFF, 0x01, 0x00, 0x04, 0x3E, 0xC0, 0xF5, 0xF3, 0x00];
// exports.INSTRUMENT.BNCEBASS = [0x00, 0x20, 0x4B, 0x7B, 0x04, 0x01, 0x0E, 0x21, 0x00, 0xF5, 0x72, 0x00];
// exports.INSTRUMENT.BRASS1 = [0x00, 0x21, 0x16, 0x71, 0xAE, 0x00, 0x0E, 0x21, 0x00, 0x81, 0x9E, 0x00];
// exports.INSTRUMENT.CBASSOON = [0x00, 0x30, 0xC5, 0x52, 0x11, 0x00, 0x00, 0x31, 0x80, 0x31, 0x2E, 0x00];
// exports.INSTRUMENT.CELESTA = [0x00, 0x33, 0x87, 0x01, 0x10, 0x00, 0x08, 0x14, 0x80, 0x7D, 0x33, 0x00];
// exports.INSTRUMENT.CLAR1 = [0x00, 0x32, 0x16, 0x73, 0x24, 0x00, 0x0E, 0x21, 0x80, 0x75, 0x57, 0x00];
// exports.INSTRUMENT.CLAR2 = [0x00, 0x31, 0x1C, 0x41, 0x1B, 0x00, 0x0C, 0x60, 0x80, 0x42, 0x3B, 0x00];
// exports.INSTRUMENT.CLARINET = [0x00, 0x32, 0x9A, 0x51, 0x1B, 0x00, 0x0C, 0x61, 0x82, 0xA2, 0x3B, 0x00];
// exports.INSTRUMENT.CLAVECIN = [0x00, 0x11, 0x0D, 0xF2, 0x01, 0x00, 0x0A, 0x15, 0x0D, 0xF2, 0xB1, 0x00];
// exports.INSTRUMENT.CROMORNE = [0x00, 0x00, 0x02, 0xF0, 0xFF, 0x00, 0x06, 0x11, 0x80, 0xF0, 0xFF, 0x00];
// exports.INSTRUMENT.CYMBAL1 = [0x09, 0x01, 0x00, 0xF5, 0xB5, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00];
// exports.INSTRUMENT.ELCLAV1 = [0x00, 0x05, 0x8A, 0xF0, 0x7B, 0x00, 0x08, 0x01, 0x80, 0xF4, 0x7B, 0x00];
// exports.INSTRUMENT.ELCLAV2 = [0x00, 0x01, 0x49, 0xF1, 0x53, 0x01, 0x06, 0x11, 0x00, 0xF1, 0x74, 0x02];
// exports.INSTRUMENT.ELECFL = [0x00, 0xE0, 0x6D, 0x57, 0x04, 0x01, 0x0E, 0x61, 0x00, 0x67, 0x7D, 0x00];
// exports.INSTRUMENT.ELECVIBE = [0x00, 0x13, 0x97, 0x9A, 0x12, 0x02, 0x0E, 0x91, 0x80, 0x9B, 0x11, 0x00];
// exports.INSTRUMENT.ELGUIT1 = [0x00, 0xF1, 0x01, 0x97, 0x17, 0x00, 0x08, 0x21, 0x0D, 0xF1, 0x18, 0x00];
// exports.INSTRUMENT.ELGUIT2 = [0x00, 0x13, 0x96, 0xFF, 0x21, 0x00, 0x0A, 0x11, 0x80, 0xFF, 0x03, 0x00];
// exports.INSTRUMENT.ELGUIT3 = [0x00, 0x07, 0x8F, 0x82, 0x7D, 0x00, 0x0C, 0x14, 0x80, 0x82, 0x7D, 0x00];
// exports.INSTRUMENT.ELGUIT4 = [0x00, 0x05, 0x8F, 0xDA, 0x15, 0x00, 0x0A, 0x01, 0x80, 0xF9, 0x14, 0x02];
// exports.INSTRUMENT.ELORGAN1 = [0x00, 0xB2, 0xCD, 0x91, 0x2A, 0x02, 0x09, 0xB1, 0x80, 0x91, 0x2A, 0x01];
// exports.INSTRUMENT.ELPIANO1 = [0x00, 0x01, 0x4F, 0xF1, 0x50, 0x00, 0x06, 0x01, 0x04, 0xD2, 0x7C, 0x00];
// exports.INSTRUMENT.ELPIANO2 = [0x00, 0x02, 0x22, 0xF2, 0x13, 0x00, 0x0E, 0x02, 0x00, 0xF5, 0x43, 0x00];
// exports.INSTRUMENT.EPIANO1A = [0x00, 0x81, 0x63, 0xF3, 0x58, 0x00, 0x00, 0x01, 0x80, 0xF2, 0x58, 0x00];
// exports.INSTRUMENT.EPIANO1B = [0x00, 0x07, 0x1F, 0xF5, 0xFA, 0x00, 0x0E, 0x01, 0x57, 0xF5, 0xFA, 0x00];
// exports.INSTRUMENT.FLUTE = [0x00, 0x21, 0x83, 0x74, 0x17, 0x00, 0x07, 0xA2, 0x8D, 0x65, 0x17, 0x00];
// exports.INSTRUMENT.FLUTE1 = [0x00, 0xA1, 0x27, 0x74, 0x8F, 0x00, 0x02, 0xA1, 0x80, 0x65, 0x2A, 0x00];
// exports.INSTRUMENT.FLUTE2 = [0x00, 0xE0, 0xEC, 0x6E, 0x8F, 0x00, 0x0E, 0x61, 0x00, 0x65, 0x2A, 0x00];
// exports.INSTRUMENT.FRHORN1 = [0x00, 0x21, 0x9F, 0x53, 0x5A, 0x00, 0x0C, 0x21, 0x80, 0xAA, 0x1A, 0x00];
// exports.INSTRUMENT.FRHORN2 = [0x00, 0x20, 0x8E, 0xA5, 0x8F, 0x02, 0x06, 0x21, 0x00, 0x36, 0x3D, 0x00];
// exports.INSTRUMENT.FSTRP1 = [0x00, 0xF0, 0x18, 0x55, 0xEF, 0x02, 0x00, 0xE0, 0x80, 0x87, 0x1E, 0x03];
// exports.INSTRUMENT.FSTRP2 = [0x00, 0x70, 0x16, 0x55, 0x2F, 0x02, 0x0C, 0xE0, 0x80, 0x87, 0x1E, 0x03];
// exports.INSTRUMENT.FUZGUIT1 = [0x00, 0xF1, 0x00, 0x97, 0x13, 0x00, 0x0A, 0x25, 0x0D, 0xF1, 0x18, 0x01];
// exports.INSTRUMENT.FUZGUIT2 = [0x00, 0x31, 0x48, 0xF1, 0x53, 0x00, 0x06, 0x32, 0x00, 0xF2, 0x27, 0x02];
// exports.INSTRUMENT.GUITAR1 = [0x00, 0x01, 0x11, 0xF2, 0x1F, 0x00, 0x0A, 0x01, 0x00, 0xF5, 0x88, 0x00];
// exports.INSTRUMENT.HARP1 = [0x00, 0x02, 0x29, 0xF5, 0x75, 0x00, 0x00, 0x01, 0x83, 0xF2, 0xF3, 0x00];
// exports.INSTRUMENT.HARP2 = [0x00, 0x02, 0x99, 0xF5, 0x55, 0x00, 0x00, 0x01, 0x80, 0xF6, 0x53, 0x00];
// exports.INSTRUMENT.HARP3 = [0x00, 0x02, 0x57, 0xF5, 0x56, 0x00, 0x00, 0x01, 0x80, 0xF6, 0x54, 0x00];
// exports.INSTRUMENT.HARPE1 = [0x00, 0x02, 0x29, 0xF5, 0x75, 0x00, 0x00, 0x01, 0x03, 0xF2, 0xF3, 0x00];
// exports.INSTRUMENT.HARPSI1 = [0x00, 0x32, 0x87, 0xA1, 0x10, 0x00, 0x08, 0x16, 0x80, 0x7D, 0x33, 0x00];
// exports.INSTRUMENT.HARPSI2 = [0x00, 0x33, 0x87, 0xA1, 0x10, 0x00, 0x06, 0x15, 0x80, 0x7D, 0x43, 0x00];
// exports.INSTRUMENT.HARPSI3 = [0x00, 0x35, 0x84, 0xA8, 0x10, 0x00, 0x08, 0x18, 0x80, 0x7D, 0x33, 0x00];
// exports.INSTRUMENT.HARPSI4 = [0x00, 0x11, 0x0D, 0xF2, 0x01, 0x00, 0x0A, 0x15, 0x0D, 0xF2, 0xB1, 0x00];
// exports.INSTRUMENT.HARPSI5 = [0x00, 0x36, 0x87, 0x8A, 0x00, 0x00, 0x08, 0x1A, 0x80, 0x7F, 0x33, 0x00];
// exports.INSTRUMENT.HELICPTR = [0x00, 0xF0, 0x00, 0x1E, 0x11, 0x01, 0x08, 0xE2, 0xC0, 0x11, 0x11, 0x01];
// exports.INSTRUMENT.HIHAT1 = [0x0A, 0x01, 0x00, 0xF7, 0xB5, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00];
// exports.INSTRUMENT.HIHAT2 = [0x0A, 0x01, 0x03, 0xDA, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00];
// exports.INSTRUMENT.JAVAICAN = [0x00, 0x87, 0x4D, 0x78, 0x42, 0x00, 0x0A, 0x94, 0x00, 0x85, 0x54, 0x00];
// exports.INSTRUMENT.JAZZGUIT = [0x00, 0x03, 0x5E, 0x85, 0x51, 0x01, 0x0E, 0x11, 0x00, 0xD2, 0x71, 0x00];
// exports.INSTRUMENT.JEWSHARP = [0x00, 0x00, 0x50, 0xF2, 0x70, 0x00, 0x0E, 0x13, 0x00, 0xF2, 0x72, 0x00];
// exports.INSTRUMENT.KEYBRD1 = [0x00, 0x00, 0x02, 0xF0, 0xFA, 0x01, 0x06, 0x11, 0x80, 0xF2, 0xFA, 0x01];
// exports.INSTRUMENT.KEYBRD2 = [0x00, 0x01, 0x8F, 0xF2, 0xBD, 0x00, 0x08, 0x14, 0x80, 0x82, 0xBD, 0x00];
// exports.INSTRUMENT.KEYBRD3 = [0x00, 0x01, 0x00, 0xF0, 0xF0, 0x00, 0x00, 0xE4, 0x03, 0xF3, 0x36, 0x00];
// exports.INSTRUMENT.LASER = [0x09, 0xE6, 0x00, 0x25, 0xB5, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00];
// exports.INSTRUMENT.LOGDRUM1 = [0x00, 0x32, 0x44, 0xF8, 0xFF, 0x00, 0x0E, 0x11, 0x00, 0xF5, 0x7F, 0x00];
// exports.INSTRUMENT.MARIMBA1 = [0x00, 0x05, 0x4E, 0xDA, 0x25, 0x00, 0x0A, 0x01, 0x00, 0xF9, 0x15, 0x00];
// exports.INSTRUMENT.MARIMBA2 = [0x00, 0x85, 0x4E, 0xDA, 0x15, 0x00, 0x0A, 0x81, 0x80, 0xF9, 0x13, 0x00];
// exports.INSTRUMENT.MDRNPHON = [0x00, 0x30, 0x00, 0xFE, 0x11, 0x01, 0x08, 0xAE, 0xC0, 0xF1, 0x19, 0x01];
// exports.INSTRUMENT.MLTRDRUM = [0x07, 0x0C, 0x00, 0xC8, 0xB6, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00];
// exports.INSTRUMENT.MOOGSYNT = [0x00, 0x20, 0x90, 0xF5, 0x9E, 0x02, 0x0C, 0x11, 0x00, 0xF4, 0x5B, 0x03];
// exports.INSTRUMENT.NOISE1 = [0x00, 0x0E, 0x40, 0xD1, 0x53, 0x00, 0x0E, 0x0E, 0x00, 0xF2, 0x7F, 0x03];
// exports.INSTRUMENT.OBOE1 = [0x00, 0xB1, 0xC5, 0x6E, 0x17, 0x00, 0x02, 0x22, 0x05, 0x8B, 0x0E, 0x00];
// exports.INSTRUMENT.ORGAN1 = [0x00, 0x65, 0xD2, 0x81, 0x03, 0x00, 0x02, 0x71, 0x80, 0xF1, 0x05, 0x00];
// exports.INSTRUMENT.ORGAN2 = [0x00, 0x24, 0x80, 0xFF, 0x0F, 0x00, 0x01, 0x21, 0x80, 0xFF, 0x0F, 0x00];
// exports.INSTRUMENT.ORGAN3 = [0x00, 0x03, 0x5B, 0xF0, 0x1F, 0x00, 0x0A, 0x01, 0x80, 0xF0, 0x1F, 0x00];
// exports.INSTRUMENT.ORGAN3A = [0x00, 0x03, 0x5B, 0xF0, 0x1F, 0x00, 0x0A, 0x01, 0x8D, 0xF0, 0x13, 0x00];
// exports.INSTRUMENT.ORGAN3B = [0x00, 0x03, 0x5B, 0xF0, 0x1F, 0x00, 0x0A, 0x01, 0x92, 0xF0, 0x12, 0x00];
// exports.INSTRUMENT.ORGNPERC = [0x00, 0x0C, 0x00, 0xF8, 0xB5, 0x00, 0x01, 0x00, 0x00, 0xD6, 0x4F, 0x00];
// exports.INSTRUMENT.PHONE1 = [0x00, 0x17, 0x4F, 0xF2, 0x61, 0x00, 0x08, 0x12, 0x08, 0xF1, 0xB2, 0x00];
// exports.INSTRUMENT.PHONE2 = [0x00, 0x17, 0x4F, 0xF2, 0x61, 0x00, 0x08, 0x12, 0x0A, 0xF1, 0xB4, 0x00];
// exports.INSTRUMENT.PIAN1A = [0x00, 0x81, 0x63, 0xF3, 0x58, 0x00, 0x00, 0x01, 0x80, 0xF2, 0x58, 0x00];
// exports.INSTRUMENT.PIAN1B = [0x00, 0x07, 0x1F, 0xF5, 0xFA, 0x00, 0x0E, 0x01, 0x26, 0xF5, 0xFA, 0x00];
// exports.INSTRUMENT.PIAN1C = [0x00, 0x07, 0x1F, 0xF5, 0xFA, 0x00, 0x0E, 0x01, 0x57, 0xF5, 0xFA, 0x00];
// exports.INSTRUMENT.PIANO = [0x00, 0x03, 0x4F, 0xF1, 0x53, 0x00, 0x06, 0x17, 0x00, 0xF2, 0x74, 0x00];
// exports.INSTRUMENT.PIANO1 = [0x00, 0x01, 0x4F, 0xF1, 0x53, 0x00, 0x06, 0x11, 0x00, 0xD2, 0x74, 0x00];
// exports.INSTRUMENT.PIANO2 = [0x00, 0x41, 0x9D, 0xF2, 0x51, 0x00, 0x06, 0x13, 0x00, 0xF2, 0xF1, 0x00];
// exports.INSTRUMENT.PIANO3 = [0x00, 0x01, 0x4F, 0xF1, 0x50, 0x00, 0x06, 0x01, 0x04, 0xD2, 0x7C, 0x00];
// exports.INSTRUMENT.PIANO4 = [0x00, 0x01, 0x4D, 0xF1, 0x60, 0x00, 0x08, 0x11, 0x00, 0xD2, 0x7B, 0x00];
// exports.INSTRUMENT.PIANOBEL = [0x00, 0x03, 0x4F, 0xF1, 0x53, 0x00, 0x06, 0x17, 0x03, 0xF2, 0x74, 0x00];
// exports.INSTRUMENT.PIANOF = [0x00, 0x01, 0xCF, 0xF1, 0x53, 0x00, 0x02, 0x12, 0x00, 0xF2, 0x83, 0x00];
// exports.INSTRUMENT.POPBASS1 = [0x00, 0x10, 0x00, 0x75, 0x93, 0x01, 0x00, 0x01, 0x00, 0xF5, 0x82, 0x01];
// exports.INSTRUMENT.RKSNARE1 = [0x07, 0x0C, 0x00, 0xC7, 0xB4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00];
// exports.INSTRUMENT.SAX1 = [0x00, 0x01, 0x4F, 0x71, 0x53, 0x00, 0x0A, 0x12, 0x00, 0x52, 0x7C, 0x00];
// exports.INSTRUMENT.SCRATCH = [0x00, 0x07, 0x00, 0xF0, 0xF0, 0x00, 0x0E, 0x00, 0x00, 0x5C, 0xDC, 0x00];
// exports.INSTRUMENT.SCRATCH4 = [0x00, 0x07, 0x00, 0xF0, 0xF0, 0x00, 0x0E, 0x00, 0x00, 0x5C, 0xDC, 0x00];
// exports.INSTRUMENT.SDRUM2 = [0x00, 0x06, 0x00, 0xF0, 0xF0, 0x00, 0x0E, 0x00, 0x00, 0xF6, 0xB4, 0x00];
// exports.INSTRUMENT.SHRTVIBE = [0x00, 0xE4, 0x0E, 0xFF, 0x3F, 0x01, 0x00, 0xC0, 0x00, 0xF3, 0x07, 0x00];
// exports.INSTRUMENT.SITAR1 = [0x00, 0x01, 0x40, 0xF1, 0x53, 0x00, 0x00, 0x08, 0x40, 0xF1, 0x53, 0x00];
// exports.INSTRUMENT.SITAR2 = [0x00, 0x01, 0x40, 0xF1, 0x53, 0x00, 0x00, 0x08, 0x40, 0xF1, 0x53, 0x01];
// exports.INSTRUMENT.SNAKEFL = [0x00, 0x61, 0x0C, 0x81, 0x03, 0x00, 0x08, 0x71, 0x80, 0x61, 0x0C, 0x00];
// exports.INSTRUMENT.SNARE1 = [0x07, 0x0C, 0x00, 0xF8, 0xB5, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00];
// exports.INSTRUMENT.SNRSUST = [0x00, 0x06, 0x00, 0xF0, 0xF0, 0x00, 0x0E, 0xC4, 0x03, 0xC4, 0x34, 0x00];
// exports.INSTRUMENT.SOLOVLN = [0x00, 0x70, 0x1C, 0x51, 0x03, 0x02, 0x0E, 0x20, 0x00, 0x54, 0x67, 0x02];
// exports.INSTRUMENT.STEELGT1 = [0x00, 0x01, 0x46, 0xF1, 0x83, 0x00, 0x06, 0x61, 0x03, 0x31, 0x86, 0x00];
// exports.INSTRUMENT.STEELGT2 = [0x00, 0x01, 0x47, 0xF1, 0x83, 0x00, 0x06, 0x61, 0x03, 0x91, 0x86, 0x00];
// exports.INSTRUMENT.STRINGS1 = [0x00, 0xB1, 0x8B, 0x71, 0x11, 0x00, 0x06, 0x61, 0x40, 0x42, 0x15, 0x01];
// exports.INSTRUMENT.STRNLONG = [0x00, 0xE1, 0x4F, 0xB1, 0xD3, 0x03, 0x06, 0x21, 0x00, 0x12, 0x74, 0x01];
// exports.INSTRUMENT.SYN1 = [0x00, 0x55, 0x97, 0x2A, 0x02, 0x00, 0x00, 0x12, 0x80, 0x42, 0xF3, 0x00];
// exports.INSTRUMENT.SYN2 = [0x00, 0x13, 0x97, 0x9A, 0x12, 0x00, 0x0E, 0x11, 0x80, 0x9B, 0x14, 0x00];
// exports.INSTRUMENT.SYN3 = [0x00, 0x11, 0x8A, 0xF1, 0x11, 0x00, 0x06, 0x01, 0x40, 0xF1, 0xB3, 0x00];
// exports.INSTRUMENT.SYN4 = [0x00, 0x21, 0x0D, 0xE9, 0x3A, 0x00, 0x0A, 0x22, 0x80, 0x65, 0x6C, 0x00];
// exports.INSTRUMENT.SYN5 = [0x00, 0x01, 0x4F, 0x71, 0x53, 0x00, 0x06, 0x19, 0x00, 0x52, 0x7C, 0x00];
// exports.INSTRUMENT.SYN6 = [0x00, 0x24, 0x0F, 0x41, 0x7E, 0x00, 0x0A, 0x21, 0x00, 0xF1, 0x5E, 0x00];
// exports.INSTRUMENT.SYN9 = [0x00, 0x07, 0x87, 0xF0, 0x05, 0x00, 0x04, 0x01, 0x80, 0xF0, 0x05, 0x00];
// exports.INSTRUMENT.SYNBAL1 = [0x00, 0x26, 0x03, 0xE0, 0xF0, 0x00, 0x08, 0x1E, 0x00, 0xFF, 0x31, 0x00];
// exports.INSTRUMENT.SYNBAL2 = [0x00, 0x28, 0x03, 0xE0, 0xF0, 0x00, 0x04, 0x13, 0x00, 0xE8, 0x11, 0x00];
// exports.INSTRUMENT.SYNBASS1 = [0x00, 0x30, 0x88, 0xD5, 0x19, 0x00, 0x0C, 0x71, 0x80, 0x61, 0x1B, 0x00];
// exports.INSTRUMENT.SYNBASS2 = [0x00, 0x81, 0x86, 0x65, 0x01, 0x00, 0x0C, 0x11, 0x00, 0x32, 0x74, 0x00];
// exports.INSTRUMENT.SYNBASS4 = [0x00, 0x81, 0x83, 0x65, 0x05, 0x00, 0x0A, 0x51, 0x00, 0x32, 0x74, 0x00];
// exports.INSTRUMENT.SYNSNR1 = [0x00, 0x06, 0x00, 0xF0, 0xF0, 0x00, 0x0E, 0x00, 0x00, 0xF8, 0xB6, 0x00];
// exports.INSTRUMENT.SYNSNR2 = [0x00, 0x06, 0x00, 0xF0, 0xF0, 0x00, 0x0E, 0x00, 0x00, 0xF6, 0xB4, 0x00];
// exports.INSTRUMENT.TINCAN1 = [0x00, 0x8F, 0x81, 0xEF, 0x01, 0x00, 0x04, 0x01, 0x00, 0x98, 0xF1, 0x00];
// exports.INSTRUMENT.TOM1 = [0x08, 0x04, 0x00, 0xF7, 0xB5, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00];
// exports.INSTRUMENT.TOM2 = [0x08, 0x02, 0x00, 0xC8, 0x97, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00];
// exports.INSTRUMENT.TRAINBEL = [0x00, 0x17, 0x4F, 0xF2, 0x61, 0x00, 0x08, 0x12, 0x08, 0xF2, 0x74, 0x00];
// exports.INSTRUMENT.TRIANGLE = [0x00, 0x26, 0x03, 0xE0, 0xF0, 0x00, 0x08, 0x1E, 0x00, 0xFF, 0x31, 0x00];
// exports.INSTRUMENT.TROMB1 = [0x00, 0xB1, 0x1C, 0x41, 0x1F, 0x00, 0x0E, 0x61, 0x80, 0x92, 0x3B, 0x00];
// exports.INSTRUMENT.TROMB2 = [0x00, 0x21, 0x1C, 0x53, 0x1D, 0x00, 0x0C, 0x61, 0x80, 0x52, 0x3B, 0x00];
// exports.INSTRUMENT.TRUMPET1 = [0x00, 0x31, 0x1C, 0x41, 0x0B, 0x00, 0x0E, 0x61, 0x80, 0x92, 0x3B, 0x00];
// exports.INSTRUMENT.TRUMPET2 = [0x00, 0x31, 0x1C, 0x23, 0x1D, 0x00, 0x0C, 0x61, 0x80, 0x52, 0x3B, 0x00];
// exports.INSTRUMENT.TRUMPET3 = [0x00, 0x31, 0x1C, 0x41, 0x01, 0x00, 0x0E, 0x61, 0x80, 0x92, 0x3B, 0x00];
// exports.INSTRUMENT.TRUMPET4 = [0x00, 0x31, 0x1C, 0x41, 0x0B, 0x00, 0x0C, 0x61, 0x80, 0x92, 0x3B, 0x00];
// exports.INSTRUMENT.TUBA1 = [0x00, 0x21, 0x19, 0x43, 0x8C, 0x00, 0x0C, 0x21, 0x80, 0x85, 0x2F, 0x00];
// exports.INSTRUMENT.VIBRA1 = [0x00, 0x84, 0x53, 0xF5, 0x33, 0x00, 0x06, 0xA0, 0x80, 0xFD, 0x25, 0x00];
// exports.INSTRUMENT.VIBRA2 = [0x00, 0x06, 0x73, 0xF6, 0x54, 0x00, 0x00, 0x81, 0x03, 0xF2, 0xB3, 0x00];
// exports.INSTRUMENT.VIBRA3 = [0x00, 0x93, 0x97, 0xAA, 0x12, 0x02, 0x0E, 0x91, 0x80, 0xAC, 0x21, 0x00];
// exports.INSTRUMENT.VIOLIN1 = [0x00, 0x31, 0x1C, 0x51, 0x03, 0x00, 0x0E, 0x61, 0x80, 0x54, 0x67, 0x00];
// exports.INSTRUMENT.VIOLIN2 = [0x00, 0xE1, 0x88, 0x62, 0x29, 0x00, 0x0C, 0x22, 0x80, 0x53, 0x2C, 0x00];
// exports.INSTRUMENT.VIOLIN3 = [0x00, 0xE1, 0x88, 0x64, 0x29, 0x00, 0x06, 0x22, 0x83, 0x53, 0x2C, 0x00];
// exports.INSTRUMENT.VLNPIZZ1 = [0x00, 0x31, 0x9C, 0xF1, 0xF9, 0x00, 0x0E, 0x31, 0x80, 0xF7, 0xE6, 0x00];
// exports.INSTRUMENT.WAVE = [0x00, 0x00, 0x02, 0x00, 0xF0, 0x00, 0x0E, 0x14, 0x80, 0x1B, 0xA2, 0x00];
// exports.INSTRUMENT.XYLO1 = [0x00, 0x11, 0x2D, 0xC8, 0x2F, 0x00, 0x0C, 0x31, 0x00, 0xF5, 0xF5, 0x00];
// exports.INSTRUMENT.XYLO2 = [0x06, 0x2E, 0x00, 0xFF, 0x0F, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00];
// exports.INSTRUMENT.XYLO3 = [0x00, 0x06, 0x00, 0xFF, 0xF0, 0x00, 0x0E, 0xC4, 0x00, 0xF8, 0xB5, 0x00];

exports.INSTRUMENT.PIANO1 = [0x00, 0x33, 0x5A, 0xB2, 0x50, 0x01, 0x00, 0x31, 0x00, 0xB1, 0xF5, 0x01];
exports.INSTRUMENT.PIANO2 = [0x00, 0x31, 0x49, 0xF2, 0x53, 0x07, 0x01, 0x11, 0x03, 0xF1, 0xF5, 0x00];
exports.INSTRUMENT.PIANO3 = [0x00, 0x31, 0x95, 0xD1, 0x83, 0x0D, 0x01, 0x32, 0x03, 0xC1, 0xF5, 0x00];
exports.INSTRUMENT.HONKTONK = [0x00, 0x34, 0x9B, 0xF3, 0x63, 0x01, 0x01, 0x11, 0x00, 0x92, 0xF5, 0x01];
exports.INSTRUMENT.EP1 = [0x00, 0x27, 0x28, 0xF8, 0xB7, 0x01, 0x02, 0x91, 0x00, 0xF1, 0xF9, 0x00];
exports.INSTRUMENT.EP2 = [0x00, 0x1A, 0x2D, 0xF3, 0xEE, 0x01, 0x01, 0x11, 0x00, 0xF1, 0xF5, 0x00];
exports.INSTRUMENT.HARPSIC = [0x00, 0x35, 0x95, 0xF2, 0x58, 0x0F, 0x01, 0x32, 0x02, 0x81, 0xF6, 0x01];
exports.INSTRUMENT.CLAVIC = [0x00, 0x31, 0x85, 0xC9, 0x40, 0x01, 0x00, 0x35, 0x00, 0xC2, 0xB9, 0x01];
exports.INSTRUMENT.CELESTA = [0x00, 0x09, 0x15, 0xC7, 0x64, 0x08, 0x00, 0x01, 0x05, 0xB2, 0x35, 0x00];
exports.INSTRUMENT.GLOCK = [0x00, 0x06, 0x03, 0xF4, 0x44, 0x00, 0x01, 0x01, 0x1B, 0xF2, 0x34, 0x00];
exports.INSTRUMENT.MUSICBOX = [0x00, 0x04, 0x06, 0xA9, 0x24, 0x0A, 0x01, 0x01, 0x01, 0xF5, 0x74, 0x00];
exports.INSTRUMENT.VIBES = [0x00, 0xD4, 0x00, 0xF6, 0x33, 0x00, 0x00, 0xF1, 0x00, 0x61, 0xE3, 0x00];
exports.INSTRUMENT.MARIMBA = [0x00, 0xD4, 0x00, 0xF7, 0xE8, 0x04, 0x00, 0xD1, 0x00, 0xA4, 0x64, 0x00];
exports.INSTRUMENT.XYLO = [0x00, 0x36, 0x16, 0xF7, 0xF7, 0x01, 0x00, 0x31, 0x07, 0xB5, 0xF5, 0x00];
exports.INSTRUMENT.TUBEBELL = [0x00, 0x03, 0x1B, 0xA2, 0x43, 0x0B, 0x00, 0x00, 0x00, 0xF3, 0x74, 0x00];
exports.INSTRUMENT.SANTUR = [0x00, 0xC3, 0x8E, 0xF8, 0x35, 0x01, 0x01, 0x11, 0x00, 0xC3, 0x94, 0x01];
exports.INSTRUMENT.ORGAN1 = [0x00, 0xE2, 0x07, 0xF4, 0x1B, 0x06, 0x01, 0xE0, 0x00, 0xF4, 0x0D, 0x01];
exports.INSTRUMENT.ORGAN2 = [0x00, 0xF2, 0x00, 0xF6, 0x2C, 0x04, 0x00, 0xF0, 0x00, 0xF5, 0x0B, 0x01];
exports.INSTRUMENT.ORGAN3 = [0x00, 0xF1, 0x06, 0xB6, 0x15, 0x0A, 0x00, 0xF0, 0x00, 0xBF, 0x07, 0x00];
exports.INSTRUMENT.PIPEORG = [0x00, 0x22, 0x03, 0x79, 0x16, 0x08, 0x01, 0xE0, 0x00, 0x6D, 0x08, 0x01];
exports.INSTRUMENT.REEDORG = [0x00, 0x31, 0x27, 0x63, 0x06, 0x01, 0x00, 0x72, 0x00, 0x51, 0x17, 0x01];
exports.INSTRUMENT.ACORDIAN = [0x00, 0xB4, 0x1D, 0x53, 0x16, 0x0F, 0x01, 0x71, 0x00, 0x51, 0x17, 0x01];
exports.INSTRUMENT.HARMONIC = [0x00, 0x25, 0x29, 0x97, 0x15, 0x01, 0x00, 0x32, 0x00, 0x53, 0x08, 0x01];
exports.INSTRUMENT.BANDNEON = [0x00, 0x24, 0x9E, 0x67, 0x15, 0x0F, 0x00, 0x31, 0x00, 0x53, 0x06, 0x01];
exports.INSTRUMENT.NYLONGT = [0x00, 0x13, 0x27, 0xA3, 0xB4, 0x05, 0x01, 0x31, 0x00, 0xD2, 0xF8, 0x00];
exports.INSTRUMENT.STEELGT = [0x00, 0x17, 0xA3, 0xF3, 0x32, 0x01, 0x00, 0x11, 0x00, 0xE2, 0xC7, 0x01];
exports.INSTRUMENT.JAZZGT = [0x00, 0x33, 0x24, 0xD2, 0xC1, 0x0F, 0x01, 0x31, 0x00, 0xF1, 0x9C, 0x00];
exports.INSTRUMENT.CLEANGT = [0x00, 0x31, 0x05, 0xF8, 0x44, 0x01, 0x00, 0x32, 0x02, 0xF2, 0xC9, 0x01];
exports.INSTRUMENT.MUTEGT = [0x00, 0x21, 0x09, 0x9C, 0x7B, 0x07, 0x00, 0x02, 0x03, 0x95, 0xFB, 0x00];
exports.INSTRUMENT.OVERDGT = [0x00, 0x21, 0x84, 0x81, 0x98, 0x07, 0x01, 0x21, 0x04, 0xA1, 0x59, 0x00];
exports.INSTRUMENT.DISTGT = [0x00, 0xB1, 0x0C, 0x78, 0x43, 0x01, 0x00, 0x22, 0x03, 0x91, 0xFC, 0x03];
exports.INSTRUMENT.GTHARMS = [0x00, 0x00, 0x0A, 0x82, 0x8C, 0x09, 0x00, 0x08, 0x02, 0xB4, 0xEC, 0x00];
exports.INSTRUMENT.ACOUBASS = [0x00, 0x21, 0x13, 0xAB, 0x46, 0x01, 0x00, 0x21, 0x00, 0x93, 0xF7, 0x00];
exports.INSTRUMENT.FINGBASS = [0x00, 0x01, 0x0A, 0xF9, 0x32, 0x01, 0x00, 0x22, 0x04, 0xC1, 0x58, 0x00];
exports.INSTRUMENT.PICKBASS = [0x00, 0x21, 0x07, 0xFA, 0x77, 0x0B, 0x00, 0x22, 0x02, 0xC3, 0x6A, 0x00];
exports.INSTRUMENT.FRETLESS = [0x00, 0x21, 0x17, 0x71, 0x57, 0x0B, 0x00, 0x21, 0x00, 0x62, 0x87, 0x00];
exports.INSTRUMENT.SLAPBAS1 = [0x00, 0x25, 0x01, 0xFA, 0x78, 0x07, 0x01, 0x12, 0x00, 0xF3, 0x97, 0x00];
exports.INSTRUMENT.SLAPBAS2 = [0x00, 0x21, 0x03, 0xFA, 0x88, 0x0D, 0x00, 0x13, 0x00, 0xB3, 0x97, 0x00];
exports.INSTRUMENT.SYNBASS1 = [0x00, 0x21, 0x09, 0xF5, 0x7F, 0x09, 0x01, 0x23, 0x04, 0xF3, 0xCC, 0x00];
exports.INSTRUMENT.SYNBASS2 = [0x00, 0x01, 0x10, 0xA3, 0x9B, 0x09, 0x00, 0x01, 0x00, 0x93, 0xAA, 0x00];
exports.INSTRUMENT.VIOLIN = [0x00, 0xE2, 0x19, 0xF6, 0x29, 0x0D, 0x01, 0xE1, 0x00, 0x78, 0x08, 0x01];
exports.INSTRUMENT.VIOLA = [0x00, 0xE2, 0x1C, 0xF6, 0x29, 0x0D, 0x01, 0xE1, 0x00, 0x78, 0x08, 0x01];
exports.INSTRUMENT.CELLO = [0x00, 0x61, 0x19, 0x69, 0x16, 0x0B, 0x01, 0x61, 0x00, 0x54, 0x27, 0x01];
exports.INSTRUMENT.CONTRAB = [0x00, 0x71, 0x18, 0x82, 0x31, 0x0D, 0x01, 0x32, 0x00, 0x61, 0x56, 0x00];
exports.INSTRUMENT.TREMSTR = [0x00, 0xE2, 0x23, 0x70, 0x06, 0x0D, 0x01, 0xE1, 0x00, 0x75, 0x16, 0x01];
exports.INSTRUMENT.PIZZ = [0x00, 0x02, 0x00, 0x88, 0xE6, 0x08, 0x00, 0x61, 0x00, 0xF5, 0xF6, 0x01];
exports.INSTRUMENT.HARP = [0x00, 0x12, 0x20, 0xF6, 0xD5, 0x0F, 0x01, 0x11, 0x80, 0xF3, 0xE3, 0x00];
exports.INSTRUMENT.TIMPANI = [0x00, 0x61, 0x0E, 0xF4, 0xF4, 0x01, 0x01, 0x00, 0x00, 0xB5, 0xF5, 0x00];
exports.INSTRUMENT.STRINGS = [0x00, 0x61, 0x1E, 0x9C, 0x04, 0x0F, 0x01, 0x21, 0x80, 0x71, 0x16, 0x00];
exports.INSTRUMENT.SLOWSTR = [0x00, 0xA2, 0x2A, 0xC0, 0xD6, 0x0F, 0x02, 0x21, 0x00, 0x30, 0x55, 0x01];
exports.INSTRUMENT.SYNSTR1 = [0x00, 0x61, 0x21, 0x72, 0x35, 0x0F, 0x01, 0x61, 0x00, 0x62, 0x36, 0x01];
exports.INSTRUMENT.SYNSTR2 = [0x00, 0x21, 0x1A, 0x72, 0x23, 0x0F, 0x01, 0x21, 0x02, 0x51, 0x07, 0x00];
exports.INSTRUMENT.CHOIR = [0x00, 0xE1, 0x16, 0x97, 0x31, 0x09, 0x00, 0x61, 0x00, 0x62, 0x39, 0x00];
exports.INSTRUMENT.OOHS = [0x00, 0x22, 0xC3, 0x79, 0x45, 0x01, 0x00, 0x21, 0x00, 0x66, 0x27, 0x00];
exports.INSTRUMENT.SYNVOX = [0x00, 0x21, 0xDE, 0x63, 0x55, 0x01, 0x01, 0x21, 0x00, 0x73, 0x46, 0x00];
exports.INSTRUMENT.ORCHIT = [0x00, 0x42, 0x05, 0x86, 0xF7, 0x0A, 0x00, 0x50, 0x00, 0x74, 0x76, 0x01];
exports.INSTRUMENT.TRUMPET = [0x00, 0x31, 0x1C, 0x61, 0x02, 0x0F, 0x00, 0x61, 0x81, 0x92, 0x38, 0x00];
exports.INSTRUMENT.TROMBONE = [0x00, 0x71, 0x1E, 0x52, 0x23, 0x0F, 0x00, 0x61, 0x02, 0x71, 0x19, 0x00];
exports.INSTRUMENT.TUBA = [0x00, 0x21, 0x1A, 0x76, 0x16, 0x0F, 0x00, 0x21, 0x01, 0x81, 0x09, 0x00];
exports.INSTRUMENT.MUTETRP = [0x00, 0x25, 0x28, 0x89, 0x2C, 0x07, 0x02, 0x20, 0x00, 0x83, 0x4B, 0x02];
exports.INSTRUMENT.FRHORN = [0x00, 0x21, 0x1F, 0x79, 0x16, 0x09, 0x00, 0xA2, 0x05, 0x71, 0x59, 0x00];
exports.INSTRUMENT.BRASS1 = [0x00, 0x21, 0x19, 0x87, 0x16, 0x0F, 0x00, 0x21, 0x03, 0x82, 0x39, 0x00];
exports.INSTRUMENT.SYNBRAS1 = [0x00, 0x21, 0x17, 0x75, 0x35, 0x0F, 0x00, 0x22, 0x82, 0x84, 0x17, 0x00];
exports.INSTRUMENT.SYNBRAS2 = [0x00, 0x21, 0x22, 0x62, 0x58, 0x0F, 0x00, 0x21, 0x02, 0x72, 0x16, 0x00];
exports.INSTRUMENT.SOPSAX = [0x00, 0xB1, 0x1B, 0x59, 0x07, 0x01, 0x01, 0xA1, 0x00, 0x7B, 0x0A, 0x00];
exports.INSTRUMENT.ALTOSAX = [0x00, 0x21, 0x16, 0x9F, 0x04, 0x0B, 0x00, 0x21, 0x00, 0x85, 0x0C, 0x01];
exports.INSTRUMENT.TENSAX = [0x00, 0x21, 0x0F, 0xA8, 0x20, 0x0D, 0x00, 0x23, 0x00, 0x7B, 0x0A, 0x01];
exports.INSTRUMENT.BARISAX = [0x00, 0x21, 0x0F, 0x88, 0x04, 0x09, 0x00, 0x26, 0x00, 0x79, 0x18, 0x01];
exports.INSTRUMENT.OBOE = [0x00, 0x31, 0x18, 0x8F, 0x05, 0x01, 0x00, 0x32, 0x01, 0x73, 0x08, 0x00];
exports.INSTRUMENT.ENGLHORN = [0x00, 0xA1, 0x0A, 0x8C, 0x37, 0x01, 0x01, 0x24, 0x04, 0x77, 0x0A, 0x00];
exports.INSTRUMENT.BASSOON = [0x00, 0x31, 0x04, 0xA8, 0x67, 0x0B, 0x00, 0x75, 0x00, 0x51, 0x19, 0x00];
exports.INSTRUMENT.CLARINET = [0x00, 0xA2, 0x1F, 0x77, 0x26, 0x01, 0x01, 0x21, 0x01, 0x74, 0x09, 0x00];
exports.INSTRUMENT.PICCOLO = [0x00, 0xE1, 0x07, 0xB8, 0x94, 0x01, 0x01, 0x21, 0x01, 0x63, 0x28, 0x00];
exports.INSTRUMENT.FLUTE1 = [0x00, 0xA1, 0x93, 0x87, 0x59, 0x01, 0x00, 0xE1, 0x00, 0x65, 0x0A, 0x00];
exports.INSTRUMENT.RECORDER = [0x00, 0x22, 0x10, 0x9F, 0x38, 0x01, 0x00, 0x61, 0x00, 0x67, 0x29, 0x00];
exports.INSTRUMENT.PANFLUTE = [0x00, 0xE2, 0x0D, 0x88, 0x9A, 0x01, 0x01, 0x21, 0x00, 0x67, 0x09, 0x00];
exports.INSTRUMENT.BOTTLEB = [0x00, 0xA2, 0x10, 0x98, 0x94, 0x0F, 0x00, 0x21, 0x01, 0x6A, 0x28, 0x00];
exports.INSTRUMENT.SHAKU = [0x00, 0xF1, 0x1C, 0x86, 0x26, 0x0F, 0x00, 0xF1, 0x00, 0x55, 0x27, 0x00];
exports.INSTRUMENT.WHISTLE = [0x00, 0xE1, 0x3F, 0x9F, 0x09, 0x00, 0x00, 0xE1, 0x00, 0x6F, 0x08, 0x00];
exports.INSTRUMENT.OCARINA = [0x00, 0xE2, 0x3B, 0xF7, 0x19, 0x01, 0x00, 0x21, 0x00, 0x7A, 0x07, 0x00];
exports.INSTRUMENT.SQUARWAV = [0x00, 0x22, 0x1E, 0x92, 0x0C, 0x0F, 0x00, 0x61, 0x06, 0xA2, 0x0D, 0x00];
exports.INSTRUMENT.SAWWAV = [0x00, 0x21, 0x15, 0xF4, 0x22, 0x0F, 0x01, 0x21, 0x00, 0xA3, 0x5F, 0x00];
exports.INSTRUMENT.SYNCALLI = [0x00, 0xF2, 0x20, 0x47, 0x66, 0x03, 0x01, 0xF1, 0x00, 0x42, 0x27, 0x00];
exports.INSTRUMENT.CHIFLEAD = [0x00, 0x61, 0x19, 0x88, 0x28, 0x0F, 0x00, 0x61, 0x05, 0xB2, 0x49, 0x00];
exports.INSTRUMENT.CHARANG = [0x00, 0x21, 0x16, 0x82, 0x1B, 0x01, 0x00, 0x23, 0x00, 0xB2, 0x79, 0x01];
exports.INSTRUMENT.SOLOVOX = [0x00, 0x21, 0x00, 0xCA, 0x93, 0x01, 0x00, 0x22, 0x00, 0x7A, 0x1A, 0x00];
exports.INSTRUMENT.FIFTHSAW = [0x00, 0x23, 0x00, 0x92, 0xC9, 0x08, 0x01, 0x22, 0x00, 0x82, 0x28, 0x01];
exports.INSTRUMENT.BASSLEAD = [0x00, 0x21, 0x1D, 0xF3, 0x7B, 0x0F, 0x00, 0x22, 0x02, 0xC3, 0x5F, 0x00];
exports.INSTRUMENT.FANTASIA = [0x00, 0xE1, 0x00, 0x81, 0x25, 0x00, 0x01, 0xA6, 0x86, 0xC4, 0x95, 0x01];
exports.INSTRUMENT.WARMPAD = [0x00, 0x21, 0x27, 0x31, 0x01, 0x0F, 0x00, 0x21, 0x00, 0x44, 0x15, 0x00];
exports.INSTRUMENT.POLYSYN = [0x00, 0x60, 0x14, 0x83, 0x35, 0x0D, 0x02, 0x61, 0x00, 0xD1, 0x06, 0x00];
exports.INSTRUMENT.SPACEVOX = [0x00, 0xE1, 0x5C, 0xD3, 0x01, 0x01, 0x01, 0x62, 0x00, 0x82, 0x37, 0x00];
exports.INSTRUMENT.BOWEDGLS = [0x00, 0x28, 0x38, 0x34, 0x86, 0x01, 0x02, 0x21, 0x00, 0x41, 0x35, 0x00];
exports.INSTRUMENT.METALPAD = [0x00, 0x24, 0x12, 0x52, 0xF3, 0x05, 0x01, 0x23, 0x02, 0x32, 0xF5, 0x01];
exports.INSTRUMENT.HALOPAD = [0x00, 0x61, 0x1D, 0x62, 0xA6, 0x0B, 0x00, 0xA1, 0x00, 0x61, 0x26, 0x00];
exports.INSTRUMENT.SWEEPPAD = [0x00, 0x22, 0x0F, 0x22, 0xD5, 0x0B, 0x01, 0x21, 0x84, 0x3F, 0x05, 0x01];
exports.INSTRUMENT.ICERAIN = [0x00, 0xE3, 0x1F, 0xF9, 0x24, 0x01, 0x00, 0x31, 0x01, 0xD1, 0xF6, 0x00];
exports.INSTRUMENT.SOUNDTRK = [0x00, 0x63, 0x00, 0x41, 0x55, 0x06, 0x01, 0xA2, 0x00, 0x41, 0x05, 0x01];
exports.INSTRUMENT.CRYSTAL = [0x00, 0xC7, 0x25, 0xA7, 0x65, 0x01, 0x01, 0xC1, 0x05, 0xF3, 0xE4, 0x00];
exports.INSTRUMENT.ATMOSPH = [0x00, 0xE3, 0x19, 0xF7, 0xB7, 0x01, 0x01, 0x61, 0x00, 0x92, 0xF5, 0x01];
exports.INSTRUMENT.BRIGHT = [0x00, 0x66, 0x9B, 0xA8, 0x44, 0x0F, 0x00, 0x41, 0x04, 0xF2, 0xE4, 0x01];
exports.INSTRUMENT.GOBLIN = [0x00, 0x61, 0x20, 0x22, 0x75, 0x0D, 0x00, 0x61, 0x00, 0x45, 0x25, 0x00];
exports.INSTRUMENT.ECHODROP = [0x00, 0xE1, 0x21, 0xF6, 0x84, 0x0F, 0x00, 0xE1, 0x01, 0xA3, 0x36, 0x00];
exports.INSTRUMENT.STARTHEM = [0x00, 0xE2, 0x14, 0x73, 0x64, 0x0B, 0x01, 0xE1, 0x01, 0x98, 0x05, 0x01];
exports.INSTRUMENT.SITAR = [0x00, 0x21, 0x0B, 0x72, 0x34, 0x09, 0x00, 0x24, 0x02, 0xA3, 0xF6, 0x01];
exports.INSTRUMENT.BANJO = [0x00, 0x21, 0x16, 0xF4, 0x53, 0x0D, 0x00, 0x04, 0x00, 0xF6, 0xF8, 0x00];
exports.INSTRUMENT.SHAMISEN = [0x00, 0x21, 0x18, 0xDA, 0x02, 0x0D, 0x00, 0x35, 0x00, 0xF3, 0xF5, 0x00];
exports.INSTRUMENT.KOTO = [0x00, 0x25, 0x0F, 0xFA, 0x63, 0x09, 0x00, 0x02, 0x00, 0x94, 0xE5, 0x01];
exports.INSTRUMENT.KALIMBA = [0x00, 0x32, 0x07, 0xF9, 0x96, 0x01, 0x00, 0x11, 0x00, 0x84, 0x44, 0x00];
exports.INSTRUMENT.BAGPIPE = [0x00, 0x20, 0x0E, 0x97, 0x18, 0x09, 0x02, 0x25, 0x03, 0x83, 0x18, 0x01];
exports.INSTRUMENT.FIDDLE = [0x00, 0x61, 0x18, 0xF6, 0x29, 0x01, 0x00, 0x62, 0x01, 0x78, 0x08, 0x01];
exports.INSTRUMENT.SHANNAI = [0x00, 0xE6, 0x21, 0x76, 0x19, 0x0B, 0x00, 0x61, 0x03, 0x8E, 0x08, 0x01];
exports.INSTRUMENT.TINKLBEL = [0x00, 0x27, 0x23, 0xF0, 0xD4, 0x01, 0x00, 0x05, 0x09, 0xF2, 0x46, 0x00];
exports.INSTRUMENT.AGOGO = [0x00, 0x1C, 0x0C, 0xF9, 0x31, 0x0F, 0x01, 0x15, 0x00, 0x96, 0xE8, 0x01];
exports.INSTRUMENT.STEELDRM = [0x00, 0x02, 0x00, 0x75, 0x16, 0x06, 0x02, 0x01, 0x00, 0xF6, 0xF6, 0x01];
exports.INSTRUMENT.WOODBLOK = [0x00, 0x25, 0x1B, 0xFA, 0xF2, 0x01, 0x00, 0x12, 0x00, 0xF6, 0x9A, 0x00];
exports.INSTRUMENT.TAIKO = [0x00, 0x02, 0x1D, 0xF5, 0x93, 0x01, 0x00, 0x00, 0x00, 0xC6, 0x45, 0x00];
exports.INSTRUMENT.MELOTOM = [0x00, 0x11, 0x15, 0xF5, 0x32, 0x05, 0x00, 0x10, 0x00, 0xF4, 0xB4, 0x00];
exports.INSTRUMENT.SYNDRUM = [0x00, 0x22, 0x06, 0xFA, 0x99, 0x09, 0x00, 0x01, 0x00, 0xD5, 0x25, 0x00];
exports.INSTRUMENT.REVRSCYM = [0x00, 0x2E, 0x00, 0xFF, 0x00, 0x0F, 0x02, 0x0E, 0x0E, 0x21, 0x2D, 0x00];
exports.INSTRUMENT.FRETNOIS = [0x00, 0x30, 0x0B, 0x56, 0xE4, 0x01, 0x01, 0x17, 0x00, 0x55, 0x87, 0x02];
exports.INSTRUMENT.BRTHNOIS = [0x00, 0x24, 0x00, 0xFF, 0x03, 0x0D, 0x00, 0x05, 0x08, 0x98, 0x87, 0x01];
exports.INSTRUMENT.SEASHORE = [0x00, 0x0E, 0x00, 0xF0, 0x00, 0x0F, 0x02, 0x0A, 0x04, 0x17, 0x04, 0x03];
exports.INSTRUMENT.BIRDS = [0x00, 0x20, 0x08, 0xF6, 0xF7, 0x01, 0x00, 0x0E, 0x05, 0x77, 0xF9, 0x02];
exports.INSTRUMENT.TELEPHON = [0x00, 0x20, 0x14, 0xF1, 0x08, 0x01, 0x00, 0x2E, 0x02, 0xF4, 0x08, 0x00];
exports.INSTRUMENT.HELICOPT = [0x00, 0x20, 0x04, 0xF2, 0x00, 0x03, 0x01, 0x23, 0x00, 0x36, 0x05, 0x01];
exports.INSTRUMENT.APPLAUSE = [0x00, 0x2E, 0x00, 0xFF, 0x02, 0x0F, 0x00, 0x2A, 0x05, 0x32, 0x55, 0x03];
exports.INSTRUMENT.GUNSHOT = [0x00, 0x20, 0x00, 0xA1, 0xEF, 0x0F, 0x00, 0x10, 0x00, 0xF3, 0xDF, 0x00];
