{
    BSD 3-Clause License
    Copyright (c) 2021-2022, Jerome Shidel
    All rights reserved.
}

{$I DANGER.DEF}
program ImgViewer;

uses Danger, SimpleUI, FFmtBMP;

const
    Version = '2022-02-10';

const
    FFmt : word = 0;
    Image : PImage = nil;
    Font : PFont = nil;
    Sprite : PSprite = nil;
    Files : PList = nil;
    Current : PListItem = nil;
    AboutVisible : boolean = false;

procedure AddFiles(FileSpec : String);
var
    Search : TSysFindRec;
    Kind : Word;
begin
    if FileExists(FileSpec) then begin
        Kind := FormatType(FileSpec);
        if (Kind = ffImage) or (Kind = ffSPrite) or (Kind = ffBitmapFont) then
            AddList(Files, StrPtr(FileSpec));
    end else begin
        if  SysFindFirst(FileSpec, faAnyFile, Search) then repeat
            Kind := FormatType(FilePath(FileSpec) + Search.Name);
            if (Kind = ffImage) or (Kind = ffSPrite) or (Kind = ffBitmapFont) then
                AddList(Files, StrPtr(FilePath(FileSpec) + Search.Name));
        until not SysFindNext(Search);
        SysFindClose(Search);
    end;
end;

procedure StatusBar;
var
    Name, S : String;
begin
    Name := PtrStr(Current^.Data);
    S := LCase(Name);
    if Assigned(Sprite) then
        Video^.Region(0, GetMaxY - Video^.TextHeight(S), GetMaxX - 1, GetMaxY - 1, 0);
    Video^.PutText(0, GetMaxY - Video^.TextHeight(S), S, 8);
    if Assigned(Sprite) then
        Video^.PutText(Video^.TextWidth(S + ' '),
            GetMaxY - Video^.TextHeight(S),
            IntStr(Sprite^.Sprites^[Sprite^.Index].Sequence), 7);
end;

procedure ShowImage;
var
    X, Y, XS, YS : integer;
    P : Pointer;
    Name : String;
    Size : LongInt;
    H : TFontSettings;
begin
    If not Assigned(Current) then Current := Files^.First;
    If not Assigned(Current) then Exit;
    AboutVisible := False;
    Video^.Fill(0);
    Name := PtrStr(Current^.Data);
    Video^.FreeImage(Image);
    Video^.FreeSprite(Sprite);
    Video^.FreeFont(Font);
    FFmt := FormatType(Name);
    ExitMessage := 'Load ' + Name;
    if FormatLoad(Name, P, Size) then begin
        case FFmt of
            ffImage : Image := P;
            ffBitmapFont : Font := P;
            ffSprite : Sprite := P;
        end;
    end else
        FatalError(GetError, Name);
    ExitMessage := '';
    if Assigned(Image) then
        Video^.PutImage(Image, GetMaxX div 2 - Image^.Width div 2,
            GetMaxY div 2 - Image^.Height div 2);
    if Assigned(Sprite) then begin
        Video^.SpriteMove(Sprite,
            GetMaxX div 2 - (Sprite^.Area.Right - Sprite^.Area.Left) div 2,
            GetMaxY div 2 - (Sprite^.Area.Bottom - Sprite^.Area.Top) div 2);
        Sprite^.Animate := -1;
        Video^.SpriteShow(Sprite);
    end;
    if Assigned(Font) then begin
        GetFontSettings(H);
        Video^.SetFont(Font);
        XS := Font^.MonoWidth + Font^.Spacing;
        YS := Font^.Height;
        for Y := 0 to 9 do
            for X := 0 to 23 do
                Video^.PutChar(
                    GetMaxX div 2 - (12 * XS) + X * XS,
                    GetMaxY div 2 - (5 * YS) + Y * YS,
                    Char(Y * 24 + X), 15);
        SetFontSettings(H);
    end;
    StatusBar;
end;

procedure NextImage;
var
    O, N : word;
begin
    If not Assigned(Current) then Current := Files^.First;
    If not Assigned(Current) then Exit;
    if Assigned(Sprite) then begin
        O := Video^.SpriteGetSeq(Sprite);
        N := Video^.SpriteSetSeq(Sprite, O + 1);
        if O <> N then begin
            StatusBar;
            exit;
        end;
    end;
    Current := Current^.Next;
    ShowImage;
end;

procedure PrevImage;
var
    O, N : word;
begin
    If not Assigned(Current) then Current := Files^.Last;
    If not Assigned(Current) then Exit;
    if Assigned(Sprite) then begin
        O := Video^.SpriteGetSeq(Sprite);
        N := Video^.SpriteSetSeq(Sprite, O - 1);
        if O <> N then begin
            StatusBar;
            exit;
        end;
    end;
    Current := Current^.Prev;
    If not Assigned(Current) then Current := Files^.Last;
    ShowImage;
    if Assigned(Sprite) then begin
        repeat
            O := Video^.SpriteGetSeq(Sprite);
            N := Video^.SpriteSetSeq(Sprite, O + 1);
        until O = N;
        if N <> 0 then StatusBar;
    end;
end;

procedure Execute;
var
    E : TEvent;
    Finished : boolean;
    LTT : LongInt;
begin
    PurgeEvents;
    Finished := not Assigned(Files^.First);
    if Finished then
        AboutBox
    else
        ShowImage;
    LTT := TimerTick;
    repeat
        if Finished then Break;
        repeat
            Video^.Update;
            repeat
                if (not AboutVisible) and Assigned(Sprite) and (LTT <> TimerTick) then begin
                    LTT := TimerTick;
                    Video^.SpriteNextAll(False);
                    Video^.Update;
                end;
                Idle;
            until GetEvent(E);
            Video^.Prepare;
            if MouseEvent(E) then MouseMove(E.Position);
        until (E.Kind = evKeyPress) or (E.Kind = evMouseClick);
        if (E.Kind = evKeyPress) then
            case E.KeyCode of
                kbEnter : begin
                    if AboutVisible then
                        ShowImage
                    else begin
                        AboutBox;
                        AboutVisible := True;
                        ShowImage
                    end;
                end;
                kbNone : begin
                    if E.ScanCode = scPgUp then PrevImage;
                    if E.ScanCode = scPgDn then NextImage;
                end;
            end;
        Finished := Finished or ((E.Kind = evKeypress) and (E.KeyCode = kbEscape ));
    until Finished;
    PurgeEvents;
end;

procedure ProcessCommandLn;
var
    Opt : String;
    I, E : integer;
begin
    I := 0;
    while I < ParamCount do begin
        Inc(I);
        Opt := UCase(ParamStr(I));
        if (Opt = '/H') or (Opt = '/?') then begin
            PrintHelp;
        end else
        if Opt = '/BM' then begin
            AppInfo.Driver.Video   := 'BIOSVID.DRV';
            Inc(I);
            if UCase(ParamStr(I)) <> 'LIST' then
                Val(ParamStr(I), AppInfo.VideoMode, E)
            else
                AppInfo.VideoMode:=$ffff;
            Inc(I);
        end else
            AddFiles(ParamStr(I));
    end;
end;

begin
    AppInfo.Title := 'ImgView';
    AppInfo.Version := Version;
    AppInfo.Year := '2021-2022';
    Files := NewList;
    FormatPaletteMode := ipmOverride;
    ProcessCommandLn;
    Ignite;
    Execute;
    Extinguish;
end.