-- rijndael.e - Rijndael encryption algorithm module
-- Created for use with the Block Cipher Library, version 1.0
-- Copyright (C) 2000  Davi Tassinari de Figueiredo
--
-- This program is distributed under the terms of the GNU General
-- Public License. Please read the documentation for more information.
--
-- You can get the latest version of this program from my Euphoria page:
-- http://www16.brinkster.com/davitf/


constant

-- Tables used for multiplication in GF(x)
Logtable = {
  0,  0, 25,  1, 50,  2, 26,198, 75,199, 27,104, 51,238,223,  3,
100,  4,224, 14, 52,141,129,239, 76,113,  8,200,248,105, 28,193,
125,194, 29,181,249,185, 39,106, 77,228,166,114,154,201,  9,120,
101, 47,138,  5, 33, 15,225, 36, 18,240,130, 69, 53,147,218,142,
150,143,219,189, 54,208,206,148, 19, 92,210,241, 64, 70,131, 56,
102,221,253, 48,191,  6,139, 98,179, 37,226,152, 34,136,145, 16,
126,110, 72,195,163,182, 30, 66, 58,107, 40, 84,250,133, 61,186,
 43,121, 10, 21,155,159, 94,202, 78,212,172,229,243,115,167, 87,
175, 88,168, 80,244,234,214,116, 79,174,233,213,231,230,173,232,
 44,215,117,122,235, 22, 11,245, 89,203, 95,176,156,169, 81,160,
127, 12,246,111, 23,196, 73,236,216, 67, 31, 45,164,118,123,183,
204,187, 62, 90,251, 96,177,134, 59, 82,161,108,170, 85, 41,157,
151,178,135,144, 97,190,220,252,188,149,207,205, 55, 63, 91,209,
 83, 57,132, 60, 65,162,109, 71, 20, 42,158, 93, 86,242,211,171,
 68, 17,146,217, 35, 32, 46,137,180,124,184, 38,119,153,227,165,
103, 74,237,222,197, 49,254, 24, 13, 99,140,128,192,247,112,  7},

Alogtable = {
  1,  3,  5, 15, 17, 51, 85,255, 26, 46,114,150,161,248, 19, 53,
 95,225, 56, 72,216,115,149,164,247,  2,  6, 10, 30, 34,102,170,
229, 52, 92,228, 55, 89,235, 38,106,190,217,112,144,171,230, 49,
 83,245,  4, 12, 20, 60, 68,204, 79,209,104,184,211,110,178,205,
 76,212,103,169,224, 59, 77,215, 98,166,241,  8, 24, 40,120,136,
131,158,185,208,107,189,220,127,129,152,179,206, 73,219,118,154,
181,196, 87,249, 16, 48, 80,240, 11, 29, 39,105,187,214, 97,163,
254, 25, 43,125,135,146,173,236, 47,113,147,174,233, 32, 96,160,
251, 22, 58, 78,210,109,183,194, 93,231, 50, 86,250, 21, 63, 65,
195, 94,226, 61, 71,201, 64,192, 91,237, 44,116,156,191,218,117,
159,186,213,100,172,239, 42,126,130,157,188,223,122,142,137,128,
155,182,193, 88,232, 35,101,175,234, 37,111,177,200, 67,197, 84,
252, 31, 33, 99,165,244,  7,  9, 27, 45,119,153,176,203, 70,202,
 69,207, 74,222,121,139,134,145,168,227, 62, 66,198, 81,243, 14,
 18, 54, 90,238, 41,123,141,140,143,138,133,148,167,242, 13, 23,
 57, 75,221,124,132,151,162,253, 28, 36,108,180,199, 82,246,  1},


S = {                                       -- S-box
 99,124,119,123,242,107,111,197, 48,  1,103, 43,254,215,171,118,
202,130,201,125,250, 89, 71,240,173,212,162,175,156,164,114,192,
183,253,147, 38, 54, 63,247,204, 52,165,229,241,113,216, 49, 21,
  4,199, 35,195, 24,150,  5,154,  7, 18,128,226,235, 39,178,117,
  9,131, 44, 26, 27,110, 90,160, 82, 59,214,179, 41,227, 47,132,
 83,209,  0,237, 32,252,177, 91,106,203,190, 57, 74, 76, 88,207,
208,239,170,251, 67, 77, 51,133, 69,249,  2,127, 80, 60,159,168,
 81,163, 64,143,146,157, 56,245,188,182,218, 33, 16,255,243,210,
205, 12, 19,236, 95,151, 68, 23,196,167,126, 61,100, 93, 25,115,
 96,129, 79,220, 34, 42,144,136, 70,238,184, 20,222, 94, 11,219,
224, 50, 58, 10, 73,  6, 36, 92,194,211,172, 98,145,149,228,121,
231,200, 55,109,141,213, 78,169,108, 86,244,234,101,122,174,  8,
186,120, 37, 46, 28,166,180,198,232,221,116, 31, 75,189,139,138,
112, 62,181,102, 72,  3,246, 14, 97, 53, 87,185,134,193, 29,158,
225,248,152, 17,105,217,142,148,155, 30,135,233,206, 85, 40,223,
140,161,137, 13,191,230, 66,104, 65,153, 45, 15,176, 84,187, 22},

Si = {                                       -- Inverse S-box
 82,  9,106,213, 48, 54,165, 56,191, 64,163,158,129,243,215,251,
124,227, 57,130,155, 47,255,135, 52,142, 67, 68,196,222,233,203,
 84,123,148, 50,166,194, 35, 61,238, 76,149, 11, 66,250,195, 78,
  8, 46,161,102, 40,217, 36,178,118, 91,162, 73,109,139,209, 37,
114,248,246,100,134,104,152, 22,212,164, 92,204, 93,101,182,146,
108,112, 72, 80,253,237,185,218, 94, 21, 70, 87,167,141,157,132,
144,216,171,  0,140,188,211, 10,247,228, 88,  5,184,179, 69,  6,
208, 44, 30,143,202, 63, 15,  2,193,175,189,  3,  1, 19,138,107,
 58,145, 17, 65, 79,103,220,234,151,242,207,206,240,180,230,115,
150,172,116, 34,231,173, 53,133,226,249, 55,232, 28,117,223,110,
 71,241, 26,113, 29, 41,197,137,111,183, 98, 14,170, 24,190, 27,
252, 86, 62, 75,198,210,121, 32,154,219,192,254,120,205, 90,244,
 31,221,168, 51,136,  7,199, 49,177, 18, 16, 89, 39,128,236, 95,
 96, 81,127,169, 25,181, 74, 13, 45,229,122,159,147,201,156,239,
160,224, 59, 77,174, 42,245,176,200,235,187, 60,131, 83,153, 97,
 23, 43,  4,126,186,119,214, 38,225,105, 20, 99, 85, 33, 12,125},

		    -- Round constants used in key expansion
rcon = {#01,#02,#04,#08,#10,#20,#40,#80,#1B,#36,#6C,#D8,#AB,#4D,#9A,
	#2F,#5E,#BC,#63,#C6,#97,#35,#6A,#D4,#B3,#7D,#FA,#EF,#C5,#91},

				       -- Permutation in ShiftRow step
ShiftRowTable = {{ 1, 6,11,16, 5,10,15, 4, 9,14, 3, 8,13, 2, 7,12},
		 { 1, 6,11,16, 5,10,15,20, 9,14,19,24,13,18,23, 4,17,22, 3, 8,21, 2, 7,12},
		 { 1, 6,15,20, 5,10,19,24, 9,14,23,28,13,18,27,32,17,22,31, 4,21,26, 3, 8,25,30, 7,12,29, 2,11,16}},

				       -- Inverse permutation
iShiftRowTable ={{ 1,14,11, 8, 5, 2,15,12, 9, 6, 3,16,13,10, 7, 4},
		 { 1,22,19,16, 5, 2,23,20, 9, 6, 3,24,13,10, 7, 4,17,14,11, 8,21,18,15,12},
		 { 1,30,23,20, 5, 2,27,24, 9, 6,31,28,13,10, 3,32,17,14, 7, 4,21,18,11, 8,25,22,15,12,29,26,19,16}}

function mul (integer a, integer b)
    if a = 0 or b = 0 then return 0
    else return Alogtable [ remainder(Logtable [a+1] + Logtable [b+1], 255)+1]
    end if
end function


constant Nb = 128 / 32, ShiftRow = ShiftRowTable [Nb/2-1], iShiftRow = iShiftRowTable [Nb/2-1]
integer Nk, Nr

procedure set_Nr ()
    if Nb = 8 or Nk = 8 then Nr = 14
    elsif Nb = 6 or Nk = 6 then Nr = 12
    else Nr = 10 end if

end procedure

sequence subkeys, i_subkeys

sequence mul2, mul3, mul9, mulE, mulB, mulD
mul2 = repeat (-1, 256)    mul3 = repeat (-1, 256)
mul9 = repeat (-1, 256)    mulE = repeat (-1, 256)
mulB = repeat (-1, 256)    mulD = repeat (-1, 256)
for z = 0 to 255 do
     mul2 [z+1] = mul (2, z)             mul3 [z+1] = mul (3, z)
     mul9 [z+1] = mul (9, z)             mulE [z+1] = mul (#E, z)
     mulB [z+1] = mul (#B, z)            mulD [z+1] = mul (#D, z)
end for

function MixColumn (sequence state)
    integer col1, col2, col3, col4

    for n = 1 to length (state) by 4 do
	col1 = state[n]
	col2 = state[n+1]
	col3 = state[n+2]
	col4 = state[n+3]
	state [n..n+3] =
	   { xor_bits(xor_bits(xor_bits(mul2[col1+1],mul3[col2+1]),col3),col4),
	     xor_bits(xor_bits(xor_bits(mul2[col2+1],mul3[col3+1]),col4),col1),
	     xor_bits(xor_bits(xor_bits(mul2[col3+1],mul3[col4+1]),col1),col2),
	     xor_bits(xor_bits(xor_bits(mul2[col4+1],mul3[col1+1]),col2),col3)}
    end for

    return state
end function


function InvMixColumn (sequence state)
    integer col1, col2, col3, col4

    for n = 1 to length (state) by 4 do
	col1 = state[n]
	col2 = state[n+1]
	col3 = state[n+2]
	col4 = state[n+3]
	state [n..n+3] =
	   { xor_bits(xor_bits(xor_bits(mulE[col1+1],mulB[col2+1]),mulD[col3+1]),mul9[col4+1]),
	     xor_bits(xor_bits(xor_bits(mulE[col2+1],mulB[col3+1]),mulD[col4+1]),mul9[col1+1]),
	     xor_bits(xor_bits(xor_bits(mulE[col3+1],mulB[col4+1]),mulD[col1+1]),mul9[col2+1]),
	     xor_bits(xor_bits(xor_bits(mulE[col4+1],mulB[col1+1]),mulD[col2+1]),mul9[col3+1])}
    end for

    return state
end function

-- Make lookup tables

sequence t0, t1, t2, t3, ti0, ti1, ti2, ti3
procedure make_lookup_tables ()
    atom Sa, Sia

    t0 = repeat (0, 256)
    t1 = repeat (0, 256)
    t2 = repeat (0, 256)
    t3 = repeat (0, 256)

    ti0 = repeat (0, 256)
    ti1 = repeat (0, 256)
    ti2 = repeat (0, 256)
    ti3 = repeat (0, 256)

    for a = 1 to 256 do
	Sa = S [a]
	t0 [a] = {mul2 [Sa + 1], Sa, Sa, mul3 [Sa + 1]}
	t1 [a] = {mul3 [Sa + 1], mul2 [Sa + 1], Sa, Sa}
	t2 [a] = {Sa, mul3 [Sa + 1], mul2 [Sa + 1], Sa}
	t3 [a] = {Sa, Sa, mul3 [Sa + 1], mul2 [Sa + 1]}

	Sia = Si [a]
	ti0 [a] = {mulE [Sia+1],mul9 [Sia+1],mulD [Sia+1],mulB [Sia+1]}
	ti1 [a] = {mulB [Sia+1],mulE [Sia+1],mul9 [Sia+1],mulD [Sia+1]}
	ti2 [a] = {mulD [Sia+1],mulB [Sia+1],mulE [Sia+1],mul9 [Sia+1]}
	ti3 [a] = {mul9 [Sia+1],mulD [Sia+1],mulB [Sia+1],mulE [Sia+1]}
    end for

end procedure
make_lookup_tables ()

-- ? ti0 [1000..1024]
-- ? T5 [1000..1024]
-- printf (1,"0x%02x, 0x%02x, 0x%02x, 0x%02x, 0x%02x, 0x%02x, 0x%02x, 0x%02x",ti3)
-- abort(0)



global procedure rijndael_init (sequence key)
    sequence temp, W

    Nk = length(key) / 4
    set_Nr ()

    W = repeat (-1, Nb * (Nr + 1))
    for i = 1 to Nk do
	W [i] = key [4*i-3 .. 4*i]
    end for

    for i = Nk to Nb * (Nr + 1) - 1 do
	temp = W [i]

	if remainder (i, Nk) = 0 then
	    temp = {xor_bits (S[temp [2]+1], rcon [i / Nk]),
		    S[temp[3]+1], S[temp[4]+1], S[temp [1]+1]}

	elsif Nk = 8 and remainder (i, Nk) = 4 then
	    temp = {S [temp[1]+1], S[temp[2]+1], S[temp[3]+1], S[temp[4]+1]}
	end if

	W [i + 1 ] = xor_bits (W [i - Nk + 1], temp )
    end for

    subkeys = repeat ( -1, Nr + 1)
    temp = repeat (-1, Nb * 4)

    for n = 1 to Nr + 1 do
	for k = 1 to Nb do
	    temp [k * 4 - 3 .. k * 4] = W [ (n-1)* Nb + k ]
	end for
	subkeys [n] = temp
    end for

    i_subkeys = subkeys
    for n = 2 to Nr do
	i_subkeys [n] = InvMixColumn (i_subkeys [n])
    end for

end procedure


global function rijndael_enc_block (sequence state)
    sequence temp

    state = xor_bits (state, subkeys [1] )

    temp = repeat (-1, length(state))

    for r = 2 to Nr do

	for n = 1 to length (state) by 4 do
	    temp [n..n+3] = xor_bits (xor_bits(xor_bits(
	       t0 [state[ShiftRow [n]]+1],
	       t1 [state[ShiftRow [n+1]]+1]),
	       t2 [state[ShiftRow [n+2]]+1]),
	       t3 [state[ShiftRow [n+3]]+1])
	end for

	-- Add round key
	state = xor_bits ( temp, subkeys [r])

    end for

    -- Final round
    -- Byte sub and Shift row
    for n = 1 to length (state) do
	temp [n] = S [state [ShiftRow[n]] + 1]
    end for

    -- Add round key
    state = xor_bits ( temp, subkeys [Nr + 1])

    return state
end function


global function rijndael_dec_block (sequence state)
    sequence temp

    state = xor_bits (state, subkeys [Nr + 1] )

    temp = repeat (-1, length(state))

    for r = Nr to 2 by -1 do

	for n = 1 to length (state) by 4 do
	    temp [n..n+3] = xor_bits (xor_bits(xor_bits(
	       ti0 [state[iShiftRow [n]]+1],
	       ti1 [state[iShiftRow [n+1]]+1]),
	       ti2 [state[iShiftRow [n+2]]+1]),
	       ti3 [state[iShiftRow [n+3]]+1])
	end for

	-- Add round key
	state = xor_bits ( temp, i_subkeys [r])

    end for

    -- Final round
    -- Byte sub and Shift row
    for n = 1 to length (state) do
	temp [n] = Si [state [iShiftRow[n]] + 1]
    end for

    -- Add round key
    state = xor_bits ( temp, subkeys [1])

    return state
end function


global procedure rijndael_clean_status ()
    subkeys = {}
end procedure

global function rijndael_get_status ()
    return {Nk, Nr, subkeys, i_subkeys}
end function

global procedure rijndael_set_status (sequence st)
    Nk = st [1]
    Nr = st [2]
    subkeys = st [3]
    i_subkeys = st [4]
end procedure

-- global constant ALG_RIJNDAEL128 = register_enc_alg ("Rijndael-128", "rijndael", 16, 16)
-- global constant ALG_RIJNDAEL192 = register_enc_alg ("Rijndael-192", "rijndael", 24, 16)
-- global constant ALG_RIJNDAEL256 = register_enc_alg ("Rijndael-256", "rijndael", 32, 16)
global constant ALG_RIJNDAEL = register_enc_alg ("Rijndael", "rijndael", 32, 16)

