/*
 * md5sum.c	- Generate/check MD5 Message Digests
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Written March 1993 by Branko Lankester
 * Modified June 1993 by Colin Plumb for altered md5.c.
 * Modified August 2005 by Blair Campbell for FreeDOS project and enhanced features
 * Modified September 2005 by Blair Campbell for bugfixes, recursing, LFNs, and SHA-1 support,
 * relicenced under GPL
 *
 * Version History:
 * 1.0 First release by Blair Campbell with binary mode as default and a text mode switch
 * 2.0 Added wildcard support, support for - and / switches, removed getopt, kitten translation
 * 3.0 Added LFN support, SHA-1 support, a recursive option, added support for @file lists,
 *     CRC-32 checksumming, max recursion levels, Turbo C support, Pacific C support,
 *     and fixed many discovered bugs
 *
 * Todo list/Wishes:
 * Add CRC(16/32) checksum support - Partially done - CRC-32 checksumming is done
 * Add RMD-160 support, may or may not do - Partially done, but not functional and will not be
 * part of an official release
 * Port to DJGPP
 *
 * Known Bugs:
 * In recursive and lfn modes, it seems to stop searching after a set amount of files
 */
#ifdef __DJGPP__
#undef USE_LFN
#endif /*DJGPP*/
#include <stdio.h>						/* Standard input and output */
#include <stdlib.h>						/* Getenv to parse %MD5% */
#include <dos.h>						/* DOS function headers */
#include <string.h>						/* String-manipulation header */
#include <ctype.h>						/* Toupper function */
#if defined(__WATCOMC__)
#include <io.h>							/* Access function */
#include <sys\stat.h>						/* stat function */
#include <direct.h>						/* getcwd, _getdrive */
#include <malloc.h>						/* Malloc function */
#elif defined(__TURBOC__)
#include "wat2tcc.h"
#include <io.h>							/* Access function */
#include <sys\stat.h>						/* stat function */
#include <alloc.h>
#include <dir.h>
#elif defined(__PACIFIC__)
#include <sys.h>
#include <unixio.h>						/* Access function */
#include <stat.h>						/* stat function */
#include "wat2pac.h"
#endif
#include "md5.h"						/* MD5 header */
#ifdef USE_SHA
#include "sha1.h"						/* SHA-1 header */
#include "sha256.h"						/* SHA-256 header */
#endif /*SHA*/
#ifdef USE_RMD
#include "rmd160.h"
#endif /*RMD*/
#ifdef USE_CRC
#include "crc.h"						/* CRC-16/32 header */
#endif /*CRC*/
#include "kitten.h"						/* Translation header */
#include "others.h"						/* DOS switchar, truename() */
#ifdef USE_LFN
#include "io95\io95.h"						/* LFN functions */
#include "io95\find95.h"					/* LFN findfirst/next/stop */
#include "lfn.h"						/* lfntruename, supportsLFN */
#endif /*LFN*/

#define	FOPRTXT	"r"						/* Text file mode */
#define	FOPRBIN	"rb"						/* Binary file mode */

int ckfile(FILE *fp, unsigned char *digest);			/* Define variables and functs */
int do_check(FILE *chkf);
void print_the_digest(unsigned char *);

short verbose = 0, bin_mode = 1, check = 0, recursive = 0, list = 0;
unsigned char digest[32];
#ifdef USE_LFN
char longnm[MAXPATH95];						/* LFN name and mode of chksum */
short lfn_mode = 0, showlfn = 0;
char listname[MAXPATH95];					/* For file lists */
#else
char listname[_MAX_PATH];					/* For short names */
#endif
char truenm[_MAX_PATH]="                             ";		/* For truename() */
#ifdef USE_SHA
short sha = 0, sha256 = 0;
#endif
#ifdef USE_RMD
short rmd = 0;
unsigned int rmdhash[4096/8];
#endif
#ifdef USE_CRC
short crc16 = 0, crc32 = 0;
unsigned short crc16sum = 0;
long crc32sum;
#endif
short md = 1;
char *dirs;							/* For original dir specified */
int levels = 0, maxlevel = 4;

int mksum(char *name, int rc, char *arg)			/* Function to create the sum */
{
	struct _find_t nameinfo;
	struct stat dir;					/* To check for directories */
#ifdef USE_LFN
	struct ffblk95 lfninfo;
#endif
	FILE *fname;						/* For findfirst and opening */
	int filename;
#ifdef USE_LFN
	if(lfn_mode) filename = findfirst95(name, &lfninfo, FA95_RDONLY|FA95_HIDDEN|FA95_SYSTEM|FA95_DIREC|FA95_ARCH);
	else
#endif
	filename = _dos_findfirst(name, _A_NORMAL|_A_SYSTEM|_A_HIDDEN|_A_SUBDIR, &nameinfo);
	if (filename != 0)					/* Check file existance */
	{
		fprintf(stderr, "%s: %s\n",			/* No matching files */
				kittengets(1,1,"File does not exist"),
				arg);
		rc = 2;						/* Set exit code and continue */
	}
	while(filename == 0)					/* Find all files matching arg */
	{
#ifdef USE_LFN
		if(lfn_mode)					/* If LFNs are supported */
		{
			strcpy(nameinfo.name,lfninfo.ff_95.ff_shortname);
			if(access(nameinfo.name, F_OK) != 0)	/* Make sure the files exists */
			{
				findstop95(&lfninfo);		/* If it doesn't, */
				filename = _dos_findfirst(name, _A_NORMAL|_A_SYSTEM|_A_HIDDEN|_A_SUBDIR, &nameinfo);
				lfn_mode = 0;			/* Use the regular findfirst */
				if(filename != 0) {_dos_findclose(&nameinfo);return 2;}
			}
		}
#endif								/* Get the true name of file */
		if(truename((char _far *)truenm, (char _far *)nameinfo.name) == NULL) return 2;
#ifdef USE_LFN
		if(showlfn && lfntruename((char *)longnm, (char *)nameinfo.name) == NULL)
			showlfn = 0;				/* Get the true LFN of file */
#endif
		stat(truenm, &dir);				/* Stat the file */
		if(S_ISDIR(dir.st_mode) && strcmp(nameinfo.name, ".") != 0 && strcmp(nameinfo.name, "..") != 0)
		{						/* Is it a directory? */
			if(!recursive)				/* If not using /R */
			{					/* Print a Warning */
#ifdef USE_LFN
				fprintf(stderr, "%s: %s\n", kittengets(1,11,"File is a subdirectory"), showlfn ? longnm : truenm);
#else
				fprintf(stderr, "%s: %s\n", kittengets(1,11,"File is a subdirectory"), truenm);
#endif
			}
			if(recursive && levels < maxlevel)	/* Recursive option specified */
			{
				char backup[PATH_MAX];
				getcwd(backup,PATH_MAX);	/* Backup current directory */
				++levels;			/* Increase the used levels */
				chdir(nameinfo.name);		/* Change dir to filename */
#ifdef USE_LFN
				strcat((showlfn)?longnm:truenm, "\\*.*");
				--levels;
				rc = mksum("*.*", rc, (showlfn)?longnm:truenm);
#else
				strcat(truenm, "\\*.*");
				--levels;
				rc = mksum("*.*", rc, truenm);
#endif
				/* Recursively make checksums */
				chdir(backup);			/* Change to former directory */
			}
		}
		else if (strcmp(nameinfo.name, ".") != 0 && strcmp(nameinfo.name, "..") != 0)
		{						/* Is it a file? */
			if (bin_mode) fname = fopen(truenm, FOPRBIN);
			else fname = fopen(truenm, FOPRTXT);
								/* Generate according to mode */
			if (ckfile(fname, digest))		/* Check if digest failed */
			{
#ifdef USE_LFN
				fprintf(stderr, "md5sum: %s %s\n",
						kittengets(1,3,"error reading"),
						showlfn ? longnm : truenm);
#else
				fprintf(stderr, "md5sum: %s %s\n",
						kittengets(1,3,"error reading"), truenm);
#endif
				rc = 1;				/* Set exit code and continue */
			}
			print_the_digest(digest);			/* Print the digest */
#ifdef USE_LFN
			printf( " %c%s\n", bin_mode ? '*' : ' ', showlfn ? longnm : truenm);
#else
			printf( " %c%s\n", bin_mode ? '*' : ' ', truenm);
#endif
			fclose(fname);				/* Close the file */
		}
#ifdef USE_LFN
		if(lfn_mode) filename = findnext95(&lfninfo);
		else
#endif
			filename = _dos_findnext(&nameinfo);	/* Find next file matching arg */
	}
#ifdef USE_LFN
	if(lfn_mode) findstop95(&lfninfo);			/* Close the LFN find */
	else
#endif
		_dos_findclose(&nameinfo);			/* All files found, close find */
	return(rc);						/* Return any errors */
}

int usage(void)							/* The help function */
{								/* Print help messages */
	printf("\n%s",  kittengets(0,0, "Usage: MD5SUM [options] [/C [file]] | [file...]"));
	printf("\n%s",  kittengets(0,1, "Generates or checks MD5 Message Digests"));
	printf("\n%s",  kittengets(0,2, "    /C           check message digests (default is generate)"));
	printf("\n%s",  kittengets(0,3, "    /[-]V        enable or disable verbose mode (default is off)"));
#ifdef USE_LFN
	printf("\n%s",  kittengets(0,4, "    /[-]L        enable or disable LFN usage (default is enable if supported)"));
#endif
	printf("\n%s",  kittengets(0,5, "    /[-]R[num]   enable or diable recursing (default is off)"));
	printf("\n%s",  kittengets(0,6, "                 num is a number from 1 to 9 and limits recursion (default=4)"));
	printf("\n%s",  kittengets(0,7, "    /[-]S        same as /[-]R"));
	printf("\n%s",  kittengets(0,8, "    /T           read files in text mode"));
	printf("\n%s",  kittengets(0,9, "    /B           read files in binary mode (default; overrides /T)"));
#ifdef USE_SHA
	printf("\n%s",  kittengets(0,10, "    /M[:|=]mode  select the digest mode (SHA or MD5)"));
#endif
	printf("\n%s",  kittengets(0,11, "The input for /C should be the list of message digests and file names"));
	printf("\n%s",  kittengets(0,12,"that is printed on stdout by this program when it generates digests."));
	printf("\n%s",  kittengets(0,13,"The argument for /M should be a digest mode, which could possibly be SHA,"));
	printf("\n%s\n",kittengets(0,14,"CRC32, SHA256, or MD5, depending on how MD5SUM was compiled."));
	kittenclose();						/* Close catalog and return */
	return 2;
}

void print_the_digest(unsigned char *p)
{
	int i;
#ifdef USE_SHA
	if(sha)for(i = 0; i < 20; i++ ) printf( "%02x", *p++);	/* Print SHA-1 digest */
#ifdef USE_SHA256
	if(sha256)for(i=0;i < 32; i++ ) printf( "%02x", *p++);	/* Print SHA-256 digest */
#endif
#endif
#ifdef USE_RMD
	if(rmd)for(i = 0; i < 512;i++ ) printf( "%02x", *p++);	/* Print RMD-160 digest */
#endif
#ifdef USE_CRC
	if(crc16)printf( "%04x", crc16sum);			/* Print the CRC-16 checksum */
	if(crc32)printf("%08lX", crc32sum);			/* Print the CRC-32 checksum */
#endif
	if(md) for (i = 0; i < 16; ++i) printf( "%02x", *p++);	/* Print MD5 digest */
}

int checkarg(const char *opt, int rc)
{
	short i, len;						/* Define variables */
	
	len=strlen(opt);					/* Find out the length of opt */
	if(len==0)						/* opt is not an arg, exit*/
	{
		printf("\n%s %s", kittengets(1,0,"Invalid option in"), opt);
		kittenclose();					/* Close catalog before exit */
		return 1;
	}
	for(i=0;i<=len;i++)					/* Parse options */
	{
		switch(toupper(opt[i]))				/* Make opt upper case */
		{
			case 'H':				/* The recognized help opts */
			case '?':
				rc = usage();			/* Call the help function */
				return rc;
			case 'C':				/* Check digests, not generate */
				check = 1;
				break;
			case 'V':				/* Verbose mode */
				verbose = 1;
				break;
			case 'S':
			case 'R':
				recursive = 1;			/* Recursive mode */
				i++;
				if(isdigit(opt[i])) maxlevel = opt[i];
				break;
			case 'T':				/* Text mode instead of binary */
				bin_mode = 0;
				break;
			case 'B':				/* Use bin mode (default) */
				bin_mode = 1;
				break;
			case '-':
				i++;
				switch(toupper(opt[i]))		/* Check for option removing */
				{
#ifdef USE_LFN
					case 'L':		/* Turn off LFN mode */
						showlfn = lfn_mode = 0;
						break;
#endif
					case 'S':
					case 'R':		/* Turn off recursive mode */
						recursive = 0;
						break;
					case 'V':		/* Turn off verbose mode */
						verbose = 0;
						break;
				}
				break;
			case 'L':				/* Use LFNs if available */
				if (supportsLFN('A' + _getdrive() - 1) == 1) showlfn = lfn_mode = 1;
				else fprintf(stderr, "\n%s", kittengets(1,12,"Warning: LFNs not supported"));
				break;				/* Check availability */
#if defined(USE_SHA) || defined(USE_CRC)
			case 'M':				/* Mode of digest */
				i++;
				if (opt[i] == ':' || opt[i] == '=') i++;
				if (opt[i] == '\0')		/* Increment i if '=' or ':' */
				{
					fprintf(stderr, "\n%s: %s", kittengets(1,13,"Option missing argument"), opt);
					return 1;		/* No mode argument */
				}
#ifdef USE_RMD
				if (strnicmp(&opt[i], "rmd", 3)== 0)
				{				/* Check for RMD-160 mode */
					rmd = 1;		/* Set RMD-160 mode */
#ifdef USE_SHA
					sha = sha256 = 0;	/* Unset SHA modes */
#endif
					md = 0;			/* Unset MD5 mode */
#ifdef USE_CRC
					crc16 = crc32 = 0;	/* Unset CRC modes */
#endif
					i += 2;			/* Increment to next option */
				}
#endif
#ifdef USE_SHA
#ifdef USE_SHA256
				if (strnicmp(&opt[i], "sha2", 4)== 0)
				{				/* Check for SHA-256 mode */
					sha256 = 1;		/* Set SHA-256 mode */
					md = sha = 0;		/* Unset MD5 and SHA modes */
#ifdef USE_CRC
					crc16 = crc32 = 0;	/* Unset CRC modes */
#endif
					i += 3;			/* Increment to next option */
				}
#endif
				else if (strnicmp(&opt[i], "sha", 3) == 0)
				{				/* Check for SHA mode */
					sha = 1;		/* Set SHA mode */
					md = sha256 = 0;	/* Unset MD5 and SHA-256 modes */
#ifdef USE_CRC
					crc16 = crc32 = 0;	/* Unset CRC modes */
#endif
					i = i + 2;		/* Increment i to next option */
				}
#endif
#ifdef USE_CRC
				else if (strnicmp(&opt[i], "crc16", 5) == 0)
				{
#ifdef USE_SHA
					sha = sha256 = 0;	/* Unset SHA modes */
#endif
					md = crc32 = 0;		/* Unset MD5 and CRC-32 modes */
					crc16 = 1;		/* Set CRC-16 mode */
					i = i + 4;		/* Increment i to next option */
				}
				else if (strnicmp(&opt[i], "crc32", 5) == 0)
				{
#ifdef USE_SHA
					sha = sha256 = 0;	/* Unet SHA mode */
#endif
					md = crc16 = 0;		/* Unset MD5 and CRC-16 modes */
					crc32 = 1;		/* Set CRC-32 mode */
					i = i + 4;		/* Increment i to next option */
				}
#endif
				else if (strnicmp(&opt[i], "md5", 3) == 0)
				{				/* Check for MD5 mode */
					md = 1;			/* Set MD5 mode */
#ifdef USE_SHA
					sha = sha256 = 0;	/* Unset MD5 mode */
#endif
#ifdef USE_CRC
					crc16 = crc32 = 0;	/* Unset CRC modes */
					i = i + 2;		/* Increment i to next option */
#endif
				}
				else
				{				/* Unknown checksum mode */
					fprintf(stderr, "\n%s", kittengets(1,14,"Invalid checksum mode"));
					return 1;		/* Return an error */
				}
				break;
#endif
		}
	}
	return 0;						/* Return success */
}

int main(int argc, char **argv)
{
	short i = 1, rc = 0;					/* Define variables */
	unsigned a;
	FILE *fp;
	char *cwd = NULL;
#ifdef USE_LFN
	char tmp[MAXPATH95];
#else
	char tmp[PATH_MAX];
#endif
	kittenopen("md5sum");					/* Open the message catalog */
#ifdef USE_LFN
	if(supportsLFN('A' + _getdrive() - 1) == 1){showlfn = 1;lfn_mode = 1;}
#endif

	if(getenv("MD5") != NULL)				/* Check for the "MD5" Env */
	{							/* Variable and parse options */
		rc=checkarg(getenv("MD5"), rc);
		if(rc>0) return rc;
	}
	while(i<argc && (argv[i][0]=='/'||argv[i][0]=='-')||argv[i][0]==get_switch_character())
	{							/* Scan for options */
		rc = checkarg(&argv[i][1], rc);			/* Run the scanning function */
		if (rc > 0) return rc;				/* If help functions were used */
		if(check)					/* Check option found */
		{
			i++;					/* The file is the next arg */
			if (strcmp(argv[i], NULL) == 0) fp = stdin;
			else					/* If no arg found, use stdin */
			{					/* Otherwise, use the file */
				if ((fp = fopen(argv[i], FOPRTXT)) == NULL)
				{
					printf("%s: %s\n", kittengets(1,1,"File does not exist"), (argv[i]));
					kittenclose();		/* Close catalog before exit */
					return 2;
				}
			}
			return (do_check(fp));			/* Check the files and exit */
		}
		i++;
	}
	if (argc < i+1)						/* Use stdin if no next arg */
	{
		fputc('\n', stderr);
		if (ckfile(stdin, digest))			/* Generate digest from stdin */
		{
			printf("\nmd5sum: %s", kittengets(1,2,"read error on stdin"));
			kittenclose();				/* Close catalog before exit */
			return 2;
		}
		print_the_digest(digest);				/* Print the digest */
		fputc('\n', stderr);					/* Print newline */
		kittenclose();					/* Close catalog before exit */
		return 0;
	}
	fputc('\n', stderr);					/* Print a newline character */
#ifdef __TURBOC__
	cwd = (char *)malloc(PATH_MAX);
#endif
	getcwd(cwd,PATH_MAX);					/* Get the current directory */
	while(i<argc && argv[i][0]=='@')			/* If using a file list */
	{
		if ((fp = fopen(&argv[i][1],"rt"))!=NULL)	/* Open the list in text mode */
		{
			while (fgets(listname,270,fp)!=NULL)	/* Read a line */
			{					/* Replace '\n' with NULL */
				listname[strlen(listname)-1]='\0';
				strcpy(tmp,unix2dos(listname));
				if(strchr(tmp, '\\') != NULL)
				{				/* Change to a certain dir */
					dirs = (char *)malloc(strlen(tmp) + 1);
					strcpy(dirs, tmp);	/* Copy the full arg to dirs */
					*&strrchr(dirs, '\\')[1] = '\0';
					if((strlen(dirs)!=1)	/* If arg isn't a dir or root */
						&& (strlen(dirs)>3&&dirs[1]!=':'&&dirs[2]!='\\'))
						dirs[strlen(dirs)-1] = '\0';
					if(dirs[1] == ':')	/* Change to a specified drive */
						_dos_setdrive((toupper(dirs[0]) - 'A') + 1, &a);
#ifdef USE_LFN
					if(lfn_mode && chdir95(dirs) != 0) printf("%s: %s\n", kittengets(1,15,"Directory does not exist"), dirs);
					else
#endif			
						if(chdir(dirs)!=0){
							printf("%s: %s\n", kittengets(1,15,"Directory does not exist"), dirs);
							return 2;}
					strcpy(tmp, &strrchr(tmp, '\\')[1]);
					free(dirs);		/* Cut arg to last filename */
				}
				rc = mksum(tmp, rc, listname);	/* Make the checksum */
				_dos_setdrive((toupper(cwd[0]) - 'A') + 1, &a);
				chdir(cwd);			/* Change to the previous cwd */
			}
			fclose(fp);				/* If error, close file */
		}
		else printf("\nmd5sum: %s [@]%s", kittengets(1,4,"error reading"), &argv[i][1]);
		i++;						/* Invalid file list */
	}
	for (; argc > i; i++)					/* Arguments found, find files */
	{
		strcpy(tmp, unix2dos(argv[i]));			/* If a unix-style path given */
		if(strchr(tmp, '\\') != NULL)
		{						/* Change to a certain dir */
			dirs = (char *)malloc(strlen(tmp) + 1);
			strcpy(dirs, tmp);			/* Copy the full arg to dirs */
			*&strrchr(dirs, '\\')[1] = '\0';	/* Cut to directories only */
			if((strlen(dirs)!=1)			/* If arg isn't a dir or root */
					&& (strlen(dirs)>3&&dirs[1]!=':'&&dirs[2]!='\\'))
				dirs[strlen(dirs)-1] = '\0';	/* Remove trailing '\\' */
			if(dirs[1] == ':')			/* Change to a specified drive */
				_dos_setdrive((toupper(dirs[0]) - 'A') + 1, &a);
#ifdef USE_LFN
			if(lfn_mode && chdir95(dirs) != 0) {printf("%s: %s\n", kittengets(1,15,"Directory does not exist"), dirs);return 2;}
			else if(!lfn_mode && chdir(dirs) != 0) {
#else
				if(chdir(dirs)!=0) {		/* Change to the directory */
#endif
					printf("%s: %s\n", kittengets(1,15,"Directory does not exist"), dirs);
					return 2;}
			strcpy(tmp, &strrchr(tmp, '\\')[1]);	/* Cut arg to last filename */
			free(dirs);
		}
		rc = mksum(tmp, rc, argv[i]);			/* Make the checksums */
		_dos_setdrive((toupper(cwd[0]) - 'A') /*+ 1*/, &a);	/* Change to previous drive */
		chdir(cwd);					/* Change to the previous cwd */
	}
	free(cwd);
	kittenclose();						/* Close the message catalog */
	return rc;						/* Use return function for int */
}

int ckfile(FILE *fp, unsigned char *digest)
{
#ifdef USE_CRC
	unsigned char buf[8192];
#elif defined(USE_RMD)
	unsigned char buf[4096];
#else
	unsigned char buf[1024];				/* Define variables */
#endif
	int n;
#ifdef USE_SHA
	if(sha)							/* If using SHA-1 digests */
	{
		sha1_context ctx;
		sha1_starts( &ctx );				/* Start digesting */
		while( ( n = fread( buf, 1, sizeof( buf ), fp ) ) > 0 )
			sha1_update( &ctx, buf, n );		/* Produce digest until eof */
		sha1_finish(&ctx, digest);			/* Finish the digest */
	}
#ifdef USE_SHA256
	if(sha256)
	{
		sha256_context ctx;
		sha256_starts( &ctx );				/* Start digesting */
		while( ( n = fread( buf, 1, 4096, fp ) ) == 4096 )
			sha256_update( &ctx, buf, n );		/* Produce digest until eof */
		sha256_finish( &ctx, digest );		/* Finish the digest */
	}
#endif
#endif
#ifdef USE_RMD
	if(rmd)
	{
		unsigned int rbuf[4096];
		unsigned long long buffer[4096 / 32], chunk[16], length[2], tmp;
		int i;
		MDinit(buffer);
		while( ( n = fread( rbuf, 1, sizeof( rbuf ), fp ) ) > 0 )
		{
			for(i = 0; i < 4096; i+=64)
			{
				pack_chunk(chunk, rbuf + i);
				MDcompress(buffer, chunk);
			}
			if ((tmp = length[0] + n) < length[0]) ++length[1];
			length[0] = tmp;
		}
		for(i = 0; i < n - 63; i +=64)
		{
			pack_chunk(chunk, rbuf + i);
			MDcompress(buffer, chunk);
		}
		if ((tmp = length[0] + n) < length[0]) ++length[1];
		length[0] = tmp;
		MDfinish(buffer, rbuf + i, length[0] << 3, length[0] >> 29 | length[1] << 3);
		for(i = 0; i < 4096 / 8; i +=4)
		{
			long long wd = buffer[i >> 2];
			rmdhash[i] = (unsigned int)wd;
			rmdhash[i+1] = (unsigned int)wd>>8;
			rmdhash[i+2] = (unsigned int)wd>>16;
			rmdhash[i+3] = (unsigned int)wd>>24;
		}
		snprintf(digest, 513, "%02x", rmdhash);
	}
#endif
#ifdef USE_CRC
	if(crc16)						/* If using CRC-16 cksums */
	{
		while( ( n = fread(buf, 1, 4096, fp ) ) > 0 )	/* Produce digest until eof */
			crc16sum = updcrc( crc16sum, buf, n );
	}
	if(crc32)
	{
		crc32sum = 0L;					/* Reset the checksum */
		while ( ( n = fread(buf, 1, 8192, fp) ) > 0 )	/* Produce checksum until eof */
			crc32sum = update_crc32((const char *)buf, n, crc32sum);
	}
#endif
	if(md)							/* If using MD5 digests */
	{
		MD5_CTX ctx;
		MD5Init(&ctx);					/* Start digesting */
		while ((n = fread(buf, 1, sizeof(buf), fp)) > 0)
			MD5Update(&ctx, buf, n);		/* Produce digest until eof */
		MD5Final(digest, &ctx);				/* Finish the digest */
	}
	if (ferror(fp))	return -1;				/* Check for file error */
	return 0;						/* Return to previous function */
}

int hex_digit(int c)						/* Check for hex digits */
{
	if (c >= '0' && c <= '9') return c - '0';		/* Check for hex numbers */
	if (c >= 'a' && c <= 'f') return c - 'a' + 10;		/* Check for hex characters */
	return -1;						/* No hex characters found */
}

int get_md5_line(FILE *fp, unsigned char *digest, char *file)
{
	char buf[1024];						/* Define variables*/
	int i, d1, d2, rc;
	char *p = buf;

	if (fgets(buf, sizeof(buf), fp) == NULL) return -1;	/* Read file; check for error */

#ifdef USE_SHA
	if (sha)for (i = 0; i < 20; ++i)			/* The lenth of an SHA digest */
	{							/* Get the hash to check */
		if ((d1 = hex_digit(*p++)) == -1) return 0;	/* Check for hex digits */
		if ((d2 = hex_digit(*p++)) == -1) return 0;
		*digest++ = d1*16 + d2;				/* Finish reading the digest */
	}
#ifdef USE_SHA256
	if (sha256)for(i=0; i < 32; ++i)			/* The lenth of SHA-256 digest */
	{							/* Get the hash to check */
		if ((d1 = hex_digit(*p++)) == -1) return 0;	/* Check for hex digits */
		if ((d2 = hex_digit(*p++)) == -1) return 0;
		*digest++ = d1*16 + d2;				/* Finish reading the digest */
	}
#endif
#endif
#ifdef USE_CRC
	if (crc32)for (i = 0; i < 8; ++i)			/* Lenth of a CRC32 checksum */
	{							/* Get the hash to check */
		*digest++ = *p++;				/* Finish reading the digest */
	}
#endif
	if (md) for (i = 0; i < 16; ++i)			/* The lenth of an md5 digest */
	{							/* Get the md5sum to check */
		if ((d1 = hex_digit(*p++)) == -1) return 0;	/* Check for hex digits */
		if ((d2 = hex_digit(*p++)) == -1) return 0;
		*digest++ = d1*16 + d2;				/* Finish reading the digest */
	}
	if (*p++ != ' ') return 0;				/* Look for whitespace */
								/*
								 * next char is an attribute
								 * char, * space means text file
								 * if it's a * '*' the file
								 * should be checked in * binary
								 * mode.
								 */
	if (*p == ' ') rc = 1;					/* Look for text mode */
	else if (*p == '*') rc = 2;				/* Look for bin mode */
	else
	{							/* Not an md5-recognized line */
		printf("\nmd5sum: %s: %s", kittengets(1,4,"unrecognized line"), buf);
		return 0;					/* Return */
	}
	++p;
	i = strlen(p);						/* i is the lenth of p */
	if (i < 2 || i > 255) return 0;				/* p must be a legitimate name */
	p[i-1] = '\0';						/* Terminate with a NULL char */
	strcpy(file, p);					/* Return the file name */
	return rc;						/* Return exit code */
}

int do_check(FILE *chkf)					/* Check digests */
{
	int rc, ex = 0, failed = 0, checked = 0;		/* Define variables */
	unsigned char chk_digest[32], file_digest[32];
#ifdef USE_LFN
	char filename[270], lfn[270];
#else
	char filename[256];
#endif
	FILE *fp;
	int flen = 14;
	short i;
								/* Read file with digests */
	while ((rc = get_md5_line(chkf, chk_digest, filename)) >= 0)
	{
		if (rc == 0) continue;				/* not an md5 line */
		if (verbose)
		{						/* Print verbose message */
			if (strlen(filename) > flen) flen = strlen(filename);
			printf("%-*s ", flen, filename);
		}
#ifdef USE_LFN
		if(lfn_mode)					/* If using LFNs */
		{
			strcpy(lfn, filename);			/* Copy variables for lfn2sfn */
			lfn2sfn95(lfn, filename);		/* Convert a lfn to a sfn */
		}
#endif
		if (rc == 1 || bin_mode == 0)
			fp = fopen(filename, FOPRTXT);		/* If text mode is defined */
		else fp = fopen(filename, FOPRBIN);		/* If bin mode is defined */
		if (fp == NULL)
		{						/* File to check doesn't exist */
			printf("\nmd5sum: %s %s", kittengets(1,5,"can't open"), filename);
			ex = 2;					/* Exit code to return */
			continue;
		}
		if (ckfile(fp, file_digest))			/* Get the file's digest */
		{						/* and if there is an error */
			printf("\nmd5sum: %s %s", kittengets(1,4,"error reading"), filename);
			ex = 2;					/* Set exit code */
			fclose(fp);				/* Close the file */
			continue;
		}
		fclose(fp);					/* Close file after digesting */
#ifdef USE_SHA
		if(sha) i = memcmp(chk_digest, file_digest, 20);/* Check if digest matches */
#ifdef USE_SHA256
		if(sha256)i=memcmp(chk_digest, file_digest, 32);/* Check if digest matches */
#endif
#endif
#ifdef USE_CRC
		if(crc32)
		{						/* Put crc32sum into  a string */
/*			snprintf((char *)file_digest, 9, "%08lX", crc32sum);*/
			sprintf((char *)file_digest, "%08lX", crc32sum);
			printf("%s %081X\n", file_digest, crc32sum);
			i = memcmp(chk_digest, file_digest, 8);	/* Compare the file digests */
		}
#endif
		if (md) i = memcmp(chk_digest, file_digest, 16);/* Check if digest matches */
		if (i != 0)
		{						/* Digest does not match */
			if (verbose) printf("%s\n", kittengets(1,6,"FAILED"));
			else printf("\nmd5sum: %s '%s'",
					kittengets(1,7,"check failed for"), filename);
			++failed;				/* Increment failed */
		}						/* Print verbose message */
		else if(verbose)printf("%s\n",kittengets(1,15,"OK"));
		++checked;					/* Increment checked */
	}
	if (verbose && failed)					/* Print verbose if failed */
	printf("\nmd5sum: %d %s %d %s", failed, kittengets(1,8, "of"), checked, kittengets(1,9,"file(s) failed check"));
	if (!checked)
	{							/* If no files got checked */
		printf("\nmd5sum: %s", kittengets(1,10,"no files checked"));
		kittenclose();					/* Close message catalog */
		return 3;					/* Exit the function */
	}
	if (!ex && failed) ex = 1;				/* Change exit code if failed */
	kittenclose();						/* Close message catalog */
	return ex;						/* Exit the function with 'ex' */
}
